/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cab.passport.passportorder.service;

import com.cab.passport.passportorder.dao.PassportOrderDao;
import com.cab.passport.passportorder.entity.PassportOrder;
import com.cab.passport.passportorder.entity.PassportOrderBusinessStateEnum;
import com.google.common.collect.Maps;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.Collections3;
import com.thinkgem.jeesite.common.utils.excel.ExportExcel;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 护照订单Service
 *
 * @author lgl
 * @version 2018-10-09
 */
@Service
@Transactional(readOnly = true)
public class PassportOrderPingYangService extends CrudService<PassportOrderDao, PassportOrder> {

    @Autowired
    PassportOrderDao passportOrderDao;
    @Transactional(readOnly = false)
    public void edit(PassportOrder passportOrder) {
        PassportOrder po = get(passportOrder.getId());
        if (PassportOrderBusinessStateEnum.WAIT_FOR_INPUT_IDENTIFICATIONFLAG.getValue().equals(po.getBusinessState())
                || PassportOrderBusinessStateEnum.WAIT_FOR_PRINT.getValue().equals(po.getBusinessState())) {
            //仅在业务状态为待录入芯片或待打印贴纸时编辑有效
            po.setIdentificationFlag(passportOrder.getIdentificationFlag());
            String lastEight = getLastEight(passportOrder.getIdentificationFlag());
            po.setCabPassport(getCabPassport(lastEight));
            po.setCabRegisterCode(getCabRegisterCode(passportOrder, lastEight));
        }
        if (PassportOrderBusinessStateEnum.WAIT_FOR_MAIL.getValue().equals(po.getBusinessState())) {
            //仅在业务状态为待寄出时编辑有效
            po.setExpressNumber(passportOrder.getExpressNumber());
        }
        if (PassportOrderBusinessStateEnum.WAIT_FOR_INPUT_IDENTIFICATIONFLAG.getValue().equals(po.getBusinessState()) && StringUtils.isNotBlank(passportOrder.getIdentificationFlag())) {
            po.setBusinessState(PassportOrderBusinessStateEnum.WAIT_FOR_PRINT.getValue());
        } else if (PassportOrderBusinessStateEnum.WAIT_FOR_MAIL.getValue().equals(po.getBusinessState()) && StringUtils.isNotBlank(passportOrder.getExpressNumber())) {
            po.setBusinessState(PassportOrderBusinessStateEnum.FINISH.getValue());
        }
        super.save(po);
    }

    @Transactional(readOnly = false)
    public void print(PassportOrder passportOrder) {
        PassportOrder po = get(passportOrder.getId());
        if (StringUtils.isBlank(passportOrder.getIdentificationFlag())) {
            throw new RuntimeException("芯片号为空，无法打印！");
        }
        if (StringUtils.isBlank(passportOrder.getCabPassport())) {
            throw new RuntimeException("护照号为空，无法打印！");
        }
        if (!passportOrder.getBusinessState().equals(PassportOrderBusinessStateEnum.WAIT_FOR_PRINT.getValue())) {
            throw new RuntimeException("该记录业务状态不合适，无法打印！");
        }
        if (PassportOrderBusinessStateEnum.WAIT_FOR_PRINT.getValue().equals(po.getBusinessState())) {
            po.setBusinessState(PassportOrderBusinessStateEnum.WAIT_FOR_MAIL.getValue());
        }
        super.save(po);
    }

    @Transactional(readOnly = false)
    public void resetPrint(PassportOrder passportOrder) {
        PassportOrder po = get(passportOrder.getId());
        if (PassportOrderBusinessStateEnum.WAIT_FOR_MAIL.getValue().equals(po.getBusinessState())) {
            po.setBusinessState(PassportOrderBusinessStateEnum.WAIT_FOR_PRINT.getValue());
        }
        super.save(po);
    }

    private String getCabPassport(String lastEight) {
        return "N" + lastEight;
    }

    private String getCabRegisterCode(PassportOrder passportOrder, String lastEight) {
        String prefix = "";
        if ("1".equals(passportOrder.getPetType())) {
            prefix = "W";
        } else if ("2".equals(passportOrder.getPetType())) {
            prefix = "S";
        }
        return prefix + lastEight;
    }

    private String getLastEight(String identificationFlag) {
        String lastSeven = getLastSeven(identificationFlag);
        String maxLastSeven = dao.getMaxCabPassportByLastSeven(lastSeven);
        int index = 0;
        if (StringUtils.isNotBlank(maxLastSeven)) {
            index = Integer.parseInt(maxLastSeven.substring(1, 2)) + 1;
        }
        return index + lastSeven;
    }

    private String getLastSeven(String identificationFlag) {
        if (identificationFlag.length() != 15 || !StringUtils.isNumeric(identificationFlag)) {
            throw new RuntimeException("芯片号长度必须为15位数字！");
        }
        String lastSix = identificationFlag.substring(9);
        char[] lastSixArray = lastSix.toCharArray();
        int[] lastSixIntArray = new int[6];
        for (int i = 0; i < 6; i++) {
            lastSixIntArray[i] = (int) lastSixArray[i] - (int) '0';
        }
        int check = (lastSixIntArray[0] * 1 + lastSixIntArray[1] * 3 + lastSixIntArray[2] * 5 + lastSixIntArray[3] * 2 + lastSixIntArray[4] * 4 + lastSixIntArray[5] * 6) % 9;
        return check + lastSix;
    }

    public void exportExcel(String ids,PassportOrder passportOrder,HttpServletResponse response) throws Exception {
        List<PassportOrder> exportList = new ArrayList<PassportOrder>();
        if(StringUtils.isNotBlank(ids)){
            String[] idList = ids.split(",");
            Map map = Maps.newHashMap();
            map.put("idList", idList);
            exportList = passportOrderDao.getByIdList(map);
        }else{
            throw new RuntimeException("请选择需要导出的内容！");
        }
        if(!Collections3.isEmpty(exportList)){
            for (PassportOrder po :exportList){
                po.setPackageType(DictUtils.getDictLabel(po.getPackageType(),"passport_order_package_type",""));
                if(StringUtils.isBlank(po.getExpressWeight())){
                    po.setExpressWeight("1Kg");
                }
            }
        }
        new ExportExcel("", PassportOrder.class, 2).setDataList(exportList).write(response, "地址信息表.xlsx").dispose();

    }
}