/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cab.passport.petstore.service;

import com.cab.passport.petstore.dao.PetStoreDao;
import com.cab.passport.petstore.entity.PetStore;
import com.cab.passport.referrer.dao.ReferrerDao;
import com.cab.passport.referrer.entity.Referrer;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.RESTResponse;
import com.thinkgem.jeesite.common.utils.ZAException;
import com.thinkgem.jeesite.common.utils.excel.ExportExcel;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import net.sf.json.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;

/**
 * 宠物店Service
 *
 * @author lgl
 * @version 2018-12-18
 */
@Service
@Transactional(readOnly = true)
public class PetStoreService extends CrudService<PetStoreDao, PetStore> {

    public static final String WX_OPEN_ID = "wxOpenId";
    public static final String PET_STORE = "petStore";
    public static final String CHANNEL_USER_ID = "channelUserId";
    public static final String INVITE_CODE = "inviteCode";
    public static final String REFERER_CODE = "refererCode";
    public static final String PET_STORE_OWNER_NAME = "petStoreOwnerName";
    public static final String PET_STORE_OWNER_MOBILE = "petStoreOwnerMobile";
    public static final String PET_STORE_NAME = "petStoreName";
    public static final String PET_STORE_CITY = "petStoreCity";
    public static final String PET_STORE_ADDRESS = "petStoreAddress";
    public static final String PET_STORE_MOBILE = "petStoreMobile";
    public static final String PET_STORE_AREA = "petStoreArea";
    public static final String MAIN_PROJECT = "mainProject";
    public static final String CKU_GROOMER_TRAINER_FLAG = "ckuGroomerTrainerFlag";
    public static final String SELL_PET_FLAG = "sellPetFlag";
    public static final String SELL_DOG_TYPE = "sellDogType";
    public static final String SELL_CAT_TYPE = "sellCatType";
    public static final String YEARLY_SALES_VOLUME = "yearlySalesVolume";
    public static final String BUSINESS_LICENSE_NUMBER = "businessLicenseNumber";
    public static final String PROVINCE_IDS = "provinceIds";

    @Autowired
    ExecutorService petStoreVisitLogQueueThreadPool;
    @Autowired
    private ReferrerDao referrerDao;

    public PetStore get(String id) {
        return super.get(id);
    }

    public List<PetStore> findList(PetStore petStore) {
        return super.findList(petStore);
    }

    public Page<PetStore> findPage(Page<PetStore> page, PetStore petStore) {
        return super.findPage(page, petStore);
    }

    @Transactional(readOnly = false)
    public void save(PetStore petStore) {
        super.save(petStore);
    }

    @Transactional(readOnly = false)
    public void edit(PetStore petStore) {
        PetStore po = dao.get(petStore.getId());
        po.setPetStoreAddress(petStore.getPetStoreAddress());
        po.setPetStoreMobile(petStore.getPetStoreMobile());
        super.save(po);
    }

    @Transactional(readOnly = false)
    public void delete(PetStore petStore) {
        super.delete(petStore);
    }

    public RESTResponse checkInviteCode(String inviteCode) {
        //校验邀请码
        checkInviteCode(dao.getByInviteCode(inviteCode));
        return new RESTResponse();
    }

    @Transactional(readOnly = false)
    public RESTResponse activatePetStore(String json, HttpServletRequest request) {
        String wxOpenId = getWxOpenId(request);
        // 获取参数
        JSONObject jo = JSONObject.fromObject(json);
        //参数校验
        validateParameter(jo, false);
        String inviteCode = jo.getString(INVITE_CODE);
        PetStore po = dao.getByInviteCode(inviteCode);
        //校验邀请码
        checkInviteCode(po);
        //如果有推荐码，则效验推荐码
        checkRefererCode(jo.get(REFERER_CODE)==null?"":jo.getString(REFERER_CODE));
        //激活
        activatePetStore(po, wxOpenId, jo);
        //缓存店铺实体
        request.getSession().setAttribute(PetStoreService.PET_STORE, po);
        return new RESTResponse("petStoreId", po.getId());
    }

    @Transactional(readOnly = false)
    public RESTResponse activatePetStoreByChannel(String json, HttpServletRequest request) {
        // 获取参数
        JSONObject jo = JSONObject.fromObject(json);
        //参数校验
        validateParameter(jo, true);
        //保存宠物店
        PetStore po = savePetStore(jo);
        //缓存店铺实体
        request.getSession().setAttribute(PetStoreService.PET_STORE, po);
        return new RESTResponse("petStoreId", po.getId());
    }
    //效验推荐码
    public void checkRefererCode(String refererCode){
        if(StringUtils.isNotBlank(refererCode)){
            Referrer referrer=referrerDao.getByCode(refererCode);
            if(null==referrer||referrer.getStatus().equals("1")){
                throw new ZAException("邀请码输入错误或该邀请码无效请核对");
            }
        }
    }

    //校验邀请码
    private void checkInviteCode(PetStore po) {
        if (po == null) {
            throw new ZAException("邀请码有误");
        }
        if (po.getActivationTime() != null) {
            throw new ZAException("邀请码已使用");
        }
    }

    private String getWxOpenId(HttpServletRequest request) {
        String wxOpenId = (String) request.getSession().getAttribute(WX_OPEN_ID);
        if (StringUtils.isBlank(wxOpenId)) {
            throw new ZAException("请从新登录");
        }
        return wxOpenId;
    }

    private void validateParameter(JSONObject jo, boolean channelFlag) {
        if (channelFlag) {
            if (!jo.containsKey(CHANNEL_USER_ID) || StringUtils.isBlank(jo.getString(CHANNEL_USER_ID))) {
                throw new ZAException("渠道用户id为空");
            }
        } else {
            if (!jo.containsKey(INVITE_CODE) || StringUtils.isBlank(jo.getString(INVITE_CODE))) {
                throw new ZAException("邀请码为空");
            }
        }
        if (!jo.containsKey(PET_STORE_OWNER_NAME) || StringUtils.isBlank(jo.getString(PET_STORE_OWNER_NAME))) {
            throw new ZAException("姓名为空");
        }
        if (!jo.containsKey(PET_STORE_OWNER_MOBILE) || StringUtils.isBlank(jo.getString(PET_STORE_OWNER_MOBILE))) {
            throw new ZAException("手机号为空");
        }
        if (!jo.containsKey(PET_STORE_NAME) || StringUtils.isBlank(jo.getString(PET_STORE_NAME))) {
            throw new ZAException("宠物店名称为空");
        }
        if (!jo.containsKey(PET_STORE_CITY) || StringUtils.isBlank(jo.getString(PET_STORE_CITY))) {
            throw new ZAException("宠物店所在城市为空");
        }
        if (!jo.containsKey(PET_STORE_ADDRESS) || StringUtils.isBlank(jo.getString(PET_STORE_ADDRESS))) {
            throw new ZAException("宠物店地址为空");
        }
        if (!jo.containsKey(PET_STORE_MOBILE) || StringUtils.isBlank(jo.getString(PET_STORE_MOBILE))) {
            throw new ZAException("宠物店联系方式为空");
        }
        if (!jo.containsKey(PET_STORE_AREA) || StringUtils.isBlank(jo.getString(PET_STORE_AREA))) {
            throw new ZAException("宠物店面积为空");
        }
        if (!jo.containsKey(MAIN_PROJECT) || StringUtils.isBlank(jo.getString(MAIN_PROJECT))) {
            throw new ZAException("主营项目为空");
        }
        if (!jo.containsKey(CKU_GROOMER_TRAINER_FLAG) || StringUtils.isBlank(jo.getString(CKU_GROOMER_TRAINER_FLAG))) {
            throw new ZAException("是否有CKU认证为空");
        }
        if (!jo.containsKey(SELL_PET_FLAG) || StringUtils.isBlank(jo.getString(SELL_PET_FLAG))) {
            throw new ZAException("是否销售活体为空");
        }
        if ("1".equals(jo.getString(SELL_PET_FLAG))) {
            if (!jo.containsKey(SELL_DOG_TYPE)) {
                throw new ZAException("销售犬品种为空");
            }
            if (!jo.containsKey(SELL_CAT_TYPE)) {
                throw new ZAException("销售猫品种为空");
            }
            if (!jo.containsKey(YEARLY_SALES_VOLUME) || StringUtils.isBlank(jo.getString(YEARLY_SALES_VOLUME))) {
                throw new ZAException("年销量为空");
            }
        }
        if (!jo.containsKey(BUSINESS_LICENSE_NUMBER)) {
            throw new ZAException("缺少营业执照号");
        }
        if (!jo.containsKey(PROVINCE_IDS) || StringUtils.isBlank(jo.getString(PROVINCE_IDS))) {
            throw new ZAException("缺少城市所属省信息");
        }
    }

    //激活
    private void activatePetStore(PetStore po, String wxOpenId, JSONObject jo) {
        po.setWxOpenId(wxOpenId);
        savePetStore(jo, po);
    }

    public static final Map<String, String> PROVINCE_CITY_MAP = new ImmutableMap.Builder<String, String>()
            .put("3", "北京市")
            .put("4", "上海市")
            .put("5", "天津市")
            .put("6", "重庆市")
            .build();

    //根据城市名获取省份id
    private String getProvinceIdsByCity(String provinceIds) {
        JSONObject jo = new JSONObject();
        String parentIdArray = provinceIds;
        String nameArray = "";
        if (PROVINCE_CITY_MAP.containsKey(provinceIds)) {
            if ("3".equals(provinceIds) || "5".equals(provinceIds)) {
                //河北省id
                parentIdArray = "7";
            }
            if ("4".equals(provinceIds)) {
                //浙江省id
                parentIdArray = "14";
            }
            if ("6".equals(provinceIds)) {
                //四川省id
                parentIdArray = "25";
            }
            nameArray = PROVINCE_CITY_MAP.get(provinceIds);
        }
        jo.put("parentIdArray", parentIdArray);
        jo.put("nameArray", nameArray);
        return jo.toString();
    }

    //主营项目转换
    public void setMainProject(PetStore po) {
        List<String> mainProjectList = Lists.newArrayList();
        for (String mainProject : po.getMainProject().split(",")) {
            mainProjectList.add(DictUtils.getDictLabel(mainProject, "pet_store_main_project", ""));
        }
        po.setMainProject(StringUtils.join(mainProjectList, ","));
    }

    //设置统计信息
    public void setStatisticsInfo(PetStore petStore) {
        if (petStore.getActivationTime() != null && StringUtils.isNotBlank(petStore.getId())) {
            List<Map<String, Object>> mapList = dao.getVisitLogByWxOpenId(petStore.getId());
            long requestUri1Count = 0l;
            long requestUri2Count = 0l;
            long requestUri3Count = 0l;
            long requestUri4Count = 0l;
            for (Map<String, Object> map : mapList) {
                String requestUri = (String) map.get("requestUri");
                long requestUriCount = (Long) map.get("requestUriCount");
                if ("1".equals(requestUri)) {
                    requestUri1Count = requestUriCount;
                } else if ("2".equals(requestUri)) {
                    requestUri2Count = requestUriCount;
                } else if ("3".equals(requestUri)) {
                    requestUri3Count = requestUriCount;
                } else if ("4".equals(requestUri)) {
                    requestUri4Count = requestUriCount;
                }
            }
            petStore.setContactPetOwnerCount(requestUri4Count);
            petStore.setViewPetOwnerCount(requestUri1Count + requestUri2Count);
            petStore.setContactPetStoreCount(requestUri3Count);
        }
    }

    public void exportExcel(PetStore petStore, HttpServletResponse response) throws Exception {
        List<PetStore> exportList = dao.findList(petStore);
        if (exportList.size() == 0) {
            throw new RuntimeException("没有符合条件的数据！");
        }
        exportList.forEach(po -> {
            if ("1".equals(po.getCkuGroomerTrainerFlag())) {
                po.setCkuGroomerTrainerFlag("是");
            } else {
                po.setCkuGroomerTrainerFlag("否");
            }
            setMainProject(po);
            setStatisticsInfo(po);
        });
        new ExportExcel("", PetStore.class, 2).setDataList(exportList).write(response, "宠集盒用户信息.xlsx").dispose();
    }

    @Transactional(readOnly = false)
    private void savePetStore(JSONObject jo, PetStore po) {
        po.setActivationTime(new Date());
        po.setPetStoreOwnerName(jo.getString(PET_STORE_OWNER_NAME));
        po.setPetStoreOwnerMobile(jo.getString(PET_STORE_OWNER_MOBILE));
        po.setPetStoreName(jo.getString(PET_STORE_NAME));
        String petStoreCity = jo.getString(PET_STORE_CITY);
        po.setPetStoreCity(petStoreCity);
        po.setPetStoreAddress(jo.getString(PET_STORE_ADDRESS));
        po.setPetStoreMobile(jo.getString(PET_STORE_MOBILE));
        po.setPetStoreArea(jo.getString(PET_STORE_AREA));
        po.setMainProject(jo.getString(MAIN_PROJECT));
        po.setCkuGroomerTrainerFlag(jo.getString(CKU_GROOMER_TRAINER_FLAG));
        po.setSellPetFlag(jo.getString(SELL_PET_FLAG));
        if ("1".equals(jo.getString(SELL_PET_FLAG))) {
            po.setSellDogType(jo.getString(SELL_DOG_TYPE));
            po.setSellCatType(jo.getString(SELL_CAT_TYPE));
            po.setYearlySalesVolume(jo.getString(YEARLY_SALES_VOLUME));
        }
        po.setBusinessLicenseNumber(jo.getString(BUSINESS_LICENSE_NUMBER));
        po.setProvinceIds(getProvinceIdsByCity(jo.getString(PROVINCE_IDS)));
        po.setRefererCode(jo.get(REFERER_CODE)==null?"":jo.getString(REFERER_CODE));
        save(po);
    }

    //保存宠物店
    @Transactional(readOnly = false)
    public PetStore savePetStore(JSONObject jo) {
        String channelUserId = jo.getString(CHANNEL_USER_ID);
        //渠道用户id查重
        PetStore po = dao.getByChannelUserId(channelUserId);
        if (po != null) {
            throw new ZAException("该用户已注册");
        }
        //先新增空记录
        po = new PetStore();
        Date date = new Date();
        User user = new User();
        user.setId("1");
        po.setCreateBy(user);
        po.setCreateDate(date);
        dao.insert(po);
        po.setChannelUserId(channelUserId);
        //更新该记录
        savePetStore(jo, po);
        return po;
    }

}