/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.show.service;

import java.time.LocalDate;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

import javax.validation.Validator;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.dao.DogBirthCertificateDao;
import com.cku.oa.dog.dao.DogDao;
import com.cku.oa.dog.dao.DogPedigreeCertifiedRedoDao;
import com.cku.oa.dog.entity.Dog;
import com.cku.oa.dog.entity.DogPedigreeCertifiedRedo;
import com.cku.oa.handler.dao.HandlerQualificationCertificateDao;
import com.cku.oa.show.dao.ChampionLoginDao;
import com.cku.oa.show.dao.ChildShowsDao;
import com.cku.oa.show.dao.MainShowsDao;
import com.cku.oa.show.dao.ShowResultsDao;
import com.cku.oa.show.dao.ShowResultsDdzDao;
import com.cku.oa.show.entity.ChampionLogin;
import com.cku.oa.show.entity.ChildShows;
import com.cku.oa.show.entity.MainShows;
import com.cku.oa.show.entity.ShowResults;
import com.thinkgem.jeesite.common.beanvalidator.BeanValidators;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.Collections3;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.common.utils.excel.ImportExcel;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;

/**
 * 犬展成绩Service
 * 
 * @author lgl
 * @version 2016-07-13
 */
@Service
@Transactional(readOnly = true)
public class ShowResultsService extends CrudService<ShowResultsDao, ShowResults> {

	/**
	 * 验证Bean实例对象
	 */
	@Autowired
	protected Validator validator;
	@Autowired
	private ShowApplyService showApplyService;
	@Autowired
	private ChildShowsDao childShowsDao;
	@Autowired
	private ShowResultsDao showResultsDao;
	@Autowired
	private ShowResultsDdzDao showResultsDdzDao;
	@Autowired
	private DogDao dogDao;
	@Autowired
	private DogBirthCertificateDao dogBirthCertificateDao;
	@Autowired
	private MainShowsDao mainShowsDao;
	@Autowired
	private ChampionLoginDao championLoginDao;
	@Autowired
	private HandlerQualificationCertificateDao handlerQualificationCertificateDao;
	@Autowired
	private DogPedigreeCertifiedRedoDao dogPedigreeCertifiedRedoDao;
	@Autowired
	private ShowScoreRankingService showScoreRankingService;
	@Autowired
	private ChildShowsService childShowsService;

	public ShowResults get(String id) {
		return super.get(id);
	}

	public List<ShowResults> findList(ShowResults showResults) {
		return super.findList(showResults);
	}

	public Page<ShowResults> findPage(Page<ShowResults> page, ShowResults showResults) {
		return super.findPage(page, showResults);
	}

	@Transactional(readOnly = false)
	public void save(ShowResults showResults) {
//		if (!StringUtils.isEmpty(showResults.getResultCc())) {
//			String cc = showResults.getResultCc();
//			if (",".equals(showResults.getResultCc().substring(0, 1))) {
//				cc = showResults.getResultCc().substring(1, showResults.getResultCc().length()).replace(",", "/");
//				showResults.setResultCc(cc);
//			} else {
//				cc = showResults.getResultCc().replace(",", "/");
//			}
//			showResults.setResultCc(cc);
//		}
		super.save(showResults);
		// 标记为未统计
		childShowsService.updateUnStatisticsState(showResults.getShowCode());
	}

	@Transactional(readOnly = false)
	public void delete(ShowResults showResults) {
		super.delete(showResults);
	}

	/**
	 *
	 * @description: 导入犬展成绩
	 * @author: laiguanglong
	 * @throws Exception
	 * @date: 2016年8月2日 下午2:26:40
	 */
	@Transactional(readOnly = false, rollbackFor = Exception.class)
	public int importShowResults(MultipartFile file) throws Exception {
		HashSet<String> showCodeSet = new HashSet<String>();
		int count = 0;
		try {
			// 模版中需要开头空一行，从第3行开始读取，否则会少读一行
			ImportExcel ei = new ImportExcel(file, 1, 0);
			List<ShowResults> list = ei.getDataList(ShowResults.class);
			boolean flag = true;
			for (int i = 0; i < list.size(); i++) {
				ShowResults showResults = list.get(i);
				// 跳过无效记录
				if (StringUtils.isBlank(showResults.getShowCode())) {
					continue;
				}
				flag = false;
				// cc
//				if(!StringUtils.isEmpty(showResults.getResultCc())) {
//					showResults.setResultCc(showResults.getResultCc().replace("/", ","));
//				}
				
				showResults.setShowCode(StringUtils.strip(showResults.getShowCode()));
				// 字典类型字段转换
				showResults.setAgeGroup(DictUtils.getDictValue(showResults.getAgeGroup(), "show_age_group", ""));
				showResults.setDogGender(DictUtils.getDictValue(showResults.getDogGender(), "dog_gender", ""));
				// 官网成绩排名所属年份（默认去犬展编号年份，当犬展跨年，可以手动修改年份为下一年）
				/**
				 * @Description: 更新此犬犬展year字段,根据主犬展设置年份year判断
				 * @author: zhangxiang
				 * @date: 2019/10/16 15:16
				 */
				ChildShows childShows = childShowsDao.getByShowCode(showResults.getShowCode());
				if (childShows == null) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,
							"未找到child" + showResults.getShowCode() + "犬展信息！");
				}
				MainShows mainShows = mainShowsDao.get(childShows.getMainShowId());
				if (mainShows == null) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "未找到main" + showResults.getShowCode() + "犬展信息！");
				}
				if (StringUtils.isBlank(mainShows.getYear())) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,
							"请前往该场" + showResults.getShowCode() + "的主犬展 “" + mainShows.getMainShowName() + "”中设置犬展年份");
				}
//				String year = "";
//				if(StringUtils.isNotBlank(showResults.getShowCode())&&showResults.getShowCode().length()>=4){
//					year = showResults.getShowCode().substring(0,4);
//				}
				showResults.setYear(mainShows.getYear());
				// 参数校验
				try {
					BeanValidators.validateWithException(validator, showResults);
				} catch (Exception e) {
					logger.warn("校验参数出错", e);
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "参数有误，请修改后再导入！");
				}
//				//判断是否报名
//				if(!showApplyService.isApply(showResults.getShowCode(), showResults.getPedigreeCertified())){
//					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号为"+showResults.getPedigreeCertified()+",犬展编号为"+showResults.getShowCode()+"的记录没有找到相关的报名记录");
//				}
				// 判断是否已有成绩
				if (checkResultByShowCodeAndPedigreeCertified(showResults.getShowCode(),
						showResults.getPedigreeCertified())) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "已有血统证书号为" + showResults.getPedigreeCertified()
							+ ",犬展编号为" + showResults.getShowCode() + "的成绩记录");
				}
				// 写数据库
				try {
					super.save(showResults);
				} catch (Exception e) {
					logger.error("第" + (i + 1) + "行数据库保存犬展成绩出错：", e);
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "第" + (i + 1) + "行" + "参数有误，无法入库，请修改后再导入！");
				}

				count++;
				// 保存犬展编号
				showCodeSet.add(showResults.getShowCode());
			}
			if (flag) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "未找到有效记录！");
			}
		} catch (ZAException e) {
			throw e;
		} catch (Exception e) {
			logger.warn("犬展成绩导入出错！", e);
			throw new Exception(e);
		}
		// 标记为未统计
		for (String showCode : showCodeSet) {
			childShowsService.updateUnStatisticsState(showCode);
		}
		return count;
	}

	/**
	 * @Description 判断是否已有成绩
	 * @author yuanshuai
	 * @date 2020/11/13 15:58
	 */
	public boolean checkResultByShowCodeAndPedigreeCertified(String showCode, String pedigreeCertified) {
		return dao.countByShowCodeAndPedigreeCertified(showCode, pedigreeCertified) > 0;
	}

	/**
	 * 更新本场成绩
	 * 
	 * @param showCode
	 */
	@Transactional(readOnly = false)
	private void updateShowResult(String showCode) {
		/****** 全犬种成绩处理 *******/
		// 初始化本场成绩
		dao.initShowResult(showCode);
		// 全犬种成绩，下面的成绩会覆盖上面的，累积到全犬种排行榜
		updateResutlBOB(showCode);// 单犬种最佳奖(成犬组)
		updateResultBIG(showCode);// 犬种组最佳奖(成犬组)
		updateResultBIS(showCode);// 全场总冠军(成犬组)
		// 全犬种成绩，单独榜单
		updateResultBBE(showCode);// 繁殖人组成绩
		updateResultOHC(showCode);// 犬主人组成绩
		// 全犬种成绩，累计到单犬种排行榜
		updateBreedResutlBOB(showCode);// bob的单犬种积分
		// 全犬种成绩，累计到单犬种排行榜、全犬种积分榜
		updateResultBISS(showCode);// 单犬种全场总冠军

		// 修正成绩为负数的情况
		dao.amendShowResult(showCode);
		// 修改成绩取消的本场积分为0
		dao.updateCancelResult(showCode);

		/****** 单独展成绩处理 *******/
		// 初始化单独展结果表 show_breed_result字段
		showResultsDdzDao.initShowBreedResult(showCode);
		// 单犬种成绩，累计到单犬种排行榜、全犬种排行榜BISS、BOV二取一，BISS优先级高
		updateBreedResutlBISS(showCode);// 单犬种全场总冠军
		updateBreedResutlDdzBOV(showCode);// 犬种组全场总冠军
		// 单犬种成绩，累计到单犬种排行榜、全犬种排行榜
//		updateBreedResutlDdzBOB(showCode);//单犬种最佳奖
		// 单犬种成绩，只算入全犬种排行榜
//		updateBreedResutlDdzBIG(showCode);//犬种组最佳奖
		// 单犬种成绩，计算幼犬特幼月榜BPBOB、PBOB
		updateBreedResutlDdzBPPBOB(showCode, "BPBOB");
		updateBreedResutlDdzBPPBOB(showCode, "PBOB");

	}

	/**
	 * 计算BOB成绩
	 * 
	 * @param showCode
	 */
	private void updateResutlBOB(String showCode) {
		dao.updateResultBPBOB(showCode);
		dao.updateResultPBOB(showCode);
		dao.updateResultBOB(showCode);
	}

	/**
	 * 计算BIG成绩
	 * 
	 * @param showCode
	 */
	private void updateResultBIG(String showCode) {
		initResultBIG(showCode,"BIG");
		initResultBIG(showCode,"PBIG");
		initResultBIG(showCode,"BPBIG");
//		for (int i = 1; i < 5; i++) {
//			dao.updateResultBPBIG(showCode, i);
//		}
//		for (int i = 1; i < 5; i++) {
//			dao.updateResultPBIG(showCode, i);
//		}
//		for(int i=1;i<5;i++){
//			dao.updateResultBIG(showCode,i);
//		}
	}

	/**
	 * 计算BBE成绩
	 * 
	 * @param showCode
	 */
	@Transactional(readOnly = false)
	private void updateResultBBE(String showCode) {
		List<ShowResults> resultsList = dao.getByShowCode(showCode);
		for (ShowResults data : resultsList) {
			ChildShows childShows = childShowsDao.getByShowCode(showCode);
			Dog dog = dogDao.getByPedigreeCertifiedCode(data.getPedigreeCertified());
			if (dog == null) {
				DogPedigreeCertifiedRedo pedigreeCertifiedRedo = dogPedigreeCertifiedRedoDao
						.getByOriginalPedigreeCertified(data.getPedigreeCertified());
				if (pedigreeCertifiedRedo != null) {
					Dog dog1 = dogDao.getByPedigreeCertifiedCode(pedigreeCertifiedRedo.getCurrentPedigreeCertified());
					if (dog1 != null) {
						dog = dog1;
					}
				}
			}
			if (dog != null) {
				// 军登录申请日期小于等于当前比赛结束日期，则不计算
//				List<String> list = championLoginDao.getLoginTypeByPedigreeCertified(dog.getPedigreeCertifiedCode());
				//【ID1006138】删除繁殖人组积分累计的判断条件
//				ChampionLogin entity = new ChampionLogin();
//				entity.setPedigreeCertified(dog.getPedigreeCertifiedCode());
//				entity.setPaymentState("2");
//				entity.setReviewState("1");
//				entity.setPaymentTimeLimit(childShows.getEndTime());
//				List<ChampionLogin> list = championLoginDao.findList(entity);
				if (getMonthAge(childShows.getStartTime(), dog.getBirthdate()) >= 15 ){//&& Collections3.isEmpty(list)) {
					Integer bbeScore = 0;
					if (("BBE-BIS1").equals(data.getResultBbeBis())) {
						bbeScore += 80;
					}
					if (("BBE-BIS2").equals(data.getResultBbeBis())) {
						bbeScore += 50;
					}
					if (("BBE-BIG1").equals(data.getResultBbeBig())) {
						bbeScore += 30;
					}
					if (("BBE-BIG2").equals(data.getResultBbeBig())) {
						bbeScore += 20;
					}
					if (("BBE-BIG3").equals(data.getResultBbeBig())) {
						bbeScore += 15;
					}
					if (("BBE-BIG4").equals(data.getResultBbeBig())) {
						bbeScore += 50;
					}
					if (("BBE-BISS").equals(data.getResultBbeBiss())) {
						bbeScore += 10;
					}
					if (("BBE-WINNER").equals(data.getResultBbeWinner())) {
						bbeScore += 5;
					}
					if (bbeScore != 0) {
						data.setBbeScore(bbeScore.toString());
						dao.update(data);
					}
				}
			}
		}
	}

	/**
	 * 计算OHC成绩
	 * 
	 * @param showCode
	 */
	@Transactional(readOnly = false)
	public void updateResultOHC(String showCode) {
		List<ShowResults> resultsList = dao.getByShowCode(showCode);

		for (ShowResults data : resultsList) {
			ChildShows childShows = childShowsDao.getByShowCode(showCode);
			Dog dog = dogDao.getByPedigreeCertifiedCode(data.getPedigreeCertified());
			if (dog == null) {
				DogPedigreeCertifiedRedo pedigreeCertifiedRedo = dogPedigreeCertifiedRedoDao
						.getByOriginalPedigreeCertified(data.getPedigreeCertified());
				if (pedigreeCertifiedRedo != null) {
					Dog dog1 = dogDao.getByPedigreeCertifiedCode(pedigreeCertifiedRedo.getCurrentPedigreeCertified());
					if (dog1 != null) {
						dog = dog1;
					}
				}
			}
			if (dog != null) {
				if (getMonthAge(childShows.getStartTime(), dog.getBirthdate()) >= 9) {
					// 犬主人或第二犬主人一人无牵犬师资格证书即可
					// 查询犬主人
					List<String> levelList = handlerQualificationCertificateDao.getABC(dog.getMemberCode());
					// 查询第二犬主人
					if (!Collections3.isEmpty(levelList)
							&& (levelList.contains("A") || levelList.contains("B") || levelList.contains("C"))) {
						if (!StringUtils.isBlank(dog.getSecondOwnerMemberNum())) {
							levelList = handlerQualificationCertificateDao.getABC(dog.getSecondOwnerMemberNum());
						}
					}
					if (!Collections3.isEmpty(levelList)
							&& (levelList.contains("A") || levelList.contains("B") || levelList.contains("C"))) {
						continue;
					}

					// 计算成绩
					int ohcScore = 0;
					if (("OHC-BIS1").equals(data.getResultOhcBis())) {
						ohcScore += 80;
					}
					if (("OHC-BIS2").equals(data.getResultOhcBis())) {
						ohcScore += 50;
					}
					if (("OHC-BIG1").equals(data.getResultOhcBig())) {
						ohcScore += 30;
					}
					if (("OHC-BIG2").equals(data.getResultOhcBig())) {
						ohcScore += 20;
					}
					if (("OHC-BIG3").equals(data.getResultOhcBig())) {
						ohcScore += 15;
					}
					if (("OHC-BIG4").equals(data.getResultOhcBig())) {
						ohcScore += 10;
					}
					if (("OHC-BISS").equals(data.getResultOhcBiss())) {
						ohcScore += 10;
					}
					if (("BOH").equals(data.getResultOhcBoh())) {
						ohcScore += 5;
					}
					if (ohcScore != 0) {
						data.setOhcScore(Integer.toString(ohcScore));
						dao.update(data);
					}
				}
			}
		}
	}

	/**
	 *
	 * @description: 计算BIS项成绩
	 * @author: laiguanglong
	 * @date: 2016年8月2日 下午2:26:06
	 */
	private void updateResultBIS(String showCode) {
		// BPBIS/PBIS计算打败只数需要区分本场比赛是否评选了BPBIG/PBIG,如果有BIG成绩统计犬种组,如果无BIG成绩统计犬种
		List<Long> bpbisCountList; // 1234【犬种】or【犬种组】里各有多少只犬参赛
		if (dao.countBIG(showCode, "BPBIG") > 0) {
			bpbisCountList = dao.countByDogBreedGroupForBIS(showCode, "BPBIS");
		} else {
			bpbisCountList = dao.countByFciCodeForBIS(showCode, "BPBIS");
		}
		List<Long> pbisCountList; // 1234【犬种】or【犬种组】里各有多少只犬参赛
		if (dao.countBIG(showCode, "PBIG") > 0) {
			pbisCountList = dao.countByDogBreedGroupForBIS(showCode, "PBIS");
		} else {
			pbisCountList = dao.countByFciCodeForBIS(showCode, "PBIS");
		}
		// BIS统计犬种组 1234【犬种组】里各有多少只犬参赛
		List<Long> bisCountList = dao.countByDogBreedGroupForBIS(showCode, "BIS");

		// 该年龄段一共有多少只狗参赛
		Long bisSum = dao.countSumBIS(showCode);
		Long bpbisSum = dao.countSumBPBIS(showCode);
		Long pbisSum = dao.countSumPBIS(showCode);
		long bisCount = 0;
		long bpbisCount = 0;
		long pbisCount = 0;
		for (int i = 1; i < 5; i++) {
			dao.updateResultBIS(showCode, (bisSum - bisCount) + "", "BIS" + i);
			dao.updateResultBIS(showCode, (bpbisSum - bpbisCount) + "", "BPBIS" + i);
			dao.updateResultBIS(showCode, (pbisSum - pbisCount) + "", "PBIS" + i);
			if (i < 4) {
				bisCount += bisCountList.size() > i - 1 ? bisCountList.get(i - 1) : 0;
				bpbisCount += bpbisCountList.size() > i - 1 ? bpbisCountList.get(i - 1) : 0;
				pbisCount += pbisCountList.size() > i - 1 ? pbisCountList.get(i - 1) : 0;
			}
		}
	}

	private int getMonthAge(Date date, Date birthdate) {
		Calendar dateCalendar = Calendar.getInstance();
		Calendar birthdateCalendar = Calendar.getInstance();
		dateCalendar.setTime(date);
		birthdateCalendar.setTime(birthdate);
		int monthAge = dateCalendar.get(Calendar.MONTH) - birthdateCalendar.get(Calendar.MONTH);
		if (dateCalendar.get(Calendar.YEAR) != birthdateCalendar.get(Calendar.YEAR)) {
			monthAge = (dateCalendar.get(Calendar.YEAR) - birthdateCalendar.get(Calendar.YEAR)) * 12 + monthAge;
		}
		birthdateCalendar.add(Calendar.MONTH, monthAge);
		if (birthdateCalendar.getTime().after(dateCalendar.getTime())) {
			monthAge = monthAge - 1;
		}
		return monthAge;
	}

	/**
	 * BIG成绩初始化
	 * 
	 * @param showCode
	 * @param resultBig 可传BPBIG、PBIG、BIG
	 */
	private void initResultBIG(String showCode,String resultBig) {
		// 本场所有的犬种组
		List<String> dogBreedGroupList = dao.findBIGDogBreedGroup(showCode, resultBig);
		ShowResults showResults = new ShowResults();
		showResults.setShowCode(showCode);
		for (String dogBreedGroup : dogBreedGroupList) {
			showResults.setDogBreedGroup(dogBreedGroup);
			showResults.setResultBig(resultBig);
			// 本犬种组内有多少只
			Long show_result = dao.countByDogBreedGroup(showResults);
			// 本犬种组所有得到BIG的犬只，从1到4排序
			List<ShowResults> showResultsList = dao.findFciCode(showResults);
			for (ShowResults show : showResultsList) {
				show.setShowCode(showCode);
				show.setShowResult(show_result.toString());
				show.setResultBig(resultBig);
				dao.updateShowResultBIG(show);
				// 查询本犬种的只数
				Long fciCount = dao.countByFciCode(show);
				show_result = show_result - fciCount;
			}
		}
	}
	
	
	@Transactional(readOnly = false)
	public void countShowResult(String showCode) {
		updateStatisticsState( showCode);
		// 合并刷新排行榜逻辑
		showScoreRankingService.createRankings(showCode,false);
	}

	/**
	 *
	 * @description: 统计犬展并修改标记
	 * @author: laiguanglong
	 * @date: 2016年11月23日 下午4:25:54
	 */
	@Transactional(readOnly = false)
	public void updateStatisticsState(String showCode) {
		// 更新本场成绩
		updateShowResult(showCode);
		// 更新本场犬展成绩更新状态
		ChildShows childShows = childShowsDao.getByShowCode(showCode);
		childShows.setStatisticsState("1");
		childShowsDao.update(childShows);
		/**
		 * @Description: 更新此犬犬展year字段,根据主犬展设置年份year判断
		 * @author: zhangxiang
		 * @date: 2019/10/16 15:16
		 */
		MainShows mainShows = mainShowsDao.get(childShows.getMainShowId());
		if (mainShows == null || StringUtils.isBlank(mainShows.getYear())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "请前往主犬展中设置该犬展年份");
		}
		// 查询此犬展的所有犬：更新本场犬展比赛成绩（不为失格）的杜高犬，将禁止繁殖状态改为可繁殖状态
		List<ShowResults> showResultsList = showResultsDao.findAllListByCode(showCode);
		for (ShowResults showResult : showResultsList) {
			// 此判断处理2020年之前导入的成绩，有的为2020年的成绩，有的为2019年数据，为了在统计时更正year字段
			if (showResult.getYear().equals("2019")) {
				showResult.setYear(mainShows.getYear());
				// 更新该报名犬只犬展年份年份
				showResultsDao.updateShowResultYear(showResult);
			}
			// 杜高犬只参赛 繁殖状态不改变
//			Dog dog = dogDao.getByPedigreeCertifiedCode(showResult.getPedigreeCertified());
//			if(dog!=null && "292".equals(dog.getDogBreed()) && !showResult.getRemarksCc().contains("失格")){
//				dog.setBusinessBan("0");
//				dogDao.updateDogBusinessBan(dog);
//				DogBirthCertificate dogBirthCertificate = dogBirthCertificateDao.getByPedigreeCertified(showResult.getPedigreeCertified());
//				//出生纸更新繁殖状态
//				if(dogBirthCertificate!=null){
//					dogBirthCertificate.setBusinessBan("0");
//					dogBirthCertificateDao.updateDogBirthCertificateBusinessBan(dogBirthCertificate);
//				}
//			}
		}

	}

	/**
	 * 查询所有BIS成绩
	 * 
	 * @author yuanshuai
	 * @date 2021/5/27 10:28
	 */
	public List<ShowResults> findAllBISByShowCode(String showCode) {
		return dao.findAllBISByShowCode(showCode);
	}

	/**
	 * 根据犬展编号和犬种组查询犬只
	 * 
	 * @author yuanshuai
	 * @date 2021/6/2 15:42
	 */
	public List<ShowResults> findAllResultByAgeGroup(String showCode, String groupType) {
		return dao.findAllResultByAgeGroup(showCode, groupType);
	}

	/**
	 * 全犬种冠军展获得BOB成绩，计算的单犬种积分
	 * 
	 * @param showCode
	 */
	private void updateBreedResutlBOB(String showCode) {
		dao.updateBreedResutlBOB(showCode);
	}

	/**
	 * 单独展成绩获得BISS成绩，计算的单犬种积分
	 * 
	 * @param showCode
	 */
	private void updateBreedResutlBISS(String showCode) {
		// bov有值，计算打败只数，计算9月龄以上；bov没有值，计算打败只数，计算6月龄以上；
		showResultsDdzDao.updateBreedResutlBISSWithBOV(showCode);
		showResultsDdzDao.updateBreedResutlBISSWithoutBOV(showCode);
	}

	/**
	 * 单独展成绩获得BISS成绩，计算的单犬种积分
	 * 
	 * @param showCode
	 */
	private void updateResultBISS(String showCode) {
		// bov有值，计算打败只数，计算9月龄以上；bov没有值，计算打败只数，计算6月龄以上；
		dao.updateResultBISSWithBOV(showCode);
		dao.updateResultBISSWithoutBOV(showCode);

	}

	/**
	 * 单独展成绩获得BOB成绩，计算的单犬种积分
	 * 
	 * @param showCode
	 */
	private void updateBreedResutlDdzBOB(String showCode) {
		showResultsDdzDao.updateBreedResutlDdzBOB(showCode);
	}

	/**
	 * 单独展成绩获得BOV成绩，计算的单犬种积分
	 * 
	 * @param showCode
	 */
	private void updateBreedResutlDdzBOV(String showCode) {
		showResultsDdzDao.updateBreedResutlDdzBOV(showCode);

	}

	/**
	 * 单独展成绩获得BIG成绩，计算的单犬种积分
	 * 
	 * @param showCode
	 */
	private void updateBreedResutlDdzBIG(String showCode) {
		showResultsDdzDao.updateBreedResutlDdzBIG(showCode);

	}

	@Transactional(readOnly = false)
	public String refreshHm(List<String> showCodes) {
		List<ChildShows> sclist = dao.refreshHm(showCodes);
		//验证是全部为去年的犬展
		if(showCodes.size()!=sclist.size()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "有犬展编号错误，只查出"+sclist.size()+"场！");
		}
		int lastyear = LocalDate.now().getYear()-1;
		List<ChildShows> errList = sclist.stream().filter(a->
			Integer.valueOf( a.getYear())<lastyear).collect(Collectors.toList());
		if(errList.size()>0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只能处理去年和今年的！");
		}
		List<String> list = sclist.stream().map(a->a.getShowCode()).collect(Collectors.toList());
		
		//先算出所有场次的成绩
		for (String showCode : list) {
			updateStatisticsState(showCode);
		}
		// 合并刷新排行榜逻辑
		String showCodeLast = list.get(list.size()-1);
		showScoreRankingService.createRankings(showCodeLast,true);
		
		return showCodeLast;
		
	}

	private void updateBreedResutlDdzBPPBOB(String showCode, String resultBob) {
		showResultsDdzDao.updateBreedResutlDdzBPPBOB(showCode, resultBob);
	}

}