/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.thinkgem.jeesite.common.utils;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Months;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 日期工具类, 继承org.apache.commons.lang.time.DateUtils类
 * 
 * @author ThinkGem
 * @version 2014-4-15
 */
public class DateUtils extends org.apache.commons.lang3.time.DateUtils {

	private static String[] parsePatterns = { "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm", "yyyy-MM",
			"yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy/MM", "yyyy.MM.dd", "yyyy.MM.dd HH:mm:ss",
			"yyyy.MM.dd HH:mm", "yyyy.MM" };
	public static String YYYYMMDDTHHMMSSZ = "yyyy-MM-dd'T'HH:mm:ssZ";
	public static final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

	/**
	 * 得到当前日期字符串 格式（yyyy-MM-dd）
	 */
	public static String getDate() {
		return getDate("yyyy-MM-dd");
	}

	/**
	 * 得到当前日期字符串 格式（yyyy-MM-dd） pattern可以为："yyyy-MM-dd" "HH:mm:ss" "E"
	 */
	public static String getDate(String pattern) {
		return DateFormatUtils.format(new Date(), pattern);
	}

	/**
	 * 得到日期字符串 默认格式（yyyy-MM-dd） pattern可以为："yyyy-MM-dd" "HH:mm:ss" "E"
	 */
	public static String formatDate(Date date, Object... pattern) {
		String formatDate = null;
		if (pattern != null && pattern.length > 0) {
			formatDate = DateFormatUtils.format(date, pattern[0].toString());
		} else {
			formatDate = DateFormatUtils.format(date, "yyyy-MM-dd");
		}
		return formatDate;
	}

	/**
	 * 得到日期时间字符串，转换格式（yyyy-MM-dd HH:mm:ss）
	 */
	public static String formatDateTime(Date date) {
		return formatDate(date, "yyyy-MM-dd HH:mm:ss");
	}
	
	public static String formatDateTimeSSS(Date date) {
		return formatDate(date, "yyyy-MM-dd-HH-mm-ss-SSS");
	}

	/**
	 * 得到当前时间字符串 格式（HH:mm:ss）
	 */
	public static String getTime() {
		return formatDate(new Date(), "HH:mm:ss");
	}

	/**
	 * 得到当前日期和时间字符串 格式（yyyy-MM-dd HH:mm:ss）
	 */
	public static String getDateTime() {
		return formatDate(new Date(), "yyyy-MM-dd HH:mm:ss");
	}

	/**
	 * 得到当前年份字符串 格式（yyyy）
	 */
	public static String getYear() {
		return formatDate(new Date(), "yyyy");
	}

	/**
	 * 得到当前月份字符串 格式（MM）
	 */
	public static String getMonth() {
		return formatDate(new Date(), "MM");
	}

	/**
	 * 得到当天字符串 格式（dd）
	 */
	public static String getDay() {
		return formatDate(new Date(), "dd");
	}

	/**
	 * 得到当前星期字符串 格式（E）星期几
	 */
	public static String getWeek() {
		return formatDate(new Date(), "E");
	}

	/**
	 * 日期型字符串转化为日期 格式 { "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm",
	 * "yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy.MM.dd",
	 * "yyyy.MM.dd HH:mm:ss", "yyyy.MM.dd HH:mm" }
	 */
	public static Date parseDate(Object str) {
		if (str == null) {
			return null;
		}
		try {
			return parseDate(str.toString(), parsePatterns);
		} catch (ParseException e) {
			return null;
		}
	}


	/**
	 * 获取过去的天数
	 * 
	 * @param date
	 * @return
	 */
	public static long pastDays(Date date) {
		long t = new Date().getTime() - date.getTime();
		return t / (24 * 60 * 60 * 1000);
	}

	/**
	 * 获取过去的小时
	 * 
	 * @param date
	 * @return
	 */
	public static long pastHour(Date date) {
		long t = new Date().getTime() - date.getTime();
		return t / (60 * 60 * 1000);
	}

	/**
	 * 获取过去的分钟
	 * 
	 * @param date
	 * @return
	 */
	public static long pastMinutes(Date date) {
		long t = new Date().getTime() - date.getTime();
		return t / (60 * 1000);
	}

	/**
	 * 转换为时间（天,时:分:秒.毫秒）
	 * 
	 * @param timeMillis
	 * @return
	 */
	public static String formatDateTime(long timeMillis) {
		long day = timeMillis / (24 * 60 * 60 * 1000);
		long hour = (timeMillis / (60 * 60 * 1000) - day * 24);
		long min = ((timeMillis / (60 * 1000)) - day * 24 * 60 - hour * 60);
		long s = (timeMillis / 1000 - day * 24 * 60 * 60 - hour * 60 * 60 - min * 60);
		long sss = (timeMillis - day * 24 * 60 * 60 * 1000 - hour * 60 * 60 * 1000 - min * 60 * 1000 - s * 1000);
		return (day > 0 ? day + "," : "") + hour + ":" + min + ":" + s + "." + sss;
	}

	/**
	 * 获取两个日期之间的天数
	 *
	 * @param before
	 * @param after
	 * @return
	 */
	public static double getDistanceOfTwoDate(Date before, Date after) {
		long beforeTime = before.getTime();
		long afterTime = after.getTime();
		return (afterTime - beforeTime) / (1000 * 60 * 60 * 24);
	}

	/**
	 * 查询一年之后的日期
	 * 
	 * @return
	 */
	public static String get1YearLater(Date date) {
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		return format.format(get1YearLaterDate(date));
	}

	/**
	 * 查询一年之后的日期,日期格式
	 */
	public static Date get1YearLaterDate(Date date) {
		Calendar calendar = Calendar.getInstance();// 日历对象
		calendar.setTime(date);// 设置当前日期
		calendar.add(Calendar.YEAR, 1);// 年份+1
		return calendar.getTime();
	}

	/**
	 * 查询一个月之后的日期
	 * 
	 * @return
	 */
	public static String get1MonthLater(Date date) {
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		Calendar calendar = Calendar.getInstance();// 日历对象
		calendar.setTime(date);// 设置当前日期
		calendar.add(Calendar.MONTH, 1);// 月份+1
		return format.format(calendar.getTime());
	}

	/**
	 * 查询多少分钟之后的日期
	 *
	 * @return
	 */
	public static Date getMinuteLater(Date date,int late) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.add(Calendar.MINUTE, late);
		return calendar.getTime();
	}

	/**
	 * 获取日期差值，获取两个日期的相差月份
	 * 
	 * @param date1
	 * @param date2
	 * @param x
	 * @return
	 */
	public static boolean isInXMounth(Date date1, Date date2, int x) {
		if (date1 == null || date2 == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		if (x < 1) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "月份数必须为大于1的正整数");
		}
		try {
			date1 = sdf.parse(sdf.format(date1));
			date2 = sdf.parse(sdf.format(date2));
		} catch (ParseException e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期有误");
		}
		Calendar startDateCalendar = Calendar.getInstance();
		Calendar endDateCalendar = Calendar.getInstance();
		if (date1.before(date2)) {
			startDateCalendar.setTime(date1);
			endDateCalendar.setTime(date2);
		} else {
			startDateCalendar.setTime(date2);
			endDateCalendar.setTime(date1);
		}

		int year = endDateCalendar.get(Calendar.YEAR) - startDateCalendar.get(Calendar.YEAR);
		int month = endDateCalendar.get(Calendar.MONTH) - startDateCalendar.get(Calendar.MONTH);
		if (year * 12 + month < x) {
			return true;
		} else if (year * 12 + month == x) {
			return endDateCalendar.get(Calendar.DAY_OF_MONTH) < startDateCalendar.get(Calendar.DAY_OF_MONTH);
		} else {
			return false;
		}
	}

	/**
	 * 获取两个日期中大的日期，然后加上X月的 新日期
	 * 
	 * @param date1
	 * @param date2
	 * @param x
	 * @return
	 */
	public static Date getMaxXMounth(Date date1, Date date2, int x) {
		if (date1 == null || date2 == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		if (x < 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "月份数必须为大于等于0的正整数");
		}
		try {
			date1 = sdf.parse(sdf.format(date1));
			date2 = sdf.parse(sdf.format(date2));
		} catch (ParseException e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期有误");
		}
		Calendar dateCalendar = Calendar.getInstance();
		if (date1.before(date2)) {
			dateCalendar.setTime(date2);
		} else {
			dateCalendar.setTime(date1);
		}
		if (x > 0) {
			dateCalendar.add(Calendar.MONTH, x);
		}
		return dateCalendar.getTime();
	}

	/**
	 * 获取两个日期中小的日期，然后加上X月的 新日期
	 * 
	 * @param date1
	 * @param date2
	 * @param x
	 * @return
	 */
	public static Date getMinXMounth(Date date1, Date date2, int x) {
		if (date1 == null || date2 == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		if (x < 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "月份数必须为大于等于0的正整数");
		}
		try {
			date1 = sdf.parse(sdf.format(date1));
			date2 = sdf.parse(sdf.format(date2));
		} catch (ParseException e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期有误");
		}
		Calendar dateCalendar = Calendar.getInstance();
		if (date1.after(date2)) {
			dateCalendar.setTime(date2);
		} else {
			dateCalendar.setTime(date1);
		}
		if (x > 0) {
			dateCalendar.add(Calendar.MONTH, x);
		}
		return dateCalendar.getTime();
	}

	/**
	 * 获取本月的结束时间
	 * 
	 * @return
	 */
	public static Date getEndDayOfMonth() {
		Calendar calendar = Calendar.getInstance();
		calendar.set(getNowYear(), getNowMonth() - 1, 1);
		int day = calendar.getActualMaximum(5);
		calendar.set(getNowYear(), getNowMonth() - 1, day);
		return getDayEndTime(calendar.getTime());
	}

	public static String getEndDayOfMonthStr() {
		Calendar calendar = Calendar.getInstance();
		calendar.set(getNowYear(), getNowMonth() - 1, 1);
		int day = calendar.getActualMaximum(5);
		calendar.set(getNowYear(), getNowMonth() - 1, day);
		return formatDate(getDayEndTime(calendar.getTime()));
	}

	/**
	 * 获取本月的结束时间
	 * 
	 * @return
	 */
	public static Date getEndDayOfMonth(Date time) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(time);
		calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), 1);
		int day = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);
		calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), day);
		return getDayEndTime(calendar.getTime());
	}

	/**
	 * 获取今年是哪一年
	 * 
	 * @return
	 */
	public static Integer getNowYear() {
		Date date = new Date();
		return getNowYear(date);
	}

	/**
	 * 获取今年是哪一年
	 * 
	 * @return
	 */
	public static Integer getNowYear(Date date) {
		GregorianCalendar gc = (GregorianCalendar) Calendar.getInstance();
		gc.setTime(date);
		return Integer.valueOf(gc.get(1));
	}

	/**
	 * 获取本月是哪一月
	 * 
	 * @return
	 */
	public static int getNowMonth(Date date) {
		GregorianCalendar gc = (GregorianCalendar) Calendar.getInstance();
		gc.setTime(date);
		return gc.get(2) + 1;
	}

	/**
	 * 获取本月是哪一月
	 * 
	 * @return
	 */
	public static int getNowMonth() {
		Date date = new Date();
		return getNowMonth(date);
	}

	/**
	 * 获取某个日期的开始时间
	 * 
	 * @param d
	 * @return
	 */
	public static Date getDayStartTime(Date d) {
		Calendar calendar = Calendar.getInstance();
		if (null != d)
			calendar.setTime(d);
		calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), calendar.get(Calendar.DAY_OF_MONTH), 0,
				0, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		// return new Timestamp(calendar.getTimeInMillis());
		return calendar.getTime();
	}

	/**
	 * 获取某个日期的结束时间
	 * 
	 * @param d
	 * @return
	 */
	public static Date getDayEndTime(Date d) {
		Calendar calendar = Calendar.getInstance();
		if (null != d)
			calendar.setTime(d);
		calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), calendar.get(Calendar.DAY_OF_MONTH), 23,
				59, 59);
		calendar.set(Calendar.MILLISECOND, 999);
		// return new Timestamp(calendar.getTimeInMillis());
		return calendar.getTime();
	}

	/**
	 * 判断当前日期是否在某个日期之前
	 * 
	 * @param date1
	 * @param date2
	 * @return
	 */
	public static boolean isBeforeMounth(Date date1, String dateStr, boolean isIncludeJudgmentDate) {
		if (date1 == null || StringUtils.isEmpty(dateStr)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		Date date2 = null;
		try {
			date1 = sdf.parse(sdf.format(date1));
			date2 = sdf.parse(dateStr);
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期有误");
		}
		return date1.before(date2) || (isIncludeJudgmentDate ? (date1.getTime() == date2.getTime()) : false);
	}

	/**
	 * 判断当前日期是否在某个日期之前
	 * 
	 * @param date1
	 * @param date2
	 * @return
	 */
	public static boolean isBeforeMounth(Date date1, Date date2, boolean isIncludeJudgmentDate) {
		if (date1 == null || date2 == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		try {
			date1 = sdf.parse(sdf.format(date1));
			date2 = sdf.parse(sdf.format(date2));
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期有误");
		}
		return date1.before(date2) || (isIncludeJudgmentDate ? (date1.getTime() == date2.getTime()) : false);
	}

	/**
	 * 判断当前日期是否在某个日期之前
	 * 
	 * @param date1
	 * @param date2
	 * @return
	 */
	public static boolean isAfterMounth(Date date1, String dateStr, boolean isIncludeJudgmentDate) {
		if (date1 == null || StringUtils.isEmpty(dateStr)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		Date date2 = null;
		try {
			date1 = sdf.parse(sdf.format(date1));
			date2 = sdf.parse(dateStr);
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期有误");
		}
		return date1.after(date2) || (isIncludeJudgmentDate ? (date1.getTime() == date2.getTime()) : false);
	}

	/**
	 * 判断当前日期是否在某个日期之前
	 * 
	 * @param date1
	 * @param date2
	 * @return
	 */
	public static boolean isAfterMounth(Date date1, Date date2, boolean isIncludeJudgmentDate) {
		if (date1 == null || date2 == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		try {
			date1 = sdf.parse(sdf.format(date1));
			date2 = sdf.parse(sdf.format(date2));
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期有误");
		}
		return date1.after(date2) || (isIncludeJudgmentDate ? (date1.getTime() == date2.getTime()) : false);
	}

	/**
	 * 计算两个日期相差月份
	 * 
	 * @param date1
	 * @param date2
	 * @return
	 */
	public static int monthsBetween(Date date1, Date date2) {
		int months = 0;
		if (date1 == null || date2 == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		try {
			DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd");
			DateTime start = formatter.parseDateTime(sdf.format(date1));
			DateTime end = formatter.parseDateTime(sdf.format(date2));
			months = Months.monthsBetween(start, end).getMonths();
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期有误");
		}
		return months > 0 ? months : 0;
	}

	/**
	 * 根据开始日期和截止日期获取两日期中间所有日期集合
	 *
	 * @param startDate 开始日期
	 * @param endDate   截止日期
	 * @return List<Date> 日期集合
	 * @author yuanshuai
	 * @date 2021/2/4 13:48
	 */
	public static List<Date> getDayListBetweenTwoDays(Date startDate, Date endDate) {
		List<Date> dayList = new ArrayList<>();
		if (startDate != null && endDate != null) {
			Calendar tempStart = Calendar.getInstance();
			tempStart.setTime(startDate);
			Calendar tempEnd = Calendar.getInstance();
			tempEnd.setTime(endDate);
			tempEnd.add(Calendar.DATE, +1);
			while (tempStart.before(tempEnd)) {
				dayList.add(tempStart.getTime());
				tempStart.add(Calendar.DAY_OF_YEAR, 1);
			}
		}
		return dayList;
	}

	/**
	 * 判断目标日期是不是在起始日期至偏移量日期之内
	 * 
	 * @param offset 便宜日期（天）
	 * @author yuanshuai
	 * @date 2021/4/2 15:53
	 */
	public static boolean isBelongDayOffset(Date target, Date start, int offset) {
		Calendar startCal = Calendar.getInstance();
		startCal.setTime(start);
		startCal.add(Calendar.DAY_OF_MONTH, offset);
		Calendar endCal = Calendar.getInstance();
		endCal.setTime(start);
		Calendar targetCal = Calendar.getInstance();
		targetCal.setTime(target);
		if (!targetCal.before(startCal) && targetCal.before(endCal)) {
			return true;
		} else {
			return false;
		}
	}

	// 日期比较方法
	public static int compareDate(Date first, Date second) {
		return first.compareTo(second);
	}

	public static String parseDate(Date date,String pattern){
		SimpleDateFormat format = new SimpleDateFormat(pattern);
		return format.format(date);
	}

	/**
	 * 将日期转成RFC3339
	 * @author yuanshuai
	 * @date 2021/11/24 17:00
	 */
	public static String parseDateByRFC3339(Date date) {
		return new DateTime(date, DateTimeZone.forTimeZone(TimeZone.getTimeZone("Asia/Shanghai"))).toString();
	}
	
}
