/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.show.service;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.text.StrBuilder;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.entity.Dog;
import com.cku.oa.dog.service.DogDnaService;
import com.cku.oa.dog.service.DogService;
import com.cku.oa.finance.dao.PaymentChargingItemDao;
import com.cku.oa.finance.dao.PaymentOrderDao;
import com.cku.oa.finance.dao.PaymentOrderDetailDao;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.finance.entity.PaymentOrder;
import com.cku.oa.finance.entity.PaymentOrderDetail;
import com.cku.oa.finance.entity.PaymentOrderProject;
import com.cku.oa.finance.entity.PaymentRecord;
import com.cku.oa.finance.service.PaymentOrderProjectService;
import com.cku.oa.finance.service.PaymentRecordService;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.kennel.service.KennelService;
import com.cku.oa.show.dao.ChampionLoginDao;
import com.cku.oa.show.entity.ChampionLogin;
import com.cku.oa.show.entity.ChildShows;
import com.cku.oa.show.entity.ShowResults;
import com.cku.oa.sys.code.SysCodeUtil;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.ExpressQueueService;
import com.cku.oa.sys.service.LoginServerService;
import com.cku.oa.sys.service.RemindService;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.oa.sys.util.MemberUtil;
import com.cku.oa.sys.util.ZtSmsUtil;
import com.cku.oa.sys.zalog.SysContext;
import com.cku.restful.v1.show.service.RestChampionLoginService;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.excel.ExportExcel;
import com.thinkgem.jeesite.common.utils.excel.ImportExcel;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import net.sf.json.JSONObject;

/**
 * 冠军登录Service
 * 
 * @author zhuoHeng
 * @version 2016-07-14
 */
@Service
@Transactional(readOnly = true)
public class ChampionLoginService extends CrudService<ChampionLoginDao, ChampionLogin> {

	@Autowired
	private RemindService remindService;
	@Autowired
	public ChampionLoginDao championLoginDao;
	@Autowired
	public PaymentOrderDao paymentOrderDao;
	@Autowired
	public PaymentOrderDetailDao paymentOrderDetailDao;
	@Autowired
	public PaymentChargingItemDao paymentChargingItemDao;
	@Autowired
	public PaymentOrderProjectService paymentOrderProjectService;
	@Autowired
	public PaymentRecordService paymentRecordService;
	@Autowired
	private ExpressQueueService expressQueueService;
	@Autowired
	private DogDnaService dogDnaService;
	@Autowired
	private DogService dogService;
	@Autowired
	private KennelService kennelService;
	@Autowired
	private MemberService memberService;
	@Autowired
	private LoginServerService loginServerService;

	public ChampionLogin get(String id) {
		return super.get(id);
	}

	public List<ChampionLogin> findList(ChampionLogin championLogin) {
		return super.findList(championLogin);
	}

	public Page<ChampionLogin> findPage(Page<ChampionLogin> page, ChampionLogin championLogin) {
		if (StringUtils.isBlank(championLogin.getDogBreed())) {
			// 根据当前登录端查询犬种
			championLogin.getSqlMap().put("dogBreedMap", loginServerService.dataScopeFilter());
		} else {
			championLogin.getSqlMap().put("dogBreedMap", " ='" + championLogin.getDogBreed() + "'");
		}
		return super.findPage(page, championLogin);
	}

	public Page<ChampionLogin> findPageNotDogBreed(Page<ChampionLogin> page, ChampionLogin championLogin) {
		return super.findPage(page, championLogin);
	}

	@Transactional(readOnly = false)
	public void save(ChampionLogin championLogin) throws ZAException {
		if (championLogin.getIsNewRecord()) {
			// 如果血统证书号为空，不保存
			if (StringUtils.isBlank(championLogin.getPedigreeCertified())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号为空，无法保存冠军登录信息");
			}
			championLogin.preInsert();
			championLogin.setPaymentState("1");
			championLogin.setProccessState("0");
			championLogin.setCreateDate(new Date());
			dao.insert(championLogin);
		} else {
			ChampionLogin model = get(championLogin.getId());
			String reviewState = model.getReviewState();
			// 若审核状态没有更改，不进入状态机
			if (model.getReviewState().equals(championLogin.getReviewState())) {
				championLogin.setUpdateDate(new Date());
			} else {
				if ("4".equals(reviewState) || "0".equals(reviewState)) {
					// FCI审核中 4 转换为未审核
					reviewState = "0";
				} else if ("1".equals(reviewState)) {
					// 通过 1
					reviewState = "1";
				} else {
					// 其他包括2不通过，3已注销都转换为未通过
					reviewState = "2";
				}
				SysContext sysContext = new SysContext(model.getPaymentState(), reviewState, model.getProccessState());
				if ("1".equals(championLogin.getReviewState())) {
					// 通过
					sysContext.confirmSuccess(sysContext, "冠军登录", model.getMemberCode(), "show_champion_login",
							model.getId(), model.getRunningNumber(), (str) -> {
							});
					String loginType = championLogin.getLoginType();
					if ("1".equals(loginType)) {
						loginType = "champion_login_cn_code";
					} else if ("2".equals(loginType)) {
						loginType = "champion_login_copper_code";
					} else if ("3".equals(loginType)) {
						loginType = "champion_login_silver_code";
					} else if ("4".equals(loginType)) {
						loginType = "champion_login_gold_code";
					} else if ("5".equals(loginType)) {
						loginType = "champion_login_super_code";
					} else if ("6".equals(loginType)) {
						loginType = "champion_login_w_code";
					} else if ("7".equals(loginType)) {
						loginType = "champion_login_yong_code";
					}
					// 登录序号
					String loginCode = SysCodeUtil.getChampionloginCode(loginType);
					if ("".equals(loginCode)) {
						throw new ZAException(-1, "冠军登录序号生成错误");
					}
					championLogin.setLoginCode(loginCode);
					championLogin.setPrintState("0");
				} else if ("2".equals(championLogin.getReviewState())) {
					// 未通过
					sysContext.confirmFail(sysContext, "冠军登录", championLogin.getReviewRemarks(), model.getMemberCode(),
							"show_champion_login", model.getId(), model.getRunningNumber(), (str) -> {
							});
				}
				championLogin.setReviewTime(new Date());
			}
			// 修改犬只表的犬英文名
			Dog dog = dogService.getByPedigreeCertifiedCode(model.getPedigreeCertified());
			String nameEn = championLogin.getDog().getNameEn();
			if (StringUtils.isNotBlank(nameEn) && !nameEn.equals(dog.getNameEn())) {
				nameEn = nameEn.replace("&amp;", "&");
				dog.setNameEn(nameEn);
				dogService.save(dog);
			}
			dao.update(championLogin);
//			sendMsg(championLogin);
		}
	}

	@Transactional(readOnly = false)
	public void orderSave(ChampionLogin orderSave) {
		// 2019年3月1日起取消金银铜登录 2019年2月28日14:22:03
		if ("2".equals(orderSave.getLoginType()) || "3".equals(orderSave.getLoginType())
				|| "4".equals(orderSave.getLoginType())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "自2019年3月1日起取消金银铜冠军登陆");
		}
		// 血统证书号必须正确
		Dog dog = dogService.getByPedigreeCertifiedCode(orderSave.getPedigreeCertified());
		if (dog == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号有误");
		}
		// 犬只状态只能为正常或已出口
		if (!"0".equals(dog.getDogState()) && !"5".equals(dog.getDogState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只状态不为正常或已出口");
		}
		// 付款会员必须是犬主
		if (!orderSave.getMemberCode().equals(dog.getMemberCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该订单付款人不为该犬只犬主");
		}
		// 必须是有效会员
		Member member = memberService.getByMemberCode(dog.getMemberCode());
		if (!MemberUtil.isActiveMember(member)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬主不是有效会员");
		}
		// 重复订单校验
		// 注销后可以从新申请
		List<ChampionLogin> championLoginList = dao.validateLogin(orderSave.getLoginType(),
				orderSave.getPedigreeCertified());
		for (Iterator<ChampionLogin> it = championLoginList.iterator(); it.hasNext();) {
			ChampionLogin item = it.next();
			if ("3".equals(item.getReviewState())) {
				it.remove();
			}
		}
		if (championLoginList.size() > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只已申请冠军登录");
		}
		ChampionLogin championLogin = new ChampionLogin();
		championLogin.setPedigreeCertified(dog.getPedigreeCertifiedCode());
		championLogin.setLoginType(orderSave.getLoginType());
		championLogin.setDogBreed(dog.getDogBreed());
		championLogin.setMembershipCode(dog.getMemberCode());
		// 订单价格取实收价格
		championLogin.setCreateDate(new Date());
		championLogin.setPaymentState("2");
		championLogin.setReviewState("0");
		championLogin.setProccessState("0");
		championLogin.setCkuState("0");
		championLogin.setPrintState("0");
		championLogin.setRunningNumber(orderSave.getRunningNumber());
		// oa订单录入价格取收费项详情中的实收价格
		List<PaymentOrderDetail> orderDetailList = paymentOrderDetailDao
				.getOrderByRunningNum(orderSave.getRunningNumber(), orderSave.getMemberCode());
		PaymentOrderDetail paymentOrderDetail = null;
		for (PaymentOrderDetail orderDetail : orderDetailList) {
			if (orderSave.getChargingItemId().equals(orderDetail.getChargingItemId())) {
				paymentOrderDetail = orderDetail;
				break;
			}
		}
		championLogin.setPrice(paymentOrderDetail.getPrice());
		championLogin.setPaymentMembershipCode(paymentOrderDetail.getMemberCode());
		championLogin.setPaymentTime(paymentOrderDetail.getCreateDate());
		super.save(championLogin);
		paymentOrderDetail.setBusinessIds(championLogin.getId());
		paymentOrderDetail.preUpdate();
		paymentOrderDetailDao.update(paymentOrderDetail);
		sendMsg(championLogin);
		
	}

	@Transactional(readOnly = false)
	public void delete(ChampionLogin championLogin) {
		super.delete(championLogin);
	}

	/**
	 * @Description：根据血统证书号获取犬只信息和成绩
	 * @author: Jack Xu
	 * @version: 2016年11月17日 下午7:06:08
	 */
	public String getChamploginType(String pedigreeCertified) {
		String loginType = championLoginDao.championLoginType(pedigreeCertified);
		return loginType;
	}

	/**
	 * @Description：根据血统证书号获取犬只信息和成绩
	 * @author: zhuoHeng
	 * @version: 2016年8月25日 下午4:36:08
	 */
	@Transactional(readOnly = false)
	public String getDogMessage(String pedigreeCertified) {
		Map<String, Object> map = new HashMap<String, Object>();
		Dog dog = championLoginDao.getDogMessage(pedigreeCertified);
		SimpleDateFormat formatter = new SimpleDateFormat("yyyy/MM/dd");
		if (dog != null && !"".equals(dog)) {
			map.put("message", "success");
			map.put("dogId", dog.getId());
			map.put("identifiSign", dog.getIdentificationFlag());
			map.put("dogEgName", dog.getNameEn());
			map.put("dogCnName", dog.getNameCn());
			map.put("dogType", dog.getDogBreed());
			map.put("dogBreed", dog.getDogType());
			map.put("birthdate", dog.getBirthdate() == null ? "" : formatter.format(dog.getBirthdate()));
			map.put("dogGender", dog.getGender());
			map.put("dogOwner", dog.getDogOwner());
			map.put("memberNum", dog.getMemberCode());
			map.put("pedigreeNum", dog.getPedigreeCertifiedCode());
			int cc = championLoginDao.getCCMessage(pedigreeCertified);
			int jcc = championLoginDao.getJCCMessage(pedigreeCertified);
			map.put("resultCc", cc );
			map.put("resultJcc", jcc);
            // map.put("resultCc", dog.getShowResults() != null ? dog.getShowResults().getResultCc() : 0);
			map.put("resultGcc", dog.getShowResults() != null ? dog.getShowResults().getResultGcc() : 0);
			// map.put("resultJcc", dog.getShowResults() != null ? dog.getShowResults().getResultJcc() : 0);
			map.put("resultRcc", dog.getShowResults() != null ? dog.getShowResults().getResultRcc() : 0);
			map.put("resultRjcc", dog.getShowResults() != null ? dog.getShowResults().getResultRjcc() : 0);
			map.put("resultCacib", dog.getShowResults() != null ? dog.getShowResults().getResultCacib() : 0);
			map.put("resultBis", dog.getShowResults() != null ? dog.getShowResults().getResultBis() : "0");
			map.put("resultBig", dog.getShowResults() != null ? dog.getShowResults().getResultBig() : "0");
			map.put("resultBob", dog.getShowResults() != null ? dog.getShowResults().getResultBob() : "0");
			map.put("resultBos", dog.getShowResults() != null ? dog.getShowResults().getResultBos() : "0");
			map.put("resultBow", dog.getShowResults() != null ? dog.getShowResults().getResultBow() : "0");
			map.put("isDogDnaArchive", "" + dogDnaService.isHaveDogDna(dog.getId()));
		} else {
			map.put("message", "failure");
		}
		return JSONObject.fromObject(map).toString();
	}

	/**
	 *
	 * @description: 给用户发送消息，根据审核状态和缴费状态来判断发送内容
	 * @author: laiguanglong
	 * @date: 2016年8月19日 下午3:45:44
	 */
	public void sendMsg(ChampionLogin championLogin) {
		String reviewState = championLogin.getReviewState();
		// 只有0，1，2的状态发消息
		if ("0".equals(reviewState) || "1".equals(reviewState) || "2".equals(reviewState)) {
			String content = "";
			String loginType = championLogin.getLoginType();
			String loginTypeLabel = "";
			if ("6".equals(loginType) || "7".equals(loginType)) {
				loginTypeLabel = DictUtils.getDictLabel(loginType, "main_login_type", "");
			} else {
				loginTypeLabel = DictUtils.getDictLabel(loginType, "champion_login_type", "");
			}
			if ("0".equals(reviewState) ) {
				if("5".equals(loginType)) {//超级登录时免费，不发收费相关短信
					return ;
				}
				SimpleDateFormat formatter = new SimpleDateFormat("yyyy年MM月dd日HH时mm分");
				String dateString = formatter.format(new Date());
				// 未审核状态
				if ("9".equals(championLogin.getPaymentState())||"1".equals(championLogin.getPaymentState())) {
					// 提交未缴费
					content = "您于" + dateString + "申请了" + loginTypeLabel + "，犬只证书号为"+championLogin.getPedigreeCertified()
					+"。登录费用"+championLogin.getPrice()+"元整，请您务必于当日24时前完成缴费，以免耽误业务办理。";
				} else if ("2".equals(championLogin.getPaymentState())) {
					// 缴费成功
					content =  "您于" + dateString + "申请了" + loginTypeLabel + "，犬只证书号为"+championLogin.getPedigreeCertified()
						+"。登录费用"+championLogin.getPrice()+"元整，请您核对登录犬只信息，以免耽误业务办理。";
				}
			} else if ("1".equals(reviewState)) {
				// 审核通过
				content = "恭喜您，您的血统证书号为" + championLogin.getPedigreeCertified() + "的犬" + loginTypeLabel + "已经审核通过。";
			} else if ("2".equals(reviewState)) {
				// 审核不通过
				content = "您的血统证书号为" + championLogin.getPedigreeCertified() + "的犬" + loginTypeLabel
						+ "申请审核未通过，具体原因请登录会员中心查看。";
			}
			
			Member member = memberService.getByMemberCode(championLogin.getPaymentMembershipCode());
			ZtSmsUtil.sendNote(content, member.getMobile());
		}
	}

	/**
	 *
	 * @Description：重置冠军登录打印状态
	 * @author: zhuoHeng
	 * @version: 2016年8月26日 上午9:56:36
	 */
	@Transactional(readOnly = false)
	public void resetPrint(ChampionLogin championLogin) {
		championLogin.setPrintState("");
		dao.resetPrintState(championLogin);
	}

	/**
	 *
	 * @Description：登录申请是否重复和是否符合申请条件验证,验证是否完成普通登录
	 * @author: zhuoHeng
	 * @version: 2016年8月29日 上午10:44:29
	 */
	public String validateLoginType(String loginType, String pedigreeCertified) {
		String str = "2345";
		String flag = "false";
		if (str.contains(loginType)) {
			List<ChampionLogin> validateList = dao.validateLoginType(pedigreeCertified);
			if (validateList.size() > 0) {
				flag = "true";
			} else {
				flag = "false";
			}
		} else {
			flag = "true";
		}
		return flag;
	}

	/**
	 *
	 * @Description：登录申请是否重复
	 * @author: zhuoHeng
	 * @version: 2016年8月29日 下午1:59:12
	 */
	public String validateLogin(String loginType, String pedigreeCertified) {
		List<ChampionLogin> validateList = dao.validateLogin(loginType, pedigreeCertified);
		if (validateList.size() > 0) {
			return "false";
		} else {
			return "true";
		}
	}

	/**
	 *
	 * @description: 获取已付款待审核的冠军登录申请记录数
	 * @author: laiguanglong
	 * @date: 2016年8月29日 下午7:54:28
	 */
	public int getAuditCount() {
		ChampionLogin championLogin = new ChampionLogin();
		// 已付款
		championLogin.setPaymentState("2");
		// 未审核
		championLogin.setReviewState("0");
		return dao.listCount(championLogin);
	}

	/**
	 *
	 * @Description：冠军登录打印
	 * @author: zhuoHeng
	 * @version: 2016年9月5日 上午9:46:21
	 */
	@Transactional(readOnly = false)
	public void print(ChampionLogin championLogin) {
		ChampionLogin model = super.get(championLogin.getId());
		String loginType = DictUtils.getDictLabels(model.getLoginType(), "main_login_type",
				DictUtils.getDictLabels(model.getLoginType(), "champion_login_type", ""));
		championLogin.setPrintState("1");
		championLogin.setPrintTime(new Date());
		dao.print(championLogin);
		expressQueueService.push(model.getPaymentMembershipCode(), model.getRunningNumber(), loginType,
				model.getLoginCode(), null); // GYH2 TODO
		SysContext sysContext = new SysContext(null, null, null);
		sysContext.noStateExecute(sysContext, "冠军登录打印", model.getMemberCode(), "show_champion_login", model.getId(),
				model.getRunningNumber(), (str) -> {
				});

		// 等级贴纸业务
		// update by zjw 2023-02-07 工单号：ID1005116 取消宠爱护照发放
//		if ("1".equals(championLogin.getStickerFlag())) {
//			Dog dog = dogService.getByPedigreeCertifiedCode(championLogin.getPedigreeCertified());
//			dogStickerApplyService.applySticker(dog, championLogin.getPaymentMembershipCode(),
//					championLogin.getRunningNumber());
//		}
	}

	/**
	 *
	 * @description: 冠军登录申请业务校验
	 * @author: laiguanglong
	 * @date: 2016年9月10日 上午11:53:56
	 */
	public Dog validateBusiness(ChampionLogin po) {
		Member member = UserUtils.getLoginMember();
		// 校验会员为有效会员
		if (!MemberUtil.isActiveMember(member)) {
			throw new ZAException(ZAErrorCode.ZA_ERC_MEMBER_INVALID, "该业务仅限有效会员办理");
		}
		Dog dog = dogService.getByPedigreeCertifiedCode(po.getPedigreeCertified());
		if (dog == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号有误");
		}
		if (!member.getMemberCode().equals(dog.getMemberCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号有误");
		}
		// 犬只状态只能为正常或已出口
		if (!"0".equals(dog.getDogState()) && !"5".equals(dog.getDogState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只状态不为正常或已出口,不允许申请冠军登录");
		}
		// 所有登录增加芯片号限制，无芯片号的犬只不能提交登录申请。
		if (StringUtils.isBlank(dog.getIdentificationFlag())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只芯片号为空，不允许申请冠军登录");
		}
		int mounthAge = getMonthAge(new Date(), dog.getBirthdate());
		if ("7".equals(po.getLoginType())) {
			if (mounthAge < 9) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "青年冠军登录犬年龄必须为9个月以上");
			}
		}else if ("1".equals(po.getLoginType())) {
			if (mounthAge < 9 ) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "中国冠军登录犬年龄必须为9个月以上");
			}
		} else {
			if (mounthAge < 15) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬年龄必须为15个月以上");
			}
		}
		// DNA存档判断
		// if (!"7".equals(po.getLoginType()) &&
		// !dogDnaService.isHaveDogDna(dog.getId())) {
		// throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬没有进行DNA存档");
		// }
		// if ("7".equals(po.getLoginType())
		// && !dogBloodSaveService.check(po.getPedigreeCertified(),
		// false).isDogDloodDna()) {
		// throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬没有进行DNA血样留存");
		// }
		// 贴纸申请业务校验
		// update by zjw 2023-02-07 工单号：ID1005116 取消宠爱护照发放
//		if ("1".equals(po.getStickerFlag())) {
//			// 由于判定能否申请ss级贴纸必须打印后，所以第一次申请当时肯定不能申请ss级，所以去掉校验，但是如果已有ss级贴纸，则不允许申请贴纸业务
//			int alreadyHaveMaxClass = dogStickerApplyService.getAlreadyHaveMaxClass(dog.getIdentificationFlag());
//			if (alreadyHaveMaxClass == 4) {
//				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只目前没有可申请的贴纸");
//			}
//		}
		// 铜制、银制、金质、超级登录需完成中国登录后才可申请
		if (!"true".equals(validateLoginType(po.getLoginType(), po.getPedigreeCertified()))) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "铜制、银制、金质、超级登录需完成中国登录后才可申请");
		}
		if ("1".equals(po.getLoginType())) {
			// 普通冠军登录校验
			validateComLogin(dog);
		} else if ("2".equals(po.getLoginType())) {
			// 铜制冠军登录校验
			validateTongLogin(dog);
		} else if ("3".equals(po.getLoginType())) {
			// 银制冠军登录校验
			validateYinLogin(dog);
		} else if ("4".equals(po.getLoginType())) {
			// 金制冠军登录校验
			validateJinLogin(dog);
		} else if ("5".equals(po.getLoginType())) {
			// 超级冠军登录校验
			// 2020-03-20添加、目前不办理此项业务、如再办理另行通知 throw new
			// ZAException(ZAErrorCode.ZA_VALID_FAILED,
			// "此业务当前调试中无法办理、请联系cku赛事部，致电：400-660-7000"); 2020-06-18产品要求放开
			validateSuperLogin(dog);
		} else if ("6".equals(po.getLoginType())) {
			// 世界冠军登录校验
			validateWorldLogin(po.getPedigreeCertified());
		} else if ("7".equals(po.getLoginType())) {
			// 青年冠军登录校验
			// 2020-03-20添加、目前不办理此项业务、如再办理另行通知 throw new
			// ZAException(ZAErrorCode.ZA_VALID_FAILED,
			// "此业务当前调试中无法办理、请联系cku赛事部，致电：400-660-7000"); 2020-06-18产品要求放开
			// validateYoungLogin(po.getPedigreeCertified());03-20之前也没有
		}
		// 业务查重
		ChampionLogin example = new ChampionLogin();
		example.setPedigreeCertified(po.getPedigreeCertified());
		example.setLoginType(po.getLoginType());
		// 注销后可以重新申请
		List<ChampionLogin> championLoginList = dao.findList(example);
		for (Iterator<ChampionLogin> it = championLoginList.iterator(); it.hasNext();) {
			ChampionLogin item = it.next();
			if ("3".equals(item.getReviewState())) {
				it.remove();
			}
		}
		if (championLoginList.size() > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只已申请过冠军登录，请勿重复申请");
		}
		return dog;
	}

	/**
	 * 计算月龄
	 *
	 * @param date
	 * @param birthdate
	 * @return
	 */
	private int getMonthAge(Date date, Date birthdate) {
		Calendar dateCalendar = Calendar.getInstance();
		Calendar birthdateCalendar = Calendar.getInstance();
		dateCalendar.setTime(date);
		birthdateCalendar.setTime(birthdate);
		int monthAge = dateCalendar.get(Calendar.MONTH) - birthdateCalendar.get(Calendar.MONTH);
		if (dateCalendar.get(Calendar.YEAR) != birthdateCalendar.get(Calendar.YEAR)) {
			monthAge = (dateCalendar.get(Calendar.YEAR) - birthdateCalendar.get(Calendar.YEAR)) * 12 + monthAge;
		}
		birthdateCalendar.add(Calendar.MONTH, monthAge);
		if (birthdateCalendar.getTime().after(dateCalendar.getTime())) {
			monthAge = monthAge - 1;
		}
		return monthAge;
	}

	/**
	 * 2、《中国冠军资质证书》 获得4张CC（至少2张CC，其余RCC），并且须获得1次BOB/ BOS/BOW其中一项成绩。
	 * 
	 * @description: 普通冠军登录校验
	 * @author: laiguanglong
	 * @date: 2016年12月1日 下午3:31:14
	 */
	public void validateComLogin(Dog dog) {
		if (!validateComLoginForBoolean(dog)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请普通冠军登录须获得4张CC（至少2张CC，其余RCC)且获得1次BOB/ BOS/BOW其中一项成绩");
		}
	}

	public boolean validateComLoginForBoolean(Dog dog){
		Map<String, Integer> ccMap = getCcMap(dog);
		int countCC = ccMap.get("countCC");
		int countRCC = ccMap.get("countRCC");
		if (countCC < 2 || (countCC + countRCC / 2) < 4) {
			return false;
		}
		int countBOBBOSBOW = dao.countBOB_BOS_BOW(dog.getPedigreeCertifiedCode());
		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
			countBOBBOSBOW += dao.countBOB_BOS_BOW(dog.getPedigreeCertifiedEn());
		}
		if (countBOBBOSBOW < 1) {
			return false;
		}
		return true;
	}

	/**
	 * 3、《铜质冠军资质证书》 获得8张CC（至少4张CC，其余RCC），并且须获得1次BOB/ BOS/BOW其中一项成绩。
	 * 
	 * @description: 铜制冠军登录校验
	 * @author: laiguanglong
	 * @date: 2016年12月1日 下午3:31:14
	 */
	private void validateTongLogin(Dog dog) {
//		int countCC = dao.countCC(dog.getPedigreeCertifiedCode());
//		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
//			countCC += dao.countCC(dog.getPedigreeCertifiedEn());
//		}
		Map<String,Integer> ccMap = getCcMap(dog);
		int countCC = ccMap.get("countCC");
		if (countCC < 4) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请铜制冠军登录须获得8张CC（至少4张CC，其余RCC）");
		} else if (countCC < 8) {
			// 至少4张CC，其余RCC
//			int countRCC = dao.countRCC(dog.getPedigreeCertifiedCode());
//			if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
//				countRCC += dao.countRCC(dog.getPedigreeCertifiedEn());
//			}
			int countRCC = ccMap.get("countRCC");
			if ((countCC + countRCC / 2) < 8) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请铜制冠军登录须获得8张CC（至少4张CC，其余RCC）");
			}
		}
		int countBOB_BOS_BOW = dao.countBOB_BOS_BOW(dog.getPedigreeCertifiedCode());
		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
			countBOB_BOS_BOW += dao.countBOB_BOS_BOW(dog.getPedigreeCertifiedEn());
		}
		if (countBOB_BOS_BOW < 1) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请铜制冠军登录须获得1次BOB/ BOS/BOW其中一项成绩");
		}
	}

	/**
	 * 4、《银质冠军资质证书》 获得12张CC（至少8张CC，其余RCC），并且须获得2次BOB成绩（不同裁判授予）。
	 * 
	 * @description: 银制冠军登录校验
	 * @author: laiguanglong
	 * @date: 2016年12月1日 下午3:31:14
	 */
	private void validateYinLogin(Dog dog) {
//		int countCC = dao.countCC(dog.getPedigreeCertifiedCode());
//		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
//			countCC += dao.countCC(dog.getPedigreeCertifiedEn());
//		}
		Map<String,Integer> ccMap = getCcMap(dog);
		int countCC = ccMap.get("countCC");
		if (countCC < 8) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请银制冠军登录须获得12张CC（至少8张CC，其余RCC）");
		} else if (countCC < 12) {
//			int countRCC = dao.countRCC(dog.getPedigreeCertifiedCode());
//			if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
//				countRCC += dao.countRCC(dog.getPedigreeCertifiedEn());
//			}
			int countRCC = ccMap.get("countRCC");
			if ((countCC + countRCC / 2) < 12) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请银制冠军登录须获得12张CC（至少8张CC，其余RCC）");
			}
		}
		int countBOB = dao.countBOB(dog.getPedigreeCertifiedCode());
		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
			countBOB += dao.countBOB(dog.getPedigreeCertifiedEn());
		}
		if (countBOB < 2) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请银制冠军登录须获得2次BOB成绩（不同裁判授予）");
		}
	}

	/**
	 * 5、《金质冠军资质证书》 获得20张CC（至少12张CC，其余RCC），并且须获得2次BIG成绩（不同裁判授予）。
	 * 
	 * @description: 金制冠军登录校验
	 * @author: laiguanglong
	 * @date: 2016年12月1日 下午3:31:14
	 */
	private void validateJinLogin(Dog dog) {
//		int countCC = dao.countCC(dog.getPedigreeCertifiedCode());
//		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
//			countCC += dao.countCC(dog.getPedigreeCertifiedEn());
//		}
		Map<String,Integer> ccMap = getCcMap(dog);
		int countCC = ccMap.get("countCC");
		if (countCC < 12) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请金制冠军登录须获得20张CC（至少12张CC，其余RCC）");
		} else if (countCC < 20) {
//			int countRCC = dao.countRCC(dog.getPedigreeCertifiedCode());
//			if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
//				countRCC += dao.countRCC(dog.getPedigreeCertifiedEn());
//			}
			int countRCC = ccMap.get("countRCC");
			if ((countCC + countRCC / 2) < 20) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请金制冠军登录须获得20张CC（至少12张CC，其余RCC）");
			}
		}
		int countBIG = dao.countBIG(dog.getPedigreeCertifiedCode());
		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
			countBIG += dao.countBIG(dog.getPedigreeCertifiedEn());
		}
		if (countBIG < 2) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请金制冠军登录须获得2次BIG成绩（不同裁判授予）");
		}
	}

	/**
	 * 6、《超级冠军资质证书》 获得20张CC（至少15张CC，其余RCC），并且须获得2次BIS成绩（不同裁判授予）。
	 * 
	 * @description: 超级冠军登录校验
	 * @author: laiguanglong
	 * @date: 2016年12月1日 下午3:31:14
	 */
	private void validateSuperLogin(Dog dog) {
//		int countCC = dao.countCC(dog.getPedigreeCertifiedCode());
//		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
//			countCC += dao.countCC(dog.getPedigreeCertifiedEn());
//		}
//		Map<String,Integer> ccMap = getCcMap(dog);
//		int countCC = ccMap.get("countCC");
//		if (countCC < 15) {
//			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请超级冠军登录须获得20张CC（至少15张CC，其余RCC）");
//		} else if (countCC < 20) {
////			int countRCC = dao.countRCC(dog.getPedigreeCertifiedCode());
////			if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
////				countRCC += dao.countRCC(dog.getPedigreeCertifiedEn());
////			}
//			int countRCC = ccMap.get("countRCC");
//			if ((countCC + countRCC / 2) < 20) {
//				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请超级冠军登录须获得20张CC（至少15张CC，其余RCC）");
//			}
//		}
//		int countBIS = dao.countBIS(dog.getPedigreeCertifiedCode());
//		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
//			countBIS += dao.countBIS(dog.getPedigreeCertifiedEn());
//		}
//		if (countBIS < 2) {
//			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请超级冠军登录须获得2次BIS成绩（不同裁判授予）");
//		}
//		// 增加5次BIG1要求 2019年2月28日14:33:31
//		int countBIGOne = dao.countBIGOne(dog.getPedigreeCertifiedCode());
//		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
//			countBIGOne += dao.countBIGOne(dog.getPedigreeCertifiedEn());
//		}
//		if (countBIGOne < 5) {
//			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请超级冠军登录须获得5次BIG1成绩（不同裁判授予）");
//		}
		//判断比赛报名冠军组 在冠军组中胜出获得GCC卡
		int gccGroup = dao.countGCCGroup(dog.getPedigreeCertifiedCode());
		if(gccGroup<=0){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "您还未在冠军组中胜出获得GCC卡");
		}
		//累计获得30张GCC卡
		int gcc = dao.countGCC(dog.getPedigreeCertifiedCode());
		if(gcc<30){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请超级冠军登录须累计获得30张GCC卡");
		}
	}

	/**
	 * @description: 世界冠军登录校验
	 * @author: laiguanglong
	 * @date: 2016年12月1日 下午2:54:54
	 */
	private void validateWorldLogin(String pedigreeCertified) {
		Dog dog = dogService.getByPedigreeCertifiedCode(pedigreeCertified);
		// 1、申请犬只为CKU注册犬。
		if (!dog.getPedigreeCertifiedCode().toUpperCase().startsWith("CKU")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只不是CKU注册犬");
		}
		// 2、血统书显示完整三代祖辈血统。
		if (StringUtils.isBlank(dog.getDogGeneration()) || Integer.parseInt(dog.getDogGeneration()) < 3) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只代数必须大于等于3");
		}
		// 3、血统书显示芯片不能为空
		if (StringUtils.isBlank(dog.getIdentificationFlag())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只芯片号为空");
		}
		// 4、犬主已在CKU登记国际犬舍名称。
		Kennel kennel = kennelService.getKennelByMemberCode(dog.getMemberCode());
		if (kennel == null || !"0".equals(kennel.getDelFlag()) || !"1".equals(kennel.getReviewState())
				|| !"1".equals(kennel.getFciReviewState()) || kennel.getEndTime() == null
				|| kennel.getEndTime().getTime() < System.currentTimeMillis()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只犬主未在CKU登记国际犬舍名称。 ");
		}
		// 6、获得4张以上CACIB卡
		int countCACIB = dao.countCACIB(pedigreeCertified);
		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
			countCACIB += dao.countCACIB(dog.getPedigreeCertifiedEn());
		}
		if (countCACIB < 4) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只获得的CACIB卡少于4张");
		}
	}

	/**
	 * 1、《青年冠军资质证书》 获得3张JCC(至少1张JCC，其余RJCC)，并且须获得1次BOB/BOS/BOW/BOJ/JBOS其中一项成绩。
	 * 
	 * @description: 青年冠军登录校验
	 * @author: laiguanglong
	 * @date: 2016年12月1日 下午3:31:14
	 */
	private void validateYoungLogin(String pedigreeCertified) {
		int coutJCC = dao.countJCC(pedigreeCertified);
		if (coutJCC < 1) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请青年冠军登录至少3张JCC(至少1张JCC，其余RJCC)");
		} else if (coutJCC < 3) {
			int countRJCC = dao.countRJCC(pedigreeCertified);
			if ((coutJCC + countRJCC / 2) < 3) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请青年冠军登录至少3张JCC(至少1张JCC，其余RJCC)");
			}
		}
		int countBOB_BOS_BOW_BOJ_JBOS = dao.countBOB_BOS_BOW_BOJ_JBOS(pedigreeCertified);
		if (countBOB_BOS_BOW_BOJ_JBOS < 1) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请青年冠军登录须获得1次BOB/BOS/BOW/BOJ/JBOS其中一项成绩");
		}
	}

	/**
	 * @Description： 冠军登陆导入模板
	 * 
	 * @author: yuanshuai
	 * @date: 2019/6/19 11:01
	 */
	public String importChampionLoginTemplate(HttpServletResponse response) throws IOException {
		String fileName = "冠军登陆导入模板.xlsx";
		List<ChampionLogin> list = Lists.newArrayList();
		ChampionLogin login = new ChampionLogin();
		login.setLoginType("普通登录/超级登录/世界冠军登录/青年冠军登录");
		login.setPedigreeCertified("CKU-xxxxxxxxxx/xx");
		login.setLoginCode("NO.CKU xxxxxxxxxxx");
		login.setImportRemark("备注");
		login.setOrderProject("订单项目名称");
		list.add(login);
		new ExportExcel("冠军登陆信息", ChampionLogin.class, 2).setDataList(list).write(response, fileName).dispose();
		return null;
	}

	/**
	 * @Description： 冠军登陆导入
	 * 
	 * @author: yuanshuai
	 * @date: 2019/6/19 11:02
	 */
	@Transactional(readOnly = false)
	public void importChampionLogin(MultipartFile file)
			throws InvalidFormatException, IOException, IllegalAccessException, InstantiationException {
		ImportExcel ei = new ImportExcel(file, 1, 0);
		List<ChampionLogin> list = ei.getDataList(ChampionLogin.class);
		boolean error = false;
		StrBuilder strBuilder = new StrBuilder();
		for (ChampionLogin login : list) {
			if (StringUtils.isBlank(login.getPedigreeCertified())) {
				continue;
			}
			// 判断登录类型
			Member member = null;
			String loginType = login.getLoginType();
			if (!StringUtils.isBlank(loginType)) {
				String tempType = DictUtils.getDictValue(login.getLoginType(), "main_login_type", null);
				if (!StringUtils.isBlank(tempType)) {
					loginType = tempType;
				} else {
					loginType = DictUtils.getDictValue(login.getLoginType(), "champion_login_type", null);
				}
			}
			if (StringUtils.isBlank(loginType)) {
				error = recordError(strBuilder, login.getPedigreeCertified() + "登录类型为空或有误");
			} else {
				login.setLoginType(loginType);
			}
			// 2019年3月1日起取消金银铜登录 2019年2月28日14:22:03
			if ("2".equals(login.getLoginType()) || "3".equals(login.getLoginType())
					|| "4".equals(login.getLoginType())) {
				error = recordError(strBuilder, login.getPedigreeCertified() + "自2019年3月1日起取消金银铜冠军登陆");
			}
			// 血统证书号必须正确
			Dog dog = dogService.getByPedigreeCertifiedCode(login.getPedigreeCertified());
			if (dog == null) {
				error = recordError(strBuilder, login.getPedigreeCertified() + "血统证书号有误");
			} else if (!"0".equals(dog.getDogState()) && !"5".equals(dog.getDogState())) {
				// 犬只状态只能为正常或已出口
				error = recordError(strBuilder, login.getPedigreeCertified() + "该犬只状态不为正常或已出口");
			} else if (!StringUtils.isBlank(dog.getMemberCode())) {
				// 判断会员
				member = memberService.getByMemberCode(dog.getMemberCode());
				if (!MemberUtil.isActiveMember(member)) {
					error = recordError(strBuilder, login.getPedigreeCertified() + "该犬主不是有效会员");
				}
			} else if (!StringUtils.isBlank(loginType)) {
				// 重复订单校验
				// 注销后可以从新申请
				List<ChampionLogin> championLoginList = dao.validateLogin(login.getLoginType(),
						login.getPedigreeCertified());
				championLoginList.removeIf(item -> "3".equals(item.getReviewState()));
				if (championLoginList.size() > 0) {
					error = recordError(strBuilder, login.getPedigreeCertified() + "该犬只已申请此级别冠军登录");
				}
			}
			if (StringUtils.isBlank(login.getLoginCode())) {
				error = recordError(strBuilder, login.getPedigreeCertified() + "登录序号为空");
			}
			// 校验订单项目
			PaymentOrderProject project = new PaymentOrderProject();
			project.setName(login.getOrderProject());
			if (!StringUtils.isBlank(login.getOrderProject())) {
				List<PaymentOrderProject> projectList = paymentOrderProjectService.findList(project);
				if (projectList == null || projectList.size() == 0) {
					error = recordError(strBuilder, login.getPedigreeCertified() + "订单项目名称有误");
				} else if (projectList.size() > 1) {
					error = recordError(strBuilder, login.getPedigreeCertified() + "订单项目名称不唯一");
				} else {
					project = projectList.get(0);
				}
			} else {
				error = recordError(strBuilder, login.getPedigreeCertified() + "订单项目名称为空");
			}
			if (!error) {
				// 获取收费项目
				String shortName = null;
				if ("1".equals(login.getLoginType())) {
					shortName = RestChampionLoginService.PAYMENTCHARGINGITEM_SHORTNAME_COM;
				} else if ("5".equals(login.getLoginType())) {
					shortName = RestChampionLoginService.PAYMENTCHARGINGITEM_SHORTNAME_CHAOJI;
				} else if ("6".equals(login.getLoginType())) {
					shortName = RestChampionLoginService.PAYMENTCHARGINGITEM_SHORTNAME_SHIJIE;
				} else if ("7".equals(login.getLoginType())) {
					shortName = RestChampionLoginService.PAYMENTCHARGINGITEM_SHORTNAME_QINGNIAN;
				}
				PaymentChargingItem item = paymentChargingItemDao.getByShortName(shortName);
				login.setDogName(login.getNameTitle().trim() + " " + dog.getNameEn().trim());
				login.setDogBreed(dog.getDogBreed());
				login.setMembershipCode(dog.getMemberCode());
				// 订单价格取实收价格
				Date now = new Date();
				login.setCreateDate(now);
				login.setPaymentState("2");
				// 默认审核通过
				login.setReviewState("1");
				login.setReviewTime(now);
				login.setProccessState("0");
				// 默认已打印
				login.setPrintState("1");
				login.setPrintTime(now);
				login.setCkuState("0");
				login.setPrice(item.getPrice());
				login.setPaymentMembershipCode(dog.getMemberCode());
				login.setPaymentTime(now);
				login.setRemarks(login.getImportRemark());
				login.preInsert();
				PaymentOrder order = buildImportOrder(login, member, item, project);
				login.setRunningNumber(order.getOrderCode());
				dao.insert(login);
				// 修改犬只表的犬英文名
				if (StringUtils.isNotBlank(login.getDogName()) && !login.getDogName().equals(dog.getNameEn())) {
					dog.setNameEn(login.getDogName());
					dogService.save(dog);
				}
			}
		}
		if (error) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, strBuilder.toString());
		}
	}

	/**
	 * @Description： 生成导入订单
	 * 
	 * @author: yuanshuai
	 * @date: 2019/6/19 11:46
	 */
	@Transactional(readOnly = false)
	private PaymentOrder buildImportOrder(ChampionLogin login, Member member, PaymentChargingItem item,
			PaymentOrderProject project) {
		String orderCode = SysCodeUtil.getPayRunningNumber();
		Date now = new Date();
		// 生成订单明细
		PaymentOrderDetail detail = new PaymentOrderDetail();
		detail.setOrderCode(orderCode);
		detail.setAddTime(now);
		detail.setProccessState("1");
		detail.setProccessTime(now);
		detail.setProjectId(project.getId());
		detail.setProjectName(project.getName());
		detail.setChargingItemId(item.getId());
		detail.setChargingItemName(item.getName());
		detail.setPrice(item.getPrice());
		detail.setTotalPrice(item.getPrice());
		detail.setNum("1");
		detail.setBusinessIds(login.getId());
		detail.setShowBusinessModelFlag("0");
		detail.setBusinessTable("show_champion_login");
		detail.setMemberCode(member.getMemberCode());
		detail.setMemberName(member.getName());
		detail.preInsert();
		paymentOrderDetailDao.insert(detail);
		// 生成订单
		PaymentOrder order = new PaymentOrder();
		order.setSource("oa");
		// 比赛现场收入
		order.setPaymentWay("21");
		order.setOrderCode(orderCode);
		order.setAddTime(now);
		order.setLocked("0");
		order.setTotalPrice(item.getPrice());
		order.setPaymentState("2");
		order.setPaymentTime(now);
		order.setMemberCode(member.getMemberCode());
		order.setMemberName(member.getName());
		order.setProjectId(project.getId());
		order.setProjectName(project.getName());
		order.setReceiver(member.getName());
		order.setReceiveAddress(member.getAddress());
		order.setReceiveCity(member.getCity());
		order.setReceiveProvince(member.getProvince());
		order.setReceiveDel(member.getMobile());
		order.setReceivePostCode(member.getPostcode());
		order.preInsert();
		paymentOrderDao.insert(order);
		// 生成流水记录
		PaymentRecord record = new PaymentRecord();
		record.setRunningNumber(orderCode);
		record.setBusinessId(member.getId());
		record.setMemberCode(member.getMemberCode());
		record.setPaymentState("2");
		record.setPaymentWay(order.getPaymentWay());
		record.setPaymentTime(now);
		record.setPrice(order.getTotalPrice());
		record.setPaymentAmount(order.getTotalPrice());
		record.setPaymentRemarks("冠军登陆导入自动生成订单");
		// 所有现金收入都为96
		record.setChargingItemId("96");
		paymentRecordService.newRecord(record);
		return order;
	}

	/**
	 * @Description： 导入统一记录错误
	 * 
	 * @author: yuanshuai
	 * @date: 2019/6/11 16:01
	 */
	private boolean recordError(StrBuilder strBuilder, String msg) {
		strBuilder.append(msg).append("##");
		return true;
	}
	
	private Map<String, Integer> getCcMap(Dog dog) {
		Map<String, Integer> map = new HashMap<>();
		List<ShowResults> results = dao.countResult(dog.getPedigreeCertifiedCode());
		int countCC = 0, countRCC = 0;
		for (ShowResults ccs : results) {
			if (!StringUtils.isEmpty(ccs.getResultCc())) {
				String[] ccStr = ccs.getResultCc().split("/");
				for (String str : ccStr) {
					if ("CC".equals(str)) {
						countCC += 1;
					} else if ("RCC".equals(str)) {
						countRCC += 1;
					}
				}
			}
		}
		if (StringUtils.isNotBlank(dog.getPedigreeCertifiedEn())) {
			List<ShowResults> enList = dao.countResult(dog.getPedigreeCertifiedEn());
			for (ShowResults ccs : enList) {
				if (!StringUtils.isEmpty(ccs.getResultCc())) {
					String[] ccStr = ccs.getResultCc().split("/");
					for (String str : ccStr) {
						if ("CC".equals(str)) {
							countCC += 1;
						} else if ("RCC".equals(str)) {
							countRCC += 1;
						}
					}
				}
			}
		}
		map.put("countCC", countCC);
		map.put("countRCC", countRCC);
		return map;
	}
}