package com.cku.restful.v1.show.service;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.PageBeanResult;
import com.cku.core.RESTResponse;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.handler.dao.HandlerShowDao;
import com.cku.oa.handler.entity.HandlerShow;
import com.cku.oa.show.dao.ChildShowsDao;
import com.cku.oa.show.dao.MainShowsDao;
import com.cku.oa.show.dao.ShowHotelDao;
import com.cku.oa.show.dao.ShowJudgeDao;
import com.cku.oa.show.dao.ShowOfficeDao;
import com.cku.oa.show.dao.ShowReportInfoDao;
import com.cku.oa.show.dao.ShowScoreWebsiteDao;
import com.cku.oa.show.dao.ShowVenueDao;
import com.cku.oa.show.entity.ChildShows;
import com.cku.oa.show.entity.MainShows;
import com.cku.oa.show.entity.ShowHotel;
import com.cku.oa.show.entity.ShowJudge;
import com.cku.oa.show.entity.ShowMainJudgeInfo;
import com.cku.oa.show.entity.ShowOffice;
import com.cku.oa.show.entity.ShowScoreWebsite;
import com.cku.oa.show.entity.ShowVenue;
import com.cku.oa.show.service.ShowMainJudgeInfoService;
import com.cku.oa.sys.util.ImgUtil;
import com.cku.restful.v1.show.model.RestMainShows;
import com.cku.restful.v1.show.model.RestShowHotel;
import com.cku.restful.v1.show.model.RestShowJudge;
import com.cku.restful.v1.show.model.RestShowOffice;
import com.cku.restful.v1.show.model.RestShowVenue;
import com.cku.restful.v1.sys.utils.BeanUtil;
import com.cku.util.DateUtils;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

/**
 * 
 * <p>
 * Title:RestShowsService
 * </p>
 * <p>
 * Description: 犬展相关服务层
 * </p>
 * <p>
 * Company:
 * </p>
 * 
 * @author zhuoHeng
 * @date 2016年9月23日 上午10:22:01
 */
@Service
@Transactional(readOnly = true)
public class CkuShowsService {

	@Autowired
	public MainShowsDao mainShowsDao;
	@Autowired
	public ChildShowsDao childShowsDao;
	@Autowired
	public ShowJudgeDao showJudgeDao;
	@Autowired
	public ShowOfficeDao showOfficeDao;
	@Autowired
	public ShowHotelDao showHotelDao;
	@Autowired
	public ShowVenueDao showVenueDao;
	@Autowired
	public ShowScoreWebsiteDao showScoreWebsiteDao;
	@Autowired
	public ShowReportInfoDao showReportInfoDao;
	@Autowired
	private ShowMainJudgeInfoService showMainJudgeInfoService;
	@Autowired
	private HandlerShowDao handlerShowDao;

	/**
	 * 
	 * @Description：按报名截止时间倒叙查询近期3场犬展（CKU官网快速报名通道使用） @author: zhuoHeng
	 * @version: 2016年9月23日 上午10:35:50
	 */
	public List<RestMainShows> getMainShows() {

		List<MainShows> list = mainShowsDao.getMainShows();
		List<RestMainShows> restList = new ArrayList<>();
		for (MainShows mainShows : list) {
			RestMainShows restMainShows = new RestMainShows();
			BeanUtil.copyProperties(restMainShows, mainShows, "yyyy-MM-dd");
			List<ChildShows> childShowsList = childShowsDao.getByMainShowId(mainShows.getId());
			if (childShowsList.size() > 0) {
				restMainShows.setShowCode(childShowsList.get(0).getShowCode());
			}
			restList.add(restMainShows);
		}
		return restList;
	}

	/**
	 *
	 * @Description：CKU官网犬展介绍接口
	 * @author: laiguanglong
	 * @version: 2017年8月1日
	 */
	public RESTResponse introduce(HttpServletRequest request) {
		String pageNo = request.getParameter("pageNo");
		String pageSize = request.getParameter("pageSize");
		int pageNoInt = 1;
		int pageSizeInt = 10;
		try {
			if (StringUtils.isNotBlank(pageNo)) {
				pageNoInt = Integer.parseInt(pageNo);
			}
			if (StringUtils.isNotBlank(pageSize)) {
				pageSizeInt = Integer.parseInt(pageSize);
			}
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
		}
		List<MainShows> mainShowsList = mainShowsDao.findUnEndMainShows((pageNoInt - 1) * pageSizeInt, pageSizeInt);
		List<JSONObject> list = new ArrayList<JSONObject>();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");
		for (MainShows mainShows : mainShowsList) {
			JSONObject jo = new JSONObject();
			jo.put("mainShowId", getString(mainShows.getId()));
			jo.put("thumbnailImage", getString(ImgUtil.getImgUrl(request, mainShows.getThumbnailImage())));
			jo.put("closingTime", getString(sdf.format(mainShows.getClosingTime())));
			jo.put("scheduleImage", getString(ImgUtil.getImgUrl(request, mainShows.getScheduleImage())));
			jo.put("mainShowName", getString(mainShows.getMainShowName()));
			list.add(jo);
		}
		long count = mainShowsDao.countUnEndMainShows();
		return new RESTResponse("list", new PageBeanResult<JSONObject>(list, count));
	}

	/**
	 *
	 * @Description：CKU官网犬展回顾接口
	 * @author: laiguanglong
	 * @version: 2017年8月1日
	 */
	public RESTResponse review(HttpServletRequest request) {
		String pageNo = request.getParameter("pageNo");
		String pageSize = request.getParameter("pageSize");
		String year = request.getParameter("year");
		int pageNoInt = 1;
		int pageSizeInt = 10;
		try {
			if (StringUtils.isNotBlank(pageNo)) {
				pageNoInt = Integer.parseInt(pageNo);
			}
			if (StringUtils.isNotBlank(pageSize)) {
				pageSizeInt = Integer.parseInt(pageSize);
			}
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
		}
		//判断年份，只可显示17年之后数据
		if (StringUtils.isNotBlank(year)
				&& 2017 > Integer.parseInt(year)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "年份需大于等于2017年");
		}
		List<MainShows> mainShowsList = mainShowsDao.findCloseMainShows(year,(pageNoInt - 1) * pageSizeInt, pageSizeInt);
		List<JSONObject> list = new ArrayList<>();
		for (MainShows mainShows : mainShowsList) {
			JSONObject jo = new JSONObject();
			jo.put("mainShowId", getString(mainShows.getId()));
			jo.put("thumbnailImage", getString(ImgUtil.getImgUrl(request, mainShows.getThumbnailImage())));
			jo.put("mainShowName", getString(mainShows.getMainShowName()));
			list.add(jo);
		}
		long count = mainShowsDao.countCloseMainShows(year);
		return new RESTResponse("list", new PageBeanResult<JSONObject>(list, count));
	}

	/**
	 *
	 * @Description：CKU官网犬展介绍接口
	 * @author: laiguanglong
	 * @version: 2017年8月1日
	 */
	public RESTResponse handlerIntroduce(HttpServletRequest request) {
		String pageNo = request.getParameter("pageNo");
		String pageSize = request.getParameter("pageSize");
		int pageNoInt = 1;
		int pageSizeInt = 10;
		try {
			if (StringUtils.isNotBlank(pageNo)) {
				pageNoInt = Integer.parseInt(pageNo);
			}
			if (StringUtils.isNotBlank(pageSize)) {
				pageSizeInt = Integer.parseInt(pageSize);
			}
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
		}
		List<HandlerShow> showsList = handlerShowDao.findUnEndMainShows((pageNoInt - 1) * pageSizeInt, pageSizeInt);
		List<JSONObject> list = new ArrayList<JSONObject>();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");
		for (HandlerShow handlerShow : showsList) {
			JSONObject jo = new JSONObject();
			jo.put("mainShowId", getString(handlerShow.getId()));
			jo.put("thumbnailImage", getString(ImgUtil.getImgUrl(request, handlerShow.getWebsitePhoto())));
			jo.put("photoRoute", getString(handlerShow.getPhotoRoute()));
			jo.put("closingTime", getString(sdf.format(handlerShow.getRegisterEndTime())));
			jo.put("mainShowName", getString(handlerShow.getShowName()));
			list.add(jo);
		}
		long count = handlerShowDao.countUnEndMainShows();
		return new RESTResponse("list", new PageBeanResult<JSONObject>(list, count));
	}

	/**
	 *
	 * @Description：CKU官网犬展回顾接口
	 * @author: laiguanglong
	 * @version: 2017年8月1日
	 */
	public RESTResponse handlerReview(HttpServletRequest request) {
		String pageNo = request.getParameter("pageNo");
		String pageSize = request.getParameter("pageSize");
		int pageNoInt = 1;
		int pageSizeInt = 10;
		try {
			if (StringUtils.isNotBlank(pageNo)) {
				pageNoInt = Integer.parseInt(pageNo);
			}
			if (StringUtils.isNotBlank(pageSize)) {
				pageSizeInt = Integer.parseInt(pageSize);
			}
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
		}
		List<HandlerShow> showsList = handlerShowDao.findCloseMainShows((pageNoInt - 1) * pageSizeInt, pageSizeInt);
		List<JSONObject> list = new ArrayList<JSONObject>();
		for (HandlerShow handlerShow : showsList) {
			JSONObject jo = new JSONObject();
			jo.put("mainShowId", getString(handlerShow.getId()));
			jo.put("thumbnailImage", getString(ImgUtil.getImgUrl(request, handlerShow.getWebsitePhoto())));
			jo.put("photoRoute", getString(handlerShow.getPhotoRoute()));
			jo.put("mainShowName", getString(handlerShow.getShowName()));
			list.add(jo);
		}
		long count = handlerShowDao.countCloseMainShows();
		return new RESTResponse("list", new PageBeanResult<JSONObject>(list, count));
	}

	/**
	 *
	 * @Description：CKU官网主犬展信息接口
	 * @author: laiguanglong
	 * @version: 2017年8月1日
	 */
	public RESTResponse mainShowInfo(HttpServletRequest request) {
		String mainShowId = request.getParameter("mainShowId");
		MainShows mainShows = null;
		if (!StringUtils.isBlank(mainShowId)) {
			mainShows = mainShowsDao.get(mainShowId);
		}
		if (mainShows == null || StringUtils.isBlank(mainShows.getId())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "主犬展信息有误");
		}
		JSONObject jo = new JSONObject();
		jo.put("id", getString(mainShows.getId()));
		jo.put("handerImage", getString(ImgUtil.getImgUrl(request, mainShows.getHanderImage())));
		jo.put("dogNumMax", getString(mainShows.getDogNumMax()));
		SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy年MM月dd日HH时mm分");
		jo.put("closingTime", getString(sdf1.format(mainShows.getClosingTime())));
		SimpleDateFormat sdf2 = new SimpleDateFormat("yyyy年MM月dd日");
		jo.put("startTime", getString(sdf2.format(mainShows.getStartTime())));
		jo.put("endTime", getString(sdf2.format(mainShows.getEndTime())));
		ShowOffice po = null;
		if (StringUtils.isNotBlank(mainShows.getOrganizer())) {
			po = showOfficeDao.get(mainShows.getOrganizer());
		} else {
			po = showOfficeDao.get(mainShows.getOfficeId());
		}
		jo.put("showOffice", new RestShowOffice(po));
		String venueName = "";
		if (StringUtils.isNotBlank(mainShows.getShowVenueId())) {
			ShowVenue showVenue = showVenueDao.get(mainShows.getShowVenueId());
			venueName = showVenue.getVenueName();
		}
		jo.put("venueName", venueName);
		jo.put("undertaker", getString(mainShows.getUndertaker()));
		// 子犬展信息
		JSONArray childShowsList = new JSONArray();
		List<ChildShows> childList = childShowsDao.getByMainShowId(mainShowId);
		for (ChildShows childShows : childList) {
			JSONObject child = new JSONObject();
			child.put("showName", getString(childShows.getShowName()));
			child.put("fee", getString(childShows.getFee()));
			child.put("startTime", getString(DateUtils.dateToStr(childShows.getStartTime())));
			child.put("endTime", getString(DateUtils.dateToStr(childShows.getEndTime())));
			childShowsList.add(child);
		}
		jo.put("childShowsList", childShowsList);
		// 犬展成绩播报记录数量
		jo.put("showReportInfoSize", showReportInfoDao.getChildListByMainId(mainShows.getId()).size());
		// 犬展负责人 自办比赛是北京办公室负责人 合办是各个办公室负责人
		if ("1".equals(mainShows.getOrgnizeType())) {// 0：合办旧模式；1：自办;2:合办新模式
			ShowOffice bj = showOfficeDao.get("1");// id是1为北京办公室
			jo.put("director", bj.getManager());
		} else {
			jo.put("director", po.getManager());
		}

		return new RESTResponse("data", jo);
	}

	/**
	 *
	 * @Description：CKU官网犬展裁判安排接口
	 * @author: laiguanglong
	 * @version: 2017年8月1日
	 */
	public RESTResponse judgeInfo(HttpServletRequest request) {
		String mainShowId = request.getParameter("mainShowId");
		if (StringUtils.isBlank(mainShowId)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "主犬展id为空");
		}
		List<ChildShows> childList = childShowsDao.getByMainShowId(mainShowId);
		JSONArray childShowsList = new JSONArray();
		for (ChildShows childShows : childList) {
			JSONObject jo = new JSONObject();
			jo.put("showName", getString(childShows.getShowName()));
			jo.put("showRefereeRemarks", getString(childShows.getRefereeRemarks()));
			JSONArray showJudges = new JSONArray();
			if (StringUtils.isNotBlank(childShows.getReferee())) {
				for (String showJudgeId : childShows.getReferee().split(",")) {
					ShowJudge showJudge = showJudgeDao.get(showJudgeId);
					showJudges.add("(" + showJudge.getCitizenship() + ")" + showJudge.getJudgeName());
				}
			}
			jo.put("showJudge", showJudges);
			childShowsList.add(jo);
		}
		JSONObject data = new JSONObject();
		data.put("childShowsList", childShowsList);
		MainShows mainShows = mainShowsDao.get(mainShowId);
		JSONArray judgeList = new JSONArray();
		if (StringUtils.isNotBlank(mainShows.getReferee())) {
			for (String showJudgeId : mainShows.getReferee().split(",")) {
				ShowJudge showJudge = showJudgeDao.get(showJudgeId);
				if (showJudge != null) {
					RestShowJudge restShowJudge = new RestShowJudge(showJudge, request);
					// 是否为show特殊类型 0国秀裁判照 1默认裁判照 2特殊裁判照
					if ("0".equals(mainShows.getIsShowType())
							&& StringUtils.isNotBlank(showJudge.getShowTypeJudgePicture())) {
						restShowJudge.setJudgePicture(ImgUtil.getImgUrl(request, showJudge.getShowTypeJudgePicture()));
					} else if ("2".equals(mainShows.getIsShowType())) {
						// 查询该犬展有无裁判特殊照片
						ShowMainJudgeInfo showMainJudgeInfo = showMainJudgeInfoService
								.findByShowMainIdAndshowJudgeId(mainShowId, showJudgeId);
						if (showMainJudgeInfo != null
								&& StringUtils.isNotBlank(showMainJudgeInfo.getShowJudgePicture())) {
							restShowJudge.setJudgePicture(
									ImgUtil.getImgUrl(request, showMainJudgeInfo.getShowJudgePicture()));
						}
					}
					//将注册商标转换成Unicode编码返给前台
					if (StringUtils.isNotBlank(restShowJudge.getJudgeTitle())) {
						String judgeTitle = StringEscapeUtils.unescapeHtml4(restShowJudge.getJudgeTitle());
						restShowJudge.setJudgeTitle(judgeTitle);
					}
					judgeList.add(restShowJudge);
				}
			}
		}
		data.put("judgeList", judgeList);
		return new RESTResponse("data", data);
	}

	/**
	 *
	 * @Description：CKU官网犬展场馆信息接口
	 * @author: laiguanglong
	 * @version: 2017年8月1日
	 */
	public RESTResponse venueInfo(HttpServletRequest request) {
		String mainShowId = request.getParameter("mainShowId");
		if (StringUtils.isBlank(mainShowId)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "主犬展id为空");
		}
		MainShows mainShows = mainShowsDao.get(mainShowId);
		ShowVenue showVenue = new ShowVenue();
		if (StringUtils.isNotBlank(mainShows.getShowVenueId())) {
			showVenue = showVenueDao.get(mainShows.getShowVenueId());
		}
		return new RESTResponse("data", new RestShowVenue(showVenue));
	}

	/**
	 *
	 * @Description：CKU官网犬展酒店信息接口
	 * @author: laiguanglong
	 * @version: 2017年8月1日
	 */
	public RESTResponse hotelInfo(HttpServletRequest request) {
		String mainShowId = request.getParameter("mainShowId");
		if (StringUtils.isBlank(mainShowId)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "主犬展id为空");
		}
		MainShows mainShows = mainShowsDao.get(mainShowId);
		List<RestShowHotel> list = new ArrayList<RestShowHotel>();
		if (StringUtils.isNotBlank(mainShows.getHospitalId())) {
			String[] hospitalIds = mainShows.getHospitalId().split(",");
			for (String hospitalId : hospitalIds) {
				ShowHotel po = showHotelDao.get(hospitalId);
				list.add(new RestShowHotel(po));
			}
		}
		long count = list.size();
		return new RESTResponse("list", new PageBeanResult<RestShowHotel>(list, count));
	}

	private static String getString(String str) {
		return str == null ? "" : str;
	}

	/**
	 *
	 * @Description：CKU官网犬展积分信息接口
	 * @author: cuihuaiyu
	 * @version: 2018年01月22日
	 */
	public RESTResponse getShowScoreWebsite(HttpServletRequest request) {
		String year = request.getParameter("year");
		String group = request.getParameter("group");
		if (StringUtils.isBlank(group)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬展组别为空");
		}
		List<ShowScoreWebsite> list = showScoreWebsiteDao.getShowScoreWebsite(year, group);
		return new RESTResponse("list", list);
	}
}
