/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.thinkgem.jeesite.modules.sys.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.UnavailableSecurityManagerException;
import org.apache.shiro.session.InvalidSessionException;
import org.apache.shiro.session.Session;
import org.apache.shiro.subject.Subject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.sys.dao.OrgDao;
import com.cku.oa.sys.entity.Org;
import com.cku.oa.sys.entity.user.CardTypeEnum;
import com.cku.oa.sys.entity.user.IEnum;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.entity.user.MemberTypeEnum;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.partner.sys.model.PartnerLoginUserInfo;
import com.cku.restful.v1.cab.sys.service.UserService;
import com.cku.util.SysConfig;
import com.thinkgem.jeesite.common.service.BaseService;
import com.thinkgem.jeesite.common.utils.CacheUtils;
import com.thinkgem.jeesite.common.utils.SpringContextHolder;
import com.thinkgem.jeesite.modules.sys.dao.AreaDao;
import com.thinkgem.jeesite.modules.sys.dao.MenuDao;
import com.thinkgem.jeesite.modules.sys.dao.OfficeDao;
import com.thinkgem.jeesite.modules.sys.dao.RoleDao;
import com.thinkgem.jeesite.modules.sys.dao.UserDao;
import com.thinkgem.jeesite.modules.sys.entity.Area;
import com.thinkgem.jeesite.modules.sys.entity.Menu;
import com.thinkgem.jeesite.modules.sys.entity.Office;
import com.thinkgem.jeesite.modules.sys.entity.Role;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.security.SystemAuthorizingRealm.Principal;

import lombok.extern.slf4j.Slf4j;

/**
 * 用户工具类
 *
 * @author ThinkGem
 * @version 2013-12-05
 */
@Slf4j
public class UserUtils {
	
	private static Logger logger = LoggerFactory.getLogger(UserUtils.class);

	private static UserDao userDao = SpringContextHolder.getBean(UserDao.class);
	private static RoleDao roleDao = SpringContextHolder.getBean(RoleDao.class);
	private static MenuDao menuDao = SpringContextHolder.getBean(MenuDao.class);
	private static AreaDao areaDao = SpringContextHolder.getBean(AreaDao.class);
	private static OfficeDao officeDao = SpringContextHolder.getBean(OfficeDao.class);
	private static MemberService memberService = SpringContextHolder.getBean(MemberService.class);

	private static OrgDao orgDao = SpringContextHolder.getBean(OrgDao.class);
	public static final String USER_CACHE = "userCache";
	public static final String USER_CACHE_ID_ = "id_";
	public static final String USER_CACHE_LOGIN_NAME_ = "ln";
	public static final String USER_CACHE_LIST_BY_OFFICE_ID_ = "oid_";

	public static final String CACHE_ROLE_LIST = "roleList";
	public static final String CACHE_MENU_LIST = "menuList";
	public static final String CACHE_AREA_LIST = "areaList";
	public static final String CACHE_OFFICE_LIST = "officeList";
	public static final String CACHE_OFFICE_ALL_LIST = "officeAllList";

	/**
	 * 根据ID获取用户
	 *
	 * @param id
	 * @return 取不到返回null
	 */
	public static User get(String id) {
		User user = (User) CacheUtils.get(USER_CACHE, USER_CACHE_ID_ + id);
		if (user == null) {
			user = userDao.get(id);
			if (user == null) {
				return null;
			}
			user.setRoleList(roleDao.findList(new Role(user)));
			CacheUtils.put(USER_CACHE, USER_CACHE_ID_ + user.getId(), user);
			CacheUtils.put(USER_CACHE, USER_CACHE_LOGIN_NAME_ + user.getLoginName(), user);
		}
		return user;
	}

	/**
	 * 根据ID获取用户
	 *
	 * @param id
	 * @return 取不到返回null
	 */
	public static User getDBUser(String id) {
		User user = userDao.get(id);
		if (user == null) {
			return null;
		}
		user.setRoleList(roleDao.findList(new Role(user)));
		CacheUtils.put(USER_CACHE, USER_CACHE_ID_ + user.getId(), user);
		CacheUtils.put(USER_CACHE, USER_CACHE_LOGIN_NAME_ + user.getLoginName(), user);
		return user;
	}

	/**
	 * 根据登录名获取用户
	 *
	 * @param loginName
	 * @return 取不到返回null
	 */
	public static User getByLoginNameAndUserType(String loginName, String userType) {
		User user = userDao.getByLoginNameAndUserType(new User(null, loginName, userType));
		if (user == null) {
			return null;
		}
		user.setRoleList(roleDao.findList(new Role(user)));
		return user;
	}

	/**
	 * 通过名称和手机号获取登录账号
	 * 
	 * @author yuanshuai
	 * @date 2021/6/16 14:29
	 */
	public static User getBackUserByNameAndMobile(String name, String mobile) {
		return userDao.getBackUserByNameAndMobile(name, mobile);
	}

	public static User getUserInfoByToken(String token) {
		User user = userDao.getUserInfoByToken(token);
		if (user == null) {
			return null;
		}
		user.setRoleList(roleDao.findList(new Role(user)));
		return user;
	}

	/**
	 * 根据邮箱获取用户
	 *
	 * @param loginName
	 * @return 取不到返回null
	 */
	public static User getByEmailAndUserType(String email, String userType) {
		User temp = new User();
		temp.setEmail(email);
		temp.setUserType(userType);
		User user = userDao.getByEmailAndUserType(temp);
		if (user == null) {
			return null;
		}
		user.setRoleList(roleDao.findList(new Role(user)));
		return user;
	}

	/**
	 * 清除当前用户缓存
	 */
	public static void clearCache() {
		removeCache(CACHE_ROLE_LIST);
		removeCache(CACHE_MENU_LIST);
		removeCache(CACHE_AREA_LIST);
		removeCache(CACHE_OFFICE_LIST);
		removeCache(CACHE_OFFICE_ALL_LIST);
		UserUtils.clearCache(getUser());
	}

	/**
	 * 清除指定用户缓存
	 *
	 * @param user
	 */
	public static void clearCache(User user) {
		CacheUtils.remove(USER_CACHE, USER_CACHE_ID_ + user.getId());
		CacheUtils.remove(USER_CACHE, USER_CACHE_LOGIN_NAME_ + user.getLoginName());
		CacheUtils.remove(USER_CACHE, USER_CACHE_LOGIN_NAME_ + user.getOldLoginName());
		if (user.getOffice() != null && user.getOffice().getId() != null) {
			CacheUtils.remove(USER_CACHE, USER_CACHE_LIST_BY_OFFICE_ID_ + user.getOffice().getId());
		}
	}

	/**
	 * 获取当前用户
	 *
	 * @return 取不到返回 new User()
	 */
	public static User getUser() {
		Principal principal = getPrincipal();
		if (principal != null) {
			User user = get(principal.getId());
			if (user != null) {
				return user;
			}
			return new User();
		}
		// 如果没有登录，则返回实例化空的User对象。
		return new User();
	}

	/**
	 * 前台获取，当前登录的帐号 是会员帐号
	 *
	 * @return 取不到返回 new User();
	 */
	public static User getLoginUser() {
		ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder
				.getRequestAttributes();
		if (servletRequestAttributes != null && servletRequestAttributes.getRequest() != null
				&& StringUtils.isNotBlank(servletRequestAttributes.getRequest().getParameter("token"))) {
			// 宠爱王国token验证用户登录 by zhuoHeng 2016年12月1日17:19:45
			HttpServletRequest request = servletRequestAttributes.getRequest();
			String token = request.getParameter("token");
			String club = request.getHeader("club");
			User user = UserUtils.getUserByToken(token, club);
			return user;
			// end
		} else {
			Principal principal = getPrincipal();
			if (principal != null) {
				User user = get(principal.getId());
				if (user != null) {
					user.setLoginType(principal.getLoginType());
					return user;
				}
				return new User();
			}
		}
		// 如果没有登录，则返回实例化空的User对象。
		return new User();
	}

	/**
	 * 获取当前登录的合作机构
	 * 
	 * @return
	 */
	public static PartnerLoginUserInfo getLoginPartner() {
		ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder
				.getRequestAttributes();
		if (servletRequestAttributes != null && servletRequestAttributes.getRequest() != null
				&& StringUtils.isNotBlank(servletRequestAttributes.getRequest().getHeader("token"))) {
			HttpServletRequest request = servletRequestAttributes.getRequest();
			String token = request.getHeader("token");
			return UserService.validatePartner(token);
		} else {
			throw new ZAException(ZAErrorCode.ZA_ERC_INVALID_STATUS, ZAErrorCode.ZA_ERC_INVALID_STATUS_MSG);
		}
	}

	/**
	 * 前台获取，当前登录帐号对应的会员号
	 *
	 * @return 取不到返回 null
	 * @throws Exception
	 */
	public static Member getLoginMember() {
		User user = UserUtils.getLoginUser();
		if (user.getId() != null) {
			Member member = new Member();
			member.setUser(user);
			if ("3".equals(user.getUserType())) {
				Org org = orgDao.findByUserId(user.getId());
				return memberService.getByMemberCode(org.getMemberCode());
			} else {
				List<Member> members = memberService.findList(member);

				if (members.size() != 0) {
					members.get(0).setLoginType(user.getLoginType());
					return member = members.get(0);
				}
			}

		}
		log.error("验证地址：" + SysConfig.getInstance().getProperty("cab_validate_url"));
		throw new ZAException(ZAErrorCode.ZA_ERC_INVALID_STATUS, "会员绑定关系发生变化，请重新登录！");
	}

	public static String getMemberTypeDesc(Member member) {
		StringBuffer memberTypeDesc = new StringBuffer();
		Integer memberType = UserUtils.getMemberType(member);
		memberTypeDesc.append(IEnum.getMessage(MemberTypeEnum.class, memberType));
		return memberTypeDesc.toString();
	}

	public static String getMemberCardTypeDesc(Member member) {
		StringBuffer memberTypeDesc = new StringBuffer();
		Integer cardType = UserUtils.getMemberCardType(member);
		memberTypeDesc.append(IEnum.getMessage(CardTypeEnum.class, cardType));
		return memberTypeDesc.toString();
	}

	/**
	 * 转换当前会员卡片类型 0:"普卡" 10: "银卡" 20:"金卡" 30:"铂金卡" 40:"紫金卡"
	 * 
	 * @return 取不到返回 0
	 * @throws Exception
	 */
	public static Integer getMemberCardType(String memberCode) {
		Member member = memberService.findMemberAndKennelByMemberCode(memberCode);
		return getMemberCardType(member);
	}

	/**
	 * 转换当前会员爱宠和专繁类型 10:"爱宠" 20: "专繁"
	 * 
	 * @return 取不到返回 10
	 * @throws Exception
	 */
	public static Integer getMemberType(Member member) {
		Integer memberType = MemberTypeEnum.BASICS_PET.code();
		if (StringUtils.isNotBlank(member.getMemberCardType())) {
			memberType = Integer.valueOf(member.getMemberCardType().substring(0, 2));
		} else {
			throw new ZAException(ZAErrorCode.ZA_ERC_NO_MEMBER, "获取会员类型错误");
		}
		return memberType;
	}

	/**
	 * 转换当前会员爱宠和专繁类型 10:"爱宠" 20: "专繁"
	 * 
	 * @return 取不到返回 10
	 * @throws Exception
	 */
	public static Integer getMemberType(String memberCode) {
		Member member = memberService.findMemberAndKennelByMemberCode(memberCode);
		return getMemberType(member);
	}

	/**
	 * 转换当前会员卡片类型 0:"普卡" 10: "银卡" 20:"金卡" 30:"铂金卡" 40:"紫金卡"
	 * 
	 * @return 取不到返回 0
	 * @throws Exception
	 */
	public static Integer getMemberCardType(Member member) {
		Integer cardType = CardTypeEnum.DEFAULT_CARD.code();
		// 会员过期时间为空 均视为 普卡会员
		if (member.getMemberEndDate() == null) {
			cardType = CardTypeEnum.DEFAULT_CARD.code();
		}
		if (StringUtils.isNotBlank(member.getMemberCardType())) {
			cardType = Integer.valueOf(member.getMemberCardType().substring(2));
		} else {
			throw new ZAException(ZAErrorCode.ZA_ERC_NO_MEMBER, "获取会员类型错误");
		}
		return cardType;
	}

	/**
	 * 获取当前用户角色列表
	 *
	 * @return
	 */
	public static List<Role> getRoleList() {
		@SuppressWarnings("unchecked")
		List<Role> roleList = (List<Role>) getCache(CACHE_ROLE_LIST);
		if (roleList == null) {
			User user = getUser();
			if (user.isAdmin()) {
				roleList = roleDao.findAllList(new Role());
			} else {
				Role role = new Role();
				role.getSqlMap().put("dsf", BaseService.dataScopeFilter(user.getCurrentUser(), "o", "u"));
				roleList = roleDao.findList(role);
			}
			putCache(CACHE_ROLE_LIST, roleList);
		}
		return roleList;
	}

	/**
	 * 获取当前用户授权菜单 权限目前俩套体系，一套基于角色授权，一套是基于用户授权，记录在用户表的permissions字段
	 *
	 * @return
	 */
	public static List<Menu> getMenuList() {
		@SuppressWarnings("unchecked")
		List<Menu> menuList = (List<Menu>) getCache(CACHE_MENU_LIST);
		if (menuList == null) {
			User user = getUser();
			if (user.isAdmin()) {
				menuList = menuDao.findAllList(new Menu());
			} else {
				// 角色授权的情况
				Menu m = new Menu();
				m.setUserId(user.getId());
				menuList = menuDao.findByUserId(m);

				// 用户授权的情况
				String permissions = user.getPermissions();
				if (permissions != null && !permissions.equals("")) {
					List<Menu> allMenu = menuDao.findAllList(new Menu());
					for (Menu menu : allMenu) {
						for (String permission : permissions.split(",")) {
							if (menu.getId().equals(permission)) {
								menuList.add(menu);
							}
						}
					}
				}
			}
			putCache(CACHE_MENU_LIST, menuList);
		}

		// 通过当前用户的permissions字段判断还有哪些权限
		return menuList;
	}

	/**
	 * 获取当前用户授权的区域
	 *
	 * @return
	 */
	public static List<Area> getAreaList() {
		@SuppressWarnings("unchecked")
		List<Area> areaList = (List<Area>) getCache(CACHE_AREA_LIST);
		if (areaList == null) {
			areaList = areaDao.findAllList(new Area());
			putCache(CACHE_AREA_LIST, areaList);
		}
		return areaList;
	}

	/**
	 * 获取所有的省的信息
	 */
	public static List<Area> getProvince() {
		List<Area> list = UserUtils.getAreaList();
		List<Area> mapList = new ArrayList<Area>();
		String rootId = "1";
		for (int i = 0; i < list.size(); i++) {
			Area e = list.get(i);
			if (e.getParentId().equals(rootId)) {
				mapList.add(e);
			}
		}
		return mapList;
	}

	/**
	 * 获取所有的城市
	 */
	public static List<Area> getCity() {
		List<Area> list = UserUtils.getAreaList();
		List<Area> mapList = new ArrayList<Area>();
		String cityType = "3";
		for (int i = 0; i < list.size(); i++) {
			Area e = list.get(i);
			// e.getType 等于3，为城市
			if (e.getType().equals(cityType)) {
				mapList.add(e);
			}
		}
		return mapList;
	}

	/**
	 * 根据区域id得到区域
	 *
	 * @param id
	 * @return
	 */
	public static Area getAreaById(String id) {
		Area result = null;
		List<Area> list = UserUtils.getAreaList();
		for (Area area : list) {
			if (id.equals(area.getId())) {
				result = area;
				break;
			}
		}
		return result;
	};

	/**
	 * 根据区域name得到区域
	 *
	 * @param name
	 * @return
	 */
	public static Area getAreaByName(String name) {
		Area result = null;
		List<Area> list = UserUtils.getAreaList();
		for (Area area : list) {
			if (name.equals(area.getName())) {
				result = area;
				break;
			}
		}
		return result;
	};

	/**
	 * 获取指定省下的所有城市
	 */
	public static List<Area> getCityByProvince(String provinceName) {

		List<Area> list = UserUtils.getAreaList();
		String provinceId = "";
		for (int i = 0; i < list.size(); i++) {
			Area e = list.get(i);
			if (e.getName().equals(provinceName)) {
				provinceId = e.getId();
				break;
			}
		}

		List<Area> mapList = new ArrayList<Area>();
		for (int i = 0; i < list.size(); i++) {
			Area e = list.get(i);
			if (e.getParentId().equals(provinceId)) {
				mapList.add(e);
			}
		}
		return mapList;
	}

	/**
	 * 获取当前用户有权限访问的部门
	 *
	 * @return
	 */
	public static List<Office> getOfficeList() {
		@SuppressWarnings("unchecked")
		List<Office> officeList = (List<Office>) getCache(CACHE_OFFICE_LIST);
		if (officeList == null) {
			User user = getUser();
			if (user.isAdmin()) {
				officeList = officeDao.findAllList(new Office());
			} else {
				Office office = new Office();
				office.getSqlMap().put("dsf", BaseService.dataScopeFilter(user, "a", ""));
				officeList = officeDao.findList(office);
			}
			putCache(CACHE_OFFICE_LIST, officeList);
		}
		return officeList;
	}

	/**
	 * 根据名称获取部门
	 *
	 * @param name
	 * @return
	 */
	public static Office getOfficeByName(String name) {
		Office result = null;
		List<Office> officeList = getOfficeList();
		for (Office o : officeList) {
			if (name.equals(o.getName())) {
				result = o;
				break;
			}
		}
		return result;
	}

	/**
	 * 获取当前用户有权限访问的部门
	 *
	 * @return
	 */
	public static List<Office> getOfficeAllList() {
		@SuppressWarnings("unchecked")
		List<Office> officeList = (List<Office>) getCache(CACHE_OFFICE_ALL_LIST);
		if (officeList == null) {
			officeList = officeDao.findAllList(new Office());
		}
		return officeList;
	}

	/**
	 * 获取授权主要对象
	 */
	public static Subject getSubject() {
		return SecurityUtils.getSubject();
	}

	/**
	 * 获取当前登录者对象
	 */
	public static Principal getPrincipal() {
		try {
			Subject subject = SecurityUtils.getSubject();
			Principal principal = (Principal) subject.getPrincipal();
			if (principal != null) {
				return principal;
			}
			// subject.logout();
		} catch (UnavailableSecurityManagerException e) {

		} catch (InvalidSessionException e) {

		}
		return null;
	}

	public static Session getSession() {
		try {
			Subject subject = SecurityUtils.getSubject();
			Session session = subject.getSession(false);
			if (session == null) {
				session = subject.getSession();
			}
			if (session != null) {
				return session;
			}
			// subject.logout();
		} catch (InvalidSessionException e) {

		}
		return null;
	}

	// ============== User Cache ==============

	public static Object getCache(String key) {
		return getCache(key, null);
	}

	public static Object getCache(String key, Object defaultValue) {
		// Object obj = getCacheMap().get(key);
		Object obj = getSession().getAttribute(key);
		return obj == null ? defaultValue : obj;
	}

	public static void putCache(String key, Object value) {
		// getCacheMap().put(key, value);
		getSession().setAttribute(key, value);
	}

	public static void removeCache(String key) {
		// getCacheMap().remove(key);
		getSession().removeAttribute(key);
	}

	// public static Map<String, Object> getCacheMap(){
	// Principal principal = getPrincipal();
	// if(principal!=null){
	// return principal.getCacheMap();
	// }
	// return new HashMap<String, Object>();
	// }

	/**
	 *
	 * @Description：通过token获取宠爱王国登录账号对应的cku账号信息
	 * @author: zhuoHeng
	 * @version: 2016年11月8日 下午3:17:57
	 */
	public static User getUserByToken(String token, String club) {
		User user = new User();
		Map<String, String> map = UserService.validate(token, club);
		String ckuUserId = map.get("ckuUserId");
		String loginType = map.get("loginType");
		if (!StringUtils.isBlank(ckuUserId) && !"-1".equals(ckuUserId)) {
			user = UserUtils.getDBUser(ckuUserId);
			if (user != null) {
				user.setLoginType(loginType);
			}
		}
		return user;
	}

}
