package com.cku.oa.org.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.org.dao.OrgSamplingDao;
import com.cku.oa.org.entity.OrgSampling;
import com.cku.oa.org.vo.OrgSamplingDetailListVo;
import com.cku.oa.org.vo.OrgSamplingGroupVo;
import com.cku.oa.sampling.entity.Sampling;
import com.cku.oa.sampling.enums.SamplingDistributionChannelEnum;
import com.cku.oa.sampling.service.SamplingService;
import com.cku.oa.sys.entity.Org;
import com.cku.oa.sys.service.OrgService;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 俱乐部采样包发放Service
 *
 * @author yuanshuai
 * @version 2023-05-28
 */
@Service
@Transactional(readOnly = true)
public class OrgSamplingService extends CrudService<OrgSamplingDao, OrgSampling> {

    @Autowired
    private OrgService orgService;
    @Autowired
    private SamplingService samplingService;

    public OrgSampling get(String id) {
        return super.get(id);
    }

    public List<OrgSampling> findList(OrgSampling orgSampling) {
        return super.findList(orgSampling);
    }

    public Page<OrgSampling> findPage(Page<OrgSampling> page, OrgSampling orgSampling) {
        return super.findPage(page, orgSampling);
    }

    /**
     * 分组分页
     *
     * @author yuanshuai
     * @date 2023/5/28 15:50
     */
    public Page<OrgSamplingGroupVo> findGroupPage(Page<OrgSamplingGroupVo> page, OrgSamplingGroupVo groupVo) {
        int pageNo = page.getPageNo();
        int pageSize = page.getPageSize();
        groupVo.setPage(page);
        Org org = orgService.get(groupVo.getOrgId());
        List<OrgSamplingGroupVo> voList = dao.findGroupList(groupVo);
        voList.forEach(s -> {
            s.setOrgName(org.getName());
        });
        page.setList(voList);
        if ((long) (pageNo - 1) * pageSize >= page.getCount()) {
            page.setList(new ArrayList<>(0));
        }
        return page;
    }

    /**
     * 详情分页
     *
     * @author yuanshuai
     * @date 2023/5/28 17:40
     */
    public Page<OrgSamplingDetailListVo> findDetailListPage(Page<OrgSamplingDetailListVo> page, OrgSamplingDetailListVo listVo) {
        int pageNo = page.getPageNo();
        int pageSize = page.getPageSize();
        listVo.setPage(page);
        Org org = orgService.get(listVo.getOrgId());
        List<OrgSamplingDetailListVo> voList = dao.findDetailList(listVo);
        voList.forEach(s -> {
            s.setOrgName(org.getName());
        });
        page.setList(voList);
        if ((long) (pageNo - 1) * pageSize >= page.getCount()) {
            page.setList(new ArrayList<>(0));
        }
        return page;
    }

    @Transactional(readOnly = false)
    public void save(OrgSampling orgSampling) {
        super.save(orgSampling);
    }

    /**
     * 添加
     *
     * @author yuanshuai
     * @date 2023/5/28 11:00
     */
    @Transactional(readOnly = false)
    public void add(OrgSampling request) {
        //参数校验
        if (StringUtils.isBlank(request.getAddress())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "地址不能为空");
        }
        if (Objects.isNull(request.getPostDate())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "发放日期不能为空");
        }
        //俱乐部
        Org org = orgService.get(request.getOrgId());
        if (Objects.isNull(org) || StringUtils.isBlank(org.getId())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "合作机构id有误");
        }
        //全部未使用列表
        List<Sampling> unuseList = samplingService.findUnuseList();
        //添加采样包列表
        List<String> samplingCodeList = Arrays.stream(request.getSamplingCode().split(","))
                .collect(Collectors.toList());
        //数据校验
        samplingCodeList.forEach(s -> {
            if (unuseList.stream()
                    .noneMatch(t -> s.equals(t.getSamplingCode()))) {
                throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "采样包" + s + "已使用，请重新选择采样包");
            }
        });
        //生成批次号
        final String batchNum = String.valueOf(System.currentTimeMillis());
        List<OrgSampling> addList = samplingCodeList.stream()
                .map(s -> {
                    OrgSampling orgSampling = new OrgSampling();
                    orgSampling.setBatchNum(batchNum);
                    orgSampling.setOrgId(request.getOrgId());
                    orgSampling.setAddress(request.getAddress());
                    orgSampling.setPostDate(request.getPostDate());
                    orgSampling.setSamplingCode(s);
                    orgSampling.preInsert();
                    return orgSampling;
                })
                .collect(Collectors.toList());
        dao.batchInsert(addList);
        //批量更新邮寄状态
        samplingService.batchPost(unuseList.stream()
                        .filter(t -> samplingCodeList.stream().anyMatch(s -> s.equals(t.getSamplingCode())))
                        .collect(Collectors.toList()),
                SamplingDistributionChannelEnum.ORG,
                org.getMemberCode(),
                org.getName(),
                null);

    }

    /**
     * 删除
     *
     * @author yuanshuai
     * @date 2023/5/29 10:55
     */
    @Transactional(readOnly = false)
    public void delete(OrgSampling request) {
        OrgSampling orgSampling = get(request.getId());
        if (Objects.isNull(orgSampling) || StringUtils.isBlank(orgSampling.getId())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "ID有误");
        }
        Sampling sampling = samplingService.findBySamplingCode(orgSampling.getSamplingCode());
        if (Objects.isNull(sampling) || StringUtils.isBlank(sampling.getId())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "采样包编号有误");
        }
        if (1 == sampling.getBindType()) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "采样包已绑定，不允许删除");
        }
        //删除俱乐部采样包记录
        dao.delete(request);
        //解绑采样包信息
        samplingService.unbindMemberInfo(sampling);
    }

    /**
     * 批量删除
     *
     * @author yuanshuai
     * @date 2023/5/29 10:55
     */
    @Transactional(readOnly = false)
    public void batchDelete(OrgSampling request) {
        List<OrgSamplingDetailListVo> detailList = dao.findDetailList(OrgSamplingDetailListVo.builder()
                .orgId(request.getOrgId())
                .batchNum(request.getBatchNum())
                .build());
        if (CollectionUtils.isEmpty(detailList)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "没有找到该批采样包");
        }
        List<Sampling> samplingList = samplingService.findListBySamplingCodeList(detailList.stream()
                .map(OrgSamplingDetailListVo::getSamplingCode)
                .collect(Collectors.toList()));
        if (CollectionUtils.isEmpty(samplingList)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "没有找到该批采样包");
        }
        if (samplingList.stream().anyMatch(s -> 1 == s.getBindType())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该批采样包已有绑定记录，不允许删除");
        }
        //根据批次号删除俱乐部采样包信息
        dao.deleteByBatchNum(request.getBatchNum());
        //解绑采样包信息
        samplingService.batchUnbindMemberInfo(samplingList);
    }
}