/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.dog.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.dao.DogBreedCertifiedDao;
import com.cku.oa.dog.dao.DogDao;
import com.cku.oa.dog.dao.DogNewbornDao;
import com.cku.oa.dog.dao.DogSemenDao;
import com.cku.oa.dog.entity.*;
import com.cku.oa.finance.dao.PaymentOrderDetailDao;
import com.cku.oa.finance.entity.PaymentOrder;
import com.cku.oa.finance.entity.PaymentOrderDetail;
import com.cku.oa.finance.service.PaymentOrderService;
import com.cku.oa.finance.service.SaPaymentDetailTotalService;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.kennel.service.KennelService;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.LoginServerService;
import com.cku.oa.sys.service.RemindService;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.oa.sys.util.MemberUtil;
import com.cku.oa.sys.zalog.SysContext;
import com.cku.restful.v1.dog.service.RestDogService;
import com.cku.util.Debugger;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import net.sf.json.JsonConfig;
import net.sf.json.util.CycleDetectionStrategy;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 配种证明Service
 * 
 * @author zhuoHeng
 * @version 2016-08-02
 */
@Service
@Transactional(readOnly = true)
public class DogNewbornService extends CrudService<DogNewbornDao, DogNewborn> {

	@Autowired
	private RemindService remindService;
	@Autowired
	private MemberService memberService;
	@Autowired
	private KennelService kennelService;
	@Autowired
	private DogDao dogDao;
	@Autowired
	private DogDnaService dogDnaService;
	@Autowired
	private DogTypeService dogTypeService;
	@Autowired
	private PaymentOrderService paymentOrderService;
	@Autowired
	private PaymentOrderDetailDao paymentOrderDetailDao;
	@Autowired
	private BusinessBanService businessBanService;
	@Autowired
	private RestDogService restDogService;
	@Autowired
	private DogBreedCertifiedDao dogBreedCertifiedDao;
	@Autowired
	public DogSemenDao dogSemenDao;
	@Autowired
	public DogSemenChangeService dogSemenChangeService;
	@Autowired
	public LoginServerService loginServerService;
	@Autowired
	public SaPaymentDetailTotalService saPaymentDetailTotalService;
	@Autowired
	public DogNewbornDao dogNewbornDao;
	@Autowired
	public DogDnaArchiveService dogDnaArchiveService;

	public DogNewborn get(String id) {
		return super.get(id);
	}

	public List<DogNewborn> findList(DogNewborn dogNewborn) {
		if (StringUtils.isBlank(dogNewborn.getDogBreed())) {
			// 根据当前登录端查询犬种
			dogNewborn.getSqlMap().put("dogBreedMap", loginServerService.dataScopeFilter());
		} else {
			dogNewborn.getSqlMap().put("dogBreedMap", " ='" + dogNewborn.getDogBreed() + "'");
		}
		return super.findList(dogNewborn);
	}

	public Page<DogNewborn> findPage(Page<DogNewborn> page, DogNewborn dogNewborn) {
		if (StringUtils.isBlank(dogNewborn.getDogBreed())) {
			// 根据当前登录端查询犬种
			dogNewborn.getSqlMap().put("dogBreedMap", loginServerService.dataScopeFilter());
		} else {
			dogNewborn.getSqlMap().put("dogBreedMap", " ='" + dogNewborn.getDogBreed() + "'");
		}
		return super.findPage(page, dogNewborn);
	}

	@Transactional(readOnly = false)
	public void save(DogNewborn dogNewborn) {
		super.save(dogNewborn);
	}

	@Transactional(readOnly = false)
	public void delete(DogNewborn dogNewborn) {
		super.delete(dogNewborn);
	}

	/**
	 *
	 * @Description：修改配种证明信息
	 * @author: zhuoHeng
	 * @version: 2016年8月11日 下午2:53:11
	 */
	@Transactional(readOnly = false)
	public void edit(DogNewborn dogNewborn) throws ZAException {
		dogNewborn.setUpdateDate(new Date());
		dao.edit(dogNewborn);
	}

	/**
	 *
	 * @Description：审核配种证明申请
	 * @author: zhuoHeng
	 * @version: 2016年8月15日 下午6:01:58
	 */
	@Transactional(readOnly = false)
	public void changeStatus(DogNewborn dogNewborn) throws ZAException {
		DogNewborn newBron = super.get(dogNewborn.getId());
		SysContext sysContext = new SysContext(newBron.getPaymentState(), newBron.getReviewState(),
				newBron.getProcessState() == null ? "0" : newBron.getProcessState());
		if ("1".equals(dogNewborn.getReviewState())) {
			sysContext.confirmSuccess(sysContext, "配种证明", newBron.getMemberCode(), "dog_newborn", newBron.getId(),
					newBron.getRunningNumber(), (str) -> {

					});
			if (!newBron.getUrgent().equals("1")) {
				// TODO 财务收入确认时点
				saPaymentDetailTotalService.financeConfirmTime(newBron.getId(), "dog_newborn",
						newBron.getRunningNumber());
			}

		} else if ("2".equals(dogNewborn.getReviewState())) {
			sysContext.confirmFail(sysContext, "配种证明", dogNewborn.getReviewRemarks(), newBron.getMemberCode(),
					"dog_newborn", newBron.getId(), newBron.getRunningNumber(), (str) -> {

					});
		}
		dao.changeStatus(dogNewborn);
		sendMsg(dogNewborn);
	}

	/**
	 *
	 * @description: 根据配种证明编号查询
	 * @author: laiguanglong
	 * @date: 2016年8月16日 下午7:23:02
	 */
	public DogNewborn getBybreedCertifiedCode(String breedCertifiedCode) {
		return dao.getByBreedCertifiedCode(breedCertifiedCode);
	}

	/**
	 *
	 * @description: 获取已付款待审核的配种证明申请记录数
	 * @author: laiguanglong
	 * @date: 2016年8月29日 下午4:52:58
	 */
	public int getAuditCount() {
		DogNewborn dogNewborn = new DogNewborn();
		// 已付款
		dogNewborn.setPaymentState("2");
		// 未审核
		dogNewborn.setReviewState("0");
		return dao.listCount(dogNewborn);
	}

	/**
	 *
	 * @Description：获取已付款待审核的配种证明申请记录数（藏獒） @author: zhuoHeng
	 * @version: 2016年9月3日 下午7:28:33
	 */
	public int getAuditCount(String dogBreed) {
		DogNewborn dogNewborn = new DogNewborn();
		// 已付款
		dogNewborn.setPaymentState("2");
		// 未审核
		dogNewborn.setReviewState("0");
		// 藏獒
		dogNewborn.setDogBreed(dogBreed);
		return dao.listCount(dogNewborn);
	}

	/**
	 *
	 * @Description：给用户发送消息，根据审核状态和缴费状态来判断发送内容
	 * @author: zhuoHeng
	 * @version: 2016年9月1日 下午3:32:44
	 */
	public void sendMsg(DogNewborn dogNewborn) {
		String reviewState = dogNewborn.getReviewState();
		if ("0".equals(reviewState) || "1".equals(reviewState) || "2".equals(reviewState)) {
			String content = "";
			// 未审核
			if ("0".equals(reviewState)) {
				// 已提交
				if ("9".equals(dogNewborn.getPaymentState())) {
					SimpleDateFormat formatter = new SimpleDateFormat("yyyy年MM月dd日HH时mm分");
					String dateString = formatter.format(new Date());
					content = "您于" + dateString + "提交配种证明申请，配种证明编号为“" + dogNewborn.getBreedCertifiedCode()
							+ "”，请您务必于当日24时前完成缴费，过期订单失效哦。";
					remindService.addRemind(content, dogNewborn.getFdogOwnerCode());
				}
			} else if ("1".equals(reviewState)) {
				content = "您提交的配种证明编号为“" + dogNewborn.getBreedCertifiedCode() + "”审核通过。";
				remindService.addRemind(content, dogNewborn.getMdogOwnerCode());
				content = "配种证明编号为“" + dogNewborn.getBreedCertifiedCode() + "”审核通过，请及时办理出生纸。";
				remindService.addRemind(content, dogNewborn.getFdogOwnerCode());
			} else if ("2".equals(reviewState)) {
				content = "您提交的配种证明编号为“" + dogNewborn.getBreedCertifiedCode() + "”审核未通过，请进入会员中心查看未通过原因，修改后重新提交。";
				remindService.addRemind(content, dogNewborn.getMdogOwnerCode());
				content = "配种证明编号为“" + dogNewborn.getBreedCertifiedCode() + "”审核未通过，请联系公犬主人修改后重新提交。";
				remindService.addRemind(content, dogNewborn.getFdogOwnerCode());
			}
		}
	}

	/**
	 *
	 * @Description：根据母犬血统证书号查询繁殖次数
	 * @author: zhuoHeng
	 * @version: 2016年9月1日 下午4:12:15
	 */
	public String getBreedNumber(String femalePedigreeCertified) {
		List<DogNewborn> BreedMessage = dao.getBreedNumber(femalePedigreeCertified);
		for (DogNewborn model : BreedMessage) {
			model.setReviewState(DictUtils.getDictLabel(model.getReviewState(), "review_state", ""));
		}
		JsonConfig jsonConfig = new JsonConfig();
		jsonConfig.setCycleDetectionStrategy(CycleDetectionStrategy.LENIENT);
		JSONArray json = JSONArray.fromObject(BreedMessage, jsonConfig);
		return json.toString();
	}

	/**
	 *
	 * @Description：配种证明业务信息保存
	 * @author: zhuoHeng
	 * @version: 2016年9月6日 下午5:23:31
	 */
	@Transactional(readOnly = false)
	public void orderSave(DogNewborn dogNewborn) {
		// oa订单录入价格取收费项详情中的实收价格
		List<PaymentOrderDetail> orderDetailList = paymentOrderDetailDao
				.getOrderByRunningNum(dogNewborn.getRunningNumber(), dogNewborn.getMemberCode());
		PaymentOrderDetail paymentOrderDetail = null;
		for (PaymentOrderDetail orderDetail : orderDetailList) {
			if (dogNewborn.getPaymentChargingItemId().equals(orderDetail.getChargingItemId())) {
				paymentOrderDetail = orderDetail;
				break;
			}
		}
		Member motherMember = memberService.getByMemberCode(dogNewborn.getFdogOwnerCode());
		dogNewborn.setArea(motherMember.getProvince() + motherMember.getCity());// 公犬主所在地区
		dogNewborn.setPrice(paymentOrderDetail.getPrice());
		dogNewborn.setBreedCertifiedCode(getBreedCertifiedCode(dogNewborn));// 配种证明编号
		dogNewborn.setReviewState("0");// 数据为未审核
		dogNewborn.setPaymentState("2");
		dogNewborn.setPrintType("1");// 协会打印
		PaymentOrder paymentOrder = paymentOrderService.getByOrderCode(dogNewborn.getRunningNumber());
		dogNewborn.setPaymentTime(paymentOrder.getCreateDate());
		dogNewborn.setPaymentTime(new Date());
		dogNewborn.setMaleDogOwnerTopup("是");
		dogNewborn.setFemaleDogOwnerTopup("是");
		dogNewborn.setBatchFee("已交");
		Dog maleDog = StringUtils.isBlank(dogNewborn.getMalePedigreeCertified()) ? null
				: dogDao.getByPedigreeCertifiedCode(dogNewborn.getMalePedigreeCertified());
		Dog femaleDog = StringUtils.isBlank(dogNewborn.getFemalePedigreeCertified()) ? null
				: dogDao.getByPedigreeCertifiedCode(dogNewborn.getFemalePedigreeCertified());// 母犬
		// zhangxiang 2019-11-07
		// 贵宾犬、腊肠犬两个犬种不同体型配种证明登记时，幼犬体型需要自动随妈妈的体型,因父犬母犬为同一犬种，取父犬母犬都可以
		if (maleDog.getDogBreed().startsWith("172") || maleDog.getDogBreed().startsWith("148")) {
			dogNewborn.setDogBreed(femaleDog.getDogBreed());
		} else {
			dogNewborn.setDogBreed(maleDog.getDogBreed());
		}
		dogNewborn.setUrgent("0");// 加急（否）
		dogNewborn.setCreateDate(new Date());
		dogNewborn.setUpdateDate(new Date());
		// OA配种证明订单录入业务校验
		oaOrderSaveValidateBusiness(dogNewborn);
		dogNewborn.setIsNewRecord(true);
		super.save(dogNewborn);
	}

	/**
	 *
	 * @description: 生成配种证明编号
	 * @author: laiguanglong
	 * @date: 2016年10月23日 下午3:12:23
	 */
	public String getBreedCertifiedCode(DogNewborn dogNewborn) {
		// Date breedTime = dogNewborn.getBreedDate();
		StringBuffer breedCertifiedNum = new StringBuffer();
		SimpleDateFormat sdf = new SimpleDateFormat("yyMMdd");
		/*
		 * 需求为配种证明编号按缴费日期编号，考虑到缴费日期在配种证明编号生成之后才会创建，此处拿不到该数据。
		 * 因每日24时会对未缴费订单进行删除，所以缴费日期必是当前日期。 zhuoHeng
		 */
		breedCertifiedNum.append(sdf.format(new Date()));
		// 排除DC会员号对配种证明编号的影响 by yuanshuai 2018年2月2日10:42:51
		breedCertifiedNum.append(dogNewborn.getMdogOwnerCode().replace("DC", "").replace("MC", ""));
		String maxBreedCertifiedNum = dao.getMaxBreedCertifiedNum(breedCertifiedNum.toString());
		Long newBreedCertifiedNum = null;
		if (maxBreedCertifiedNum != null && !"".equals(maxBreedCertifiedNum)) {
			newBreedCertifiedNum = new Long(maxBreedCertifiedNum) + 1;
			breedCertifiedNum.setLength(0);
			breedCertifiedNum.append(newBreedCertifiedNum.toString());
		} else {
			breedCertifiedNum.append("01");
		}
		return breedCertifiedNum.toString();
	}

	public void checkFatherMember(String fatherDogMemberCode) {
		Member fatherMember = memberService.getByMemberCode(fatherDogMemberCode);
		if (!"professional".equals(fatherMember.getMemberType())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬主不是专业会员");
		}
		Debugger.doAssert(MemberUtil.isActiveProfessionalMember(fatherMember), ZAErrorCode.ZA_VALID_FAILED,
				"父犬主不是有效专业会员");
	}

	public void checkMotherMember(String motherDogMemberCode) {
		Member motherMember = memberService.getByMemberCode(motherDogMemberCode);

		if (!"professional".equals(motherMember.getMemberType())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬主不是专业会员");
		}
		// 缺少父犬主、母犬主专业会员是否有效校验
		Debugger.doAssert(MemberUtil.isActiveProfessionalMember(motherMember), ZAErrorCode.ZA_VALID_FAILED,
				"母犬主不是有效专业会员");
	}

	public void checkMotherMemberKennel(String motherDogMemberCode) {
		Kennel kennel = kennelService.getKennelByMemberCode(motherDogMemberCode);
		if (kennel == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬主没有犬舍");
		}

		// 母犬主犬舍是否有效校验
		boolean flag = false;
		if (kennel != null) {
			Date endTime = kennel.getEndTime();
			if (endTime != null) {
				if ((endTime.getTime() + 86399000) > System.currentTimeMillis()) {// 有效
					flag = true;
				}
			}
		}
		if (!flag) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬主犬舍无效");
		}
		if (!"1".equals(kennel.getReviewState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬主犬舍必须审核通过才能申请配种证明");
		}
	}

	/**
	 *
	 * @description: 配种证明业务校验
	 * @author: laiguanglong
	 * @date: 2016年9月13日 下午4:03:23
	 */
	public void validateBusiness(DogNewborn po) {
		// 冷冻精子配种不校验公犬犬主是否有有效专业会员
		if (StringUtils.isBlank(po.getSemenCode())) {
			checkFatherMember(po.getMdogOwnerCode());
		}
		checkMotherMember(po.getFdogOwnerCode());

		/*
		 * 0此方法涉及包括在定时任务里 1validateDog(po,"vip");方法是本类方法，抛出异常不会出现marked as rollback-only
		 * 2如果是其他类里抛异常，会被标记为marked as rollback-only 3所以这样要提取出判断逻辑来
		 * kennelService.validKennel(motherMember.getMemberCode());
		 */

		// 母犬配种间隔校验（5个月）
		Calendar startDay = Calendar.getInstance();
		startDay.setTime(po.getBreedDate());
		startDay.add(Calendar.MONTH, -5);
		Calendar endDay = Calendar.getInstance();
		endDay.setTime(po.getBreedDate());
		endDay.add(Calendar.MONTH, 5);
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
		int count = dao.getByFemalePedigreeCertified(po.getFemalePedigreeCertified(), sdf.format(startDay.getTime()),
				sdf.format(endDay.getTime()), po.getId());
		if (count > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬距上次配种间隔少于5个月");
		}
		// 藏獒每年只能申请一次配种
		if ("230".equals(po.getDogBreed())) {
			SimpleDateFormat sdfZangao = new SimpleDateFormat("yyyy");
			int countZangao = dao.getByFemalePedigreeCertifiedZangao(po.getFemalePedigreeCertified(),
					sdfZangao.format(po.getBreedDate()), po.getId());
			if (countZangao > 0) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "藏獒犬种每年只能申请一次配种");
			}
		}

		checkMotherMemberKennel(po.getFdogOwnerCode());

		// 配种证明犬只校验
		validateDog(po, "vip");
		// 近亲繁殖校验--仅会员中心检验，oa不检验，所以不放入validateDog方法里
//		JSONObject jo = restDogService.bloodline(po.getMalePedigreeCertified(),po.getFemalePedigreeCertified());
		JSONObject jo = restDogService.bloodlineNew(po.getMalePedigreeCertified(), po.getFemalePedigreeCertified());
		JSONArray infos = jo.getJSONArray("infos");
		for (int i = 0; i < infos.size(); i++) {
			JSONObject jsonObject = infos.getJSONObject(i);
			if ("1".equals(jsonObject.getString("state"))) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "当前公母犬为近亲关系，不可提交配种证明，如有疑问请与协会客服联系");
			}
		}
		// 当前时间距离配种日期必须在6个月内
		Date now = new Date();
		if (po.getBreedDate().getTime() > now.getTime()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "配种日期已大于当前日期，请重新输入");
		}
		if (po.getCreateDate() == null) {
			if (!com.cku.oa.sys.util.DateUtils.isInXMounth(po.getBreedDate(), 6)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "配种证明需在配种后六个月内申请，逾期将无法办理");
			}
		} else {
			// 拿记录创建时间和出生日期判断
			Calendar dateCalendar = Calendar.getInstance();
			dateCalendar.setTime(po.getBreedDate());
			dateCalendar.add(Calendar.MONTH, 6); // 月份加x
			dateCalendar.add(Calendar.DATE, 1); // 日期加x
			if (po.getCreateDate().getTime() > dateCalendar.getTimeInMillis()) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "配种证明需在配种后六个月内申请，逾期将无法办理");
			}
		}
		// 冷冻精子配种不允许加急
		if (StringUtils.isNotBlank(po.getSemenCode()) && "1".equals(po.getUrgent())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "冷冻精子配种不允许加急");
		}
		// 冷冻精子配种申请人必须是精子主人

		// 杜高父母犬配种规则：判断出生日期大于2018年6月1日并且报名缴费过比赛
		// 【ID1003486】 杜高犬繁殖规则的修改
//		if("292".equals(po.getDogBreed())){
//			if (dogNewbornDao.birthdayCheck(po.getMalePedigreeCertified())>0){
//				if (dogNewbornDao.showPaymentCheck(po.getMalePedigreeCertified()) == 0 ){
//					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"公犬不符合标准，2018年6月之后出生的犬只，繁殖要求必须参加一场赛事并不判定为失格才可以申请繁殖登记。详情请查看俱乐部繁殖规则");
//				}
//			}
//
//			if (dogNewbornDao.birthdayCheck(po.getFemalePedigreeCertified())>0){
//				if (dogNewbornDao.showPaymentCheck(po.getFemalePedigreeCertified()) == 0 ){
//					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"母犬不符合标准，2018年6月之后出生的犬只，繁殖要求必须参加一场赛事并不判定为失格才可以申请繁殖登记。详情请查看俱乐部繁殖规则");
//				}
//			}
//		}

	}

	public void test() {

	}

	/**
	 *
	 * @description: 配种证明犬只校验
	 * @author: laiguanglong
	 * @date: 2016年10月19日 下午7:54:03
	 */
	public void validateDog(DogNewborn po, String type) throws ZAException {
		Dog fatherDog = StringUtils.isBlank(po.getMalePedigreeCertified()) ? null
				: dogDao.getByPedigreeCertifiedCode(po.getMalePedigreeCertified());
		Dog motherDog = StringUtils.isBlank(po.getFemalePedigreeCertified()) ? null
				: dogDao.getByPedigreeCertifiedCode(po.getFemalePedigreeCertified());
		if (fatherDog == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬血统证书号有误");
		}
		if (motherDog == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬血统证书号有误");
		}
		Date breedDate = po.getBreedDate();
		if (!dogTypeService.isSameType(fatherDog.getDogBreed(), motherDog.getDogBreed())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父母犬犬种不同");
		}
		if ("230".equals(fatherDog.getDogBreed())) {
			// 藏獒提交配种证明时，公母犬需完成DNA位点分析(DNA存档)
			if (!dogDnaService.isHaveDogDna(fatherDog.getId())) {
				;
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "公犬没有进行DNA存档");
			}
			if (!dogDnaService.isHaveDogDna(motherDog.getId())) {
				;
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬没有进行DNA存档");
			}
			// 藏獒配种需大于20个月
			if (getMonthAge(breedDate, fatherDog.getBirthdate()) < 20
					|| getMonthAge(breedDate, motherDog.getBirthdate()) < 20) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "藏獒犬配种年龄不能低于20个月");
			}
			// 藏獒公母犬一方为2016年1月1日之后出生的犬只，繁殖人需要参加一次繁育考试，取得繁育证书
			validateDogBreedCertified(fatherDog, motherDog);
		}
		validateFatherDog(fatherDog, po.getSemenCode(), breedDate, type);
		validateMotherDog(motherDog, breedDate, type);
		// 冷冻精子校验
		if (StringUtils.isNotBlank(po.getSemenCode())) {
			Dog dog = validataSemenCode(po.getSemenCode(), null, po.getSemenOwnerMemberCode());
			if (!fatherDog.getId().equals(dog.getId())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子号有误");
			}
		}
	}

	public void validateFatherDog(Dog fatherDog, String semenCode, Date breedDate, String type) {
		if (StringUtils.isBlank(semenCode)) {
			// 血统证书必须以CKU开头
			if (!fatherDog.getPedigreeCertifiedCode().toUpperCase().startsWith("CKU")) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只有证书号以CKU开头的犬只才可以申请配种证明业务");
			}
			if ("vip".equals(type)) {
				if (getMonthAge(breedDate, fatherDog.getBirthdate()) < 12) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬配种年龄不能低于12个月");
				}
			}
			// 种公要小于10岁，种母要小于8岁 、2019年四月以后取消该限制
			if (StringUtils.isBlank(semenCode) && !isInXMounth(fatherDog.getBirthdate(), breedDate, 120)
					&& new Date().getTime() < DogBirthCertificateService.KENNEL_UNACTIVATION_MILLIS) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬配种年龄不能大于10岁");
			}
		}
		// 犬只状态都得为正常
		if (!"0".equals(fatherDog.getDogState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬状态不为正常，不允许申请配种证明");
		}
		if (!"1".equals(fatherDog.getGender())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬的性别不为公");
		}
		// 业务禁止校验
		businessBanService.validateBusinessBan(fatherDog, "父犬为禁止繁殖状态，无法办理该业务");
	}

	public void validateMotherDog(Dog motherDog, Date breedDate, String type) {
		// 血统证书必须以CKU开头
		if (!motherDog.getPedigreeCertifiedCode().toUpperCase().startsWith("CKU")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只有证书号以CKU开头的犬只才可以申请配种证明业务");
		}
		// 犬只状态都得为正常
		if (!"0".equals(motherDog.getDogState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬状态不为正常，不允许申请配种证明");
		}
		if (!"2".equals(motherDog.getGender())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬的性别不为母");
		}
		if (breedDate != null) {
			if ("vip".equals(type)) {
				if (getMonthAge(breedDate, motherDog.getBirthdate()) < 12) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬配种年龄不能低于12个月");
				}
			}
			// 种公要小于10岁，种母要小于8岁 2019年四月以后取消该限制
			if (!isInXMounth(motherDog.getBirthdate(), breedDate, 96)
					&& new Date().getTime() < DogBirthCertificateService.KENNEL_UNACTIVATION_MILLIS) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬配种年龄不能大于8岁");
			}
		}
		// 业务禁止校验
		businessBanService.validateBusinessBan(motherDog, "母犬为禁止繁殖状态，无法办理该业务");
	}

	public Dog validataSemenCode(String semenCode, List<String> semenCodeList, String semenOwnerMemberCode) {
		String dogId = "";
		String memberCode = "";
		String[] semenCodes = semenCode.split(",");
		List<String> list = Lists.newArrayList();
		for (int i = 0, len = semenCodes.length; i < len; i++) {
			DogSemen dogSemen = dogSemenDao.getBySemenCode(semenCodes[i]);
			if (dogSemen == null) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子编号有误");
			} else {
				// 判断是否有重复
				if (list.contains(semenCodes[i])) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子编号重复");
				} else {
					list.add(semenCodes[i]);
				}
				if (i == 0) {
					dogId = dogSemen.getDogId();
					memberCode = dogSemen.getMemberCode();
				} else {
					if (!dogId.equals(dogSemen.getDogId())) {
						throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子编号必须属于同一只犬");
					}
					if (!memberCode.equals(dogSemen.getMemberCode())) {
						throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子编号必须属于同一会员");
					}
				}
				// 未使用
				if (StringUtils.isNotBlank(dogSemen.getBreedCertifiedCode())
						&& !(semenCodeList != null && semenCodeList.contains(semenCodes[i]))) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子已被使用");
				}
				// 不在精子变更流程中,状态除了已处理或已撤销，别的状态只要有记录就算在流程中
				if (dogSemenChangeService.isApply(semenCodes[i])) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子" + semenCodes[i] + "正在办理精子变更业务，不允许进行配种");
				}
			}
		}
		if (!memberCode.equals(semenOwnerMemberCode)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该业务仅限精子所有者办理");
		}
		// 精子所有会员为有效专业会员
		Member member = memberService.getByMemberCode(memberCode);
		if (!"professional".equals(member.getMemberType())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子所有者会员不是专业会员");
		}
		Debugger.doAssert(MemberUtil.isActiveProfessionalMember(member), ZAErrorCode.ZA_VALID_FAILED, "精子所有者不是有效专业会员");
		return dogDao.get(dogId);
	}

	/**
	 * @description: 配种证明申请，藏獒校验繁育证书
	 * @author: laiguanglong
	 * @date: 2017/2/10 16:47
	 */
	private void validateDogBreedCertified(Dog fatherDog, Dog motherDog) {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
		Date date;
		try {
			date = sdf.parse("20160101");
		} catch (ParseException e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "格式化20160101日期出错，请联系协会");
		}
		if (fatherDog.getBirthdate().getTime() > date.getTime()) {
			DogBreedCertified dogBreedCertified = dogBreedCertifiedDao.getByDogId(fatherDog.getId());
			if (dogBreedCertified == null || StringUtils.isBlank(dogBreedCertified.getBreedCertifiedCode())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "出生日期在2016年1月1日之后的藏獒申请配种证明需取得繁育证书");
			}
		}
		if (motherDog.getBirthdate().getTime() > date.getTime()) {
			DogBreedCertified dogBreedCertified = dogBreedCertifiedDao.getByDogId(motherDog.getId());
			if (dogBreedCertified == null || StringUtils.isBlank(dogBreedCertified.getBreedCertifiedCode())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "出生日期在2016年1月1日之后的藏獒申请配种证明需取得繁育证书");
			}
		}
	}

	/**
	 * 计算月龄
	 * 
	 * @param date
	 * @param birthdate
	 * @return
	 */
	private int getMonthAge(Date date, Date birthdate) {
		if (date == null || birthdate == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		Calendar dateCalendar = Calendar.getInstance();
		Calendar birthdateCalendar = Calendar.getInstance();
		dateCalendar.setTime(date);
		birthdateCalendar.setTime(birthdate);
		int monthAge = dateCalendar.get(Calendar.MONTH) - birthdateCalendar.get(Calendar.MONTH);
		if (dateCalendar.get(Calendar.YEAR) != birthdateCalendar.get(Calendar.YEAR)) {
			monthAge = (dateCalendar.get(Calendar.YEAR) - birthdateCalendar.get(Calendar.YEAR)) * 12 + monthAge;
		}
		birthdateCalendar.add(Calendar.MONTH, monthAge);
		if (birthdateCalendar.getTime().after(dateCalendar.getTime())) {
			monthAge = monthAge - 1;
		}
		return monthAge;
	}

	/**
	 * @description: 判断出生日期和配种日期是否在X个月内 出生日期和配种日期都是没有时分秒的
	 * @author: laiguanglong
	 * @date: 2017/1/11 16:17
	 */
	public boolean isInXMounth(Date birthdate, Date breedDate, int x) {
		boolean flag = true;
		if (birthdate == null || breedDate == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "日期为空");
		}
		if (x < 1) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "月份数必须为大于1的正整数");
		}
		Calendar dateCalendar = Calendar.getInstance();
		dateCalendar.setTime(birthdate);
		dateCalendar.add(Calendar.MONTH, x); // 月份加x
		if (breedDate.getTime() > dateCalendar.getTime().getTime()) {
			flag = false;
		}
		return flag;
	}

	/**
	 * 根据配种证明号 查询父母犬只姓名和证书号
	 * 
	 * @Author chaixueteng
	 * @2016年9月21日下午7:25:43
	 */
	public Object getFM(String breedCertifiedCode) {
		HashMap<String, String> map = new HashMap<String, String>();
		DogNewborn po = dao.getFM(breedCertifiedCode);
		if (po != null) {
			map.put("rc", "0");
			map.put("fdogName", po.getFemaleDogName());
			map.put("femalePedigreeCertified", po.getFemalePedigreeCertified());
			map.put("mdogName", po.getMaleDogName());
			map.put("malePedigreeCertified", po.getMalePedigreeCertified());
		} else {
			map.put("rc", "1");
			map.put("error", "无数据!");
		}
		return map;
	}

	/**
	 *
	 * @description: OA配种证明订单录入业务校验
	 * @author: laiguanglong
	 * @date: 2016年11月17日 上午10:53:42
	 */
	private void oaOrderSaveValidateBusiness(DogNewborn po) {
		// 母犬配种间隔校验（5个月）
		Calendar startDay = Calendar.getInstance();
		startDay.setTime(po.getBreedDate());
		startDay.add(Calendar.MONTH, -5);
		Calendar endDay = Calendar.getInstance();
		endDay.setTime(po.getBreedDate());
		endDay.add(Calendar.MONTH, 5);
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
		int count = dao.getByFemalePedigreeCertified(po.getFemalePedigreeCertified(), sdf.format(startDay.getTime()),
				sdf.format(endDay.getTime()), po.getId());
		if (count > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬距上次配种间隔少于5个月");
		}

		Dog fatherDog = StringUtils.isBlank(po.getMalePedigreeCertified()) ? null
				: dogDao.getByPedigreeCertifiedCode(po.getMalePedigreeCertified());
		Dog motherDog = StringUtils.isBlank(po.getFemalePedigreeCertified()) ? null
				: dogDao.getByPedigreeCertifiedCode(po.getFemalePedigreeCertified());
		if (fatherDog == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬血统证书号有误");
		}
		if (motherDog == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬血统证书号有误");
		}
		// cku证书号，订单人和公犬主同一个人
		if (fatherDog.getPedigreeCertifiedCode().toUpperCase().startsWith("CKU")) {
			if (!fatherDog.getMemberCode().equals(po.getMemberCode())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该公犬主与缴费会员不为同一人！");
			}
		}
		// start-20180202-chaixueteng 冷冻精子配种不校验公犬犬主是否有有效专业会员
		if (StringUtils.isBlank(po.getSemenCode())) {
			// 只有公犬血统证书为CKU开头的需要校验公犬犬主为有效专业会员
			if (po.getMalePedigreeCertified().toUpperCase().startsWith("CKU")) {
				checkFatherMember(po.getMdogOwnerCode());
			}
		}
		// 校验母犬会员
		checkMotherMember(po.getFdogOwnerCode());
		// 校验母犬犬舍
		checkMotherMemberKennel(po.getFdogOwnerCode());
		// end
		// 犬只状态都得为正常
		if (!"0".equals(fatherDog.getDogState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬状态不为正常，不允许申请配种证明");
		}
		if (!"0".equals(motherDog.getDogState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬状态不为正常，不允许申请配种证明");
		}
		// 业务禁止校验
		businessBanService.validateBusinessBan(fatherDog, "父（母）犬为禁止繁殖状态");
		businessBanService.validateBusinessBan(motherDog, "父（母）犬为禁止繁殖状态");
		if (!"1".equals(fatherDog.getGender())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬的性别不为公");
		}
		if (!"2".equals(motherDog.getGender())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬的性别不为母");
		}
		if (!dogTypeService.isSameType(fatherDog.getDogBreed(), motherDog.getDogBreed())
				&& !dogTypeService.isSameTypeForOABreed(fatherDog.getDogBreed(), motherDog.getDogBreed())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父母犬犬种不同");
		}
		if ("230".equals(fatherDog.getDogBreed())) {
			// 藏獒提交配种证明时，公母犬需完；成DNA位点分析(DNA存档)
//			DogDna dogDna = new DogDna();
//			dogDna.setDogId(fatherDog.getId());
			// update by zhangjunwu 2024/06/18 增加藏獒犬 非CKU开头证书号 不校验是否有DNA存档
			DogDnaArchive fatherDnaArchive = dogDnaArchiveService
					.getByPedigreeCertifiedCode(fatherDog.getPedigreeCertifiedCode());
			if (fatherDog.getPedigreeCertifiedCode().toUpperCase().startsWith("CKU")
					&& (Objects.isNull(fatherDnaArchive) || !"2".equals(fatherDnaArchive.getPaymentState())
							|| "1".equals(fatherDnaArchive.getIsCanceled()))) {
//			if (fatherDog.getPedigreeCertifiedCode().toUpperCase().startsWith("CKU")
//					&& dogDnaService.findList(dogDna).size() == 0) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "公犬没有进行DNA存档");
			}
//			dogDna.setDogId(motherDog.getId());
//			if (dogDnaService.findList(dogDna).size() == 0) {
			DogDnaArchive motherDnaArchive = dogDnaArchiveService
					.getByPedigreeCertifiedCode(motherDog.getPedigreeCertifiedCode());
			if (Objects.isNull(motherDnaArchive) || !"2".equals(motherDnaArchive.getPaymentState())
					|| "1".equals(motherDnaArchive.getIsCanceled())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬没有进行DNA存档");
			}
			Date breedDate = po.getBreedDate();
			// 藏獒配种需大于20个月
			if (getMonthAge(breedDate, fatherDog.getBirthdate()) < 20
					|| getMonthAge(breedDate, motherDog.getBirthdate()) < 20) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "藏獒犬配种年龄不能低于20个月");
			}
		}

		// 杜高父母犬配种规则：判断出生日期大于2018年6月1日并且报名缴费过比赛
		// 【ID1003486】 杜高犬繁殖规则修改
//    	if("292".equals(po.getDogBreed())){
//			if (dogNewbornDao.birthdayCheck(po.getMalePedigreeCertified())>0){
//				if (dogNewbornDao.showPaymentCheck(po.getMalePedigreeCertified()) == 0 ){
//					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"公犬不符合标准，2018年6月之后出生的犬只，繁殖要求必须参加一场赛事并不判定为失格才可以申请繁殖登记。详情请查看俱乐部繁殖规则");
//				}
//			}
//
//			if (dogNewbornDao.birthdayCheck(po.getFemalePedigreeCertified())>0){
//				if (dogNewbornDao.showPaymentCheck(po.getFemalePedigreeCertified()) == 0 ){
//					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"母犬不符合标准，2018年6月之后出生的犬只，繁殖要求必须参加一场赛事并不判定为失格才可以申请繁殖登记。详情请查看俱乐部繁殖规则");
//				}
//			}
//		}
	}

	public HashMap<String, String> getDogByPedigreeCertified(String pedigreeCertified) {
		HashMap<String, String> map = new HashMap<String, String>();
		Dog dog = StringUtils.isBlank(pedigreeCertified) ? null : dogDao.getByPedigreeCertifiedCode(pedigreeCertified);
		if (dog != null) {
			map.put("dogId", dog.getId());
			map.put("dogOwner", dog.getDogOwner());
			map.put("dogOwnerMemberCode", dog.getMemberCode());
			map.put("nameEn", dog.getNameEn());
			map.put("gender", DictUtils.getDictLabel(dog.getGender(), "dog_gender", ""));
			map.put("colorFlag", dog.getColorFlag());
			map.put("identificationFlag", dog.getIdentificationFlag());
			map.put("dogState", dog.getDogState());// 犬只状态
			map.put("DNA", dog.getDna());
			map.put("dogBreed", dog.getDogBreed());
			String dogGeneration = "false";
			if (!"".equals(dog.getDogGeneration()) && dog.getDogGeneration() != null
					&& new Integer(dog.getDogGeneration()) >= 3) {
				dogGeneration = "true";
			} else {
				dogGeneration = "false";
			}
			map.put("dogGeneration", dogGeneration);// 犬只代数是否满足3代标识
			if (dog.getPedigreeCertifiedCode().toUpperCase().startsWith("CKU")) {
				Member member = memberService.getByMemberCode(dog.getMemberCode() == null ? "" : dog.getMemberCode());
				map.put("memberType", member.getMemberType());
			}
		}
		return map;
	}

	/**
	 * @description: 配种证明自动审核
	 * @author: laiguanglong
	 * @date: 2017/2/7 16:28
	 */
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public void autoAudit(DogNewborn review) {
		// 默认审核通过
		String reviewState = "1";
		String reviewRemarks = "系统自动审核-通过";
		try {
			validateBusiness(review);
		} catch (Exception e) {
			// 审核校验抛异常，审核不通过
			reviewState = "2";
			reviewRemarks = "系统自动审核-不通过：" + e.getMessage();
		}
		// 处理审核结果
		DogNewborn copy = new DogNewborn();
		BeanUtils.copyProperties(review, copy);
		copy.setReviewState(reviewState);
		copy.setReviewRemarks(reviewRemarks);
		changeStatus(copy);
	}

	/**
	 * @description: 配种证明付款后业务
	 * @author: laiguanglong
	 * @date: 2017/2/8 9:43
	 */
	public void afterPay(String id) {
		DogNewborn po = dao.get(id);
		// 加急自动审核
		if ("1".equals(po.getUrgent())) {
			autoAudit(po);
		}
	}

	public List<DogNewborn> getDogBreedList(String pedigreeCertified) {
		List<DogNewborn> list = dao.getDogBreedList(pedigreeCertified);
		return list;
	}
}