package com.cku.oa.dogsporting.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.entity.DogType;
import com.cku.oa.dog.service.DogTypeService;
import com.cku.oa.dogsporting.dao.DogSportingDogsDao;
import com.cku.oa.dogsporting.entity.DogSportingDogs;
import com.cku.oa.dogsporting.entity.DogSportingMembers;
import com.cku.oa.sys.code.SysCodeUtil;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.excel.ExportExcel;
import com.thinkgem.jeesite.common.utils.excel.ImportExcel;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 犬运动犬只Service
 *
 * @author yuanshuai
 * @version 2021-06-15
 */
@Service
@Transactional(readOnly = true)
public class DogSportingDogsService extends CrudService<DogSportingDogsDao, DogSportingDogs> {

	@Autowired
	private DogTypeService dogTypeService;
	@Autowired
	private DogSportingMembersService dogSportingMembersService;

	public DogSportingDogs get(String id) {
		return super.get(id);
	}

	public List<DogSportingDogs> findList(DogSportingDogs dogSportingDogs) {
		return super.findList(dogSportingDogs);
	}

	public Page<DogSportingDogs> findPage(Page<DogSportingDogs> page, DogSportingDogs dogSportingDogs) {
		return super.findPage(page, dogSportingDogs);
	}

	@Transactional(readOnly = false)
	public void save(DogSportingDogs dogSportingDogs) {
		parameterValidate(dogSportingDogs);
		if (dogSportingDogs.getIsNewRecord()) {
			dogSportingDogs.setCertificateCode(SysCodeUtil.getCDSCDogCertificateCode());
			dogSportingDogs.setPrintState("0");
		}
		super.save(dogSportingDogs);
	}

	private void parameterValidate(DogSportingDogs dogSportingDogs) {
		DogSportingMembers sportingMembers = dogSportingMembersService.getByMemberCode(dogSportingDogs.getMemberCode());
		if (sportingMembers == null || StringUtils.isBlank(sportingMembers.getId())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只" + dogSportingDogs.getDogName() + "：犬主会员号有误");
		}
		if (StringUtils.isBlank(dogSportingDogs.getDogGender())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只" + dogSportingDogs.getDogName() + "：犬只性别不能为空");
		}
		if (StringUtils.isBlank(dogSportingDogs.getDogType())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只" + dogSportingDogs.getDogName() + "：犬只品种不能为空");
		}
		if (StringUtils.isBlank(dogSportingDogs.getSomatotype())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只" + dogSportingDogs.getDogName() + "：犬只体型不能为空");
		}
		if (dogSportingDogs.getDogName().length() > 40) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只" + dogSportingDogs.getDogName() + "：犬只名称需为40个字符内");
		}
		if (dogSportingDogs.getBirthday() == null) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只" + dogSportingDogs.getDogName() + "：犬只出生日期不能为空");
		}
		if (!StringUtils.isBlank(dogSportingDogs.getIdentificationFlag()) && checkIdentificationFlagByCertificateCode(dogSportingDogs)) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只" + dogSportingDogs.getDogName() + "：芯片号信息已存在");
		}
		if (!StringUtils.isBlank(dogSportingDogs.getNoseCode()) && checkNoseCodeByCertificateCode(dogSportingDogs)) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只" + dogSportingDogs.getDogName() + "：鼻纹编号信息已存在");
		}
	}

	/**
	 * 芯片号验重
	 *
	 * @author yuanshuai
	 * @date 2021/6/22 19:35
	 */
	private boolean checkIdentificationFlagByCertificateCode(DogSportingDogs dogSportingDogs) {
		return dao.getIdentificationFlagCountByCertificateCode(dogSportingDogs.getIdentificationFlag(), dogSportingDogs.getCertificateCode()) > 0;
	}

	/**
	 * 芯片号验重
	 *
	 * @author yuanshuai
	 * @date 2021/6/22 19:35
	 */
	private boolean checkNoseCodeByCertificateCode(DogSportingDogs dogSportingDogs) {
		return dao.getNoseCodeCountByCertificateCode(dogSportingDogs.getNoseCode(), dogSportingDogs.getCertificateCode()) > 0;
	}

	@Transactional(readOnly = false)
	public void delete(DogSportingDogs dogSportingDogs) {
		super.delete(dogSportingDogs);
	}

	/**
	 * 导入模板
	 *
	 * @author yuanshuai
	 * @date 2021/6/22 17:18
	 */
	public String importFileTemplate(HttpServletResponse response) throws IOException {
		List<DogSportingDogs> list = Lists.newArrayList();
		DogSportingDogs dog = new DogSportingDogs();
		dog.setDogName("犬名");
		dog.setMemberCode("会员号");
		dog.setDogGender("公/母");
		dog.setDogType("犬种编号");
		dog.setSomatotype("大/中/小");
		dog.setBirthday(new Date());
		dog.setColorFlag("毛色");
		dog.setIdentificationFlag("芯片号");
		dog.setNoseCode("鼻纹编号");
		dog.setRemarks("备注");
		list.add(dog);
		new ExportExcel("犬运动犬只信息", DogSportingDogs.class, 2).setDataList(list).write(response, "犬运动犬只数据导入模板.xlsx").dispose();
		return null;
	}

	/**
	 * 导入
	 *
	 * @author yuanshuai
	 * @date 2021/6/22 17:30
	 */
	@Transactional(readOnly = false)
	public void importMember(MultipartFile file) throws IOException, InvalidFormatException, IllegalAccessException, InstantiationException {
		//计数器
		AtomicInteger count = new AtomicInteger();
		//读取excel
		ImportExcel ei = new ImportExcel(file, 1, 0);
		List<DogSportingDogs> list = ei.getDataList(DogSportingDogs.class);
		list.forEach(dog -> {
			if (StringUtils.isBlank(dog.getDogName())) {
				return;
			}
			parameterValidate(dog);
			if (!"大".equals(dog.getSomatotype()) && !"中".equals(dog.getSomatotype()) && !"小".equals(dog.getSomatotype())) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只体型需为大/中/小");
			}
			if (!"公".equals(dog.getDogGender()) && !"母".equals(dog.getDogGender())) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "犬只性别需为公或母");
			}
			dog.setDogGender(DictUtils.getDictValue(dog.getDogGender(), "dog_gender", null));
			dog.setSomatotype(DictUtils.getDictValue(dog.getSomatotype(), "dog_somatotype", null));
			save(dog);
			count.getAndIncrement();
		});
		if (count.get() <= 0) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "导入失败，Excel中无数据");
		}
	}

	/**
	 * 修改打印状态
	 *
	 * @author yuanshuai
	 * @date 2021/6/23 13:53
	 */
	@Transactional(readOnly = false)
	public DogSportingDogs updatePrintState(String id, String state) {
		DogSportingDogs dog = get(id);
		dog.setPrintState(state);
		dao.updatePrintState(dog);
		return dog;
	}

	/**
	 * 获得犬运动犬种列表
	 *
	 * @author yuanshuai
	 * @date 2021/7/8 12:08
	 */
	public List<DogType> getDogSportingDogTypeList() {
		List<DogType> dogTypeList = dogTypeService.findAllList();
		DogType other = new DogType();
		other.setBreedCnName("其他");
		other.setDogBreedCode("000");
		dogTypeList.add(other);
		return dogTypeList;
	}

	/**
	 * 导出
	 *
	 * @author yuanshuai
	 * @date 2021/7/8 15:50
	 */
	public List<DogSportingDogs> export(DogSportingDogs dogSportingDogs) {
		List<DogSportingDogs> list = findList(dogSportingDogs);
		list.forEach(dog -> {
			if (StringUtils.isBlank(dog.getDogName())) {
				return;
			}
			dog.setDogName(StringEscapeUtils.unescapeHtml4(dog.getDogName()));
			dog.setDogGender(DictUtils.getDictLabel(dog.getDogGender(), "dog_gender", null));
			dog.setSomatotype(DictUtils.getDictLabel(dog.getSomatotype(), "dog_somatotype", null));
		});
		return list;
	}
}