/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.thinkgem.jeesite.modules.sys.security;

import com.cku.oa.sys.util.IPLimitUtil;
import com.cku.util.MD5Generator;
import com.thinkgem.jeesite.common.config.Global;
import com.thinkgem.jeesite.common.servlet.ValidateCodeServlet;
import com.thinkgem.jeesite.common.utils.SpringContextHolder;
import com.thinkgem.jeesite.common.web.Servlets;
import com.thinkgem.jeesite.modules.sys.entity.Dict;
import com.thinkgem.jeesite.modules.sys.entity.Menu;
import com.thinkgem.jeesite.modules.sys.entity.Role;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.service.SystemService;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import com.thinkgem.jeesite.modules.sys.utils.LogUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;
import com.thinkgem.jeesite.modules.sys.web.LoginController;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationInfo;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.authc.SimpleAuthenticationInfo;
import org.apache.shiro.authz.AuthorizationInfo;
import org.apache.shiro.authz.Permission;
import org.apache.shiro.authz.SimpleAuthorizationInfo;
import org.apache.shiro.realm.AuthorizingRealm;
import org.apache.shiro.session.Session;
import org.apache.shiro.subject.PrincipalCollection;
import org.apache.shiro.util.ByteSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.DependsOn;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.Serializable;
import java.util.Collection;
import java.util.List;

/**
 * 系统安全认证实现类
 * @author ThinkGem
 * @version 2014-7-5
 */
@Service
//@DependsOn({"userDao","roleDao","menuDao"})
public class SystemAuthorizingRealm extends AuthorizingRealm {

	private Logger logger = LoggerFactory.getLogger(getClass());

	@Value("${mode}")
	private String mode;

	private SystemService systemService;

	/**
	 * 认证回调函数, 登录时调用
	 */
	@Override
	protected AuthenticationInfo doGetAuthenticationInfo(AuthenticationToken authcToken) {

		String username = ""; 
		char[] password = null;
		String type = "2";
		boolean isMobileLogin = false;
		boolean isMD5 = false;
		String loginType = "";
		if(authcToken instanceof OAuthToken){
			//oauth登录
			OAuthToken oauth = (OAuthToken)authcToken;
			//oauthIP访问限制，每一个ip，每天限制只能登录30次
			try{
				IPLimitUtil.CheckLimit("oauth", oauth.getHost(), 30);
			}catch(Exception e){
				logger.warn(oauth.getHost()+"请求oauth登录次数过多，被限制访问,使用帐号"+oauth.getUsername());
				throw new AuthenticationException("msg:该IP请求次数过多，请明天再试.");
			}
			username = oauth.getUsername();
			password = oauth.getPassword();
			type = oauth.getType();
		}else{
			//普通登录
			UsernamePasswordToken token = (UsernamePasswordToken) authcToken;
			
			// 校验登录验证码
			if (token.getType().equals("1")&&LoginController.isValidateCodeLogin(token.getUsername(), false, false)){
				Session session = UserUtils.getSession();
				String code = (String)session.getAttribute(ValidateCodeServlet.VALIDATE_CODE);
				if (token.getCaptcha() == null || !token.getCaptcha().toLowerCase().equals(code.toLowerCase())){
					throw new AuthenticationException("msg:验证码错误, 请重试.");
				}
			}
			username = token.getUsername();
			password = token.getPassword();
			type = token.getType();
			isMobileLogin = token.isMobileLogin();
			isMD5 = token.isMD5();
			loginType = token.getLoginType();
		}

		//判断loginType
		List<Dict> dictList = DictUtils.getDictList("login_switch");
		if("oa".equals(loginType) && "product".equals(mode) && "1".equals(dictList.get(0).getValue())){
			throw new AuthenticationException("msg:请从钉钉扫码登录");
		}

		// 校验用户名密码
		User user = getSystemService().getUserByLoginNameAndUserType(username,type);
		if (user != null) {
			if (Global.NO.equals(user.getLoginFlag())){
				throw new AuthenticationException("msg:该帐号已禁止登录.");
			}
			SimpleAuthenticationInfo au = new SimpleAuthenticationInfo(new Principal(user, isMobileLogin,loginType),user.getPassword(), ByteSource.Util.bytes(""), getName());
			if(isMD5){
				au = new SimpleAuthenticationInfo(new Principal(user, isMobileLogin,loginType),MD5Generator.generate16(user.getPassword()), ByteSource.Util.bytes(""), getName());
			}
			return au;
		} else {
			return null;
		}
	}

	/**
	 * 授权查询回调函数, 进行鉴权但缓存中无用户的授权信息时调用
	 */
	@Override
	protected AuthorizationInfo doGetAuthorizationInfo(PrincipalCollection principals) {
		Principal principal = (Principal) getAvailablePrincipal(principals);
		User user = UserUtils.get(principal.getId());
		if (user != null&& "1".equals(user.getUserType())) {
			SimpleAuthorizationInfo info = new SimpleAuthorizationInfo();
			List<Menu> list = UserUtils.getMenuList();
			for (Menu menu : list){
				if (StringUtils.isNotBlank(menu.getPermission())){
					// 添加基于Permission的权限信息
					for (String permission : StringUtils.split(menu.getPermission(),",")){
						info.addStringPermission(permission);
					}
				}
			}

			// 添加用户权限
			String userType = user.getUserType();
			if(userType.equals("1")){
				//登录oa的权限
				info.addStringPermission("system:loginAdmin");
			}
			info.addStringPermission("user");
			// 添加用户角色信息
			for (Role role : user.getRoleList()){
				info.addRole(role.getEnname());
			}
			// 更新登录IP和时间
			getSystemService().updateUserLoginInfo(user);
			// 记录登录日志
			LogUtils.saveLog(Servlets.getRequest(), "系统登录");
			return info;
		} else {
			return null;
		}
	}
	
	@Override
	protected void checkPermission(Permission permission, AuthorizationInfo info) {
		authorizationValidate(permission);
		super.checkPermission(permission, info);
	}
	
	@Override
	protected boolean[] isPermitted(List<Permission> permissions, AuthorizationInfo info) {
		if (permissions != null && !permissions.isEmpty()) {
            for (Permission permission : permissions) {
        		authorizationValidate(permission);
            }
        }
		return super.isPermitted(permissions, info);
	}
	
	@Override
	public boolean isPermitted(PrincipalCollection principals, Permission permission) {
		authorizationValidate(permission);
		return super.isPermitted(principals, permission);
	}
	
	@Override
	protected boolean isPermittedAll(Collection<Permission> permissions, AuthorizationInfo info) {
		if (permissions != null && !permissions.isEmpty()) {
            for (Permission permission : permissions) {
            	authorizationValidate(permission);
            }
        }
		return super.isPermittedAll(permissions, info);
	}
	
	/**
	 * 授权验证方法
	 * @param permission
	 */
	private void authorizationValidate(Permission permission){
		// 模块授权预留接口
	}
	
	/**
	 * 设定密码校验的Hash算法与迭代次数
	 */
	@PostConstruct
	public void initCredentialsMatcher() {
		MD5CredentialMatcher matcher = new MD5CredentialMatcher(SystemService.HASH_ALGORITHM);
		setCredentialsMatcher(matcher);
	}
	

	/**
	 * 清空所有关联认证
	 * @Deprecated 不需要清空，授权缓存保存到session中
	 */
	@Deprecated
	public void clearAllCachedAuthorizationInfo() {

	}

	/**
	 * 获取系统业务对象
	 */
	public SystemService getSystemService() {
		if (systemService == null){
			systemService = SpringContextHolder.getBean(SystemService.class);
		}
		return systemService;
	}
	
	/**
	 * 授权用户信息
	 */
	public static class Principal implements Serializable {

		private static final long serialVersionUID = 1L;
		
		private String id; // 编号
		private String loginName; // 登录名
		private String name; // 姓名
		private boolean mobileLogin; // 是否手机登录
		private String loginType; //登录端
		
//		private Map<String, Object> cacheMap;

		public Principal(User user, boolean mobileLogin,String loginType) {
			this.id = user.getId();
			this.loginName = user.getLoginName();
			this.name = user.getName();
			this.mobileLogin = mobileLogin;
			this.loginType = loginType;
		}

		public String getId() {
			return id;
		}

		public String getLoginName() {
			return loginName;
		}

		public String getName() {
			return name;
		}

		public boolean isMobileLogin() {
			return mobileLogin;
		}
		
		

//		@JsonIgnore
//		public Map<String, Object> getCacheMap() {
//			if (cacheMap==null){
//				cacheMap = new HashMap<String, Object>();
//			}
//			return cacheMap;
//		}

		public String getLoginType() {
			return loginType;
		}

		public void setLoginType(String loginType) {
			this.loginType = loginType;
		}

		/**
		 * 获取SESSIONID
		 */
		public String getSessionid() {
			try{
				return (String) UserUtils.getSession().getId();
			}catch (Exception e) {
				return "";
			}
		}
		
		@Override
		public String toString() {
			return id;
		}

	}
}
