package com.cku.oa.goods.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.goods.dao.SkuItemRelDao;
import com.cku.oa.goods.entity.SkuItemRel;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * SKU收费项关联Service
 *
 * @author yuanshuai
 * @version 2023-06-16
 */
@Service
@Transactional(readOnly = true)
public class SkuItemRelService extends CrudService<SkuItemRelDao, SkuItemRel> {

    public SkuItemRel get(String id) {
        return super.get(id);
    }

    public List<SkuItemRel> findList(SkuItemRel skuItemRel) {
        return super.findList(skuItemRel);
    }

    public Page<SkuItemRel> findPage(Page<SkuItemRel> page, SkuItemRel skuItemRel) {
        return super.findPage(page, skuItemRel);
    }

    /**
     * 根据SKU查找关联
     *
     * @author yuanshuai
     * @date 2023/7/28 17:22
     */
    public List<SkuItemRel> findListBySku(String skuId) {
        if (StringUtils.isBlank(skuId)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "商品ID为空");
        }
        SkuItemRel rel = new SkuItemRel();
        rel.setSkuId(skuId);
        return findList(rel);
    }

    /**
     * 根据SKU查找收费项ID
     *
     * @author yuanshuai
     * @date 2023/7/28 17:29
     */
    public String findItemIdsBySku(String skuId) {
        List<SkuItemRel> relList = findListBySku(skuId);
        return relList.stream()
                .map(SkuItemRel::getChargingItemId)
                .collect(Collectors.joining(","));
    }

    @Transactional(readOnly = false)
    public void save(SkuItemRel skuItemRel) {
        super.save(skuItemRel);
    }

    @Transactional(readOnly = false)
    public void save(String skuId, String itemIds) {
        //拼装item
        List<SkuItemRel> relList = Stream.of(itemIds.split(","))
                .map(i -> {
                    SkuItemRel rel = new SkuItemRel();
                    rel.setSkuId(skuId);
                    rel.setChargingItemId(i);
                    return rel;
                })
                .collect(Collectors.toList());
        //查找原item
        List<SkuItemRel> originList = findListBySku(skuId);
        //处理新增
        relList.stream()
                .filter(o ->  originList.stream().noneMatch(rel -> rel.getChargingItemId().equals(o.getChargingItemId())))
                .forEach(this::save);
        //处理删除
        originList.stream()
                .filter(o ->  relList.stream().noneMatch(rel -> rel.getChargingItemId().equals(o.getChargingItemId())))
                .forEach(this::delete);
        relList.forEach(this::save);
    }

    @Transactional(readOnly = false)
    public void delete(SkuItemRel skuItemRel) {
        super.delete(skuItemRel);
    }

}