/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.dog.service;

import com.cku.global.service.RedisService;
import com.cku.oa.dog.dao.DogTypeDao;
import com.cku.oa.dog.entity.DogType;
import com.cku.oa.sys.service.LoginServerService;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 犬只种类Service
 *
 * @author zhuoHeng
 * @version 2016-07-21
 */
@Service
@Transactional(readOnly = true)
public class DogTypeService extends CrudService<DogTypeDao, DogType> {

    @Autowired
    private RedisService redisService;
    @Autowired
    private LoginServerService loginServerService;

    //redis-key
    private static final String REDIS_KEY_DOG_TYPE_ALL = "dog_type_all";

    /**
     * 初始化
     *
     * @author yuanshuai
     * @date 2023/7/5 10:16
     */
    @PostConstruct
    public void initDogType() {
        findAllList();
    }

    public DogType get(String id) {
        return super.get(id);
    }

    public List<DogType> findList(DogType dogType) {
        return super.findList(dogType);
    }

    public Page<DogType> findPage(Page<DogType> page, DogType dogType) {
        return super.findPage(page, dogType);
    }

    @Transactional(readOnly = false)
    public void save(DogType dogType) {
        super.save(dogType);
        redisService.remove(REDIS_KEY_DOG_TYPE_ALL);
    }

    @Transactional(readOnly = false)
    public void delete(DogType dogType) {
        super.delete(dogType);
    }

    public DogType getByDogBreedCode(String dogBreedCode) {
        return dao.getByDogBreedCode(dogBreedCode);
    }

    /**
     * @Description：获取犬种数据列表并将breedEnName拼接为：编号-中文名-英文名的形式
     * @author: zhuoHeng
     * @version: 2016年8月16日 下午5:44:03
     */
    public List<DogType> findAllList() {
        List<DogType> dogTypeList;
        if (redisService.exists(REDIS_KEY_DOG_TYPE_ALL)) {
            dogTypeList = redisService.get(REDIS_KEY_DOG_TYPE_ALL);
        } else {
            dogTypeList = dao.findAllList(new DogType());
            redisService.put(REDIS_KEY_DOG_TYPE_ALL, dogTypeList);
        }
        for (DogType d : dogTypeList) {
            d.setBreedEnName(d.getDogBreedCode() + "-" + d.getBreedCnName() + "-" + d.getBreedEnName());
        }
        return filterList(dogTypeList);
    }

    public List<DogType> findFormalAndNBList() {
        List<DogType> dogTypeList = dao.findFormalAndNBList(new DogType());
        filterList(dogTypeList);
        return dogTypeList;
    }

    private List<DogType> filterList(List<DogType> dogTypeList) {
        if (dogTypeList == null) {
            return null;
        }
        String dogTypeToList = loginServerService.getByDogTypeList();
        if (StringUtils.isBlank(dogTypeToList)) {
            return dogTypeList;
        }
        String[] split = dogTypeToList.replace("!", "").split(",");
        return dogTypeList.stream()
                .filter(d -> {
                    if (dogTypeToList.contains("!")) {
                        // 不支持的
                        return Stream.of(split).noneMatch(s -> s.equals(d.getDogBreedCode()));
                    } else {
                        // 支持的
                        return Stream.of(split).anyMatch(s -> s.equals(d.getDogBreedCode()));
                    }
                })
                .collect(Collectors.toList());
    }

    /**
     * 校验犬种代码的唯一性
     *
     * @Author chaixueteng
     * @2016年9月18日下午3:36:33
     */
    public String dogBreedCodeCheck(String dogBreedCode, String id) {
        List<DogType> list = dao.codeCheck(dogBreedCode);
        if (list.size() > 0) {
            String id2 = list.get(0).getId();
            if (id == null) {
                return "{\"rc\":false}";
            } else {
                if (!id.equals(id2)) {
                    return "{\"rc\":false}";
                }
            }
        }
        return "{\"rc\":true}";
    }

    /**
     * @description: 判断是否同一犬种, 是同一犬种返回true
     * @author: laiguanglong
     * @date: 2016年10月29日 上午11:46:48
     */
    public boolean isSameType(String dogBreedCode1, String dogBreedCode2) {
        boolean flag = false;
        if (dogBreedCode1.equals(dogBreedCode2)) {
            return true;
        }
        if (dogBreedCode1.startsWith("181") && dogBreedCode2.startsWith("181")) {
            return true;
        }
        if (dogBreedCode1.startsWith("182") && dogBreedCode2.startsWith("182")) {
            return true;
        }
        if (dogBreedCode1.startsWith("183") && dogBreedCode2.startsWith("183")) {
            return true;
        }
        return flag;
    }

    /**
     * @Description： OA后台配种规则校验
     * @author: yuanshuai
     * @date: 2019/8/15 11:05
     */
    public boolean isSameTypeForOABreed(String dogBreedCode1, String dogBreedCode2) {
        boolean flag = false;
        //148/172 允许配种
        if (dogBreedCode1.startsWith("148") && dogBreedCode2.startsWith("148")) {
            return true;
        }
        if (dogBreedCode1.startsWith("172") && dogBreedCode2.startsWith("172")) {
            return true;
        }
        return flag;
    }


    /**
     * @description:校验该犬种是否有该毛色，返回true表示有
     * @author: laiguanglong
     * @date: 2017/2/8 17:33
     */
    public boolean validateColorFlag(String dogBreed, String colorFlag) {
        boolean flag = false;
        DogType dogType = dao.getByDogBreedCode(dogBreed);
        if (StringUtils.isNotBlank(dogType.getDogColor())) {
            String[] dogColorArray = dogType.getDogColor().split(",");
            for (String dogColor : dogColorArray) {
                if (dogColor.equals(colorFlag)) {
                    flag = true;
                    break;
                }
            }
        }
        return flag;
    }

    /**
     * @description: 获取犬种表已经没有了的老犬种的犬种名
     * @author: laiguanglong
     * @date: 2017/4/17 15:09
     */
    public String getUnMapBreedCnName(String dogBreedCode) {
        String breedCnName = "";
        if ("11".equals(dogBreedCode)) {
            breedCnName = "牛头梗犬";
        } else if ("15".equals(dogBreedCode)) {
            breedCnName = "比利时牧羊犬";
        } else if ("97".equals(dogBreedCode)) {
            breedCnName = "德国尖嘴犬";
        } else if ("148".equals(dogBreedCode)) {
            breedCnName = "腊肠犬";
        } else if ("172".equals(dogBreedCode)) {
            breedCnName = "贵宾犬";
        } else if ("181".equals(dogBreedCode)) {
            breedCnName = "巨型雪纳瑞";
        } else if ("182".equals(dogBreedCode)) {
            breedCnName = "雪纳瑞";
        } else if ("183".equals(dogBreedCode)) {
            breedCnName = "迷你雪纳瑞";
        }
        return breedCnName;
    }

    /**
     * @description: 根据犬种代码获取犬种中文名
     * @author: laiguanglong
     * @date: 2017/10/10 16:56
     */
    public String getBreedCnName(String dogBreed) {
        String breedCnName = "";
        DogType dogType = dao.getByDogBreedCode(dogBreed);
        if (dogType == null) {
            breedCnName = getUnMapBreedCnName(dogBreed);
        } else {
            breedCnName = dogType.getBreedCnName();
        }
        return breedCnName;
    }

    public String getBreedEnName(String dogBreed) {
        String breedEnName = "";
        DogType dogType = dao.getByDogBreedCode(dogBreed);
        if (dogType != null) {
            breedEnName = dogType.getBreedEnName();
        } else {
            breedEnName = "";
        }
        return breedEnName;
    }


    /**
     * @Description： 查询无单犬种项目的犬种列表
     * @author: yuanshuai
     * @date: 2019/10/16 10:20
     */
    public List<DogType> findNoSingleList() {
        List<DogType> list = super.findList(new DogType());
        Iterator<DogType> iterator = list.iterator();
        while (iterator.hasNext()) {
            DogType type = iterator.next();
            if ("15C".equals(type.getDogBreedCode())
                    || "292".equals(type.getDogBreedCode())) {
                iterator.remove();
            }
        }
        return list;
    }

    /**
     * @Description： 查询单犬种俱乐部犬种列表
     * @author: yuanshuai
     * @date: 2019/10/21 13:58
     */
    public List<DogType> findSingOrgDogTypeList(String orgId) {
        List<DogType> dogTypeList = dao.findSingOrgDogTypeList(orgId);
        for (DogType dogType : dogTypeList) {
            dogType.setBreedEnName(dogType.getDogBreedCode() + "-" + dogType.getBreedCnName() + "-" + dogType.getBreedEnName());
        }
        return dogTypeList;
    }
}