/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.dog.service;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.ui.Model;
import org.springframework.util.CollectionUtils;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.geneTree.utils.GeneTreeReportUtils;
import com.cku.geneTree.vo.response.SamplingGeneticDiseaseResponse;
import com.cku.logUtil.JSONObjectUtils;
import com.cku.oa.dog.dao.DogBloodSaveDao;
import com.cku.oa.dog.dao.DogDnaArchiveDao;
import com.cku.oa.dog.dao.DogDnaArchiveReissueSamplingDao;
import com.cku.oa.dog.dao.DogPedigreeCertifiedChangeDao;
import com.cku.oa.dog.entity.Dog;
import com.cku.oa.dog.entity.DogBirthCertificate;
import com.cku.oa.dog.entity.DogBloodSave;
import com.cku.oa.dog.entity.DogDna;
import com.cku.oa.dog.entity.DogDnaArchive;
import com.cku.oa.dog.entity.DogDnaArchiveReissueSampling;
import com.cku.oa.dog.entity.DogPedigreeCertifiedChange;
import com.cku.oa.dog.entity.DogType;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.finance.entity.PaymentOrderDetail;
import com.cku.oa.finance.service.PaymentChargingItemService;
import com.cku.oa.finance.service.PaymentOrderRefundService;
import com.cku.oa.finance.service.PaymentOrderService;
import com.cku.oa.finance.service.SaPaymentDetailTotalService;
import com.cku.oa.sampling.entity.Sampling;
import com.cku.oa.sampling.enums.SamplingDistributionChannelEnum;
import com.cku.oa.sampling.enums.SamplingDogSourceTypeEnum;
import com.cku.oa.sampling.enums.SamplingFlowStateEnum;
import com.cku.oa.sampling.service.SamplingService;
import com.cku.oa.sys.entity.Org;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.ExpressQueueService;
import com.cku.oa.sys.service.LoginServerService;
import com.cku.oa.sys.service.OrgService;
import com.cku.oa.sys.zalog.SysContext;
import com.cku.partner.club.service.ClubDogChipDeleteService;
import com.cku.restful.v1.finance.service.OADeleteService;
import com.cku.util.DateUtils;
import com.thinkgem.jeesite.common.config.Global;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import edu.emory.mathcs.backport.java.util.Arrays;
import jodd.util.StringUtil;

/**
 * 犬只DNA存档Service
 *
 * @author lgl
 * @version 2016-08-08
 */
@Service
@Transactional(readOnly = true)
public class DogDnaArchiveService extends CrudService<DogDnaArchiveDao, DogDnaArchive> {

	private static final String BUSINESS_NAME = "DNA存档";

	@Autowired
	private DogService dogService;
	@Autowired
	private DogBirthCertificateService dogBirthCertificateService;
	@Autowired
	private DogDnaService dogDnaService;
	@Autowired
	private DogTypeService dogTypeService;
	@Autowired
	private PaymentChargingItemService paymentChargingItemService;
	@Autowired
	private ExpressQueueService expressQueueService;
	@Autowired
	private PaymentOrderService paymentOrderService;
	@Autowired
	private OADeleteService oADeleteService;
	@Autowired
	private ClubDogChipDeleteService clubDogChipDeleteService;
	@Autowired
	private DogPedigreeCertifiedChangeService dogPedigreeCertifiedChangeService;
	@Autowired
	private DogPedigreeCertifiedChangeDao dogPedigreeCertifiedChangeDao;
	@Autowired
	private LoginServerService loginServerService;
	@Autowired
	private SaPaymentDetailTotalService saPaymentDetailTotalService;
	@Autowired
	private SamplingService samplingService;
	@Autowired
	private OrgService orgService;
	@Autowired
	private DogBloodSaveService dogBloodSaveService;
	@Autowired
	private DogDnaArchiveReissueSamplingDao dogDnaArchiveReissueSamplingDao;
	@Autowired
	private DogBloodSaveDao dogBloodSaveDao;
	@Autowired
	private PaymentOrderRefundService paymentOrderRefundService;
	@Autowired
	private DogDnaParentsCompareAuditService dogDnaParentsCompareAuditService;

	public DogDnaArchive get(String id) {
		return super.get(id);
	}

	/**
	 * @description: DNA存档打印
	 * @author: laiguanglong
	 * @date: 2016年8月30日 下午2:30:04
	 */
	@Transactional(readOnly = false)
	public DogDnaArchive print(String id) throws ZAException {
		DogDnaArchive dogDnaArchive = super.get(id);
		String printState = "1";
		dao.updatePrintFlay(id, printState);
		DogType dogType = dogTypeService.getByDogBreedCode(dogDnaArchive.getDogBreed());
		if (dogType == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,
					"该犬只的犬种" + dogDnaArchive.getDogBreed() + ",在犬种表无法找到对应的记录，请检查是否有该犬种");
		}
		dogDnaArchive.setBreedCnName(dogType.getBreedCnName());
		DogDna dogDna = dogDnaService.getByDogId(dogDnaArchive.getDogId());
		String[] dnaResults = dogDna.getDnaResult().split("	");
		StringBuilder sb = new StringBuilder(
				"<tr><th>PEZ1</th><th>PEZ2</th><th>FH2010</th><th>PEZ5</th><th>PEZ12</th><th>PEZ21</th><th>PEZ3</th><th>PEZ6</th><th>PEZ8</th><th>FH2054</th><th>FH2611</th><th>FH2132</th><th>PEZ20</th><th>PEZ15</th><th>FH2079</th></tr><tr>");
		for (int i = 0; i < dnaResults.length; i = i + 2) {
			sb.append("<td>" + dnaResults[i] + "</td>");
		}
		sb.append("</tr><tr>");
		for (int i = 1; i < dnaResults.length; i = i + 2) {
			sb.append("<td>" + dnaResults[i] + "</td>");
		}
		sb.append("</tr>");
		dogDnaArchive.setPrintTable(sb.toString());
		// 通过状态机记录日志
		SysContext sysContext = new SysContext(null, null, null);
		sysContext.noStateExecute(sysContext, BUSINESS_NAME + "打印", dogDnaArchive.getPaymentMemberCode(),
				"dogDnaArchive", dogDnaArchive.getId(), dogDnaArchive.getRunningNumber(), (str) -> {
				});
		// 添加待邮寄物品
		expressQueueService.push(dogDnaArchive.getPaymentMemberCode(), dogDnaArchive.getRunningNumber(), "DNA位点分析证书",
				dogDnaArchive.getPedigreeCertified(), null); // GYH2 TODO
		return dogDnaArchive;
	}

	public List<DogDnaArchive> findList(DogDnaArchive dogDnaArchive) {
		if (StringUtils.isBlank(dogDnaArchive.getDogBreed())) {
			// 根据当前登录端查询犬种
			dogDnaArchive.getSqlMap().put("dogBreedMap", loginServerService.dataScopeFilter());
		} else {
			dogDnaArchive.getSqlMap().put("dogBreedMap", " ='" + dogDnaArchive.getDogBreed() + "'");
		}
		return super.findList(dogDnaArchive);
	}

	public Page<DogDnaArchive> findPage(Page<DogDnaArchive> page, DogDnaArchive dogDnaArchive) {
		if (StringUtils.isBlank(dogDnaArchive.getDogBreed())) {
			// 根据当前登录端查询犬种
			dogDnaArchive.getSqlMap().put("dogBreedMap", loginServerService.dataScopeFilter());
		} else {
			dogDnaArchive.getSqlMap().put("dogBreedMap", " ='" + dogDnaArchive.getDogBreed() + "'");
		}
		return super.findPage(page, dogDnaArchive);
	}

	@Transactional(readOnly = false)
	public void save(DogDnaArchive dogDnaArchive) {
		super.save(dogDnaArchive);
	}

	@Transactional(readOnly = false)
	public void delete(DogDnaArchive dogDnaArchive) {
		// 已缴费的订单不允许删除
		if ("2".equals(dogDnaArchive.getPaymentState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该订单已缴费，不允许删除");
		}
		// OA删除订单
		oADeleteService.deleteUnpay(dogDnaArchive.getPaymentState(), "dog_dna_archive", dogDnaArchive.getId(),
				dogDnaArchive.getRunningNumber());
		// 如果是oa报名的话，没有订单，故没走上面的删除，所以需要再次删除
		super.delete(dogDnaArchive);
		// 地方俱乐部未交费订单删除
		clubDogChipDeleteService.deleteUnpayDogChipBusiness(dogDnaArchive.getBirthCode());
		// 解绑采样包
		if (StringUtils.isNotBlank(dogDnaArchive.getSamplingCode())) {
			samplingService.unbindDogInfoBySamplingCode(dogDnaArchive.getSamplingCode());
		}
	}

	/**
	 * @description: DNA存档审核页面
	 * @author: laiguanglong
	 * @date: 2016年8月18日 下午2:21:52
	 */
	public String auditForm(String id, Model model) throws ZAException {
		DogDnaArchive dogDnaArchive = dao.get(id);
		if (StringUtils.isNotBlank(dogDnaArchive.getOverseasPedigreeCertified())) {
			DogPedigreeCertifiedChange dogPedigreeCertifiedChange = dogPedigreeCertifiedChangeDao
					.getByOverseasPedigreeCertified(dogDnaArchive.getOverseasPedigreeCertified());
			if (dogPedigreeCertifiedChange != null) {
				dogDnaArchive.setDogName(dogPedigreeCertifiedChange.getDogName());
				dogDnaArchive.setDogBirthday(dogPedigreeCertifiedChange.getDogBirthday());
				dogDnaArchive.setDogGender(dogPedigreeCertifiedChange.getDogGender());
			}
		}
		if (StringUtils.isNotBlank(dogDnaArchive.getBirthCode())) {
			DogBirthCertificate dogBirthCertificate = dogBirthCertificateService
					.getByBirthCerRegCode(dogDnaArchive.getBirthCode());
			if (dogBirthCertificate != null) {
				dogDnaArchive.setDogName(dogBirthCertificate.getDogEnName());
				dogDnaArchive.setDogBirthday(dogBirthCertificate.getDogBirthday());
				dogDnaArchive.setDogGender(dogBirthCertificate.getDogGender());
			}
		}
		model.addAttribute("dogDnaArchive", dogDnaArchive);
		return "oa/dog/dnaarchive/dogDnaArchiveAuditForm";
	}

	/**
	 * @description: DNA存档审核
	 * @author: laiguanglong
	 * @date: 2016年8月18日 下午2:21:52
	 */
	@Transactional(readOnly = false)
	public void audit(DogDnaArchive audit) throws ZAException {
		try {
			DogDnaArchive dogDnaArchive = get(audit.getId());
			// 未通过可继续审核
			if ("2".equals(dogDnaArchive.getReviewState())) {
				dogDnaArchive.setReviewState("0");
			}
			SysContext sysContext = new SysContext(dogDnaArchive.getPaymentState(), dogDnaArchive.getReviewState(),
					dogDnaArchive.getProcessState());
			if ("1".equals(audit.getReviewState())) {
				// 通过
				dogDnaArchive.setReviewState("1");
				sysContext.confirmSuccess(sysContext, BUSINESS_NAME, dogDnaArchive.getPaymentMemberCode(),
						"dogDnaArchive", dogDnaArchive.getId(), dogDnaArchive.getRunningNumber(), (str) -> {
						});
				// TODO 财务收入确认时点
				saPaymentDetailTotalService.financeConfirmTime(dogDnaArchive.getId(), "dog_dna_archive");
			} else if ("2".equals(audit.getReviewState())) {
				// 未通过
				dogDnaArchive.setReviewState("2");
				sysContext.confirmFail(sysContext, BUSINESS_NAME, audit.getReviewRemarks(),
						dogDnaArchive.getPaymentMemberCode(), "dogDnaArchive", dogDnaArchive.getId(),
						dogDnaArchive.getRunningNumber(), (str) -> {
						});
			}
			dogDnaArchive.setReviewRemarks(audit.getReviewRemarks());
			dogDnaArchive.setReviewTime(new Date());
			super.save(dogDnaArchive);
		} catch (Exception e) {
			throw e;
		}
	}

	/**
	 * @description: 显示处理页面
	 * @author: laiguanglong
	 * @date: 2016年8月22日 下午2:11:20
	 */
	public DogDna resolveForm(DogDnaArchive resolveForm) {
		DogDnaArchive dogDnaArchive = get(resolveForm.getId());
		Dog dog = dogService.get(dogDnaArchive.getDogId());
		DogDna dogDna = new DogDna();
		dogDna.setDnaArchiveId(dogDnaArchive.getId());
		dogDna.setDogId(dog.getId());
		dogDna.setPedigreeCertifiedCode(dog.getPedigreeCertifiedCode());
		dogDna.setNameCn(dog.getNameEn());
		dogDna.setBirthdate(dog.getBirthdate());
		dogDna.setGender(dog.getGender());
		dogDna.setDogBreed(dog.getDogBreed());
		if (StringUtils.isNotBlank(dogDnaArchive.getDnaCode())) {
			DogDna dna = dogDnaService.getByDnaCode(dogDnaArchive.getDnaCode());
			if (dna != null) {
				dogDna.setDnaCode(dna.getDnaCode());
				dogDna.setDnaResult(dna.getDnaResult());
				dogDna.setDnaResultRemark(dna.getDnaResultRemark());
			}
		}
		return dogDna;
	}

	/**
	 * @description: DNA存档处理
	 * @author: laiguanglong
	 * @date: 2016年8月18日 下午2:53:14
	 */
	@Transactional(readOnly = false)
	public void resolve(DogDna dogDna) throws ZAException {
		try {
			DogDnaArchive dogDnaArchive = get(dogDna.getDnaArchiveId());
			if (StringUtils.isBlank(dogDnaArchive.getDnaCode())) {
				// 处理
				SysContext sysContext = new SysContext(dogDnaArchive.getPaymentState(), dogDnaArchive.getReviewState(),
						dogDnaArchive.getProcessState());
				sysContext.updateData(sysContext, BUSINESS_NAME, dogDnaArchive.getPaymentMemberCode(), "dogDnaArchive",
						dogDnaArchive.getId(), dogDnaArchive.getRunningNumber(), (str) -> {
						});
				// 初始化dna信息
				dogDna.setDetectionType(Integer.parseInt(dogDnaArchive.getIsOEC()));
				dogDna.setSamplingCode(dogDnaArchive.getSamplingCode());
				// 新增dna信息
				dogDnaService.buildAndInsert(dogDna, Boolean.TRUE);
			} else {
				// 更新
				DogDna po = dogDnaService.getByDnaCode(dogDnaArchive.getDnaCode());
				po.setDnaResult(dogDna.getDnaResult());
				po.setDnaResultRemark(dogDna.getDnaResultRemark());
				// 如果是NB开头的犬种更新DNA编码  	ID: 1006553 页面不再录入DNA编号
//				if (dogDna.getDogBreed().startsWith("NB")) {
//					po.setDnaCode(dogDna.getDnaCode());
//					dogDnaArchive.setDnaCode(dogDna.getDnaCode());
					// 更新DNA存档表的处理状态
//					save(dogDnaArchive);
					// 更新犬只信息
					// 修改不更新采样包编号
//					updateDog(po, null);
//				}
				dogDnaService.save(po);
			}
		} catch (Exception e) {
			throw e;
		}
	}

	/**
	 * @description: 更新DNA存档表的处理状态
	 * @author: laiguanglong
	 * @date: 2016年8月18日 下午3:47:12
	 */
	public void updateDogDnaArchive(DogDna dogDna, Boolean processState) {
		DogDnaArchive dogDnaArchive = findByDogId(dogDna.getDogId());
		dogDnaArchive.setDnaCode(dogDna.getDnaCode());
		dogDnaArchive.setSamplingCode(dogDna.getSamplingCode());
		if (processState) {
			dogDnaArchive.setProcessState("1");
			dogDnaArchive.setProcessTime(new Date());
		}
		super.save(dogDnaArchive);
	}

	/**
	 * @description: 更新犬只打印状态
	 * @author: laiguanglong
	 * @date: 2016年8月18日 下午3:06:11
	 */
	public void updateDog(DogDna dogDna, String samplingCode) {
		Dog dog = dogService.get(dogDna.getDogId());
		dog.setDnaPayState("1");
		dog.setDna(dogDna.getDnaCode());
		if (StringUtils.isNotBlank(samplingCode)) {
			dog.setSamplingCode(samplingCode);
		}
		dogService.save(dog);
	}

	@Transactional(readOnly = false)
	public void editApplyChannel(DogDnaArchive dogDnaArchive) {
		dogDnaArchive = this.get(dogDnaArchive.getId());
		if (Objects.isNull(dogDnaArchive)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "DNA存档记录不存在");
		}
		if (Objects.isNull(dogDnaArchive.getApplyChannel()) || 0 != dogDnaArchive.getApplyChannel()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "非机构办理的DNA存档无法更改");
		}
		if (StringUtils.isNotBlank(dogDnaArchive.getSamplingCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "DNA存档已绑定采样包，无法修改");
		}
		// 1 为自检
		User user = UserUtils.getUser();

		dogDnaArchive.setApplyChannel(1);
		String remarks = StringUtils.isBlank(dogDnaArchive.getRemarks()) ? "" : (dogDnaArchive.getRemarks() + " | ");
		remarks = String.format("%s%s修改办理方式为自检(%s)", remarks, user.getName(), DateUtils.getStringDate());
		dogDnaArchive.setRemarks(remarks);
		dogDnaArchive.preUpdate();
		this.dao.updateApplyChannel(dogDnaArchive);
		saveSamplingExpressQueue(dogDnaArchive,DOG_DNA_ARCHIVE_OEC_FZ);
	}

	private void saveSamplingExpressQueue(DogDnaArchive dogDnaArchive, String chargingItemId) {
		// 生成待邮寄物品
		String certificateNo = "";
		if (StringUtils.isNotBlank(dogDnaArchive.getPedigreeCertified())) {
			// certificateName = "血统证书号";
			certificateNo = dogDnaArchive.getPedigreeCertified();
		} else if (StringUtils.isNotBlank(dogDnaArchive.getBirthCode())) {
			// certificateName = "出生纸编号";
			certificateNo = dogDnaArchive.getBirthCode();
		} else if (StringUtils.isNotBlank(dogDnaArchive.getBirthCode())) {
			// certificateName = "国外证书编号";
			certificateNo = dogDnaArchive.getOverseasPedigreeCertified();
		}
		expressQueueService.push(dogDnaArchive.getPaymentMemberCode(), dogDnaArchive.getRunningNumber(), "采样包",
				certificateNo, "dog_dna_archive", dogDnaArchive.getId(), chargingItemId);
	}

	private void pushExpress(PaymentOrderDetail paymentOrderDetail, Member oldMember) {
		String[] idArr = paymentOrderDetail.getBusinessIds().split(",");
		Stream.of(idArr).forEach(id -> {
			DogDnaArchive dogdnaArchive = this.get(id);

		});
	}

	public static final String DOG_DNA_ARCHIVE_OEC_FZ = "4625f60e8ac44d37b23ce2d972d1b532";
	
	public final static List<String> DOG_DNA_ARCHIVE_OEC_FZ_IDS = Arrays
			.asList(new String[] { "56444644f3a64d1b96e53236a85e1948","4df8d3ef339a4ea8b6f1167894567eea","4625f60e8ac44d37b23ce2d972d1b532" });

	/**
	 * @description: 保存DNA存档申请
	 * @author: laiguanglong
	 * @date: 2016年8月24日 下午4:08:43
	 */
	@Transactional(readOnly = false, rollbackFor = Exception.class)
	public void add(DogDnaArchive add) throws ZAException {
		if (StringUtils.isBlank(add.getPedigreeCertified()) && StringUtils.isBlank(add.getBirthCode())
				&& StringUtils.isBlank(add.getOverseasPedigreeCertified())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "请输入血统证书号或出生纸编号或外国血统证书号");
		}
		DogDnaArchive dogDnaArchive = new DogDnaArchive();
		Dog dog = null;
		DogBirthCertificate dogBirthCertificate = null;
		if (StringUtils.isNotBlank(add.getPedigreeCertified())) {
			dog = dogService.getByPedigreeCertifiedCode(add.getPedigreeCertified());
			if (dog == null) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号有误");
			}
			// 0根据血统证书号查重
			if (dao.getByDogId(dog.getId()) != null) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只已有DNA位点分析业务记录");
			}
			dogDnaArchive.setDogId(dog.getId());
			dogDnaArchive.setPedigreeCertified(dog.getPedigreeCertifiedCode());
			dogDnaArchive
					.setBirthCode(dog.getBirthCertificateLevelCode() == null ? "" : dog.getBirthCertificateLevelCode());
		}
		if (StringUtils.isNotBlank(add.getBirthCode())) {
			dogBirthCertificate = dogBirthCertificateService.getByBirthCerRegCode(add.getBirthCode());
			if (dogBirthCertificate == null) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "出生纸编号有误");
			}
			// 0根据出生证明查重
			if (dao.getByBirthCode(dogBirthCertificate.getBirthCerRegCode()) != null) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该出生纸已有DNA位点分析业务记录");
			}
			dogDnaArchive.setBirthCode(dogBirthCertificate.getBirthCerRegCode());
			dog = dogService.getByBirthCertificateLevelCode(dogBirthCertificate.getBirthCerRegCode());
			if (dog != null) {
				dogDnaArchive.setDogId(dog.getId());
				dogDnaArchive.setPedigreeCertified(dog.getPedigreeCertifiedCode());
			}
		}
		if (dog != null && dogBirthCertificate != null
				&& !dog.getBirthCertificateLevelCode().equals(dogBirthCertificate.getBirthCerRegCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该血统证书号与出生纸编号不匹配，请检查");
		}
		// 证书换发业务位点分析录入
		if (StringUtils.isNotBlank(add.getOverseasPedigreeCertified())) {
			DogPedigreeCertifiedChange dogPedigreeCertifiedChange = dogPedigreeCertifiedChangeService
					.getByOverseasPedigreeCertified(add.getOverseasPedigreeCertified());
			if (dogPedigreeCertifiedChange == null) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "国外血统证书编号有误");
			}
			if (StringUtils.isNotBlank(dogPedigreeCertifiedChange.getDogDnaArchiveId())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该外国血统证书已有DNA位点分析业务记录");
			}
			// 0根据国外血统证书编号查重
			if (dao.getByOverseasPedigreeCertified(dogPedigreeCertifiedChange.getOverseasPedigreeCertified()) != null) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该国外血统证书编号已有DNA位点分析业务记录");
			}
			dogDnaArchive.setOverseasPedigreeCertified(dogPedigreeCertifiedChange.getOverseasPedigreeCertified());
		}
		dogDnaArchive.setPaymentMemberCode(add.getPaymentMemberCode());
		String price = null;
		if (DOG_DNA_ARCHIVE_OEC_FZ_IDS.contains(add.getPaymentChargingItemId())) {
			dogDnaArchive.setIsOEC("1");
			DogBloodSave dogBloodSave = new DogBloodSave();
			if (0 == add.getApplyChannel()) {// 机构办理
				Org org = orgService.get(add.getOrgId());
				dogDnaArchive.setOrgId(add.getOrgId());
				dogDnaArchive.setOrganizationName(org.getName());
				dogBloodSave.setOrgId(add.getOrgId());
				dogBloodSave.setOrganizationName(org.getName());
			} else if (1 == add.getApplyChannel()) {// 自检
				// 插入待邮寄记录
				saveSamplingExpressQueue(dogDnaArchive,add.getPaymentChargingItemId());
			}
			price = getPriceByRunningNumber(add.getRunningNumber()).toString();
			// 插入一条血样留存记录
			dogBloodSave.setBirthCode(add.getBirthCode());
			dogBloodSave.setPedigreeCertified(add.getPedigreeCertified());
			dogBloodSave.setOverseasPedigreeCertified(add.getOverseasPedigreeCertified());
			dogBloodSave.setRunningNumber(add.getRunningNumber());
			dogBloodSave.setPaymentMemberCode(add.getPaymentMemberCode());
			dogBloodSaveService.addDogBloodSaveOEC(dogBloodSave);
		} else {
			PaymentChargingItem paymentChargingItem = paymentChargingItemService.get(add.getPaymentChargingItemId());
			price = paymentChargingItem.getPrice();
		}
		dogDnaArchive.setPrice(price);
		dogDnaArchive.setReviewState("0");
		dogDnaArchive.setRunningNumber(add.getRunningNumber());
		dogDnaArchive.setPaymentState("2");
		dogDnaArchive.setPaymentTime(paymentOrderService.getPaymentTimeByOrderCode(add.getRunningNumber()));
		dogDnaArchive.setProcessState("0");
		dogDnaArchive.setPrintState("0");
		dogDnaArchive.setApplyChannel(add.getApplyChannel());
		super.save(dogDnaArchive);
		// 更新证书换发表的dogDnaArchiveId、dogChipCheck字段
		if (StringUtils.isNotBlank(add.getOverseasPedigreeCertified())) {
			DogPedigreeCertifiedChange dogPedigreeCertifiedChange = dogPedigreeCertifiedChangeService
					.getByOverseasPedigreeCertified(add.getOverseasPedigreeCertified());
			dogPedigreeCertifiedChange.setDogDnaArchiveId(dogDnaArchive.getId());
			dogPedigreeCertifiedChange.setDogChipCheck("1");
			dogPedigreeCertifiedChangeService.save(dogPedigreeCertifiedChange);
		}
	}

	public Double getPriceByRunningNumber(String runningNumber) {
		List<PaymentOrderDetail> list = paymentOrderService.findOrderDetailByOrderCode(runningNumber);
		// 位点分析口腔上皮细胞 涉及的计算金额收费项有三个
		List<String> DNA_ARCHIVE_FEE = java.util.Arrays.asList("4625f60e8ac44d37b23ce2d972d1b532", "3f53d07d1006491189b0a0974fb2e443");

		return list.stream().filter(a -> DNA_ARCHIVE_FEE.contains(a.getChargingItemId()))
				.mapToDouble(a -> Double.valueOf(a.getPrice())).sum();
	}

	/**
	 * @description: 添加DNA存档订单页面，根据血统证书号查询犬只信息
	 * @author: laiguanglong
	 * @date: 2016年8月23日 下午5:22:03
	 */
	public HashMap<String, String> getDogByPedigreeCertified(String pedigreeCertified) {
		HashMap<String, String> map = new HashMap<String, String>();
		Dog dog = dogService.getByPedigreeCertifiedCode(pedigreeCertified);
		if (dog != null) {
			map.put("nameEn", dog.getNameEn());
			map.put("birthdate", DateUtils.dateToStr(dog.getBirthdate()));
			map.put("gender", DictUtils.getDictLabel(dog.getGender(), "dog_gender", ""));
			map.put("colorFlag", dog.getColorFlag());
			map.put("identificationFlag", dog.getIdentificationFlag());
			DogType dogType = dogTypeService.getByDogBreedCode(dog.getDogBreed());
			if (dogType != null) {
				map.put("dogBreed", dogType.getBreedCnName());
			}
		}
		return map;
	}

	/**
	 * @description: 添加DNA存档订单页面，根据出生纸编号查询犬只信息
	 * @author: laiguanglong
	 * @date: 2016年8月23日 下午5:22:03
	 */
	public HashMap<String, String> getDogByBirthCode(String birthCode) {
		HashMap<String, String> map = new HashMap<String, String>();
		DogBirthCertificate dogBirthCertificate = dogBirthCertificateService.getByBirthCerRegCode(birthCode);
		if (dogBirthCertificate != null) {
			map.put("nameEn", dogBirthCertificate.getDogEnName());
			map.put("birthdate", DateUtils.dateToStr(dogBirthCertificate.getDogBirthday()));
			map.put("gender", DictUtils.getDictLabel(dogBirthCertificate.getDogGender(), "dog_gender", ""));
			map.put("colorFlag", dogBirthCertificate.getColorFlag());
			map.put("identificationFlag", dogBirthCertificate.getIdentificationFlag());
			DogType dogType = dogTypeService.getByDogBreedCode(dogBirthCertificate.getDogBreed());
			if (dogType != null) {
				map.put("dogBreed", dogType.getBreedCnName());
			}
		}
		return map;
	}

	/**
	 * @description: 添加DNA存档订单页面，根据国外血统证书编号查询犬只信息
	 * @author: laiguanglong
	 * @date: 2016年8月23日 下午5:22:03
	 */
	public HashMap<String, String> getDogByOverseasPedigreeCertified(String overseasPedigreeCertified) {
		HashMap<String, String> map = new HashMap<String, String>();
		DogPedigreeCertifiedChange dogPedigreeCertifiedChange = dogPedigreeCertifiedChangeService
				.getByOverseasPedigreeCertified(overseasPedigreeCertified);
		if (dogPedigreeCertifiedChange != null) {
			map.put("nameEn", dogPedigreeCertifiedChange.getDogName());
			map.put("birthdate", DateUtils.dateToStr(dogPedigreeCertifiedChange.getDogBirthday()));
			map.put("gender", DictUtils.getDictLabel(dogPedigreeCertifiedChange.getDogGender(), "dog_gender", ""));
			map.put("colorFlag", dogPedigreeCertifiedChange.getDogFurColor());
			map.put("identificationFlag", dogPedigreeCertifiedChange.getDogChip());
			DogType dogType = dogTypeService.getByDogBreedCode(dogPedigreeCertifiedChange.getDogBreed());
			if (dogType != null) {
				map.put("dogBreed", dogType.getBreedCnName());
			}
		}
		return map;
	}

	/**
	 * @description: 获取已付款待审核的DNA存档申请记录数
	 * @author: laiguanglong
	 * @date: 2016年8月29日 下午3:54:04
	 */
	public int getAuditCount() {
		DogDnaArchive dogDnaArchive = new DogDnaArchive();
		// 已付款
		dogDnaArchive.setPaymentState("2");
		// 未审核
		dogDnaArchive.setReviewState("0");
		return dao.listCount(dogDnaArchive);
	}

	/**
	 * 重置打印
	 *
	 * @Author chaixueteng
	 * @2016年9月9日上午10:57:15
	 */
	@Transactional(readOnly = false)
	public void resetPrintprint(String id) {
		String printState = "0";
		dao.updatePrintFlay(id, printState);
	}

	public DogDnaArchive getByBirthCode(String birthCode) {
		return dao.getByBirthCode(birthCode);
	}

	/**
	 * 验证犬只DNA存档记录
	 *
	 * @Author chaixueteng
	 * @2016年11月25日下午2:54:03
	 */
	public DogDnaArchive getByDogId(String id) {
		return dao.getDnaMsg(id);
	}

	/**
	 * 合作机构采集Dna 支付后操作
	 *
	 * @param businessIds
	 */
	@Transactional(readOnly = false)
	public void afterPay(String businessIds) {
		String[] idArr = businessIds.split(",");
		Stream.of(idArr).forEach(id -> {
			DogDnaArchive dogDnaArchive = dao.get(id);
			if (StringUtils.isBlank(dogDnaArchive.getBirthCode())
					&& StringUtils.isNotBlank(dogDnaArchive.getOverseasPedigreeCertified())) {// 国外换发
				String overseasPedigreeCertified = dogDnaArchive.getOverseasPedigreeCertified();
				DogPedigreeCertifiedChange dogPedigreeCertifiedChange = dogPedigreeCertifiedChangeService
						.getByUnPayPedigreeCertified(overseasPedigreeCertified);
				if (dogPedigreeCertifiedChange != null) {
					// 修改dna标记位
					dogPedigreeCertifiedChange.setDogDnaArchiveId(dogDnaArchive.getId());
					dogPedigreeCertifiedChangeService.save(dogPedigreeCertifiedChange);
				}
			}
		});
	}

	/**
	 * 采样包补寄 支付后操作 补寄采样包 需要1、修改原采样包状态为失败，2、更新DNA存档表新采样包编号，3、生成采样包邮寄记录
	 *
	 * @param businessIds
	 */
	@Transactional(readOnly = false)
	public void afterPayReissueSampling(String businessIds, String charingItemId) {
		DogDnaArchiveReissueSampling reissueSampling = dogDnaArchiveReissueSamplingDao.get(businessIds);
		if (Objects.isNull(reissueSampling)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "采样包补寄记录有误");
		}
		// 采样包换绑
		Sampling sampling = samplingService.reissueBind(reissueSampling.getSamplingCode(),
				reissueSampling.getOldSamplingCode(), reissueSampling.getPaymentMemberCode(),
				reissueSampling.getPaymentMemberName(), reissueSampling.getRemarks());
		// 更新DNA存档表新采样包编号
		DogDnaArchive dogDnaArchive = this.get(reissueSampling.getDnaArchiveId());
		dogDnaArchive.setSamplingCode(reissueSampling.getSamplingCode());
		dogDnaArchive.setUpdateBy(reissueSampling.getCreateBy());
		dogDnaArchive.setUpdateDate(new Date());
		dao.updateSamplingInfo(dogDnaArchive);
		// 生成待邮寄物品
		expressQueueService.push(reissueSampling.getPaymentMemberCode(), reissueSampling.getRunningNumber(), "采样包补寄",
				sampling.getDogCode(), "dog_dna_archive_reissue_sampling", reissueSampling.getId(), charingItemId);
	}

	/**
	 * 根据芯片号/血统证书号查询存档记录
	 *
	 * @Author chaixueteng
	 * @2017年9月6日下午4:36:40
	 */
	public DogDnaArchive getByPediCodeOrChipCode(String pedigreeCertifiedCode) {
		return dao.getByPediCodeOrChipCode(pedigreeCertifiedCode);
	}

	/**
	 * 根据证书号查DNA存档
	 *
	 * @author yuanshuai
	 * @date 2023/8/2 16:43
	 */
	public DogDnaArchive getByPedigreeCertifiedCode(String pedigreeCertifiedCode) {
		return dao.getByPedigreeCertifiedCode(pedigreeCertifiedCode);
	}

	/**
	 * 根据国外证书号/出生纸编号/血统证书号查询存档记录
	 *
	 * @Author zhangjunwu
	 * @2023年07月03日下午4:36:40
	 */
	public DogDnaArchive getByOverseasPedigreeCertifiedOrBirthCode(String keyword) {
		return dao.getByOverseasPedigreeCertifiedOrBirthCode(keyword);
	}

	@Transactional(readOnly = false)
	public void updateOrg(String birthCode, String orgId) {
		dao.updateOrg(birthCode, orgId);
	}

	private void validIsRefundOrder(String orderCode) {
		if (paymentOrderRefundService.validOrderIsRefundingOrRefunded(orderCode)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该业务已申请退款，不可绑定");
		}
	}

	/**
	 * 绑定采样包
	 *
	 * @author yuanshuai
	 * @date 2023/5/26 15:22
	 */
	@Transactional(readOnly = false)
	public void bindSampling(DogDnaArchive dogDnaArchive, Boolean isRebind) {
		DogDnaArchive po = get(dogDnaArchive.getId());
		validIsRefundOrder(po.getRunningNumber());
		if (Objects.isNull(po) || StringUtils.isBlank(po.getId())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "DNA存档信息有误，请重试");
		}
		if (!"2".equals(po.getPaymentState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "未付款信息不可绑定");
		}
		if (!isRebind && !StringUtils.isBlank(po.getSamplingCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该DNA存档已绑定采样包");
		}
		if (isRebind && StringUtils.isBlank(po.getSamplingCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该DNA存档尚未绑定采样包");
		}
		// 是否含有双亲对比项目，含有的需要验证父母双方是否同意
		if (StringUtils.isNotBlank(dogDnaArchive.getRunningNumber())) {
			dogDnaParentsCompareAuditService.cherkByOrderCode(dogDnaArchive.getRunningNumber(),
					dogDnaArchive.getPedigreeCertified());
		}
		Sampling sampling = samplingService.findBySamplingCode(dogDnaArchive.getSamplingCode());
		if (Objects.isNull(sampling) || StringUtils.isBlank(sampling.getId())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "没有找到编号对应的采样包");
		}
		if (0 != sampling.getBindType() || SamplingFlowStateEnum.SEND.code() < sampling.getFlowState()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该采样包已使用或已注销");
		}
		if (isRebind) {
			if (Objects.equals(SamplingDistributionChannelEnum.MEMBER.code(), sampling.getDistributionChannel())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只有机构办理可以进行换绑操作");
			}
			Org org = orgService.get(dogDnaArchive.getOrgId());
			if (Objects.isNull(org)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "办理机构信息有误");
			}
			if (!Objects.equals(org.getMemberCode(), sampling.getMemberCode())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "采样包归属机构与当前办理机构不一致");
			}
		}
		// 换绑操作
		if (isRebind) {
			Sampling origin = samplingService.findBySamplingCode(po.getSamplingCode());
			// 作废旧采样包
			samplingService.logout(origin, dogDnaArchive.getRemarks(), "换绑-新采样包：" + sampling.getSamplingCode());
		}
		// 更新DNA存档表
		if (!isRebind) {
			// 初次绑定更新完成时间
			dogDnaArchive.setFinishTime(DateUtils.getNow());
			dogDnaArchive.setTypeYes("2");
		}
		dogDnaArchive.preUpdate();
		dao.updateSamplingInfo(dogDnaArchive);
		// 判断犬只来源
		String dogCode = null;
		SamplingDogSourceTypeEnum dogSourceType = null;
		if (StringUtils.isNotBlank(dogDnaArchive.getPedigreeCertified())) {
			dogCode = dogDnaArchive.getPedigreeCertified();
			dogSourceType = SamplingDogSourceTypeEnum.PEDIGREE_CERTIFIED;
		} else if (StringUtils.isNotBlank(dogDnaArchive.getBirthCode())) {
			dogCode = dogDnaArchive.getBirthCode();
			dogSourceType = SamplingDogSourceTypeEnum.BIRTH_CODE;
		} else if (StringUtils.isNotBlank(dogDnaArchive.getOverseasPedigreeCertified())) {
			dogCode = dogDnaArchive.getOverseasPedigreeCertified();
			dogSourceType = SamplingDogSourceTypeEnum.OVERSEAS_PEDIGREE_CERTIFIED;
		}
		// 没有归属人同时变更归属人为犬主人
		if (StringUtils.isBlank(sampling.getMemberCode())) {
			samplingService.post(sampling.getSamplingCode(), dogDnaArchive.getPaymentMemberCode(),
					SamplingDistributionChannelEnum.MEMBER, Boolean.TRUE);
		}
		// 更新采样包表
		samplingService.bind(dogDnaArchive.getSamplingCode(), dogCode, dogSourceType, Boolean.TRUE);
	}

	/**
	 * 采样包绑定-前端使用 有归属人
	 *
	 * @param dogCode
	 * @param samplingCode
	 * @param memberCode   当前用户
	 */
	@Transactional(readOnly = false)
	public void bindSamplingFront(String dogCode, SamplingDogSourceTypeEnum dogSourceType, String samplingCode,
								  String memberCode) {
		// dna存档表记录
		DogDnaArchive dogDnaArchive = null;
		if (SamplingDogSourceTypeEnum.BIRTH_CODE.equals(dogSourceType)) {
			dogDnaArchive = getByBirthCode(dogCode);
		} else if (SamplingDogSourceTypeEnum.OVERSEAS_PEDIGREE_CERTIFIED.equals(dogSourceType)) {
			dogDnaArchive = dao.getByOverseasPedigreeCertified(dogCode);
		} else if (SamplingDogSourceTypeEnum.PEDIGREE_CERTIFIED.equals(dogSourceType)) {
			dogDnaArchive = dao.getByPedigreeCertifiedCode(dogCode);
		}
		// 是否为口腔上皮细胞测试法:否0；是1
		if ("0".equals(dogDnaArchive.getIsOEC()) || StringUtil.isNotBlank(dogDnaArchive.getSamplingCode())) {
			return;
		} else if ("1".equals(dogDnaArchive.getIsOEC()) && StringUtil.isBlank(samplingCode)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "位点分析(口腔上皮细胞) 必须绑定采样包！");
		}
		validIsRefundOrder(dogDnaArchive.getRunningNumber());
		samplingCode = samplingCode.trim();
		// 验证是否为本人
		samplingService.checkSamplingCode(samplingCode, memberCode);
		// 绑定
		samplingService.bind(samplingCode, dogCode, dogSourceType, Boolean.TRUE);

		dogDnaArchive.setSamplingCode(samplingCode);
		dogDnaArchive.preUpdate();
		save(dogDnaArchive);

	}

	public static final List<String> DNA_ARCHIVE_FEE = java.util.Arrays.asList("4625f60e8ac44d37b23ce2d972d1b532");

	/**
	 * 点位分析（口腔上皮细胞）收费项
	 *
	 * @return
	 */
	public List<PaymentChargingItem> getDnaArchiveItem() {
		return paymentChargingItemService.findListByIds(DNA_ARCHIVE_FEE);
	}

	@Transactional(readOnly = false)
	public void bindSamplingFront(DogDnaArchive dogDnaArchive, String samplingCode, String memberCode) {
		validIsRefundOrder(dogDnaArchive.getRunningNumber());
		// dna存档表记录
		String dogCode = null;
		SamplingDogSourceTypeEnum dogSourceType = null;
		if (StringUtils.isNotBlank(dogDnaArchive.getPedigreeCertifiedCode())) {
			dogCode = dogDnaArchive.getPedigreeCertifiedCode();
			dogSourceType = SamplingDogSourceTypeEnum.PEDIGREE_CERTIFIED;
		} else if (StringUtils.isNotBlank(dogDnaArchive.getBirthCode())) {
			dogCode = dogDnaArchive.getBirthCode();
			dogSourceType = SamplingDogSourceTypeEnum.BIRTH_CODE;
		} else if (StringUtils.isNotBlank(dogDnaArchive.getOverseasPedigreeCertified())) {
			dogCode = dogDnaArchive.getOverseasPedigreeCertified();
			dogSourceType = SamplingDogSourceTypeEnum.OVERSEAS_PEDIGREE_CERTIFIED;
		}
		// 验证是否为本人
		samplingService.checkSamplingCode(samplingCode, memberCode);
		// 绑定
		samplingService.bind(samplingCode, dogCode, dogSourceType, Boolean.TRUE);
		dogDnaArchive.setSamplingCode(samplingCode);
		dogDnaArchive.preUpdate();
		save(dogDnaArchive);

	}

	/**
	 * 解绑采用拿报告
	 *
	 * @author yuanshuai
	 * @since 2024/1/16 15:50
	 */
	@Transactional(readOnly = false)
	public void unbindSampling(DogDnaArchive dogDnaArchive) {
		if (Objects.isNull(dogDnaArchive) || StringUtils.isBlank(dogDnaArchive.getId())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "DNA存档信息有误，请重试");
		}
		if (!"2".equals(dogDnaArchive.getPaymentState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "未付款信息不可解绑");
		}
		if (StringUtils.isBlank(dogDnaArchive.getSamplingCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该DNA存档尚未绑定采样包");
		}
		Sampling sampling = samplingService.findBySamplingCode(dogDnaArchive.getSamplingCode());
		if (Objects.isNull(sampling) || StringUtils.isBlank(sampling.getId())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "没有找到编号对应的采样包");
		}
		// 清空采样包编号
		dogDnaArchive.setSamplingCode(null);
		dogDnaArchive.preUpdate();
		dao.updateSamplingInfo(dogDnaArchive);
		// 解绑采样包表
		samplingService.unbindDogInfo(sampling);

	}

	public DogDnaArchive getByRunningNumber(String runningNumber) {
		return dao.getByRunningNumber(runningNumber);
	}

	@Transactional(readOnly = false)
	public void updateSamplingInfo(DogDnaArchive dogDnaArchive) {
		dao.updateSamplingInfo(dogDnaArchive);
	}

	/**
	 * 同步遗传疾病数据
	 *
	 * @author yuanshuai
	 * @date 2023/8/8 11:13
	 */
	public int syncGeneticDisease() {
		int count = 0;
		// 获取需要同步的数据
		List<DogDnaArchive> dnaArchiveList = dao.findNoGeneticDiseaseList();
		List<String> samplingCodeList = dnaArchiveList.stream().map(DogDnaArchive::getSamplingCode)
				.collect(Collectors.toList());
		if (CollectionUtils.isEmpty(samplingCodeList)) {
			return count;
		}
		// 请求同步数据
		List<SamplingGeneticDiseaseResponse> syncList = GeneTreeReportUtils
				.findGeneticDiseaseBySamplingCodeList(samplingCodeList);
		// 更新遗传疾病信息
		if (!CollectionUtils.isEmpty(syncList)) {
			for (SamplingGeneticDiseaseResponse s : syncList) {
				dao.updateGeneticDiseaseBySamplingCode(s.getSamplingCode(), s.getGeneticDiseaseJson());
				count++;
			}
		}
		return count;
	}

	@Transactional(readOnly = false)
	public void cancel(DogDnaArchive dogDnaArchive) {

		if (!"2".equals(dogDnaArchive.getPaymentState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "已缴费才可撤销");
		}
		if (Global.YES.equals(dogDnaArchive.getIsCanceled())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该DNA存档已是撤销状态");
		}
		if (StringUtils.isNotBlank(dogDnaArchive.getSamplingCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "已绑定采样包不允许撤销");
		}
		// 状态置为撤销
		dogDnaArchive.setIsCanceled(Global.YES);
		dogDnaArchive.preUpdate();
		dao.update(dogDnaArchive);
		// 查询犬只DNA存档信息
		DogDnaArchive dogDnaArchiveResult = dao.get(dogDnaArchive);
		logger.info("-------------------- : {}", JSONObjectUtils.toJsonString(dogDnaArchiveResult));
		// 逻辑删除待邮寄物品信息
		expressQueueService.deleteByBusinessId(dogDnaArchiveResult.getId());
		// 口腔上皮的DNA存档需要同事撤销血样留存
		if (Global.YES.equals(dogDnaArchive.getIsOEC())) {
			DogBloodSave dogBloodSave = dogBloodSaveDao.getOneByParam(dogDnaArchive.getBirthCode(),
					dogDnaArchive.getOverseasPedigreeCertified(), null, dogDnaArchive.getPedigreeCertifiedCode());
			if (!Objects.isNull(dogBloodSave) && Global.YES.equals(dogBloodSave.getIsOEC())
					&& Global.NO.equals(dogBloodSave.getIsCanceled())) {
				dogBloodSave.setIsCanceled(Global.YES);
				dogBloodSave.preUpdate();
				dogBloodSaveDao.update(dogBloodSave);
			}
		}
	}

	/**
	 * 修改采集方式
	 *
	 * @author yuanshuai
	 * @since 2024/1/11 15:09
	 */
	@Transactional(readOnly = false)
	public void updateBlood2OEC(DogDnaArchive dogDnaArchive) {
		if ("1".equals(dogDnaArchive.getIsOEC())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该DNA存档采集方式已为唾液");
		}
		if (!"0".equals(dogDnaArchive.getReviewState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只有审核状态为待审核时才可变更采集方式");
		}
		// 修改dna存档采集状态
		dogDnaArchive.setIsOEC("1");
		updateIsOEC(dogDnaArchive);
		// 同步修改血样留存采集状态
		if (StringUtils.isNotBlank(dogDnaArchive.getRunningNumber())) {
			DogBloodSave dogBloodSave = dogBloodSaveService.getByRunningMumber(dogDnaArchive.getRunningNumber());
			if (Objects.nonNull(dogBloodSave) && StringUtils.isNotBlank(dogBloodSave.getId())) {
				dogBloodSave.setIsOEC("1");
				dogBloodSaveService.updateIsOEC(dogBloodSave);
			}
		}
	}

	/**
	 * 修改DNA存档采集状态
	 *
	 * @author yuanshuai
	 * @since 2024/1/11 15:09
	 */
	@Transactional(readOnly = false)
	public void updateIsOEC(DogDnaArchive dogDnaArchive) {
		dogDnaArchive.preUpdate();
		dao.updateIsOec(dogDnaArchive);
	}

	/**
	 * 根据订单号查询dna存单列表
	 *
	 * @author yuanshuai
	 * @since 2024/3/29 17:57
	 */
	public List<DogDnaArchive> findListByOrderCode(String orderCode) {
		if (StringUtils.isBlank(orderCode)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "订单编号有误");
		}
		return dao.findListByOrderCode(orderCode);
	}

	/**
	 * 根据犬只id查询dna信息
	 *
	 * @author yuanshuai
	 * @since 2024/4/22 13:50
	 */
	public DogDnaArchive findByDogId(String dogId) {
		return dao.getDnaMsg(dogId);
	}
}