package com.cku.oa.sampling.web;

import com.cku.core.ResultDto;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.sampling.entity.Sampling;
import com.cku.oa.sampling.service.SamplingService;
import com.cku.oa.sampling.vo.SamplingImportVo;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.common.config.Global;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.common.utils.excel.ExportExcel;
import com.thinkgem.jeesite.common.web.BaseController;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Objects;

/**
 * 采样包Controller
 *
 * @author yuanshuai
 * @version 2023-05-22
 */
@Controller
@RequestMapping(value = "${adminPath}/sampling/sampling")
public class SamplingController extends BaseController {

    @Autowired
    private SamplingService samplingService;

    @ModelAttribute
    public Sampling get(@RequestParam(required = false) String id) {
        Sampling entity = null;
        if (StringUtils.isNotBlank(id)) {
            entity = samplingService.get(id);
        }
        if (entity == null) {
            entity = new Sampling();
        }
        return entity;
    }

    @RequestMapping(value = "samplingCode")
    @ResponseBody
    public ResultDto<Sampling> findBySamplingCode(@RequestParam String samplingCode) {
        Sampling sampling;
        try {
            sampling = samplingService.findBySamplingCodeWithCase(samplingCode);
            if (Objects.isNull(sampling)) {
                throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "采样包编号有误");
            }
            return ResultDto.success(sampling);
        } catch (Exception e) {
            e.printStackTrace();
            return ResultDto.error(ZAErrorCode.ZA_VALID_FAILED, "查询采样包失败：" + e.getMessage());
        }
    }

    @RequiresPermissions("sampling:sampling:view")
    @RequestMapping(value = {"list", ""})
    public String list(Sampling sampling, HttpServletRequest request, HttpServletResponse response, Model model) {
        Page<Sampling> page = samplingService.findPage(new Page<>(request, response), sampling);
        model.addAttribute("page", page);
        return "oa/sampling/samplingList";
    }

    @RequiresPermissions("sampling:sampling:add")
    @RequestMapping(value = "addForm")
    public String addForm(Sampling sampling, Model model) {
        model.addAttribute("sampling", sampling);
        return "oa/sampling/samplingAddForm";
    }

    @RequiresPermissions("sampling:sampling:add")
    @RequestMapping(value = "add")
    @ResponseBody
    public ResultDto<String> add(Sampling sampling) {
        try {
            samplingService.add(sampling);
        } catch (ZAException e) {
            e.printStackTrace();
            return ResultDto.error(ZAErrorCode.ZA_VALID_FAILED, "添加采样包失败：" + e.getMessage());
        }
        return ResultDto.success("添加采样包成功");
    }

    @RequiresPermissions("sampling:sampling:view")
    @RequestMapping(value = "viewForm")
    public String viewForm(Sampling sampling, Model model) {
        model.addAttribute("isView", 1);
        model.addAttribute("sampling", sampling);
        return "oa/sampling/samplingForm";
    }

    @RequiresPermissions("sampling:sampling:edit")
    @RequestMapping(value = "editForm")
    public String editForm(Sampling sampling, Model model) {
        model.addAttribute("isView", 0);
        model.addAttribute("sampling", sampling);
        return "oa/sampling/samplingForm";
    }

    @RequiresPermissions("sampling:sampling:edit")
    @RequestMapping(value = "update")
    public String update(Sampling sampling, Model model, RedirectAttributes redirectAttributes) {
        if (!beanValidator(model, sampling)) {
            return editForm(sampling, model);
        }
        try {
            samplingService.update(sampling);
            addMessage(redirectAttributes, "保存采样包成功");
        } catch (ZAException e) {
            e.printStackTrace();
            addMessage(redirectAttributes, "保存采样包失败");
        }
        return "redirect:" + Global.getAdminPath() + "/sampling/sampling/?repage";
    }

    @RequiresPermissions("sampling:sampling:del")
    @RequestMapping(value = "delete")
    public String delete(Sampling sampling, RedirectAttributes redirectAttributes) {
        try {
            samplingService.delete(sampling);
            addMessage(redirectAttributes, "删除采样包成功");
        } catch (ZAException e) {
            addMessage(redirectAttributes, "删除采样包失败");
        }
        return "redirect:" + Global.getAdminPath() + "/sampling/sampling/?repage";
    }

    @RequiresPermissions("sampling:sampling:send")
    @RequestMapping(value = "send")
    public String send(Sampling sampling, RedirectAttributes redirectAttributes) {
        try {
            samplingService.send(sampling.getId());
            addMessage(redirectAttributes, "送检采样包成功");
        } catch (ZAException e) {
            e.printStackTrace();
            addMessage(redirectAttributes, "送检采样包失败");
        }
        return "redirect:" + Global.getAdminPath() + "/sampling/sampling/?repage";
    }

    @RequiresPermissions("sampling:sampling:send")
    @RequestMapping(value = "batchSend")
    @ResponseBody
    public ResultDto<String> batchSend(@RequestBody List<Sampling> samplingList) {
        try {
            samplingService.batchSend(samplingList);
        } catch (ZAException e) {
            e.printStackTrace();
            return ResultDto.error(ZAErrorCode.ZA_VALID_FAILED, "批量送检采样包失败：" + e.getMessage());
        }
        return ResultDto.success("批量送检采样包成功");
    }

    @RequiresPermissions("sampling:sampling:logout")
    @RequestMapping(value = "logout")
    public String logout(Sampling sampling, RedirectAttributes redirectAttributes) {
        try {
            samplingService.logout(sampling.getId(), sampling.getRemarks());
            addMessage(redirectAttributes, "注销采样包成功");
        } catch (ZAException e) {
            e.printStackTrace();
            addMessage(redirectAttributes, "注销采样包失败");
        }
        return "redirect:" + Global.getAdminPath() + "/sampling/sampling/?repage";
    }

    @RequiresPermissions("sampling:sampling:result")
    @RequestMapping(value = "resultForm")
    public String resultForm(Sampling sampling, Model model) {
        model.addAttribute("sampling", sampling);
        return "oa/sampling/samplingResultForm";
    }

    @RequiresPermissions("sampling:sampling:result")
    @RequestMapping(value = "result")
    @ResponseBody
    public ResultDto<String> result(Sampling sampling) {
        try {
            samplingService.resultById(sampling.getId(), sampling.getFlowState(), sampling.getRemarks(), null, Boolean.TRUE);
        } catch (ZAException e) {
            e.printStackTrace();
            return ResultDto.error(ZAErrorCode.ZA_VALID_FAILED, "采样包上传结果失败：" + e.getMessage());
        }
        return ResultDto.success("采样包上传结果成功");
    }

    @RequiresPermissions("sampling:sampling:import")
    @RequestMapping(value = "import", method = RequestMethod.POST)
    public String importFile(MultipartFile file, RedirectAttributes redirectAttributes) {
        try {
            samplingService.importSampling(file);
            addMessage(redirectAttributes, "导入采样包信息成功！");
        } catch (Exception e) {
            redirectAttributes.addFlashAttribute("importError", "导入采样包信息失败！失败信息：##" + e.getMessage());
        }
        return "redirect:" + Global.getAdminPath() + "/sampling/sampling/?repage";
    }

    @RequiresPermissions("sampling:sampling:import")
    @RequestMapping(value = "import/template")
    public String importFileTemplate(HttpServletResponse response, RedirectAttributes redirectAttributes) {
        try {
            String fileName = "采样包导入模板.xlsx";
            List<SamplingImportVo> list = Lists.newArrayList();
            SamplingImportVo vo = new SamplingImportVo();
            vo.setSamplingCode("1234567890");
            list.add(vo);
            new ExportExcel("采样包", SamplingImportVo.class, 2).setDataList(list).write(response, fileName).dispose();
            return null;
        } catch (Exception e) {
            addMessage(redirectAttributes, "导入模板下载失败！失败信息：" + e.getMessage());
        }
        return "redirect:" + Global.getAdminPath() + "/sampling/sampling/?repage";
    }

    @RequiresPermissions("sampling:sampling:export")
    @RequestMapping(value = "export")
    public String export(Sampling sampling, HttpServletResponse response, RedirectAttributes redirectAttributes) {
        try {
            samplingService.export(sampling, response);
        } catch (Exception e) {
            addMessage(redirectAttributes, "导出采样包信息失败！失败信息：" + e.getMessage());
            return "redirect:" + Global.getAdminPath() + "/sampling/sampling/?repage";
        }
        return null;
    }

    @RequiresPermissions("sampling:sampling:dogInfoExport")
    @RequestMapping(value = "dogInfo/export")
    public String exportDogInfo(Sampling sampling, HttpServletResponse response, RedirectAttributes redirectAttributes) {
        try {
            samplingService.exportDogInfo(sampling, response);
        } catch (Exception e) {
            addMessage(redirectAttributes, "导出犬只信息失败！失败信息：" + e.getMessage());
            return "redirect:" + Global.getAdminPath() + "/sampling/sampling/?repage";
        }
        return null;
    }

}