package com.cku.oa.dog.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.dao.DogDnaDao;
import com.cku.oa.dog.entity.Dog;
import com.cku.oa.dog.entity.DogDna;
import com.cku.oa.dog.entity.DogDnaArchive;
import com.cku.oa.sampling.entity.Sampling;
import com.cku.oa.sampling.enums.SamplingDistributionChannelEnum;
import com.cku.oa.sampling.enums.SamplingDogSourceTypeEnum;
import com.cku.oa.sampling.enums.SamplingFlowStateEnum;
import com.cku.oa.sampling.service.SamplingService;
import com.cku.oa.sys.code.SysCodeUtil;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;

/**
 * 犬DNAService
 *
 * @author lgl
 * @version 2016-08-08
 */
@Service
@Transactional(readOnly = true)
public class DogDnaService extends CrudService<DogDnaDao, DogDna> {

	@Autowired
	@Lazy
	public DogService dogService;
	@Autowired
	@Lazy
	public DogDnaArchiveService dogDnaArchiveService;
	@Autowired
	@Lazy
	public SamplingService samplingService;

	public DogDna get(String id) {
		return super.get(id);
	}

	public List<DogDna> findList(DogDna dogDna) {
		return super.findList(dogDna);
	}

	public Page<DogDna> findPage(Page<DogDna> page, DogDna dogDna) {
		return super.findPage(page, dogDna);
	}

	/**
	 * 根据犬只id获取dna列表
	 *
	 * @author yuanshuai
	 * @since 2024/4/18 14:22
	 */
	private List<DogDna> findListByDogId(String dogId) {
		if (StringUtils.isBlank(dogId)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬只ID为空");
		}
		DogDna search = new DogDna();
		search.setDogId(dogId);
		return findList(search);
	}

	/**
	 * 保存dna信息
	 *
	 * @author yuanshuai
	 * @since 2024/4/22 15:31
	 */
	@Transactional(readOnly = false)
	public void buildAndSave(DogDna dogDna) {
		if (StringUtils.isBlank(dogDna.getId())) {
			//新增
			buildAndInsert(dogDna, Boolean.FALSE);
		} else {
			//修改
			save(dogDna);
			//更新采样包状态
			Sampling sampling = samplingService.findBySamplingCode(dogDna.getSamplingCode());
			if (!SamplingFlowStateEnum.SUCCESS.code().equals(sampling.getFlowState())
					&& !SamplingFlowStateEnum.FAILURE.code().equals(sampling.getFlowState())) {
				samplingService.result(sampling, SamplingFlowStateEnum.SUCCESS, null, dogDna.getDnaResult(), Boolean.TRUE);
			}
		}
	}

	/**
	 * 新增dna信息
	 *
	 * @param firstFlag 是否第一次录入DNA信息（默认dna存档的处理为第一次）
	 * @author yuanshuai
	 * @since 2024/4/22 16:37
	 */
	@Transactional(readOnly = false)
	public void buildAndInsert(DogDna dogDna, Boolean firstFlag) {
		//生成dna编号-不处理nb开头犬种 	ID: 1006553取消nb的特殊处理
//		if (!dogDna.getDogBreed().startsWith("NB") && StringUtils.isBlank(dogDna.getDnaCode())) {
			dogDna.setDnaCode(SysCodeUtil.getDnaCode(dogDna.getDogBreed()));
//		}
//		if (StringUtils.isBlank(dogDna.getDnaCode())) {
//			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "DNA编号有误");
//		}
		//如果有dna结果，则更新激活状态，犬只信息dna存档信息
		if (!StringUtils.isBlank(dogDna.getDnaResult())) {
			//该犬只其他DNA信息该为待激活
			updateActiveStateByDogId(dogDna.getDogId(), 0);
			//初始化已激活
			dogDna.setActiveState(1);
			// 更新DNA存档表的处理状态
			dogDnaArchiveService.updateDogDnaArchive(dogDna, firstFlag);
			// 更新犬只打印状态
			dogDnaArchiveService.updateDog(dogDna, dogDna.getSamplingCode());
		} else {
			//初始化未激活
			dogDna.setActiveState(0);
		}
		//保存
		save(dogDna);
		//如果是唾液则更新采样包信息
		if (1 == dogDna.getDetectionType()) {
			//如果不是第一个DNA结果则更新采样包状态
			if (!firstFlag) {
				Dog dog = dogService.get(dogDna.getDogId());
				samplingService.post(dogDna.getSamplingCode(), dog.getMemberCode(), SamplingDistributionChannelEnum.MEMBER, Boolean.FALSE);
				samplingService.bind(dogDna.getSamplingCode(), dog.getPedigreeCertifiedCode(), SamplingDogSourceTypeEnum.PEDIGREE_CERTIFIED, Boolean.FALSE);
			}
			if (!StringUtils.isBlank(dogDna.getDnaResult())) {
				samplingService.resultBySamplingCode(dogDna.getSamplingCode(), SamplingFlowStateEnum.SUCCESS, null, dogDna.getDnaResult(), Boolean.TRUE);
			}
		}
	}

	@Transactional(readOnly = false)
	public void save(DogDna dogDna) {
		super.save(dogDna);
	}

	@Transactional(readOnly = false)
	public void delete(DogDna dogDna) {
		if (1 == dogDna.getActiveState()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "无法删除已激活DNA信息");
		}
		super.delete(dogDna);
	}

	public DogDna getByDogId(String dogId) {
		return dao.getByDogId(dogId);
	}

	public DogDna getByDnaCode(String dnaCode) {
		return dao.getByDnaCode(dnaCode);
	}

	/**
	 * @description: 判断是否进行DNA存档了
	 * 先查看DogDna是否有记录，如果有，返回true；
	 * 如果没有查看是否有已交费的DogDnaArchive，如果有，返回true
	 * @author: laiguanglong
	 * @date: 2016年12月16日 上午10:17:09
	 */
	public boolean isHaveDogDna(String dogId) {
		DogDna dogDna = dao.getByDogId(dogId);
		if (dogDna != null) {
			return true;
		} else {
			DogDnaArchive dogDnaArchive = dogDnaArchiveService.findByDogId(dogId);
			return Objects.nonNull(dogDnaArchive);
		}
	}

	/**
	 * 根据DNA结果生成基因编码
	 *
	 * @author yuanshuai
	 * @date 2023/8/3 14:11
	 */
	public String convertGeneCodeByDNAResult(String dnaResult) {
		if (StringUtils.isBlank(dnaResult)) {
			return dnaResult;
		}
		String geneCode = dnaResult
				.replace(" ", "")
				.replace("\r", "")
				.replace("\t", "")
				.replace("\n", "")
				.toUpperCase();
		if (geneCode.contains("X") && !geneCode.contains("X##")) {
			geneCode = geneCode.replace("X", "X##");
		}
		if (geneCode.contains("Y") && !geneCode.contains("Y##")) {
			geneCode = geneCode.replace("Y", "Y##");
		}
		return geneCode;
	}

	/**
	 * 激活DNA信息
	 *
	 * @author yuanshuai
	 * @since 2024/4/18 10:57
	 */
	@Transactional(readOnly = false)
	public void active(DogDna dogDna) {
		if (1 == dogDna.getActiveState()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该DNA信息已激活");
		}
		if (StringUtils.isBlank(dogDna.getDnaResult())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该信息尚未录入DNA结果");
		}
		//该犬只其他DNA信息该为待激活
		updateActiveStateByDogId(dogDna.getDogId(), 0);
		//激活当前DNA信息
		dogDna.setActiveState(1);
		save(dogDna);
		// 更新DNA存档表的处理状态
		dogDnaArchiveService.updateDogDnaArchive(dogDna, Boolean.FALSE);
		// 更新犬只打印状态
		dogDnaArchiveService.updateDog(dogDna, dogDna.getSamplingCode());
	}

	/**
	 * 根据犬只id修改激活状态
	 *
	 * @author yuanshuai
	 * @since 2024/4/18 17:29
	 */
	@Transactional(readOnly = false)
	public void updateActiveStateByDogId(String dogId, Integer activeState) {
		if (StringUtils.isBlank(dogId)
				&& Objects.isNull(activeState)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "参数有误");
		}
		dao.updateActiveStateByDogId(dogId, activeState, UserUtils.getLoginUser().getId());
	}

}