/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.restful.v1.dog.service;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.PageResultDto;
import com.cku.core.RESTResponse;
import com.cku.core.ResultDto;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.dao.DogAppraisalDao;
import com.cku.oa.dog.entity.DogAppraisal;
import com.cku.oa.dog.entity.DogColorMark;
import com.cku.oa.dog.entity.DogType;
import com.cku.oa.dog.entity.DogVideoAppraisal;
import com.cku.oa.dog.service.DogAppraisalService;
import com.cku.oa.dog.service.DogColorMarkService;
import com.cku.oa.dog.service.DogTypeService;
import com.cku.oa.dog.service.DogVideoAppraisalService;
import com.cku.oa.finance.entity.PaymentCart;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.finance.service.PaymentCartService;
import com.cku.oa.finance.service.PaymentChargingItemService;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.LoginServerService;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.oa.sys.util.MemberUtil;
import com.cku.restful.v1.dog.model.RestCheckMemberVo;
import com.cku.restful.v1.dog.model.RestDogAppraisal;
import com.cku.restful.v1.dog.model.RestDogAppraisalVo;
import com.cku.restful.v1.dog.vo.RestDogAppraisaRespVo;
import com.cku.restful.v1.sys.utils.BeanUtil;
import com.cku.util.DateUtils;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.utils.IdGen;
import com.thinkgem.jeesite.common.utils.JedisUtils;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import net.sf.json.JSONObject;

/**
 * 赛事报名表Service
 *
 * @author lgl
 * @version 2016-07-15
 */
@Service
@Transactional(readOnly = true)
public class RestDogAppraisalService {

	private static final String DOG_APPRAISAL_FEE_ALL = "dog_appraisal@All";
	
	private static final String DOG_REG_APPRAISAL = "dog_reg_appraisal@ALL";
	
	private static final String DOG_BLOOD_SAVE = "dog_blood_save@New";
	
	private static final String DOG_APPRAISAL_FREE = "dog_appraisal@free";


	@Autowired
	private PaymentChargingItemService paymentChargingItemService;
	@Autowired
	private PaymentCartService paymentCartService;
	@Autowired
	private DogAppraisalService dogAppraisalService;
	@Autowired
	private DogAppraisalDao dogAppraisalDao;
	@Autowired
	private DogTypeService dogTypeService;
	@Autowired
	private DogColorMarkService dogColorMarkService;
	@Autowired
	private LoginServerService loginServerService;
	@Autowired
	private DogVideoAppraisalService dogVideoAppraisalService;
	@Autowired
	private MemberService memberService;

	public ResultDto<RestCheckMemberVo> checkMember(HttpServletRequest request) {
		RestCheckMemberVo vo = new RestCheckMemberVo();
//		vo.setNext(true);
		vo.setRenew(false);
		vo.setUp(false);

		Member member = UserUtils.getLoginMember();
		
		Integer type = UserUtils.getMemberCardType(member.getMemberCode());
		if (type <= 0) { // 需要升级
//			vo.setNext(false);
			vo.setUp(true);
		} else  if (!MemberUtil.isActiveMember(member)) { // 过期会员
//			vo.setNext(false);
			vo.setRenew(true);
//			Integer type = UserUtils.getMemberCardType(member.getMemberCode());
//			if (type < 40) {
//				vo.setUp(true);
//			}
		}
		
		return ResultDto.success(vo);
	}

	public PageResultDto<RestDogAppraisalVo> list(HttpServletRequest request) {
		String pageNo = request.getParameter("pageNo");
		String pageSize = request.getParameter("pageSize");
		// 参数校验
		int pageNoInt = 1;
		int pageSizeInt = 10;
		try {
			if (StringUtils.isNotBlank(pageNo)) {
				pageNoInt = Integer.parseInt(pageNo);
			}
			if (StringUtils.isNotBlank(pageSize)) {
				pageSizeInt = Integer.parseInt(pageSize);
			}
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
		}
		// 组装查询
		Member member = UserUtils.getLoginMember();
		DogAppraisal dogAppraisal = new DogAppraisal();
		dogAppraisal.setPaymentMemberCode(member.getMemberCode());
		Page<DogAppraisal> page = new Page<DogAppraisal>();
		page.setPageNo(pageNoInt);
		page.setPageSize(pageSizeInt);
		dogAppraisal.setPage(page);
		Page<DogAppraisal> resultPage = dogAppraisalService.findPage(page, dogAppraisal);
		List<DogAppraisal> data = resultPage.getList();
		// resp
		List<RestDogAppraisalVo> resultVo = new ArrayList<>();
		for (DogAppraisal dog : data) {
			RestDogAppraisalVo vo = new RestDogAppraisalVo();
			BeanUtil.copyProperties(vo, dog, "yyyy-MM-dd");
			vo.setDogBreedCn(dog.getBreedCnName());
			resultVo.add(vo);
		}
		return PageResultDto.success(resultVo, resultPage.getCount());
	}

	@Transactional(readOnly = false)
	public ResultDto<RestDogAppraisaRespVo> post(String json) {
		// 获取参数
		RestDogAppraisal vo = BeanUtil.toBean(json, RestDogAppraisal.class);
		// 参数校验
		validateParameter(vo);
		// 业务校验
		DogAppraisal po = buildPo(vo);
		// 业务校验
		dogAppraisalService.validateBusiness(po);
		// 写入业务表
		dogAppraisalService.save(po);
		// 写入购物车
		List<PaymentCart> carts = buildCartIds(po);

		return ResultDto.success(buildResp(carts, vo, po));
	}

	private RestDogAppraisaRespVo buildResp(List<PaymentCart> carts, RestDogAppraisal vo, DogAppraisal po) {
		RestDogAppraisaRespVo resp = new RestDogAppraisaRespVo();
		StringBuffer cartIds = new StringBuffer();

		BigDecimal bd = new BigDecimal(0);
		for (PaymentCart cart : carts) {
			cartIds.append(cart.getId() + ",");
//			orderTotal += Double.valueOf(cart.getTotalPrice());
			bd = bd.add(new BigDecimal(cart.getTotalPrice()));
		}
		String ids = cartIds.toString().endsWith(",") ? cartIds.toString().substring(0, cartIds.length() - 1)
				: cartIds.toString();
		resp.setCartId(ids);
		resp.setPrice(bd.toPlainString());
		resp.setDogBirth(vo.getBirthday());
		resp.setDogBreed(vo.getDogBreedCn());
		resp.setDogGender(vo.getDogGender());
		resp.setDogName(vo.getDogName());
		resp.setDogNose(vo.getDogNose());
		resp.setDogColor(vo.getDogColorCn());
		resp.setMemberCode(po.getDogOwnerCode());
		resp.setMemberName(po.getDogOwnerName());
		resp.setIdentifyDate(po.getIdentifyDate().toLocaleString());
		resp.setIdentifySite(po.getIdentifySite());
		resp.setMemberType(UserUtils.getMemberCardType(po.getDogOwnerCode()).toString());
		return resp;
	}

	/**
	 * @description: 参数校验
	 * @author: laiguanglong
	 * @date: 2016年9月9日 上午9:41:47
	 */
	private void validateParameter(RestDogAppraisal vo) {
		if (StringUtils.isBlank(vo.getDogName())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬名为空");
		}
		if (StringUtils.isBlank(vo.getDogBreed())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬种为空");
		}
		// 校验犬种是否符合当前用户
		loginServerService.findDogType(vo.getDogBreed(), null);

		if (StringUtils.isBlank(vo.getDogColor())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "毛色为空");
		}
		if (StringUtils.isBlank(vo.getDogGender())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "性别为空");
		}
		if (!"1".equals(vo.getDogGender()) && !"2".equals(vo.getDogGender())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "性别只能为1或2");
		}
		if (StringUtils.isBlank(vo.getBirthday())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "出生日期为空");
		}
		Date date = null;
		try {
			date = new SimpleDateFormat("yyyy-MM-dd").parse(vo.getBirthday());
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "出生日期格式有误");
		}
		if (new Date().before(date)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "出生日期必须小于当前日期");
		}
		if (StringUtils.isBlank(vo.getMainShowId())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "主赛事ID为空");
		}

		if (StringUtils.isBlank(vo.getDogNose())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "鼻纹信息为空");
		}
	}

	private List<PaymentCart> buildCartIds(DogAppraisal po) {
		List<PaymentCart> carts = new ArrayList<>();
		/*
			2020-07-20更新
			已有视频鉴定结果犬只纯种犬鉴定对应收费项：
			纯种犬鉴定服务费 300元
			纯种犬鉴定证书登记费 100元
			登记服务费（纯血犬DNA样本） 100元
			纯种犬鉴定服务费（减免）-250元
			合计收费：250元
			
			无视频鉴定犬只（手动填写犬只信息）收费项：
			纯种犬鉴定服务费 300元
			合计收费300元
		*/
		
		String businessOrderCode = IdGen.uuid();
		// 纯种犬鉴定服务费 300元
		PaymentCart cart = buildPaymentCart(po,businessOrderCode);
		paymentCartService.addToCart(cart, "1010");
		carts.add(cart);
		
		String videoAppraisalId = po.getVideoAppraisalId();
		// 参与视频鉴定犬只收费
		if (StringUtils.isNotEmpty(videoAppraisalId)) {
			// 纯种犬鉴定证书登记费 100元
			PaymentCart appraisalRegCart = buildAppraisalRegCart(po,businessOrderCode);
			paymentCartService.addToCart(appraisalRegCart, "1010");
			carts.add(appraisalRegCart);
			
			// 登记服务费（纯血犬DNA样本） 100元
			PaymentCart bloodCart = buildBlooddCart(po,businessOrderCode);
			paymentCartService.addToCart(bloodCart, "1010");
			carts.add(bloodCart);
			// 纯种犬鉴定服务费（减免）-250元
			PaymentCart appraisalFreeCart = appraisalFreeCart(po,businessOrderCode);
			paymentCartService.addToCart(appraisalFreeCart, "1010");
			carts.add(appraisalFreeCart);

		}
			
		return carts;
	}

	private PaymentCart appraisalFreeCart(DogAppraisal po,String businessOrderCode) {
		PaymentCart paymentCart = new PaymentCart();
		PaymentChargingItem paymentChargingItem = getAppraisalFreeItem();
		paymentCart.setChargingItemId(paymentChargingItem.getId());
		paymentCart.setChargingItemName(paymentChargingItem.getName());
		paymentCart.setNum("1");
		paymentCart.setPrice(paymentChargingItem.getPrice());
		paymentCart.setTotalPrice(paymentChargingItem.getPrice());
		paymentCart.setBusinessTable("dog_appraisal");
		paymentCart.setBusinessOrderCode(businessOrderCode);
		paymentCart.setBusinessIds(po.getId());
		return paymentCart;
	}
	
	private PaymentCart buildBlooddCart(DogAppraisal po,String businessOrderCode) {
		PaymentCart paymentCart = new PaymentCart();
		PaymentChargingItem paymentChargingItem = getBloodItem();
		paymentCart.setChargingItemId(paymentChargingItem.getId());
		paymentCart.setChargingItemName(paymentChargingItem.getName());
		paymentCart.setNum("1");
		paymentCart.setPrice(paymentChargingItem.getPrice());
		paymentCart.setTotalPrice(paymentChargingItem.getPrice());
		paymentCart.setBusinessOrderCode(businessOrderCode);
		paymentCart.setBusinessTable("dog_appraisal");
		paymentCart.setBusinessIds(po.getId());
		return paymentCart;
	}
	
	private PaymentCart buildAppraisalRegCart(DogAppraisal po,String businessOrderCode) {
		PaymentCart paymentCart = new PaymentCart();
		PaymentChargingItem paymentChargingItem = getAppraisalRegItem();
		paymentCart.setChargingItemId(paymentChargingItem.getId());
		paymentCart.setChargingItemName(paymentChargingItem.getName());
		paymentCart.setNum("1");
		paymentCart.setPrice(paymentChargingItem.getPrice());
		paymentCart.setTotalPrice(paymentChargingItem.getPrice());
		paymentCart.setBusinessTable("dog_appraisal");
		paymentCart.setBusinessIds(po.getId());
		paymentCart.setBusinessOrderCode(businessOrderCode);
		return paymentCart;
	}

	private PaymentCart buildPaymentCart(DogAppraisal po,String businessOrderCode) {
		PaymentCart paymentCart = new PaymentCart();
		PaymentChargingItem paymentChargingItem = getDogAppraisalItem();
		paymentCart.setChargingItemId(paymentChargingItem.getId());
		paymentCart.setChargingItemName(paymentChargingItem.getName());
		paymentCart.setNum("1");
		paymentCart.setPrice(paymentChargingItem.getPrice());
		paymentCart.setTotalPrice(paymentChargingItem.getPrice());
		paymentCart.setBusinessOrderCode(businessOrderCode);
		paymentCart.setBusinessTable("dog_appraisal");
		paymentCart.setBusinessIds(po.getId());
		return paymentCart;
	}

	private DogAppraisal buildPo(RestDogAppraisal vo) {
		String videoAppraisalId = vo.getVideoAppraisalId();
		
		// 同时申请一只犬
		if(!StringUtils.isEmpty(videoAppraisalId)) {
			boolean result = JedisUtils.exists(videoAppraisalId);
			if(result) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "视频鉴定犬已被申请");
			}else {
				JedisUtils.set(videoAppraisalId, videoAppraisalId, 15);
			}
		}
		
		DogAppraisal po = new DogAppraisal();
		Member member = null;
		if (!StringUtils.isEmpty(videoAppraisalId)) {
			DogAppraisal dogAppraisal  = dogAppraisalService.getbyVideoId(videoAppraisalId);
			if(dogAppraisal != null) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该记录已被申请");
			}
			DogVideoAppraisal dogVideoAppraisal = dogVideoAppraisalService.get(videoAppraisalId);
			member = memberService.getByMemberCode(dogVideoAppraisal.getMemberCode());
			po.setDogChip(dogVideoAppraisal.getChipNo());
		} else {
			member = UserUtils.getLoginMember();
		}
		po.setDogOwnerCode(member.getMemberCode());
		po.setDogOwnerName(member.getName());
		po.setPaymentMemberCode(member.getMemberCode());
		po.setIdentifyType("0");
		// 设置赛场鉴定相关字段
		dogAppraisalService.setMainshowsInfo(po, vo.getMainShowId());
		po.setDogBreed(vo.getDogBreed());
		po.setDogName(vo.getDogName());
		po.setDogGender(vo.getDogGender());
		po.setDogFurColor(vo.getDogColor());
		try {
			po.setBirthday(new SimpleDateFormat("yyyy-MM-dd").parse(vo.getBirthday()));
		} catch (ParseException e) {
			e.printStackTrace();
		}
//		PaymentChargingItem item = getDogAppraisalItem();
		// 获取收费项
//		po.setPrice(item.getPrice());
		// 犬只鉴定表审核状态5为待审核
		po.setReviewState("5");
		po.setPaymentState("1");
		po.setProcessState("0");
		po.setVideoAppraisalId(videoAppraisalId);
		po.setDogNose(vo.getDogNose());
//		if ("1".equals(vo.getStickerFlag())) {
		// update by zjw 2023-02-07 工单号：ID1005116 取消宠爱护照发放
//		po.setStickerFlag("1");
//		}
		po.setStickerFlag("0");
		return po;
	}

	/**
	 * @description: 获取犬只鉴定收费项
	 * @author: laiguanglong
	 * @date: 2017/5/10 14:25
	 */
	public PaymentChargingItem getDogAppraisalItem() {
		return paymentChargingItemService.getByShortName(DOG_APPRAISAL_FEE_ALL);
	}

	public PaymentChargingItem getAppraisalRegItem() {
		return paymentChargingItemService.getByShortName(DOG_REG_APPRAISAL);
	}
	
	public PaymentChargingItem getBloodItem() {
		return paymentChargingItemService.getByShortName(DOG_BLOOD_SAVE);
	}

	public PaymentChargingItem getAppraisalFreeItem() {
		return paymentChargingItemService.getByShortName(DOG_APPRAISAL_FREE);
	}
	

	
	
	public RESTResponse getPrice() {
		JSONObject jo = new JSONObject();
		PaymentChargingItem item = getDogAppraisalItem();
		jo.put("price", item.getPrice());
		jo.put("name", item.getName());
		return new RESTResponse("data", jo);
	}

	public RESTResponse checkByMainShowId(HttpServletRequest request) {
		boolean flag = true;
		String msg = "";
		String mainShowId = request.getParameter("mainShowId");
		if (StringUtils.isBlank(mainShowId)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "赛事编号为空");
		}
		// 是有效会员
		Member member = UserUtils.getLoginMember();
		if (!MemberUtil.isActiveMember(member)) {
			flag = false;
			msg = "会员无效，无法办理该业务";
		}
		int count = dogAppraisalDao.countByMainShowId(mainShowId);
		if (count >= 40) {
			flag = false;
			msg = "报名人数已达上限，请选择其他赛事报名。";
		}
		JSONObject jo = new JSONObject();
		jo.put("rc", flag);
		jo.put("msg", msg);
		return new RESTResponse("data", jo);
	}

	/**
	 *
	 * @Description：订单详细中业务数据查询
	 * @author: zhuoHeng
	 * @version: 2016年12月29日 上午9:20:08
	 */
	public RESTResponse orderDetail(String id) {

		DogAppraisal po = dogAppraisalDao.get(id);

		if (po == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
		}
		String breedCnName = "";
		if (StringUtils.isNotBlank(po.getDogBreed())) {
			DogType dogType = dogTypeService.getByDogBreedCode(po.getDogBreed());
			if (dogType == null) {
				breedCnName = dogTypeService.getUnMapBreedCnName(po.getDogBreed());
			} else {
				breedCnName = dogType.getBreedCnName();
			}
		}
		String dogFurColor = "";
		if (StringUtils.isNotBlank(po.getDogFurColor())) {
			DogColorMark dogColorMark = dogColorMarkService.getByDogColorMarkCode(po.getDogFurColor());
			if (dogColorMark != null) {
				dogFurColor = dogColorMark.getDogColorMarkCn();
			}
		}
		Map<String, String> map = new HashMap<>();
		map.put("key1", "犬名：" + po.getDogName());
		map.put("key2", "犬种：" + breedCnName);
		map.put("key3", "毛色：" + dogFurColor);
		map.put("key4", "性别：" + DictUtils.getDictLabel(po.getDogGender(), "dog_gender", ""));
		map.put("key5", "出生日期：" + DateUtils.dateToStr(po.getBirthday()));
		return new RESTResponse("data", map);
	}

	public RESTResponse getRefundList(String ids) {
		DogAppraisal dogAppraisal = new DogAppraisal();
		dogAppraisal.setDelFlag("0");
		dogAppraisal.setId(ids);
		List<DogAppraisal> lists = dogAppraisalDao.findList(dogAppraisal);

		List<RestDogAppraisal> list = new ArrayList<>();

		for(DogAppraisal dogAppraisalList:lists){
			RestDogAppraisal vo = new RestDogAppraisal();
			BeanUtil.copyProperties(vo, dogAppraisalList, "yyyy-MM-dd");
			vo.setReviewState(DictUtils.getDictLabel(vo.getReviewState(), "review_state", ""));
			list.add(vo);
		}
		return new RESTResponse("data", list);
	}
}