/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.restful.v1.groomer.web;

import java.io.IOException;
import java.util.Date;
import java.util.Objects;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;

import com.cku.annotation.MultiSubmit;
import com.cku.oa.groomer.entity.*;
import com.cku.oa.groomer.service.GroomerQualificationCertificateService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import com.cku.core.ListResultDto;
import com.cku.core.QueryByIdRequest;
import com.cku.core.RESTResponse;
import com.cku.core.ResultDto;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.logUtil.JSONObjectUtils;
import com.cku.oa.constant.Constants;
import com.cku.oa.groomer.request.CommitWrittenExamRequest;
import com.cku.oa.groomer.service.GroomerRegisterService;
import com.cku.oa.groomer.service.GroomerShowService;
import com.cku.oa.groomer.service.GroomerWrittenExamInfoService;
import com.cku.oa.groomer.vo.GroomerShowBreedsGroupVO;
import com.cku.oa.groomer.vo.GroomerShowBreedsNumVO;
import com.cku.oa.sys.entity.user.Member;
import com.cku.restful.v1.groomer.model.RestGroomerRegister;
import com.cku.restful.v1.groomer.service.RestGroomerRegisterService;
import com.cku.restful.v1.groomer.vo.GroomerWrittenExamInfoVo;
import com.cku.restful.v1.sys.web.BaseRestController;
import com.cku.util.DateUtils;
import com.cku.util.ServletUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import io.swagger.annotations.ApiOperation;

/**
 * 美容师报名表Controller
 * @author cxt
 * @version 2016-07-14
 */
@Controller
@RequestMapping(value = "/api/v1/groomer")
public class RestGroomerRegisterController extends BaseRestController {

	@Autowired
	private RestGroomerRegisterService restGroomerRegisterService;

	@Autowired
	private GroomerRegisterService groomerRegisterService;

	@Autowired
	private GroomerWrittenExamInfoService groomerWrittenExamInfoService;

	@Autowired
	private GroomerShowService groomerShowService;

	@Autowired
	private GroomerQualificationCertificateService groomerQualificationCertificateService;

	protected Logger logger = LoggerFactory.getLogger(getClass());

	@ApiOperation(value = "报名接口")
	@RequestMapping(value = "/groomerRegister", method = RequestMethod.POST)
	@ResponseBody
	public void post(@RequestBody String json, HttpServletResponse response)
			throws IOException {
		RESTResponse result = restGroomerRegisterService.post(json);
		ServletUtils.writeResponse(response, result);
	}


	/**
	 * 活动报名(校验)
	 * @throws Exception
	 * @Author chaixueteng
	 * @2016年9月22日下午2:45:13
	 */
	@ApiOperation(value = "美容洗护师考试报名页请求")
	@RequestMapping(value = "/apply", method = RequestMethod.GET)
	@ResponseBody
	public void apply(HttpServletRequest request, HttpServletResponse response)
			throws Exception {
		RESTResponse result = restGroomerRegisterService.apply(request);
		ServletUtils.writeResponse(response, result);
	}


	/**
	 * 资格证书申领列表
	 * @throws Exception
	 * @Author chaixueteng
	 * @2016年9月22日下午2:45:13
	 */
	@RequestMapping(value = "/certificateList", method = RequestMethod.GET)
	@ResponseBody
	public void certificateList(HttpServletRequest request, HttpServletResponse response)
			throws Exception {
		RESTResponse result = restGroomerRegisterService.certificateList(request);
		ServletUtils.writeResponse(response, result);
	}

	/**
	 * 资格证书在线申领
	 * @Author chaixueteng
	 * @2016年9月22日下午2:45:13
	 */
	@RequestMapping(value = "/receiveCertificate/{id}", method = RequestMethod.GET)
	@ResponseBody
	public void receiveCertificate(@PathVariable("id") String id,HttpServletRequest request, HttpServletResponse response)
			throws IOException {
		RESTResponse result = restGroomerRegisterService.receiveCertificate(id, "1");
		ServletUtils.writeResponse(response, result);
	}

	/**
	 *
	 * @description: 美容师续费
	 * @author: laiguanglong
	 * @date: 2016年10月12日 上午11:49:53
	 */
	@RequestMapping(value = "/renew", method = RequestMethod.POST)
	@ResponseBody
	public void renew(HttpServletResponse response)
			throws IOException {
		RESTResponse result = restGroomerRegisterService.renew();
		ServletUtils.writeResponse(response, result);
	}

	/**
	 *
	 * @Description：资格证书申领（宠爱王国使用）
	 * @author: zhuoHeng
	 * @version: 2016年11月10日 下午5:26:18
	 */
	@RequestMapping(value = "/receiveCertificate/applyToGet", method = RequestMethod.GET)
	@ResponseBody
	public void applyToGet(HttpServletRequest request, HttpServletResponse response)
			throws IOException {
		String id = request.getParameter("id");
		String qualificationCertificateType = request.getParameter("qualificationCertificateType");
		RESTResponse result = restGroomerRegisterService.receiveCertificate(id, qualificationCertificateType);
		ServletUtils.writeResponse(response, result);
	}
	/**
	* @Description:   免费申领
	* @author: zhangxiang
	* @date: 2021/4/16 11:01
	*/
	@RequestMapping(value = "/receiveCertificate/applyFree", method = RequestMethod.GET)
	@ResponseBody
	public void applyFree(HttpServletRequest request, HttpServletResponse response)
			throws IOException {
		String id = request.getParameter("id");
		RESTResponse result = restGroomerRegisterService.receiveCertificateFree(id);
		ServletUtils.writeResponse(response, result);
	}
	/**
	 *
	 * @Description：获取美容报名历史数据（宠爱王国使用）
	 * @author: zhuoHeng
	 * @version: 2016年12月14日 上午11:06:03
	 */
	@RequestMapping(value = "/getGroomerRegister", method = RequestMethod.GET)
	@ResponseBody
	public void getGroomerRegister(HttpServletRequest request, HttpServletResponse response)
			throws IOException {
		RESTResponse result = restGroomerRegisterService.getGroomerRegister(request);
		ServletUtils.writeResponse(response, result);
	}

	/**
	 *
	 * @Description：根据id获取美容报名详细
	 * @author: zhuoHeng
	 * @version: 2016年12月27日 下午4:01:45
	 */
	@RequestMapping(value = "/getGroomerRegisterById", method = RequestMethod.GET)
	@ResponseBody
	public void getGroomerRegisterById(HttpServletRequest request, HttpServletResponse response)
			throws IOException {
		RESTResponse result = restGroomerRegisterService.getGroomerRegisterById(request);
		ServletUtils.writeResponse(response, result);
	}

	/**
	 *
	 * @Description：修改用户证件照并重置审核状态
	 * @author: zhuoHeng
	 * @version: 2016年12月15日 上午9:53:53
	 */
	@RequestMapping(value = "/resetReviewState", method = RequestMethod.PUT)
	@ResponseBody
	public void resetReviewState(HttpServletRequest request, HttpServletResponse response)
			throws IOException {
		RESTResponse result = restGroomerRegisterService.resetReviewState(request);
		ServletUtils.writeResponse(response, result);
	}

	/**
	 *
	 * @Description：订单详细中业务数据查询
	 * @author: zhuoHeng
	 * @version: 2017年1月11日 下午5:18:52
	 */
	@RequestMapping(value = "/orderDetail", method = RequestMethod.GET)
	@ResponseBody
	public void orderDetail(HttpServletRequest request, HttpServletResponse response)
			throws IOException {
		String id = request.getParameter("id");
		RESTResponse result = restGroomerRegisterService.orderDetail(id);
		ServletUtils.writeResponse(response, result);
	}
	
	
	
	@ApiOperation(value = "历史成绩查询接口")
	@RequestMapping(value = "/nurserRgisters", method = RequestMethod.GET)
	@ResponseBody
	public ListResultDto<RestGroomerRegister> getNurserRegisters(
			HttpServletRequest request,
			HttpServletResponse response) throws Exception {
		return restGroomerRegisterService.getNurserRegisters();
    }
	
	@ApiOperation(value = "犬种选择页面接口")
	@RequestMapping(value = "/nurseGroomerShowBreeds/{showLevel}", method = RequestMethod.GET)
	@ResponseBody
	public ListResultDto<GroomerShowBreedsGroupVO> findNurseGroomerShowBreeds(@PathVariable("showLevel") String showLevel,
			HttpServletRequest request,
			HttpServletResponse response) throws Exception {
		return restGroomerRegisterService.findNurseGroomerShowBreeds(showLevel);
    }
	
	@ApiOperation(value = "犬种选择数量限制")
	@RequestMapping(value = "/nurseGroomerShowBreedsNum/{showLevel}", method = RequestMethod.GET)
	@ResponseBody
	public ResultDto<GroomerShowBreedsNumVO> getShowLevelNum(@PathVariable("showLevel") String showLevel,
			HttpServletRequest request,
			HttpServletResponse response) throws Exception {
		return restGroomerRegisterService.getShowLevelNum(showLevel);
    }

	/**
	 * 开始考试app
	 */
	@ApiOperation(value = "美容资格证书开始考试")
	@PostMapping(value = "/beginExam")
	@ResponseBody
	@MultiSubmit
	public ResultDto<GroomerExamQuestionBank> beginExam(@RequestBody GroomerRegister groomerRegister){
		if (Objects.isNull(groomerRegister)) {
			return ResultDto.error(ZAErrorCode.ZA_VALID_FAILED, "参数为空");
		}
		//log.info("RestGroomerRegisterController beginExam groomerRegister : {}", JSON.toJSONString(groomerRegister));
		GroomerRegister groomerRegisterResult = groomerRegisterService.get(groomerRegister.getId());
		if (Objects.isNull(groomerRegisterResult)) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "没找到美容活动信息！");
		}

		if (!Objects.equals(groomerRegisterResult.getReviewState(), Constants.STRING_NUM_ONE)) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "审核未通过");
		}

		Member loginMember = UserUtils.getLoginMember();
		if (!Objects.equals(groomerRegisterResult.getMemberCode(), loginMember.getMemberCode())) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "当前会员账号不符合活动报名会员账号");
		}
		//判断活动状态和笔试状态是否为待考试
		if (!groomerRegisterResult.getEligibility()) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "当前没有资格考试!");
		}
		GroomerShow showCode = groomerShowService.getShowCode(groomerRegisterResult.getShowCode());
		if (Objects.isNull(showCode)) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "找不到活动！");
		}

		if (StringUtils.isBlank(groomerRegisterResult.getShowLevel())
				|| Objects.equals("无", groomerRegisterResult.getShowLevel())
				|| Objects.equals("none", groomerRegisterResult.getShowLevel())) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "报名信息无参展级别，请联系工作人员！");
		}
		if (Objects.equals("MASTER", groomerRegisterResult.getShowLevel())
				|| Objects.equals("教师级", groomerRegisterResult.getShowLevel())) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "教师级别无法考试，请联系工作人员！");
		}

		//logger.info("考试考试接口查询考试信息 groomerWrittenExamInfoResult ： {}", JSONObject.toJSONString(groomerWrittenExamInfoResult));
		Boolean unexamined = Boolean.TRUE;
		//考试重置不走补考逻辑
		if (!groomerRegisterResult.getReset()) {
			if (groomerRegisterResult.getExamNum() >= Constants.NUM_ONE
					|| (DateUtils.nowDateAddTargetDateUnitMonth(groomerRegisterResult.getGradeTime(), Constants.NUM_TWO).compareTo(new Date()) < 0)
					&& Constants.NUM_ZERO.equals(groomerRegisterResult.getExamNum())) {
				unexamined = Boolean.FALSE;

			}
		}



		//查询考卷相关信息
		GroomerExamQuestionBank groomerExamQuestionBank = groomerRegisterService.getExamQuestions(groomerRegisterResult);
		logger.info("GroomerRegisterController beginExam groomerRegisterService.getExamQuestions result : {}", JSONObjectUtils.toJsonString(groomerExamQuestionBank));
		if (Objects.isNull(groomerExamQuestionBank)) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "无考题信息，请联系相关人员!");
		}
		groomerRegisterService.generateGroomerWrittenExamInfo(groomerRegisterResult, groomerExamQuestionBank, unexamined);
		
		return ResultDto.success(groomerExamQuestionBank);
	}

	/**
	 * 提交考试
	 */
	@ApiOperation(value = "美容资格证书提交考试")
	@PostMapping(value = "/commitWrittenExam")
	@ResponseBody
	@MultiSubmit
	public ResultDto<Boolean> commitWrittenExam(@RequestBody CommitWrittenExamRequest commitWrittenExamRequest) {
		logger.info("GroomerRegisterController commitWrittenExam commitWrittenExamRequest param : {}"
				, JSONObjectUtils.toJsonString(commitWrittenExamRequest));

		if (Objects.isNull(commitWrittenExamRequest)) {
			return ResultDto.error(ZAErrorCode.ZA_VALID_FAILED, "参数为空");
		}

		GroomerWrittenExamInfo groomerWrittenExamInfo = groomerWrittenExamInfoService.get(commitWrittenExamRequest.getWrittenInfoId());

		if (Objects.isNull(groomerWrittenExamInfo)) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "无考试信息!");
		}

		Boolean result = groomerRegisterService.commitWrittenExam(groomerWrittenExamInfo, commitWrittenExamRequest);
		return ResultDto.success(result);
	}

	/**
	 * 校验是否缴费
	 */
	@ApiOperation(value = "校验补考支付状态")
	@GetMapping(value = "/validGroomerExamPaymentStatus")
	@ResponseBody
	@MultiSubmit
	public ResultDto<GroomerWrittenExamInfoVo> validGroomerExamPaymentStatus(@Valid QueryByIdRequest groomerRegister) {
		GroomerRegister groomerRegisterBo = groomerRegisterService.get(groomerRegister.getId());
		if (Objects.isNull(groomerRegisterBo)) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "活动注册信息找不到!");
		}

		GroomerWrittenExamInfoVo groomerWrittenExamInfoVo = new GroomerWrittenExamInfoVo();
		GroomerWrittenExamInfo groomerWrittenExamInfo = new GroomerWrittenExamInfo();
		groomerWrittenExamInfo.setRegisterId(groomerRegisterBo.getId());
		groomerWrittenExamInfo.setMemberCode(UserUtils.getLoginMember().getMemberCode());
		logger.info("validGroomerExamPaymentStatus groomerWrittenExamInfo : {}", JSONObjectUtils.toJsonString(groomerWrittenExamInfo));
		GroomerWrittenExamInfo groomerWrittenExamInfoBo = groomerWrittenExamInfoService.queryWrittenExamInfoByCondition(groomerWrittenExamInfo);
		logger.info("validGroomerExamPaymentStatus groomerWrittenExamInfo : {}", JSONObjectUtils.toJsonString(groomerWrittenExamInfoBo));
		if (Objects.isNull(groomerWrittenExamInfoBo)) {
			groomerWrittenExamInfoVo.setResitPaymentStatus(Constants.STRING_ZERO);
			return ResultDto.success(groomerWrittenExamInfoVo);
		}
		if (StringUtils.isBlank(groomerWrittenExamInfoBo.getPaymentState())) {
			groomerWrittenExamInfoVo.setResitPaymentStatus(Constants.STRING_ZERO);
		} else {
			if (groomerWrittenExamInfoBo.getStartExamTime() != null) {
				groomerWrittenExamInfoVo.setResitPaymentStatus(Constants.STRING_ZERO);
			} else {
				groomerWrittenExamInfoVo.setResitPaymentStatus(groomerWrittenExamInfoBo.getPaymentState());
			}
		}
		return ResultDto.success(groomerWrittenExamInfoVo);
	}
}