/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.trainschool.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringEscapeUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.PageBeanResult;
import com.cku.core.RESTResponse;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.handler.service.HandlerDiplomaService;
import com.cku.oa.sys.code.SysCodeUtil;
import com.cku.oa.sys.dao.user.MemberDao;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.entity.user.MemberType;
import com.cku.oa.sys.util.DateUtils;
import com.cku.oa.sys.zalog.SysContext;
import com.cku.oa.trainschool.dao.TrainingInstitutionDao;
import com.cku.oa.trainschool.entity.TrainingInstitution;
import com.cku.restful.v1.dog.vo.TrainingInstitutionVO;
import com.cku.util.ServletUtils;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.service.SystemService;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

/**
 * 合作学校Service
 * 
 * @author 柴雪腾
 * @version 2016-08-08
 */
@Service
@Transactional(readOnly = true)
public class TrainingInstitutionService extends CrudService<TrainingInstitutionDao, TrainingInstitution> {

	@Autowired
	public HandlerDiplomaService handlerDiplomaService;
	@Autowired
	public SystemService systemService;
	@Autowired
	public MemberDao memberDao;

	public TrainingInstitution get(String id) {
		return super.get(id);
	}

	public List<TrainingInstitution> findList(TrainingInstitution trainingInstitution) {
		return super.findList(trainingInstitution);
	}

	public Page<TrainingInstitution> findPage(Page<TrainingInstitution> page, TrainingInstitution trainingInstitution) {
		return super.findPage(page, trainingInstitution);
	}

	@Transactional(readOnly = false)
	public void save(TrainingInstitution trainingInstitution) {
		// 批量更新同名培训机构的电话和地址
		if (trainingInstitution.getIsNewRecord()) {
			trainingInstitution.setAddress(trainingInstitution.getProvince() + trainingInstitution.getCity()
					+ trainingInstitution.getStreet());
			super.save(trainingInstitution);
		} else {
			dao.updateBySchoolNameCn(trainingInstitution);
			trainingInstitution.setAddress(trainingInstitution.getProvince() + trainingInstitution.getCity()
					+ trainingInstitution.getStreet());
			super.save(trainingInstitution);
		}
	}

	@Transactional(readOnly = false)
	public void delete(TrainingInstitution trainingInstitution) {
		super.delete(trainingInstitution);
	}

	@Transactional(readOnly = false)
	public void groomerSchoolDelete(TrainingInstitution trainingInstitution) {
		// 需要把美容师学校的登录帐号标记为关闭，不允许登录
		User user = systemService.getUser(trainingInstitution.getSchoolUserId());
		user.setLoginFlag("0");
		systemService.saveUser(user);
		super.delete(trainingInstitution);
	}

	/**
	 * 美容合作学校
	 * 
	 * @Author chaixueteng
	 * @2016年8月15日下午4:20:06
	 */
	public Page<TrainingInstitution> findGroomerSchoolPage(Page<TrainingInstitution> page,
			TrainingInstitution trainingInstitution, HttpServletResponse response) {
		trainingInstitution.setPage(page);
		return page.setList(dao.findGroomerList(trainingInstitution));
	}

	/**
	 * 美容合作学校
	 * 
	 * @Author chaixueteng
	 * @2016年8月15日下午4:20:06
	 */
	public List<TrainingInstitution> findGroomerSchoolList(TrainingInstitution trainingInstitution) {
		return dao.findGroomerList(trainingInstitution);
	}

	public void findGroomerSchoolPageAPP(Page<TrainingInstitution> page, HttpServletRequest request,
			HttpServletResponse response) throws IOException {
		List<TrainingInstitutionVO> list = new ArrayList<>();
		long count = 0;
//		类型1牵犬师2训犬师3美容师板块
		String type = request.getParameter("type");
		if ("1".equals(type) || "2".equals(type)) {
			TrainingInstitution trainingInstitution = new TrainingInstitution();
			trainingInstitution.setFirstSchoolType(type);
			Page<TrainingInstitution> trainingInstitutionPage = this.getHandlerOrgList(page, trainingInstitution);
			trainingInstitutionPage.getList().forEach(po -> {
				TrainingInstitutionVO trainingInstitutionVO = new TrainingInstitutionVO();
				BeanUtils.copyProperties(po, trainingInstitutionVO);
				if (org.apache.commons.lang3.StringUtils.isNotBlank(trainingInstitutionVO.getAvatar())) {
					String path = request.getContextPath();
					String imgPath = request.getScheme() + "://" + request.getServerName() + ":"
							+ request.getServerPort() + path + "/api/v1/file/image/"
							+ trainingInstitutionVO.getAvatar();
					trainingInstitutionVO.setAvatar(imgPath);
				}

				// 输送学员数量
				String schoolNameCn = po.getSchoolNameCn();
				int c = handlerDiplomaService.getCount(schoolNameCn);
				trainingInstitutionVO.setStudent(c);

				list.add(trainingInstitutionVO);
			});
			count = trainingInstitutionPage.getCount();
		} else {
			Page<TrainingInstitution> trainingInstitutionPage = this.findPageBYApp(page, new TrainingInstitution());
			trainingInstitutionPage.getList().forEach(po -> {
				TrainingInstitutionVO trainingInstitutionVO = new TrainingInstitutionVO();
				BeanUtils.copyProperties(po, trainingInstitutionVO);
				if (org.apache.commons.lang3.StringUtils.isNotBlank(trainingInstitutionVO.getAvatar())) {
					String path = request.getContextPath();
					String imgPath = request.getScheme() + "://" + request.getServerName() + ":"
							+ request.getServerPort() + path + "/api/v1/file/image/"
							+ trainingInstitutionVO.getAvatar();
					trainingInstitutionVO.setAvatar(imgPath);
				}

				list.add(trainingInstitutionVO);
			});
			count = trainingInstitutionPage.getCount();
		}

		ServletUtils.writeResponse(response,
				new RESTResponse("list", new PageBeanResult<TrainingInstitutionVO>(list, count)));

	}

	public Page<TrainingInstitution> findPageBYApp(Page<TrainingInstitution> page,
			TrainingInstitution trainingInstitution) {
		int pageNo = page.getPageNo();
		int pageSize = page.getPageSize();
		trainingInstitution.setPage(page);
		page.setList(dao.findGroomerListAPP(trainingInstitution));
		if ((pageNo - 1) * pageSize >= page.getCount()) {
			page.setList(new ArrayList<TrainingInstitution>(0));
		}
		return page;
	}

	/**
	 * 培训合作学校
	 * 
	 * @Author chaixueteng
	 * @2016年8月15日下午4:20:06
	 */
	public Page<TrainingInstitution> findHandlerSchoolPage(Page<TrainingInstitution> page,
			TrainingInstitution trainingInstitution) {
		trainingInstitution.setPage(page);
		List<TrainingInstitution> handlerList = dao.findHandlerList(trainingInstitution);
		for (TrainingInstitution ti : handlerList) {
			String schoolNameCn = ti.getSchoolNameCn();
			int count = handlerDiplomaService.getCount(schoolNameCn);
			ti.setStudent(count);
		}
		return page.setList(handlerList);
	}

	public TrainingInstitution getByMemberCode(String memberCode) {
		return dao.getByMemberCode(memberCode);
	}

	@Transactional(readOnly = false)
	public void saveGroomerSchool(TrainingInstitution vo) {
		// 美容学校中文名和英文名遇到特殊字符不转意
		vo.setSchoolNameCn(StringEscapeUtils.unescapeHtml4(vo.getSchoolNameCn()));
		vo.setSchoolNameEn(StringEscapeUtils.unescapeHtml4(vo.getSchoolNameEn()));
		// 详细介绍不转意
		vo.setIntroduce(StringEscapeUtils.unescapeHtml4(vo.getIntroduce()));
		vo.setSchoolNameEn(StringEscapeUtils.unescapeHtml4(vo.getSchoolNameEn()));
		vo.setAddress(vo.getProvince() + vo.getCity() + vo.getStreet());
		if (vo.getIsNewRecord()) {
			// 保存大写
			vo.setSchoolShortName(vo.getSchoolShortName().toUpperCase());
			// 英文名缩写查重
			TrainingInstitution example = new TrainingInstitution();
			example.setSchoolShortName(vo.getSchoolShortName());
			if (dao.findGroomerList(example).size() > 0) {
				throw new ZAException(ZAErrorCode.ZA_ERC_INVALID_PARAMETER, "英文名缩写已经存在");
			}
			vo.setSchoolMemberCode(SysCodeUtil.getMRHZMemberCode());
			// 新增网站用户记录
			User user = addUser(vo);
			vo.setSchoolUserId(user.getId());
			// 新增会员记录
			addMember(vo, user);
			vo.preInsert();
			vo.setAddress(vo.getProvince() + vo.getCity() + vo.getStreet());
			dao.insert(vo);
		} else {
			// 更新网站用户信息
			updateUser(vo);
			// 更新会员信息
			updateMember(vo);
			vo.preUpdate();
			dao.update(vo);
			// 批量更新同名培训机构的电话和地址
			dao.updateBySchoolNameCn(vo);
		}
	}

	/**
	 * @description: 新增网站用户记录
	 * @author: laiguanglong
	 * @date: 2017/6/15 17:23
	 */
	private User addUser(TrainingInstitution vo) {
		// 登录名查重
		if (UserUtils.getByLoginNameAndUserType(vo.getUser().getLoginName(), "3") != null) {
			throw new ZAException(ZAErrorCode.ZA_ERC_INVALID_PARAMETER, "用户帐号已经存在");
		}
		User user = new User();
		user.setUserType("3"); // 3代表合作机构与合作医院
		user.setLoginName(vo.getUser().getLoginName());
		user.setLoginFlag(vo.getUser().getLoginFlag());
		user.setName(vo.getSchoolNameCn());
		user.setPassword(systemService.entryptPassword(vo.getPassword()));
		systemService.saveUser(user);
		return user;
	}

	/**
	 * @description: 新增会员记录
	 * @author: laiguanglong
	 * @date: 2017/6/15 17:52
	 */
	private Member addMember(TrainingInstitution vo, User user) {
		Member member = new Member();
		member.setUser(user);
		member.setName(vo.getSchoolNameCn());
		member.setNameEn(vo.getSchoolNameEn());
		member.setMemberType(MemberType.basic.toString());// 基础会员
		member.setGender("male");// 默认初始化为male
		member.setPaymentState("1"); // 未付费
		member.setMemberEndDate(vo.getEndDate());
		member.setKuFlag(null);
		member.setKuEndDate(null);
		member.setGroomerType(null);
		member.setGroomerEndDate(null);
		member.setDgMemberType(null);
		member.setDgEndDate(null);
		member.setAccountBalance("0");
		member.setMemberCode(vo.getSchoolMemberCode());
		member.setProvince(vo.getProvince());
		member.setCity(vo.getCity());
		member.setStreet(vo.getStreet());
		member.preInsert();
		memberDao.insert(member);
		return member;
	}

	/**
	 * @description: 更新网站用户信息
	 * @author: laiguanglong
	 * @date: 2017/6/16 16:07
	 */
	private void updateUser(TrainingInstitution vo) {
		User user = systemService.getUser(vo.getSchoolUserId());
		user.setLoginFlag(vo.getUser().getLoginFlag());
		user.setName(vo.getSchoolNameCn());
		if (StringUtils.isNotBlank(vo.getPassword())) {
			user.setPassword(systemService.entryptPassword(vo.getPassword()));
		}
		systemService.saveUser(user);
	}

	/**
	 * @description: 更新会员信息
	 * @author: laiguanglong
	 * @date: 2017/6/16 16:09
	 */
	private void updateMember(TrainingInstitution vo) {
		Member member = memberDao.getByMemberCode(vo.getSchoolMemberCode());
		TrainingInstitution dbVo = dao.get(vo.getId());
		boolean upDateFlag = Boolean.FALSE;
		// 1970-01-01
		Date defaultDate = new Date(70, 0, 1);
		Date dbEndDate = Objects.nonNull(dbVo.getEndDate()) ? dbVo.getEndDate() : defaultDate;
		Date newEndDate = Objects.nonNull(vo.getEndDate()) ? vo.getEndDate() : defaultDate;
		if (!DateUtils.dateToStr(dbEndDate).equals(DateUtils.dateToStr(newEndDate))) {
			upDateFlag = true;
		}
		if (upDateFlag) {
			// 通过状态机记录日志
			SysContext sysContext = new SysContext(null, null, null);
			String successInfo = String.format("(会员前:%s,会员新:%s,学校前:%s,学校新:%s)",
					DateUtils.dateToStr(
							Objects.nonNull(member.getMemberEndDate()) ? member.getMemberEndDate() : defaultDate),
					DateUtils.dateToStr(newEndDate), DateUtils.dateToStr(dbEndDate), DateUtils.dateToStr(newEndDate));

			sysContext.noStateExecute(sysContext, "美容学校有效期修改".concat(successInfo), dbVo.getSchoolMemberCode(),
					"training_institution", dbVo.getId(), null, (str) -> {
						// add by zjw 2023-01-10 解决修改美容有效期后，造成的学校后台与OA后台，学校有效期不一致的情况
						member.setMemberEndDate(vo.getEndDate());
					});
		}
		member.setName(vo.getSchoolNameCn());
		member.setNameEn(vo.getSchoolNameEn());
		member.setProvince(vo.getProvince());
		member.setCity(vo.getCity());
		member.setStreet(vo.getStreet());
		member.preUpdate();
		memberDao.update(member);

	}

	/**
	 * @description: 定时任务 处理过期的美容学校
	 * @author: laiguanglong
	 * @date: 2017/7/14 17:28
	 */
	@Transactional(readOnly = false)
	public int expireGroomerSchool() {
		int count = 0;
		// 1查询过期的美容学校
		List<String> userIdList = dao.getExpireGroomerSchoolUserId();
		// 2修改登录标记位为不允许登录
		for (String userId : userIdList) {
			expireOneGroomerSchool(userId);
			count++;
		}
		return count;
	}

	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public void expireOneGroomerSchool(String userId) {
		User user = systemService.getUser(userId);
		user.setLoginFlag("0");
		systemService.saveUser(user);
	}

	/**
	 * 美容学校停止合作 设置开放为否 ,并清空截止日期
	 * 
	 * @Author chaixueteng
	 * @2017年9月19日下午2:55:23
	 */
	@Transactional(readOnly = false)
	public void stopSchoolFlag(TrainingInstitution trainingInstitution) {
		String userId = trainingInstitution.getSchoolUserId();
		User user = systemService.getUser(userId);
		user.setLoginFlag("0");
		systemService.saveUser(user);
		trainingInstitution.setStartDate(null);
		trainingInstitution.setEndDate(null);
		Member byMemberCode = memberDao.getByMemberCode(trainingInstitution.getSchoolMemberCode());
		byMemberCode.setMemberEndDate(null);
		memberDao.update(byMemberCode);
		dao.update(trainingInstitution);
	}

	@Transactional(readOnly = false)
	public void updateIntroduce(TrainingInstitution trainingInstitution) {
		trainingInstitution.setIntroduce(StringEscapeUtils.unescapeHtml4(trainingInstitution.getIntroduce()));
		dao.updateIntroduce(trainingInstitution);
	}

	public Page<TrainingInstitution> getHandlerOrgList(Page<TrainingInstitution> page,
			TrainingInstitution trainingInstitution) {
		int pageNo = page.getPageNo();
		int pageSize = page.getPageSize();
		trainingInstitution.setPage(page);
		page.setList(dao.getHandlerOrgList(trainingInstitution));
		if ((pageNo - 1) * pageSize >= page.getCount()) {
			page.setList(new ArrayList<TrainingInstitution>(0));
		}
		return page;
	}
}