/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.restful.v1.breeder.service;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import com.cku.core.PageBeanResult;
import com.cku.core.PageResultDto;
import com.cku.core.RESTResponse;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.breeder.dao.StudDogDao;
import com.cku.oa.breeder.entity.StudDog;
import com.cku.oa.breeder.service.StudDogService;
import com.cku.oa.dog.dao.DogDao;
import com.cku.oa.dog.dao.DogTypeDao;
import com.cku.oa.dog.entity.Dog;
import com.cku.oa.dog.entity.DogType;
import com.cku.oa.dog.service.DogTypeService;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.kennel.service.KennelService;
import com.cku.oa.show.dao.ShowResultsDao;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.oa.sys.util.ImgUtil;
import com.cku.restful.v1.breeder.dto.RestQueryDogDto;
import com.cku.restful.v1.breeder.dto.RestStudDogDto;
import com.cku.restful.v1.breeder.model.StudDogSearchLabelEnum;
import com.cku.restful.v1.breeder.model.StudDogSearchLabelOtherEnum;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thinkgem.jeesite.modules.sys.entity.Dict;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

/**
 * @description: 种公业务
 * @author: laiguanglong
 * @date: 2017/10/16 10:43
 */
@Service
@Transactional(readOnly = true)
public class RestStudDogService {

    @Autowired
    private StudDogDao studDogDao;
    @Autowired
    private DogDao dogDao;
    @Autowired
    private ShowResultsDao showResultsDao;
    @Autowired
    private DogTypeDao dogTypeDao;
    @Autowired
    private RestBreederService restBreederService;
    @Autowired
    private StudDogService studDogService;
    @Autowired
    private DogTypeService dogTypeService;
    @Autowired
    private MemberService memberService;
    @Autowired
    private KennelService kennelService;
    @Autowired
    private RestCityService restCityService;

    public RESTResponse getList(HttpServletRequest request) {
        String pageNo = request.getParameter("pageNo");
        String pageSize = request.getParameter("pageSize");
        // 参数校验
        int pageNoInt = 1;
        int pageSizeInt = 10;
        try {
            if (StringUtils.isNotBlank(pageNo)) {
                pageNoInt = Integer.parseInt(pageNo);
            }
            if (StringUtils.isNotBlank(pageSize)) {
                pageSizeInt = Integer.parseInt(pageSize);
            }
        } catch (Exception e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
        }
        List<StudDog> studDogList = null;
        Map map = Maps.newHashMap();
        map.put("pageNo", (pageNoInt - 1) * pageSizeInt);
        map.put("pageSize", pageSizeInt);
        User user = UserUtils.getLoginUser();
        if (StringUtils.isNotBlank(user.getId())) {
            List<String> dogBreedList = dogDao.getFemaleDogBreedListByMemberCode(UserUtils.getLoginMember().getMemberCode());
            if (dogBreedList.size() > 0) {
                map.put("dogBreed", dogBreedList);
                studDogList = studDogDao.getListOrderByDogBreed(map);
            } else {
                studDogList = studDogDao.getList(map);
            }
        } else {
            studDogList = studDogDao.getList(map);
        }
        List<JSONObject> list = buildStudDogList(studDogList, request);
        return new RESTResponse("list", new PageBeanResult<>(list, studDogDao.getCount(map)));
    }

    @Transactional(readOnly = false)
    public RESTResponse studDogSearch(HttpServletRequest request) {
        String pageNo = request.getParameter("pageNo");
        String pageSize = request.getParameter("pageSize");
        // 参数校验
        int pageNoInt = 1;
        int pageSizeInt = 10;
        try {
            if (StringUtils.isNotBlank(pageNo)) {
                pageNoInt = Integer.parseInt(pageNo);
            }
            if (StringUtils.isNotBlank(pageSize)) {
                pageSizeInt = Integer.parseInt(pageSize);
            }
        } catch (Exception e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
        }
        String priceMin = request.getParameter("priceMin");
        if (StringUtils.isNotBlank(priceMin) &&(Integer.parseInt(priceMin) < 0  || Integer.parseInt(priceMin) > 99999999)  ) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "价格应在0到为99999999之间");
        }
        String priceMax = request.getParameter("priceMax");
        if (StringUtils.isNotBlank(priceMax) && (Integer.parseInt(priceMax) > 99999999 || Integer.parseInt(priceMax) < 0 )) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "价格应在0到为99999999之间");
        }
        Map map = Maps.newHashMap();
        map.put("pageNo", (pageNoInt - 1) * pageSizeInt);
        map.put("pageSize", pageSizeInt);
        map.put("dogBreed", request.getParameter("dogBreed"));
        map.put("priceMin", priceMin);
        map.put("priceMax", priceMax);
        map.put("city", request.getParameter("city"));
        String other = request.getParameter("other");
        if (StringUtils.isNotBlank(other)) {
            for (String value : other.split(",")) {
                StudDogSearchLabelOtherEnum e = StudDogSearchLabelOtherEnum.getByValue(value);
                if (e != null) {
                    map.put(e.getParameter(), "1");
                }
            }
        }
        //保存搜索参数
        studDogDao.saveSearchParameter(map);
        List<StudDog> studDogList = studDogDao.search(map);
        List<JSONObject> list = buildStudDogList(studDogList, request);
        return new RESTResponse("list", new PageBeanResult<>(list, studDogDao.getSearchCount(map)));
    }

    private List<JSONObject> buildStudDogList(List<StudDog> studDogList, HttpServletRequest request) {
        List<JSONObject> list = Lists.newArrayList();
        for (StudDog studDog : studDogList) {
            JSONObject jo = new JSONObject();
            jo.put("id", getString(studDog.getId()));
            jo.put("coverImgId", getString(ImgUtil.getImgUrl(request, studDog.getCoverImgId())));
            jo.put("dogBreed", getString(dogTypeService.getBreedCnName(studDog.getDogBreed())));
            jo.put("price", getString(getInt(studDog.getPrice())));
            String labels = StringUtils.join(studDog.getLabel(), ",", studDog.getOtherLabel());
            JSONArray ja = new JSONArray();
            for (String label : labels.split(",")) {
                if (StringUtils.isNotBlank(label)) {
                    ja.add(DictUtils.getDictLabel(label, "stud_dog_label", ""));
                    if (ja.size() >= 4) {
                        break;
                    }
                }
            }
            jo.put("label", ja);
            jo.put("age", getString(studDogService.getAge(studDog.getBirthdate())));
            jo.put("nameEn", getString(studDog.getNameEn()));
            jo.put("city", getString(studDog.getCity()));
            jo.put("viewCount", getString(studDog.getViewCount()));
            list.add(jo);
        }
        return list;
    }

    public RESTResponse getMyStudDog(HttpServletRequest request) {
        String pageNo = request.getParameter("pageNo");
        String pageSize = request.getParameter("pageSize");
        // 参数校验
        int pageNoInt = 1;
        int pageSizeInt = 10;
        try {
            if (StringUtils.isNotBlank(pageNo)) {
                pageNoInt = Integer.parseInt(pageNo);
            }
            if (StringUtils.isNotBlank(pageSize)) {
                pageSizeInt = Integer.parseInt(pageSize);
            }
        } catch (Exception e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
        }
        String memberCode = UserUtils.getLoginMember().getMemberCode();
        List<StudDog> studDogList = studDogDao.getByMemberCode(memberCode,null,(pageNoInt - 1) * pageSizeInt, pageSizeInt);
        List<JSONObject> list = Lists.newArrayList();
        for (StudDog studDog : studDogList) {
            JSONObject jo = new JSONObject();
            jo.put("id", getString(studDog.getId()));
            String state = "";
            if ("1".equals(studDog.getState())) {
                state = "已发布";
            }
            jo.put("state", getString(state));
            jo.put("coverImgId", getString(ImgUtil.getImgUrl(request, studDog.getCoverImgId())));
            jo.put("nickname", getString(studDog.getNickname()));
            jo.put("price", getString(getInt(studDog.getPrice())));
            jo.put("pedigreeCertified", getString(studDog.getPedigreeCertified()));
            jo.put("dogBreed", getString(dogTypeService.getBreedCnName(studDog.getDogBreed())));
            jo.put("age", studDogService.getAge(studDog.getDogId()));
            list.add(jo);
        }
        return new RESTResponse("list", new PageBeanResult<>(list, studDogDao.getCountByMemberCode(memberCode,null)));
    }

    @Transactional(readOnly = false)
    public RESTResponse post(String json) {
        // 获取参数
        JSONObject jo = JSONObject.fromObject(json);
        //参数校验
        postValidateParameter(jo);
        //基本数据校验
        Dog dog = dogDao.get(jo.getString("dogId"));
        if (dog == null || !dog.getMemberCode().equals(UserUtils.getLoginMember().getMemberCode())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬id有误");
        }
        if (!"1".equals(dog.getGender())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只有公犬可以发布");
        }
        if (!studDogService.isAgeOneToTen(dog.getBirthdate())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬只年龄不在1岁到10岁之间");
        }
        //构造实体类
        StudDog po = buildPo(jo);
        //业务校验
        studDogService.validateBusiness(po);
        StudDog example = new StudDog();
        example.setDogId(po.getDogId());
        List<StudDog> list = studDogDao.findList(example);
        if (list.size() > 0) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只已经添加到种犬列表中");
        }
        //更新dog表昵称
        if (!dog.getNickname().equals(jo.getString("nickname"))) {
            dog.setNickname(jo.getString("nickname"));
            dog.preUpdate();
            dogDao.update(dog);
        }
        //写入业务表
        studDogDao.insert(po);
        return new RESTResponse("data", po.getId());
    }

    @Transactional(readOnly = false)
    public RESTResponse put(String json) {
        Member member = UserUtils.getLoginMember();
        // 获取参数
        JSONObject jo = JSONObject.fromObject(json);
        //参数校验
        putValidateParameter(jo);
        //基本数据校验
        StudDog po = studDogDao.get(jo.getString("id"));
        if (po == null || !po.getMemberCode().equals(member.getMemberCode())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
        }
        Dog dog = dogDao.get(po.getDogId());
        //更新dog表昵称
        if (!dog.getNickname().equals(jo.getString("nickname"))) {
            dog.setNickname(jo.getString("nickname"));
            dog.preUpdate();
            dogDao.update(dog);
        }
        //更新种犬信息相关字段
        po.setPrice(jo.getString("price"));
        //业务校验
        studDogService.validateBusiness(po);
        po.setDoorToDoorFlag(jo.getString("doorToDoorFlag"));
        po.setLabel(getLabel(dog.getPedigreeCertifiedCode(), po));
        po.setOtherLabel(jo.getString("otherLabel"));
        po.setRecommendInfo(jo.getString("recommendInfo"));
        po.setCoverImgId(jo.getString("coverImgId"));
        po.setOtherImgId(jo.getString("otherImgId"));
        if (jo.containsKey("city") && StringUtils.isNotBlank(jo.getString("city"))) {
            po.setCity(jo.getString("city"));
        }
        studDogService.dogChangeInfo(po, member);
        po.setAddTime(new Date());
        //修改后立刻上架
        po.setState("1");
        po.preUpdate();
        studDogDao.update(po);
        return new RESTResponse();
    }

    @Transactional(readOnly = false)
    public RESTResponse studDogState(String json) {
        Member member = UserUtils.getLoginMember();
        JSONObject jo = JSONObject.fromObject(json);
        if (!jo.containsKey("id") || StringUtils.isBlank(jo.getString("id"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id为空");
        }
        if (!jo.containsKey("state") || StringUtils.isBlank(jo.getString("state"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "state为空");
        }
        if (!"0".equals(jo.getString("state")) && !"1".equals(jo.getString("state"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "state只能为0或1");
        }
        StudDog po = studDogDao.get(jo.getString("id"));
        if (po == null || !member.getMemberCode().equals(po.getMemberCode())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
        }
        //业务校验
        studDogService.validateBusiness(po);
        po.setState(jo.getString("state"));
        if ("1".equals(jo.getString("state"))) {
            Dog dog = dogDao.get(po.getDogId());
            po.setLabel(getLabel(dog.getPedigreeCertifiedCode(), po));
            studDogService.dogChangeInfo(po, member);
            po.setAddTime(new Date());
        }
        po.preUpdate();
        studDogDao.update(po);
        return new RESTResponse();
    }

    public RESTResponse label() {
        List<JSONObject> list = Lists.newArrayList();
        List<Dict> dictList = DictUtils.getDictList("stud_dog_label");
        for (Dict dict : dictList) {
            //排序大于20的是可编辑标签
            if (dict.getSort() > 20) {
                JSONObject jo = new JSONObject();
                jo.put("label", getString(dict.getLabel()));
                jo.put("value", getString(dict.getValue()));
                list.add(jo);
            }
        }
        return new RESTResponse("list", new PageBeanResult<>(list, new Long(list.size())));
    }

    public RESTResponse maleDogBreed() {
        List<JSONObject> list = Lists.newArrayList();
        List<String> dogBreedList = dogDao.getMaleDogBreedListByMemberCode(UserUtils.getLoginMember().getMemberCode());
        for (String dogBreed : dogBreedList) {
            if (StringUtils.isNotBlank(dogBreed)) {
                DogType dogType = dogTypeDao.getByDogBreedCode(dogBreed);
                if (dogType != null) {
                    JSONObject jo = new JSONObject();
                    jo.put("dogBreedCode", getString(dogType.getDogBreedCode()));
                    jo.put("breedCnName", getString(dogType.getBreedCnName()));
                    list.add(jo);
                }
            }
        }
        return new RESTResponse("list", new PageBeanResult<>(list, new Long(list.size())));
    }

    public RESTResponse getMaleDogByDogBreed(HttpServletRequest request) {
        String dogBreed = request.getParameter("dogBreed");
        if (StringUtils.isBlank(dogBreed)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬种号为空");
        }
        String pageNo = request.getParameter("pageNo");
        String pageSize = request.getParameter("pageSize");
        // 参数校验
        int pageNoInt = 1;
        int pageSizeInt = 10;
        try {
            if (StringUtils.isNotBlank(pageNo)) {
                pageNoInt = Integer.parseInt(pageNo);
            }
            if (StringUtils.isNotBlank(pageSize)) {
                pageSizeInt = Integer.parseInt(pageSize);
            }
        } catch (Exception e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,
                    "pageNo和pageSize必须都为数字");
        }
        Member member = UserUtils.getLoginMember();
        List<Dog> dogList = dogDao.getMaleDogByMemberCodeAndDogBreed(member.getMemberCode(), dogBreed, (pageNoInt - 1) * pageSizeInt, pageSizeInt);
        List<JSONObject> list = restBreederService.buildDogList(dogList, member);
        int count = dogDao.getMaleDogCountByMemberCodeAndDogBreed(member.getMemberCode(), dogBreed);
        return new RESTResponse("list", new PageBeanResult<>(list, new Long(count)));
    }

    private StudDog buildPo(JSONObject jo) {
        Member member = UserUtils.getLoginMember();
        String memberCode = member.getMemberCode();
        Dog dog = dogDao.get(jo.getString("dogId"));
        if (dog == null || !memberCode.equals(dog.getMemberCode())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬id有误");
        }
        StudDog po = new StudDog();
        po.setDogId(dog.getId());
        po.setState("1");
        po.setPedigreeCertified(dog.getPedigreeCertifiedCode());
        po.setDogBreed(dog.getDogBreed());
        po.setMemberCode(memberCode);
        po.setPrice(jo.getString("price"));
        po.setDoorToDoorFlag(jo.getString("doorToDoorFlag"));
        po.setLabel(getLabel(dog.getPedigreeCertifiedCode(), po));
        po.setOtherLabel(jo.getString("otherLabel"));
        po.setRecommendInfo(jo.getString("recommendInfo"));
        po.setCoverImgId(jo.getString("coverImgId"));
        po.setOtherImgId(jo.getString("otherImgId"));
        String city = restCityService.getCity(member);
        if (jo.containsKey("city") && StringUtils.isNotBlank(jo.getString("city"))) {
            city = jo.getString("city");
        }
        po.setCity(city);
        studDogService.dogChangeInfo(po, member);
        po.setAddTime(new Date());
        po.preInsert();
        return po;
    }

    /**
     * @description: 获取标签
     * @author: laiguanglong
     * @date: 2017/10/16 10:44
     */
    private String getLabel(String pedigreeCertified, StudDog po) {
        List<String> label = Lists.newArrayList();
        String[] loginTypeArray = studDogService.getloginTypeArray(pedigreeCertified);
        if (StringUtils.isNotBlank(loginTypeArray[0])) {
            label.add(loginTypeArray[0]);
        }
        if (StringUtils.isNotBlank(loginTypeArray[1])) {
            label.add(loginTypeArray[1]);
        }
        String label3 = null;
        int bisCount = showResultsDao.getBisCountByPedigreeCertified(pedigreeCertified);
        if (bisCount > 0) {
            label3 = "11";
        }
        if (StringUtils.isBlank(label3)) {
            int bigCount = showResultsDao.getBigCountByPedigreeCertified(pedigreeCertified);
            if (bigCount > 0) {
                label3 = "12";
            }
        }
        if (StringUtils.isBlank(label3)) {
            int bobCount = showResultsDao.getBobCountByPedigreeCertified(pedigreeCertified);
            if (bobCount > 0) {
                label3 = "13";
            }
        }
        if (StringUtils.isNotBlank(label3)) {
            label.add(label3);
        }
        if (StringUtils.isNotBlank(po.getDogPedigreeCertifiedChangeId())) {
            label.add("15");
        }
        return StringUtils.join(label, ",");
    }

    private void postValidateParameter(JSONObject jo) {
        if (!jo.containsKey("dogId") || StringUtils.isBlank(jo.getString("dogId"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "dogId为空");
        }
        if (!jo.containsKey("nickname") || StringUtils.isBlank(jo.getString("nickname"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "nickname为空");
        }
        if (!jo.containsKey("price") || StringUtils.isBlank(jo.getString("price"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "price为空");
        }
        if (!StringUtils.isNumeric(jo.getString("price"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "price不是整数");
        }
        if (!jo.containsKey("coverImgId") || StringUtils.isBlank(jo.getString("coverImgId"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "coverImgId为空");
        }
        if (!jo.containsKey("otherImgId")) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "otherImgId为空");
        }
        if (!jo.containsKey("doorToDoorFlag") || StringUtils.isBlank(jo.getString("doorToDoorFlag"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "doorToDoorFlag为空");
        }
        if (!"0".equals(jo.getString("doorToDoorFlag")) && !"1".equals(jo.getString("doorToDoorFlag"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "doorToDoorFlag只能为0或1");
        }
        if (!jo.containsKey("otherLabel")) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "otherLabel为空");
        }
        if (!jo.containsKey("recommendInfo")) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "recommendInfo为空");
        }
    }

    private void putValidateParameter(JSONObject jo) {
        if (!jo.containsKey("id") || StringUtils.isBlank(jo.getString("id"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id为空");
        }
        if (!jo.containsKey("nickname") || StringUtils.isBlank(jo.getString("nickname"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "nickname为空");
        }
        if (!jo.containsKey("price") || StringUtils.isBlank(jo.getString("price"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "price为空");
        }
        if (!StringUtils.isNumeric(jo.getString("price"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "price不是整数");
        }
        if (!jo.containsKey("coverImgId") || StringUtils.isBlank(jo.getString("coverImgId"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "coverImgId为空");
        }
        if (!jo.containsKey("otherImgId")) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "otherImgId为空");
        }
        if (!jo.containsKey("doorToDoorFlag") || StringUtils.isBlank(jo.getString("doorToDoorFlag"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "doorToDoorFlag为空");
        }
        if (!"0".equals(jo.getString("doorToDoorFlag")) && !"1".equals(jo.getString("doorToDoorFlag"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "doorToDoorFlag只能为0或1");
        }
        if (!jo.containsKey("otherLabel")) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "otherLabel为空");
        }
        if (!jo.containsKey("recommendInfo")) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "recommendInfo为空");
        }
    }

    @Transactional(readOnly = false)
    public RESTResponse get(String id, HttpServletRequest request) {
        StudDog studDog = studDogDao.get(id);
        if (studDog == null) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
        }
        Dog dog = dogDao.get(studDog.getDogId());
        JSONObject data = studDogService.studDogOtherInfo(dog);
        //犬只呼名
        data.put("nickname", getString(dog.getNickname()));
        //配种次数
        data.put("dogNewbornCount", getString(studDog.getDogNewbornCount()));
        //城市
        data.put("city", getString(studDog.getCity()));

        Member member = memberService.getByMemberCode(dog.getMemberCode());
        Kennel kennel = kennelService.getKennelByMemberCode(dog.getMemberCode());
        //犬种和年龄
        data.put("dogBreed", getString(dogTypeService.getBreedCnName(dog.getDogBreed())));
        data.put("age", getString(studDogService.getAge(dog.getBirthdate())));
        //犬英文名
        data.put("nameEn", getString(dog.getNameEn()));
        //犬舍名和价格
        String kennelName = "";
        if (kennel != null) {
            kennelName = kennel.getName();
        }
        data.put("kennelName", getString(kennelName));
        //犬主电话
        data.put("mobile", getString(member.getMobile()));
        data.put("memberName", getString(member.getName()));
        data.put("memberGender", getString(member.getGender()));
        data.put("pedigreeCertifiedCode", getString(dog.getPedigreeCertifiedCode()));

        //id
        data.put("id", studDog.getId());
        //标签
        JSONArray label = new JSONArray();
        if (StringUtils.isNotBlank(studDog.getLabel())) {
            for (String s : studDog.getLabel().split(",")) {
                label.add(DictUtils.getDictLabel(s, "stud_dog_label", ""));
            }
        }
        if (StringUtils.isNotBlank(studDog.getOtherLabel())) {
            for (String s : studDog.getOtherLabel().split(",")) {
                label.add(DictUtils.getDictLabel(s, "stud_dog_label", ""));
            }
        }
        data.put("label", label);
        data.put("price", getString(getInt(studDog.getPrice())));
        //是否可上门配种
        data.put("doorToDoorFlag", getString(studDog.getDoorToDoorFlag()));
        //犬主推荐
        data.put("recommendInfo", getString(studDog.getRecommendInfo()));
        //风采展示
        JSONArray img = new JSONArray();
        if (StringUtils.isNotBlank(studDog.getCoverImgId())) {
            img.add(getString(ImgUtil.getImgUrl(request, studDog.getCoverImgId())));
        }
        if (StringUtils.isNotBlank(studDog.getOtherImgId())) {
            for (String otherImgId : studDog.getOtherImgId().split(",")) {
                if (StringUtils.isNotBlank(otherImgId)) {
                    img.add(getString(ImgUtil.getImgUrl(request, otherImgId)));
                }
            }
        }
        data.put("img", img);
        data.put("dogBreedCode", studDog.getDogBreed());
        //更新信息
        studDogService.dogChangeInfo(studDog, member);
        studDog.preUpdate();
        //点击次数加一
        studDog.setViewCountFlag("1");
        studDogDao.update(studDog);
        return new RESTResponse("data", data);
    }

    public String getInt(String price) {
        Double priceDouble = Double.parseDouble(price);
        return priceDouble.intValue() + "";
    }

    public RESTResponse studDogSearchLabel() {
        List<JSONObject> list = Lists.newArrayList();
        for (StudDogSearchLabelEnum studDogSearchLabelEnum : StudDogSearchLabelEnum.values()) {
            JSONObject jo = new JSONObject();
            jo.put("label", studDogSearchLabelEnum.getLabel());
            jo.put("value", studDogSearchLabelEnum.getValue());
            jo.put("type", studDogSearchLabelEnum.getType());
            list.add(jo);
        }
        return new RESTResponse("list", new PageBeanResult<>(list, new Long(list.size())));
    }

    public RESTResponse studDogSearchLabelOther() {
        List<JSONObject> list = Lists.newArrayList();
        for (StudDogSearchLabelOtherEnum studDogSearchLabelOtherEnum : StudDogSearchLabelOtherEnum.values()) {
            JSONObject jo = new JSONObject();
            jo.put("label", studDogSearchLabelOtherEnum.getLabel());
            jo.put("value", studDogSearchLabelOtherEnum.getValue());
            list.add(jo);
        }
        return new RESTResponse("list", new PageBeanResult<>(list, new Long(list.size())));
    }

    public RESTResponse studDogTopList(HttpServletRequest request) {
        String pageSize = request.getParameter("pageSize");
        // 参数校验
        int pageSizeInt = 10;
        try {
            if (StringUtils.isNotBlank(pageSize)) {
                pageSizeInt = Integer.parseInt(pageSize);
            }
        } catch (Exception e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageSize必须为数字");
        }
        List<StudDog> studDogList = studDogDao.topList(pageSizeInt);
        List<JSONObject> list = buildStudDogList(studDogList, request);
        return new RESTResponse("list", new PageBeanResult<>(list, new Long(list.size())));
    }

    private String getString(String str) {
        return str == null ? "" : str;
    }
    
    
    public PageResultDto<RestStudDogDto> findMarketStudDogByKennelCode(RestQueryDogDto queryDto,HttpServletRequest request) {
        String kennelCode = queryDto.getKennelCode();
        if(StringUtils.isBlank(kennelCode)) {
        	throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍编号不能为空");
        }
        //查询已上架种工
        String state = "1";
        List<StudDog> studDogList = studDogDao.getByMemberCode(kennelCode,state,(queryDto.getPageNo() - 1) * queryDto.getPageSize(),queryDto.getPageSize());
        long totalCount = studDogDao.getCountByMemberCode(kennelCode,state);
        List<RestStudDogDto> resultList = Collections.emptyList();
        if(!CollectionUtils.isEmpty(studDogList)) {
        	resultList = studDogList.stream().map(studDog->{
        		RestStudDogDto restDog = new RestStudDogDto();
        		restDog.setId(studDog.getId());
        		restDog.setDogId(studDog.getDogId());
        		restDog.setAge(studDogService.getAge(studDog.getDogId()));
        		restDog.setCoverImgId(getString(ImgUtil.getImgUrl(request, studDog.getCoverImgId())));
        		restDog.setDogBreed(getString(dogTypeService.getBreedCnName(studDog.getDogBreed())));
        		restDog.setNickname(getString(studDog.getNickname()));
        		restDog.setPedigreeCertified(getString(studDog.getPedigreeCertified()));
        		restDog.setPrice(getString(getInt(studDog.getPrice())));
        		restDog.setState(studDog.getState());
        		return restDog;
        	}).collect(Collectors.toList());
        }
        return PageResultDto.success(resultList,totalCount);
    }
}