package com.cku.oa.order.service.impl;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.order.service.BaseSkuBusinessService;
import com.cku.oa.order.service.IOrderBusinessService;
import com.cku.oa.order.vo.ChargingItemVo;
import com.cku.oa.order.vo.OrderSkuVo;
import com.cku.oa.order.vo.ShowApplyFeeVO;
import com.cku.oa.order.vo.SingleIdOrderBusinessVo;
import com.cku.oa.order.vo.request.SingleIdRequest;
import com.cku.oa.show.entity.MainShows;
import com.cku.oa.show.entity.ShowApply;
import com.cku.oa.show.entity.ShowApplyDiscount;
import com.cku.oa.show.service.MainShowsService;
import com.cku.oa.show.service.ShowApplyDiscountService;
import com.cku.oa.show.service.ShowApplyService;
import com.cku.oa.sys.entity.user.Member;
import com.cku.restful.v2.order.vo.request.OrderApplyRequest;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;

/**
 * 后台赛事报名（目前仅支持后台报名）
 * 
 * @author xx
 *
 */
@Service
public class ShowApplyOrderServiceImpl extends BaseSkuBusinessService<SingleIdOrderBusinessVo>
		implements IOrderBusinessService<SingleIdRequest> {

	@Autowired
	private ShowApplyService showApplyService;

	@Autowired
	private MainShowsService mainShowsService;

	@Autowired
	private ShowApplyDiscountService showApplyDiscountService;

	private static final String PAYMENT_CHARGING_ITEM_BJZB = "犬只参赛费（北京自办）";

	private static final String PAYMENT_CHARGING_ITEM_DWWL = "单位往来";

	private static final String SHOW_ORGNIZE_TYPE_XHB = "3";// 合办新模式2025
	
	private static final String SHOW_APPLY = "show_apply";

	private static final ImmutableMap<String, String> SKU_MAP = ImmutableMap.<String, String>builder()
			.put("犬只参赛费（北京自办）", "18cf6323f9bb43db8c6d08541bd04cac")
			.put("犬只参赛费（北京新合办）", "29c240bdfcfc416bafd2b2ce219ac491")
			.put("犬只参赛费（北方区新合办）", "3227b36102024a3c89394dd1faca073a")
			.put("犬只参赛费（北方区自办）", "4d91cacdacb7466abe021c1526f37438")
			.put("犬只参赛费（南方区新合办）", "e95b1a7fdf414ac4b7e2552b72be65aa")
			.put("犬只参赛费（南方区自办）", "f9a632a1b11040e88045e60f30523fa3").put("单位往来", "d5c2a6b3e5b740bd8e0813880df24ad8")
			.put("犬只参赛费（北京自办）+单位往来", "a31cb2bd5bd24fed8adbbd2561a31075").build();

	@Override
	public void businessValidate(OrderApplyRequest orderRequest, SingleIdRequest businessRequest, Member member) {
		if ("1".equals(member.getFreezeFlag()) && !member.getMemberCode().contains("HZ")) {
			throw new ZAException(ZAErrorCode.ZA_ERC_MEMBER_INVALID, "会员已冻结！");
		}
		if (StringUtils.isBlank(businessRequest.getId())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "参数为空");
		}
		ShowApply showApply = showApplyService.get(businessRequest.getId());
		if (Objects.isNull(showApply)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请记录未找到");
		}
		if (!"1".equals(showApply.getPaymentState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请记录缴费状态不正确");
		}
		// 如果是后台报名且犬主人是外籍会员则不校验未缴费订单
		if (!showApply.getApplyCode().startsWith("BM")
				|| (showApply.getApplyCode().startsWith("BM") && !"foreign".equals(member.getMemberType()))) {
			if (showApplyService.isUnpaidShowApplyById(businessRequest.getId())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "当前犬主名下犬只已有未缴费的赛事订单，请先处理。");
			}
		}
	}

	@Override
	public List<OrderSkuVo> preOrder(OrderApplyRequest orderRequest, SingleIdRequest businessRequest, Member member) {
		List<OrderSkuVo> list = Lists.newArrayList();
		ShowApply showApply = showApplyService.get(businessRequest.getId());
		MainShows mainShows = mainShowsService.get(showApply.getMainShowId());
		if (SHOW_ORGNIZE_TYPE_XHB.equals(mainShows.getOrgnizeType())) {
			// 合办新模式2025 收费项固定为北京自办、单位往来
			ShowApplyFeeVO fee = showApplyService.countFeeForNew(showApply);
			if (fee.getBjzb().compareTo(BigDecimal.ZERO) > 0 && fee.getDwwl().compareTo(BigDecimal.ZERO) > 0) {
				list.add(OrderSkuVo.builder()
						.skuId(SKU_MAP.get(PAYMENT_CHARGING_ITEM_BJZB + "+" + PAYMENT_CHARGING_ITEM_DWWL)).num(1)
						.businessList(Stream.of(SingleIdOrderBusinessVo.builder().id(businessRequest.getId()).build())
								.collect(Collectors.toList()))
						.build());
			} else if (fee.getBjzb().compareTo(BigDecimal.ZERO) == 0) {
				list.add(OrderSkuVo.builder().skuId(SKU_MAP.get(PAYMENT_CHARGING_ITEM_DWWL)).num(1)
						.businessList(Stream.of(SingleIdOrderBusinessVo.builder().id(businessRequest.getId()).build())
								.collect(Collectors.toList()))
						.build());
			}

		} else {
			String organizer = DictUtils.getDictLabel(mainShows.getOrganizer(), "show_organizer", "").replace("市", "");
			String orgnizeType = DictUtils.getDictLabel(mainShows.getOrgnizeType(), "show_orgnize_type", "");
			// 判断新旧合办
			if ("0".equals(mainShows.getOrgnizeType()) && "1".equals(mainShows.getBusinessModelFlag())) {
				orgnizeType = "新" + orgnizeType;
			}
			list.add(OrderSkuVo.builder().skuId(SKU_MAP.get("犬只参赛费（"+ organizer + orgnizeType+"）")).num(1)
					.businessList(Stream.of(SingleIdOrderBusinessVo.builder().id(businessRequest.getId()).build())
							.collect(Collectors.toList()))
					.build());
		}
		return list;

	}

	@Override
	public void processBusinessPrePay(OrderApplyRequest orderRequest, SingleIdOrderBusinessVo businessData,
			OrderSkuVo skuVo) {
		// 如果是赛事满减 需要填入订单号
		ShowApplyDiscount showApplyDiscount = showApplyDiscountService.getByShowApplyId(businessData.getId());
		if (showApplyDiscount != null) {
			showApplyDiscount.setRunningNumber(skuVo.getOrderCode());
			showApplyDiscount.preUpdate();
			showApplyDiscountService.save(showApplyDiscount);
		}
		//修改apply表的订单号
		ShowApply showApply = showApplyService.get(businessData.getId());
		showApply.setRunningNumber(skuVo.getOrderCode());
		showApplyService.save(showApply);
	}

	@Override
	@Transactional(readOnly = false)
	public void findChargingItemBusinessTableId(OrderApplyRequest orderRequest, OrderSkuVo skuVo,
			List<ChargingItemVo> itemList) {
		// 获取业务信息
		SingleIdOrderBusinessVo businessData = (SingleIdOrderBusinessVo) skuVo.getBusinessList().get(0);
		ShowApply showApply = showApplyService.get(businessData.getId());
		itemList.forEach(item->{
			 item.setBusinessIds(showApply.getId());
			 item.setBusinessTable(SHOW_APPLY);
		});
	}

	@Override
	public void processChargingItemPrice(OrderApplyRequest orderRequest, List<SingleIdOrderBusinessVo> businessRequest,
			List<ChargingItemVo> itemList) {
		// 获取业务参数
		SingleIdOrderBusinessVo businessVo = businessRequest.get(0);
		ShowApply showApply = showApplyService.get(businessVo.getId());
		MainShows mainShows = mainShowsService.get(showApply.getMainShowId());
		if (SHOW_ORGNIZE_TYPE_XHB.equals(mainShows.getOrgnizeType())) {
			// 合办新模式2025 收费项固定为北京自办、单位往来
			ShowApplyFeeVO fee = showApplyService.countFeeForNew(showApply);
			for (ChargingItemVo vo : itemList) {
				if (PAYMENT_CHARGING_ITEM_BJZB.equals(vo.getName())) {
					vo.setPrice(fee.getBjzb());
				}
				if (PAYMENT_CHARGING_ITEM_DWWL.equals(vo.getName())) {
					vo.setPrice(fee.getDwwl());
				}
			}

		} else {
			BigDecimal fee = showApplyService.countFee(showApply);
			itemList.get(0).setPrice(fee);// 修改收费项金额
		}
	}

}
