package com.cab.passport.marketing.service;

import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestTemplate;

import com.alibaba.fastjson.JSON;
import com.cab.passport.marketing.entity.PaymentOrderPayLog;
import com.cab.passport.sys.payment.dao.PaymentOrderDao;
import com.cab.passport.sys.payment.entity.PaymentOrder;
import com.cab.passport.sys.payment.service.PaymentOrderService;
import com.github.wxpay.sdk.WXPay;
import com.github.wxpay.sdk.WXPayConfig;
import com.github.wxpay.sdk.WXPayConstants.SignType;
import com.github.wxpay.sdk.WXPayUtil;
import com.thinkgem.jeesite.common.utils.IdGen;
import com.thinkgem.jeesite.common.utils.RESTResponse;
import com.thinkgem.jeesite.common.utils.ZAException;

import me.chanjar.weixin.common.bean.WxAccessToken;
import net.sf.json.JSONObject;

@Service
@Transactional(readOnly = true)
public class WXpayServiceImpl {

	private static final Logger logger = LoggerFactory.getLogger(WXpayServiceImpl.class);

	// @Autowired
	// private Environment env;

	@Autowired
	private RestTemplate restTemplate;

	@Autowired
	private MyWXPayConfig wxPayConfig;
	
	@Autowired
	private CatConfig catConfig;
	
	@Autowired
	private MyCatWXPayConfig catWxPayConfig;

	@Autowired
	private PaymentOrderDao paymentOrderDao;

	@Autowired
	private PaymentOrderPayLogService paymentOrderPayLogService;

	@Autowired
	private PaymentOrderService paymentOrderService;

	@Transactional(readOnly = false)
	public RESTResponse createPayOrder(String orderId, String userIp, String openId, String type) throws Exception {
		// 1、查询订单 获取金额
		PaymentOrder order = paymentOrderDao.get(orderId);
		if (order == null) {
			throw new ZAException("订单ID错误！");
		}
		if ("2".equals(order.getPaymentState())) {
			throw new ZAException("订单已经支付！");
		}

		// 2、记录支付记录，并修改订单支付状态
		PaymentOrderPayLog log = savePayLog(order.getTotalPrice(), order.getId(), openId);

		// 3、调用微信接口下支付单
		Map<String, String> map = getPrepayId(log.getPayNo(), new BigDecimal(order.getTotalPrice()), userIp, openId,
				type);
		map.put("payLogId", log.getId());
		return new RESTResponse(JSONObject.fromObject(map));
	}

	private PaymentOrderPayLog savePayLog(String payMoney, String orderId, String openId) {
		// 构建支付记录
		PaymentOrderPayLog entity = new PaymentOrderPayLog();
		entity.setPayType("24");
		entity.setPayTypeName("微信");
		entity.setPayMoney(payMoney);
		entity.setPayStatus("0");
		entity.setPayNo(IdGen.uuid());
		entity.setOrderId(orderId);
		entity.setCreateTime(new Date());
		entity.setBuyerLogonId(openId);
		// 保存支付记录
		paymentOrderPayLogService.save(entity);
		return entity;
	}

	public Map<String, String> getPrepayId(String outTradeNo, BigDecimal totalAmount, String userIp, String openId,
			String type) throws Exception {

		if (StringUtils.isBlank(openId)) {
			throw new ZAException("openId 为空！");
		}

		if (StringUtils.isBlank(userIp)) {
			throw new ZAException("userIp 为空！");
		}
		Map<String, String> reqData = new HashMap<>();
		reqData.put("body", "cku-鉴宝活动");
		reqData.put("out_trade_no", outTradeNo);
		// 方便测试
		// if (Arrays.asList(env.getActiveProfiles()).contains("prod")) {
		reqData.put("total_fee", String
				.valueOf(totalAmount.multiply(new BigDecimal(100)).setScale(0, BigDecimal.ROUND_HALF_UP).intValue()));
		// } else {
//		 reqData.put("total_fee", "1");
		// }
		
		reqData.put("spbill_create_ip", userIp);

		WXPayConfig config = getConfig(type);
		
		reqData.put("notify_url", getNotifyUrl(type));
		reqData.put("trade_type", "JSAPI");

		reqData.put("openid", openId);

		if("cat".equals(type)) {
			config = catConfig;
		}
		Map<String, String> weixinResponse = new WXPay(config).unifiedOrder(reqData);
		if (!weixinResponse.get("return_code").equals("SUCCESS")
				|| !weixinResponse.get("result_code").equals("SUCCESS")) {
			logger.error("获取prepay_id失败" + weixinResponse);
			throw new ZAException("获取prepay_id失败");
		}
		// 最后参与签名的参数有appId, timeStamp, nonceStr, package, signType

		Map<String, String> parameterMap2 = new HashMap<>();
		parameterMap2.put("appId", config.getAppID());
		parameterMap2.put("timeStamp", String.valueOf(System.currentTimeMillis()).toString().substring(0, 10));
		parameterMap2.put("nonceStr", WXPayUtil.generateNonceStr());
		parameterMap2.put("package", "prepay_id=" + weixinResponse.get("prepay_id"));
		parameterMap2.put("signType", SignType.MD5.toString());

		parameterMap2.put("sign", WXPayUtil.generateSignature(parameterMap2, config.getKey(), SignType.MD5));
		parameterMap2.put("timestamp", parameterMap2.get("timeStamp"));
		return parameterMap2;
	}

	public Map<String, String> getAccessToken(String code, String type) throws Exception {
		WXPayConfig config = getConfig(type);
		String URL = "https://api.weixin.qq.com/sns/oauth2/access_token?grant_type=authorization_code&appid="
				+ config.getAppID() + "&secret=" + config.getKey() + "&code=";
		String body = restTemplate.getForEntity(URL + code, String.class).getBody();
		logger.info("===== cat getAccessToken response:{}",body);
		HashMap<String, String> map = JSON.parseObject(body, HashMap.class);
		if (StringUtils.isBlank(map.get("openid"))) {
			throw new ZAException("获取openID失败！请重新获取code！");
		}
		HashMap<String, String> mapR = new HashMap<>();
		mapR.put("openid", map.get("openid"));
		return mapR;
	}
	
	

	public JSONObject getTicket(String url, String type) throws Exception {
		WXPayConfig config = getConfig(type);
		String URL = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=" + config.getAppID()
				+ "&secret=" + config.getKey();

		String t = restTemplate.getForEntity(URL, String.class).getBody();
		
		HashMap<String, String> tmap = JSON.parseObject(t, HashMap.class);
		String URL2 = "https://api.weixin.qq.com/cgi-bin/ticket/getticket?access_token=" + tmap.get("access_token")
				+ "&type=jsapi";
		String body = restTemplate.getForEntity(URL2, String.class).getBody();
		HashMap<String, String> map = JSON.parseObject(body, HashMap.class);
		if (StringUtils.isBlank(map.get("ticket"))) {
			throw new ZAException("获取票据失败!");
		}
		String noncestr = WXPayUtil.generateNonceStr();
		String timestamp = String.valueOf(System.currentTimeMillis()).toString().substring(0, 10);

		StringBuilder sb = new StringBuilder("jsapi_ticket=");
		sb.append(map.get("ticket"));
		sb.append("&noncestr=");
		sb.append(noncestr);
		sb.append("&timestamp=");
		sb.append(timestamp);
		sb.append("&url=");
		sb.append(url);

		JSONObject jsonObject = new JSONObject();
		jsonObject.put("appId", config.getAppID());
		jsonObject.put("noncestr", noncestr);
		jsonObject.put("timestamp", timestamp);
		jsonObject.put("signature", DigestUtils.sha1Hex(sb.toString()));
		return jsonObject;

	}

	@Transactional(readOnly = false)
	public RESTResponse check(String payLogId, String type) throws Exception {
		JSONObject j = new JSONObject();
		// 查询库中是否已经记录付款成功
		PaymentOrderPayLog entity = new PaymentOrderPayLog();
		entity.setId(payLogId);
		PaymentOrderPayLog log = paymentOrderPayLogService.get(entity);
		if (log == null) {
			// 已成功直接返回
			return new RESTResponse(10001, "无支付记录");
		}
		PaymentOrder order = paymentOrderService.get(log.getOrderId());
		if ("1".equals(log.getPayStatus()) && "2".equals(order.getPaymentState())) {
			// 已成功直接返回
			j.put("result", "SUCCESS");
			return new RESTResponse(j);
		}
		// 未成功向微支接口确认
		Map<String, String> reqData = new HashMap<>();
		reqData.put("out_trade_no", log.getPayNo());
		Map<String, String> weixinResponse = new WXPay(getConfig(type)).orderQuery(reqData);
		if ("SUCCESS".equals(weixinResponse.get("return_code")) && "SUCCESS".equals(weixinResponse.get("result_code"))
				&& "SUCCESS".equals(weixinResponse.get("trade_state"))) {
			// 查询成功 修改支付表
			log.setOutsidePayNo(weixinResponse.get("transaction_id"));
			log.setPayStatus("1");
			log.setFinishTime(new Date());
			log.setBankType(weixinResponse.get("bank_type"));
			log.setBuyerLogonId(weixinResponse.get("openid"));
			this.paymentOrderPayLogService.save(log);
			order.setPaymentWay("24");
			// 修改业务表
			paymentOrderService.afterPay(order);
			// 返回成功
			j.put("result", "SUCCESS");
			return new RESTResponse(j);
		}

		j.put("result", "FAIL");
		return new RESTResponse(j);
	}

	@Transactional(readOnly = false)
	public Boolean paySuccess(Map<String, String> paySuccessResult) throws Exception {
		// 先确定该通知是不是属实
		PaymentOrderPayLog entity = new PaymentOrderPayLog();
		entity.setPayNo(paySuccessResult.get("out_trade_no"));
		PaymentOrderPayLog pay = paymentOrderPayLogService.get(entity);
		// 没有查到支付单知会微信再次通知，因为可能出现存入数据库比微信处理慢的情况
		if (pay != null) {
			PaymentOrder order = paymentOrderDao.get(pay.getOrderId());
			// 检查支付金额,支付金额对不上则返回
			BigDecimal total = BigDecimal.ONE;
			// if (Arrays.asList(this.env.getActiveProfiles()).contains("prod")) {
			total = new BigDecimal(pay.getPayMoney()).multiply(new BigDecimal(100));
			// }
			if (!(total.compareTo(new BigDecimal(paySuccessResult.get("total_fee"))) == 0)) {
				return false;
			}
			// 接到重复通知不做任何操作
			if ("0".equals(pay.getPayStatus())) {
				pay.setOutsidePayNo(paySuccessResult.get("transaction_id"));
				pay.setPayStatus("1");
				pay.setFinishTime(new Date());
				pay.setBankType(paySuccessResult.get("bank_type"));
				pay.setBuyerLogonId(paySuccessResult.get("openid"));
				this.paymentOrderPayLogService.save(pay);
				// 修改订单
				paymentOrderService.afterPay(order);
			}
			return true;
		}
		return false;
	}

	private WXPayConfig getConfig(String type) {
		WXPayConfig config = null;
		switch (type) {
		case "cat":
			config = catWxPayConfig;
			break;
		case "dog":
			config = wxPayConfig;
			break;
		default:
			config = wxPayConfig;
			break;
		}
		return config;
	}

	private String getNotifyUrl(String type) {
		String url = "";
		switch (type) {
		case "cat":
			url = catWxPayConfig.getPayNotifyUrl();
			break;
		case "dog":
			url = wxPayConfig.getPayNotifyUrl();
			break;
		default:
			url = wxPayConfig.getPayNotifyUrl();
			break;
		}
		return url;
	}

}
