package com.cku.restful.v1.sys.service;

import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.http.HttpResponse;
import org.apache.http.util.EntityUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.sys.dao.IDCardDao;
import com.cku.oa.sys.dao.IDCardLogsDao;
import com.cku.oa.sys.dao.user.MemberDao;
import com.cku.oa.sys.entity.IDCard;
import com.cku.oa.sys.entity.IDCardLogs;
import com.cku.oa.sys.util.HttpUtils;
import com.cku.oa.sys.util.IPLimitUtil;
import com.cku.restful.v1.sys.model.OCRMessage;
import com.cku.restful.v1.sys.model.OCRResult;
import com.cku.util.Debugger;
import com.cku.util.MD5Generator;
import com.google.gson.Gson;
import com.thinkgem.jeesite.common.utils.CacheUtils;
import com.thinkgem.jeesite.common.utils.DateUtils;
import com.thinkgem.jeesite.common.utils.StringUtils;

import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

@Slf4j
@Service
public class RestIDCardService {
	// 身份证号次数验证字段
	private final static String IDCARD_CACHE_NAME = "IDCardCache";
	// 手机号次数验证字段
	private final static String MOBILE_CACHE_NAME = "MobileCache";
	// 请求host
	private final static String OCR_HOST = "http://netocr.com";
	// 请求uri
	private final static String VALID_PATH = "/verapi/veridenNo.do";

	private final static String REMAINDER_PATH = "/api/accountLift.do";
	// 请求方式
	private final static String HTTP_METHOD = "POST";

	private final static String OCR_KEY = "MP915Koiidyozjuk1NeK6B"; // 用户ocrKey
	private final static String OCR_SECRET = "7dfe8c3f61144e51a15a2aae79ffe89e"; // 用户ocrSecret

	@Autowired
	private IDCardDao idCardDao;
	@Autowired
	private IDCardLogsDao idCardLogsDao;
	@Autowired
	private MemberDao memberDao;

	/**
	 * 实名认证
	 *
	 * @param request
	 * @param idCard   身份证号
	 * @param realName 真实姓名
	 * @return 0为认证不通过，1为认证通过
	 */
	@Transactional(readOnly = false, rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
	public int validateIDCard(HttpServletRequest request, String idCard, String realName, String memberSite,
			boolean checkRepeat) {
		// 查看证件号是否重复
		if (StringUtils.isNotBlank(memberSite)) {
			if (checkRepeat && memberDao.getByCardNo(idCard, memberSite) != null) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "您的身份证号与其他会员的身份证号码重复，请您联系客服，致电400-660-7000");
			}
		}
		// 检查是否可以进行实名认证
		if (request != null) {
			checkValidateState(request);
			// 判断是否大于18岁
			checkAge(idCard);
		}
		// 正则判断身份证号是否符合规则
		if (!isIDcard(idCard)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "身份证号码格式错误！");
		}
		// 从本地数据库中查询记录
		Integer qualified = idCardDao.getIDCardByData(idCard, realName);
		if (qualified != null) {
			return qualified;
		} else {
			// 通过第三方服务验证
			return validateByThird(new IDCard(idCard, realName), request).getQualified().intValue();
		}
	}

	/**
	 * 判断是否大于18岁
	 * 
	 * @param idCard 身份证号
	 */
	private void checkAge(String idCard) {
		try {
			int birthYearSpan = 4;
			// 如果是15位的证件号码
			if (idCard.length() == 15) {
				birthYearSpan = 2;
			}
			// 出生日期
			String year = (birthYearSpan == 2 ? "19" : "") + idCard.substring(6, 6 + birthYearSpan);
			String month = idCard.substring(6 + birthYearSpan, 6 + birthYearSpan + 2);
			String day = idCard.substring(8 + birthYearSpan, 8 + birthYearSpan + 2);
			// 年龄
			Calendar certificateCal = Calendar.getInstance();
			Calendar currentTimeCal = Calendar.getInstance();
			certificateCal.set(Integer.parseInt(year), Integer.parseInt(month) - 1, Integer.parseInt(day));
			int yearAge = (currentTimeCal.get(currentTimeCal.YEAR)) - (certificateCal.get(certificateCal.YEAR));
			certificateCal.set(currentTimeCal.get(Calendar.YEAR), Integer.parseInt(month) - 1, Integer.parseInt(day));
			int monthFloor = (currentTimeCal.before(certificateCal) ? 1 : 0);
			int age = yearAge - monthFloor;
			Debugger.doAssert(age >= 18, ZAErrorCode.ZA_VALID_FAILED, "根据身份证信息，您的年龄未满18岁，不能注册CKU会员，请与客服联系");
		} catch (Exception e) {
			Debugger.doAssert(false, ZAErrorCode.ZA_VALID_FAILED, "根据身份证信息，您的年龄未满18岁，不能注册CKU会员，请与客服联系");
			e.printStackTrace();
		}

	}

	/**
	 * 检查是否可以进行实名认证
	 *
	 * @param request
	 */
	private void checkValidateState(HttpServletRequest request) {
		// 次数校验
		HttpSession session = request.getSession();
		// 手机号
		String mobile = (String) session.getAttribute("mobile");
		if (StringUtils.isEmpty(mobile)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "您已离开了太长时间，请重试");
		}
		// 手机号验证
		String mobileValidate = (String) session.getAttribute("validate");
		if (!mobileValidate.equals("yes")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "手机号验证失败，请先验证手机号");
		}
		// 手机号次数
		String cacheKey = MOBILE_CACHE_NAME + mobile;
		String date = (String) CacheUtils.get(IDCARD_CACHE_NAME, cacheKey + "-date");
		Date currentDate = new Date();
		String cDate = DateUtils.formatDate(currentDate, "yyyy-MM-dd");
		if (date != null && date.equals(cDate)) {
			Integer mobileCount = (Integer) CacheUtils.get(IDCARD_CACHE_NAME, cacheKey);
			if (mobileCount != null && mobileCount >= 4) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该账号今天实名认证次数已用尽，请明天再试");
			}
		}
		// ip次数
		IPLimitUtil.CheckLimit(IDCARD_CACHE_NAME, request, 8);
	}

	@Autowired
	private RedissonClient redissonClient;

	/**
	 * 通过第三方服务验证身份证信息
	 *
	 * @param idCard 身份证号和真实姓名
	 * @return IDCard
	 */
	public IDCard validateByThird(IDCard idCard, HttpServletRequest request) {
		int response = 0, cost = 0;
		RLock rLock = null;
		try {
			String key = idCard.getIdCard() + "_" + MD5Generator.generate(idCard.getRealName());
			log.info("锁开始:key:" + key + ":" + DateUtils.formatDateTime(new Date()));
			rLock = redissonClient.getFairLock(key);
			// 活锁避免死锁
			rLock.lock(5, TimeUnit.MINUTES);
			Map<String, String> headers = new HashMap<>();
			headers.put("accept", "application/json");
			// 根据API的要求，定义相对应的Content-Type
			headers.put("Content-Type", "application/x-www-form-urlencoded; charset=UTF-8");
			Map<String, String> querys = new HashMap<>();
			Map<String, String> bodys = new HashMap<>();
			bodys.put("idenNo", idCard.getIdCard());
			bodys.put("trueName", idCard.getRealName());
			bodys.put("key", OCR_KEY);
			bodys.put("secret", OCR_SECRET);
			bodys.put("typeId", "3001");
			bodys.put("format", "json");
			// 网络请求
			HttpResponse responseDate = HttpUtils.doPost(OCR_HOST, VALID_PATH, HTTP_METHOD, headers, querys, bodys);
			String jsonStr = EntityUtils.toString(responseDate.getEntity());
			// 有返回结果
			if (StringUtils.isBlank(jsonStr)) {
				response = 1;
				cost = 1;
			}
			// 解析JSON
			JSONObject jsonObject = JSONObject.fromObject(jsonStr);
			JSONArray arr = jsonObject.getJSONArray("policeCheckInfo");
			boolean a = false;
			String validDesc = "";
			if (arr != null) {
				JSONArray arr2 = (arr.getJSONObject(0)).getJSONArray("veritem");
				for (int i = 0; i < arr2.size(); i++) {
					JSONObject jo = arr2.getJSONObject(i);
					if ("verify_result_status".equals(jo.getString("desc"))) {
						a = "3".equals(jo.getString("content"));
					}

					if ("verify_result_desc".equals(jo.getString("desc"))) {
						validDesc = jo.getString("content");
					}
				}
			}

			// 判断是否为正确数据
			if (a) {
				idCard.setQualified(1);
			} else {
				idCard.setQualified(0);
			}
			idCard.setRemarks(validDesc);
			// 添加到本地数据库
			idCard.preInsert();
			idCardDao.insert(idCard);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			// 手机号次数缓存+1
			mobileCacheCountAdd(request);
			// 添加日志进数据库
			insertIDCardLogs(idCard, response, cost);
			if (Objects.nonNull(rLock)) {
				rLock.unlock();
				log.info("锁结束:" + DateUtils.formatDateTime(new Date()));
			}
		}
		return idCard;
	}

	/**
	 * 添加Log进数据库
	 */
	private void insertIDCardLogs(IDCard idcard, int response, int cost) {
		idcard.setQualified(idcard.getQualified() == null ? 0 : idcard.getQualified());
		IDCardLogs po = new IDCardLogs(idcard.getIdCard(), idcard.getRealName(), idcard.getQualified(), response, cost);
		po.setRemarks("netocr");// 新服务商标记
		po.preInsert();
		idCardLogsDao.insert(po);

	}

	/**
	 * 手机号次数缓存加1
	 */
	private void mobileCacheCountAdd(HttpServletRequest request) {
		if (request != null) {
			String mobile = (String) request.getSession().getAttribute("mobile");
			if (!StringUtils.isEmpty(mobile)) {
				// 手机号次数+1
				String cacheKey = MOBILE_CACHE_NAME + mobile;
				String date = (String) CacheUtils.get(IDCARD_CACHE_NAME, cacheKey + "-date");
				Date currentDate = new Date();
				String cDate = DateUtils.formatDate(currentDate, "yyyy-MM-dd");
				if (date != null && date.equals(cDate)) {
					Integer mobileCount = (Integer) CacheUtils.get(IDCARD_CACHE_NAME, cacheKey);
					mobileCount = mobileCount == null ? 0 : mobileCount;
					CacheUtils.put(IDCARD_CACHE_NAME, cacheKey, mobileCount + 1);
				} else {
					CacheUtils.put(IDCARD_CACHE_NAME, cacheKey, 1);
					CacheUtils.put(IDCARD_CACHE_NAME, cacheKey + "-date", cDate);
				}
			}
		}
	}

	public OCRResult queryOCRRemainder() {
		OCRResult ocrResult = new OCRResult();
		try {
			// 活锁避免死锁
			Map<String, String> headers = new HashMap<>();
			headers.put("accept", "application/json");
			// 根据API的要求，定义相对应的Content-Type
			headers.put("Content-Type", "application/x-www-form-urlencoded; charset=UTF-8");
			Map<String, String> querys = new HashMap<>();
			Map<String, String> bodys = new HashMap<>();
			bodys.put("key", OCR_KEY);
			bodys.put("secret", OCR_SECRET);
			bodys.put("format", "json");
			// 网络请求
			HttpResponse responseDate = HttpUtils.doPost(OCR_HOST, REMAINDER_PATH, HTTP_METHOD, headers, querys, bodys);
			String jsonStr = EntityUtils.toString(responseDate.getEntity());
			// 有返回结果
			if (StringUtils.isNoneBlank(jsonStr)) {
				// 解析JSON
				ocrResult = new Gson().fromJson(jsonStr, OCRResult.class);
			} else {
				ocrResult.setMessage(OCRMessage.builder().status(-100).value("OCR查询失败，查询剩余身份证识别条数未返回结果").build());
			}

		} catch (Exception e) {
			e.printStackTrace();
		}
		return ocrResult;
	}

	/**
	 * 验证输入身份证号
	 *
	 * @param str 待验证的字符串
	 * @return 如果是符合格式的字符串, 返回true, 否则为false
	 */
	public boolean isIDcard(String str) {
		String regex = "(^[1-9]\\d{5}(18|19|([23]\\d))\\d{2}((0[1-9])|(10|11|12))(([0-2][1-9])|10|20|30|31)\\d{3}[0-9Xx]$)|(^[1-9]\\d{5}\\d{2}((0[1-9])|(10|11|12))(([0-2][1-9])|10|20|30|31)\\d{2}$)";
		Pattern pattern = Pattern.compile(regex);
		Matcher matcher = pattern.matcher(str);
		return matcher.matches();
	}

	// 验证程序
	public static void main(String[] args) {
		IDCard idCard = new IDCard("身份证号", "姓名");

		try {
			Map<String, String> headers = new HashMap<>();
			headers.put("accept", "application/json");
			// 根据API的要求，定义相对应的Content-Type
			headers.put("Content-Type", "application/x-www-form-urlencoded; charset=UTF-8");
			Map<String, String> querys = new HashMap<>();
			Map<String, String> bodys = new HashMap<>();
			bodys.put("idenNo", idCard.getIdCard());
			bodys.put("trueName", idCard.getRealName());
			bodys.put("key", OCR_KEY);
			bodys.put("secret", OCR_SECRET);
			bodys.put("typeId", "3001");
			bodys.put("format", "json");
			// 网络请求
			HttpResponse responseDate = HttpUtils.doPost(OCR_HOST, VALID_PATH, HTTP_METHOD, headers, querys, bodys);
			String jsonStr = EntityUtils.toString(responseDate.getEntity());
			System.out.println(jsonStr);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

}
