package com.cku.oa.statistics.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.statistics.dao.MemberVipDao;
import com.cku.oa.statistics.entity.MemberVip;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.common.utils.excel.ImportExcel;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.text.NumberFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;

/**
 * 大客户数据统计Service
 *
 * @author yuanshuai
 * @version 2020-06-29
 */
@Service
@Transactional(readOnly = true)
public class MemberVipService extends CrudService<MemberVipDao, MemberVip> {

	public MemberVip get(String id) {
		return super.get(id);
	}

	public List<MemberVip> findList(MemberVip memberVip) {
		return super.findList(memberVip);
	}

	public Page<MemberVip> findPage(Page<MemberVip> page, MemberVip memberVip) {
		return super.findPage(page, memberVip);
	}

	@Transactional(readOnly = false)
	public void save(MemberVip memberVip) {
		super.save(memberVip);
	}

	@Transactional(readOnly = false)
	public void delete(MemberVip memberVip) {
		super.delete(memberVip);
	}

	/**
	 * 大客户血统证书进度查询
	 *
	 * @author: yuanshuai
	 * @date: 2020/6/29 15:27
	 */
	public List<MemberVip> findScheduleStat(MemberVip search) {
		//获取并设置查询参数
		if (search.getEndDate() == null) {
			search.setEndDate(Date.from(LocalDateTime.now().with(LocalTime.MAX).atZone(ZoneId.systemDefault()).toInstant()));
		}
		//默认只查询返利客户
		search.setContractType("2");
		//获取大客户列表
		List<MemberVip> list = findList(search);
		//设置百分比format
		NumberFormat format = NumberFormat.getPercentInstance();
		format.setMaximumFractionDigits(2);
		//查询数量
		for (MemberVip memberVip : list) {
			memberVip.setEndDate(search.getEndDate());
			Integer pedigreeCertifiedCompleteNum;
			if ("1".equals(memberVip.getContractType())) {
				//预付费
				pedigreeCertifiedCompleteNum = dao.findPrepaidPedigreeCertifiedCompleteCount(memberVip);
			} else if ("2".equals(memberVip.getContractType())) {
				//返利
				pedigreeCertifiedCompleteNum = dao.findCashBackPedigreeCertifiedCompleteCount(memberVip);
			} else {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + memberVip.getMemberCode() + "客户类型有误");
			}
			//会员75160有两种合同需要特殊处理
			String doubleTypeMember = "75160";
			if (doubleTypeMember.equals(memberVip.getMemberCode())) {
				pedigreeCertifiedCompleteNum = pedigreeCertifiedCompleteNum - dao.findPrepaidPedigreeCertifiedCompleteCount(memberVip);
			}
			memberVip.setPedigreeCertifiedCompleteNum(pedigreeCertifiedCompleteNum == null ? new Integer(0) : pedigreeCertifiedCompleteNum);
			//计算百分比
			memberVip.setPedigreeCertifiedCompleteRatio(format.format(new Double(memberVip.getPedigreeCertifiedCompleteNum()) / new Double(memberVip.getContractNum())));
			//查询新会员数量
			Integer memberCount = dao.findNewMemberCount(memberVip);
			memberVip.setMemberCount(memberCount == null ? new Integer(0) : memberCount);
		}
		return list;
	}

	/**
	 * @Description： 导出大客户血统证书进度
	 * @author: yuanshuai
	 * @date: 2020/6/30 11:18
	 */
	public List<MemberVip> exportScheduleStat(MemberVip search) {
		List<MemberVip> vipList = findScheduleStat(search);
		for (MemberVip memberVip : vipList) {
			memberVip.setContractType(DictUtils.getDictLabel(memberVip.getContractType(), "contract_type", null));
		}
		return vipList;
	}

	/**
	 * @Description： 构造导入模板
	 * @author: yuanshuai
	 * @date: 2020/6/30 11:27
	 */
	public List<MemberVip> importFileTemplate() {
		List<MemberVip> memberVipList = Lists.newArrayList();
		MemberVip memberVip = new MemberVip();
		memberVip.setMemberCode("2268");
		memberVip.setMemberName("刘载春（可空）");
		memberVip.setContractType("预付费/返利");
		memberVip.setManInCharge("许博然");
		memberVip.setStartDate(Date.from(LocalDate.now().atStartOfDay(ZoneId.systemDefault()).toInstant()));
		memberVip.setContractNum(200);
		memberVip.setPedigreeCertifiedCompleteNum(0);
		memberVip.setPedigreeCertifiedCompleteRatio("空着即可");
		memberVipList.add(memberVip);
		return memberVipList;
	}

	/**
	 * @Description： 大客户数据导入
	 * @author: yuanshuai
	 * @date: 2020/6/30 11:36
	 */
	@Transactional(readOnly = false)
	public void importMemberVip(MultipartFile file) throws IllegalAccessException, InstantiationException, IOException, InvalidFormatException {
		ImportExcel ei = new ImportExcel(file, 0, 0);
		List<MemberVip> list = ei.getDataList(MemberVip.class);
		//处理数据字典
		for (MemberVip memberVip : list) {
			if (memberVip == null
					|| StringUtils.isBlank(memberVip.getMemberCode())
					|| "会员号".equals(memberVip.getMemberCode())) {
				continue;
			}
			memberVip.setContractType(DictUtils.getDictValue(memberVip.getContractType(), "contract_type", null));
			if (StringUtils.isBlank(memberVip.getContractType())) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + memberVip.getMemberCode() + "客户类型有误");
			}
			try {
				save(memberVip);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
}