package com.cku.restful.v2.order.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.logUtil.JSONObjectUtils;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.finance.entity.PaymentOrder;
import com.cku.oa.finance.entity.PaymentOrderDetail;
import com.cku.oa.finance.entity.SaFinanceSubjectMapping;
import com.cku.oa.finance.service.PaymentChargingItemService;
import com.cku.oa.finance.service.PaymentOrderService;
import com.cku.oa.finance.service.SaFinanceSubjectMappingService;
import com.cku.oa.goods.entity.Sku;
import com.cku.oa.goods.entity.SysBusinessCate;
import com.cku.oa.goods.entity.SysBusinessService;
import com.cku.oa.goods.service.SkuService;
import com.cku.oa.goods.service.SysBusinessCateService;
import com.cku.oa.goods.service.SysBusinessServiceService;
import com.cku.oa.order.entity.BusinessOrder;
import com.cku.oa.order.service.BusinessOrderService;
import com.cku.oa.order.service.IOrderBusinessService;
import com.cku.oa.order.service.ISkuBusinessService;
import com.cku.oa.order.service.OrderServiceFactory;
import com.cku.oa.order.vo.ChargingItemVo;
import com.cku.oa.order.vo.OrderBusinessVo;
import com.cku.oa.order.vo.OrderSkuVo;
import com.cku.oa.order.vo.request.OrderBusinessRequest;
import com.cku.oa.sys.code.SysCodeUtil;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.restful.v1.finance.service.RestOrderService;
import com.cku.restful.v1.sys.utils.BeanUtil;
import com.cku.restful.v2.order.enums.OrderSourceEnum;
import com.cku.restful.v2.order.vo.request.OrderApplyRequest;
import com.cku.restful.v2.order.vo.response.OrderApplyInfoResponse;
import com.cku.restful.v2.order.vo.response.OrderCommitResponse;
import com.cku.restful.v2.order.vo.response.OrderSkuResponse;
import com.cku.util.DateUtils;
import com.thinkgem.jeesite.common.utils.IdGen;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.awt.event.WindowFocusListener;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 订单接口service-v2
 *
 * @author yuanshuai
 * @date 2023/6/16 14:23
 */
@Service
@Transactional(readOnly = true)
public class RestOrderServiceV2 {

    @Autowired
    private OrderServiceFactory orderServiceFactory;
    @Autowired
    private MemberService memberService;
    @Autowired
    private SysBusinessCateService sysBusinessCateService;
    @Autowired
    private SysBusinessServiceService sysBusinessServiceService;
    @Autowired
    private SkuService skuService;
    @Autowired
    private BusinessOrderService businessOrderService;
    @Autowired
    private PaymentChargingItemService paymentChargingItemService;
    @Autowired
    private PaymentOrderService paymentOrderService;
    @Autowired
    private RestOrderService restOrderService;
    @Autowired
    private SaFinanceSubjectMappingService saFinanceSubjectMappingService;

    /**
     * 日志对象
     */
    protected Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * 下单前参数校验
     *
     * @author yuanshuai
     * @date 2023/6/19 10:18
     */
    public void businessValidate(OrderApplyRequest request) {
        //获取会员
        Member member = getMemberByOrderSource(request);
        //获取业务服务类
        IOrderBusinessService<OrderBusinessRequest> businessService = orderServiceFactory.getBusinessService(request.getBusinessID());
        //构建请求参数
        OrderBusinessRequest businessRequest = buildBusinessRequest(businessService, request);
        //下单前校验
        businessService.businessValidate(request, businessRequest, member);
    }

    /**
     * 下单前参数校验并根据业务ID和业务参数查询商品相关信息
     *
     * @author yuanshuai
     * @date 2023/6/16 14:37
     */
    public OrderApplyInfoResponse preOrder(OrderApplyRequest request) {
        logger.info("RestOrderServiceV2.preOrder 入参param : {}", JSONObjectUtils.toJsonString(request));
        //获取会员
        Member member = getMemberByOrderSource(request);
        //获取业务服务类
        IOrderBusinessService<OrderBusinessRequest> businessService = orderServiceFactory.getBusinessService(request.getBusinessID());
        //构建请求参数
        OrderBusinessRequest businessRequest = buildBusinessRequest(businessService, request);
        //下单前校验并获取商品列表
        List<OrderSkuVo> skuVoList = businessService.preOrder(request, businessRequest, member);
        logger.info("RestOrderServiceV2.preOrder skuVoList : {}", JSONObjectUtils.toJsonString(skuVoList));
        if (CollectionUtils.isEmpty(skuVoList)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "参数校验有误，请重试");
        }
        //查找SKU列表
        List<Sku> skuList = skuService.findListByIdList(skuVoList.stream()
                .map(OrderSkuVo::getSkuId)
                .collect(Collectors.toList()));
        //更新SKU信息
        skuVoList.forEach(skuVo -> {
            //根据SkuId获取SKU
            Sku sku = findSkuBySkuId(skuVo.getSkuId(), skuList);
            //获取业务服务类
            ISkuBusinessService<OrderBusinessVo> skuBusinessService = orderServiceFactory.getSkuServiceByName(sku.getServiceName());
            //根据SKUID获取收费项
            List<ChargingItemVo> itemVoList = findItemListBySkuId(skuVo.getSkuId());
            //处理收费项价格
            skuBusinessService.processChargingItemPrice(request, skuVo.getBusinessList(), itemVoList);
            //更新SKU
            skuVo.setSkuName(sku.getName());
            //更新sku价格
            skuVo.setUnitPrice(itemVoList.stream()
                    .map(ChargingItemVo::getPrice)
                    .reduce(BigDecimal.ZERO, BigDecimal::add));
            skuVo.setTotalPrice(skuVo.getUnitPrice().multiply(BigDecimal.valueOf(skuVo.getNum())));
        });
        return OrderApplyInfoResponse.builder()
                .businessId(request.getBusinessID())
                .skuList(skuVoList.stream()
                        .map(vo -> {
                            OrderSkuResponse skuResp = OrderSkuResponse.builder().build();
                            BeanUtil.copyProperties(skuResp, vo, "yyyy-MM-dd");
                            return skuResp;
                        })
                        .collect(Collectors.toList()))
                .build()
                //计算总价
                .calcTotalPrice();
    }

    /**
     * 根据不同渠道获取会员信息
     *
     * @author yuanshuai
     * @date 2023/6/27 17:24
     */
    private Member getMemberByOrderSource(OrderApplyRequest request) {
        //获取登录会员
        Member member = null;
        if (OrderSourceEnum.IOS.code().equals(request.getOrderSource().code())
                || OrderSourceEnum.ANDROID.code().equals(request.getOrderSource().code())) {
            member = UserUtils.getLoginMember();
        } else if (OrderSourceEnum.OA.code().equals(request.getOrderSource().code())) {
            if (StringUtils.isBlank(request.getMemberCode())) {
                throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "会员号为空");
            }
            User user = UserUtils.getLoginUser();
            if (!"1".equals(user.getUserType())) {
                throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "OA渠道仅限后台工作人员使用");
            }
            member = memberService.getByMemberCode(request.getMemberCode());
        }
        if (Objects.isNull(member)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "会员号信息有误");
        }
        return member;
    }

    /**
     * 构建业务请求参数
     *
     * @author yuanshuai
     * @date 2023/6/18 15:10
     */
    private OrderBusinessRequest buildBusinessRequest(IOrderBusinessService<OrderBusinessRequest> businessService, OrderApplyRequest request) {
        try {
            System.out.println("--------------------------------------------------" + businessService.getClass().getName());
            //由于cglib代理生成的类无法获取interface，故通过名称重新获取
            Class<?> clazz = null;
            if (businessService.getClass().getName().contains("$")) {
                clazz = Class.forName(businessService.getClass().getName().substring(0, businessService.getClass().getName().indexOf("$")));
            } else {
                clazz = Class.forName(businessService.getClass().getName());
            }
            Type[] genericType2 = ((ParameterizedType) clazz.getGenericInterfaces()[0]).getActualTypeArguments();
            Class<?> t2 = (Class<?>) genericType2[0];
            return BeanUtil.toBean(request.getBusinessData(), t2);
        } catch (ClassNotFoundException e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "获取业务参数失败");
        }
    }

    /**
     * 提交订单
     *
     * @author yuanshuai
     * @date 2023/6/18 18:02
     */
    @Transactional(readOnly = false)
    public OrderCommitResponse commitOrder(OrderApplyRequest request) {
        //获取登录会员
        Member member = getMemberByOrderSource(request);
        validateMember(member);
        //获取业务信息
        if (StringUtils.isBlank(request.getBusinessID())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "业务ID为空");
        }
        SysBusinessService businessService = sysBusinessServiceService.get(request.getBusinessID());
        if (Objects.isNull(businessService)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "业务ID有误");
        }
        //获取业务分类
        SysBusinessCate businessCate = sysBusinessCateService.get(businessService.getCateId());
        //没配置服务无法下单
        if (StringUtils.isBlank(businessService.getServiceName())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该业务尚未完成配置，无法下单");
        }
        //获取业务服务类
        IOrderBusinessService<OrderBusinessRequest> orderBusinessService = orderServiceFactory.getBusinessServiceByName(businessService.getServiceName());
        //构建请求参数
        OrderBusinessRequest businessRequest = buildBusinessRequest(orderBusinessService, request);
        //下单前校验并获取商品列表
        List<OrderSkuVo> skuVoList = orderBusinessService.preOrder(request, businessRequest, member);
        if (CollectionUtils.isEmpty(skuVoList)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "参数校验有误，请重试");
        }
        //当前时间
        Date now = DateUtils.getNow();
        //订单业务编码
        String businessOrderCode = IdGen.uuid();
        //构建订单信息
        PaymentOrder paymentOrder = buildOrder(member, now, request.getOrderSource().code());
        //查找SKU列表
        List<Sku> skuList = skuService.findListByIdList(skuVoList.stream()
                .map(OrderSkuVo::getSkuId)
                .collect(Collectors.toList()));
        //循环SKU处理付款前业务逻辑
        for (OrderSkuVo skuVo : skuVoList) {
            //获取SKU
            Sku sku = findSkuBySkuId(skuVo.getSkuId(), skuList);
            logger.info("-------商品信息 sku : {}", JSONObjectUtils.toJsonString(sku));
            //获取业务服务类
            ISkuBusinessService<OrderBusinessVo> skuBusinessService = orderServiceFactory.getSkuServiceByName(sku.getServiceName());
            //根据SKUID获取收费项
            List<ChargingItemVo> itemVoList = findItemListBySkuId(skuVo.getSkuId());
            logger.info("-------------itemVoList : {}", JSONObjectUtils.toJsonString(itemVoList));
            //处理收费项价格
            skuBusinessService.processChargingItemPrice(request, skuVo.getBusinessList(), itemVoList);
            //更新SKU
            skuVo.setSkuName(sku.getName());
            //更新sku价格
            skuVo.setUnitPrice(itemVoList.stream()
                    .map(ChargingItemVo::getPrice)
                    .reduce(BigDecimal.ZERO, BigDecimal::add));
            skuVo.setTotalPrice(skuVo.getUnitPrice().multiply(BigDecimal.valueOf(skuVo.getNum())));
            //处理业务逻辑
            if (!CollectionUtils.isEmpty(skuVo.getBusinessList())) {
                skuVo.getBusinessList().forEach(business -> {
                    skuVo.setOrderCode(paymentOrder.getOrderCode());
                    skuVo.setMember(member);
                    skuBusinessService.processBusinessPrePay(request, business, skuVo);
                });
            }
            //添加业务名:目前看业务名没有实际的用处，后续需要优化
            //业务名无对应，且之前用于用友审计，目前应该已废弃
            //具体问题请联系张君武
			itemVoList.forEach(i -> i.setBusinessId(businessService.getFinanceId()));
            //根据不同业务单独处理表ID方法
            skuBusinessService.findChargingItemBusinessTableId(request, skuVo, itemVoList);
            //根据收费项生成订单明细
            itemVoList.forEach(item -> {
                //统一处理表ID
                //添加业务表名表ID:如果每个收费项对应的业务表唯一后续应更改为通过关联收费项查询
                //理论上获取业务名、表名表id的功能可以拉到外层，无需在实现类里处理，只有获取的收费项价格需要变更时才需要使用实现类处理
                //业务表id可能不单纯为表ID而是配种证明编号、证书号或#分隔等复杂语句
                //故先把获取此类字段放在实现类中以便不同业务自行处理
                //更新业务表名表ID
                if (StringUtils.isNotBlank(item.getBusinessTable())
                        && StringUtils.isBlank(item.getBusinessIds())) {
                    //查找业务表ID
                    String businessIds = paymentOrderService.findBusinessIdsByTableNameAndOrderCode(item.getBusinessTable(), paymentOrder.getOrderCode());
                    if (StringUtils.isNotBlank(businessIds)) {
                        item.setBusinessIds(businessIds);
                    }
                }
                //设置收费项数量
                item.setNum(skuVo.getNum());
                //生成订单明细
                saveOrderDetail(paymentOrder.getOrderCode(), businessOrderCode, member, item, sku.getRefundFlag(), now);
            });
            //根据SKU生成业务订单
            BusinessOrder businessOrder = buildBusinessOrder(businessCate, businessService, paymentOrder, skuVo, itemVoList);
            businessOrderService.save(businessOrder);
        }
        //保存订单
        saveOrder(paymentOrder, skuVoList.stream()
                .map(sku -> {
                    sku.setTotalPrice(sku.getUnitPrice().multiply(BigDecimal.valueOf(sku.getNum())));
                    return sku.getTotalPrice();
                })
                .reduce(BigDecimal.ZERO, BigDecimal::add));
        //构建返回值
        OrderCommitResponse resp = new OrderCommitResponse();
        resp.setBusinessId(request.getBusinessID());
        resp.setOrderId(paymentOrder.getId());
        resp.setOrderCode(paymentOrder.getOrderCode());
        resp.setSkuList(skuVoList.stream()
                .map(vo -> {
                    OrderSkuResponse skuResp = OrderSkuResponse.builder().build();
                    BeanUtil.copyProperties(skuResp, vo, "yyyy-MM-dd");
                    return skuResp;
                })
                .collect(Collectors.toList()));
        resp.calcTotalPrice();
        return resp;
    }

    /**
     * 根据SKUID获取SKU
     *
     * @author yuanshuai
     * @date 2023/6/28 14:43
     */
    private static Sku findSkuBySkuId(String skuId, List<Sku> skuList) {
        Sku sku = skuList.stream()
                .filter(s -> s.getId().equals(skuId))
                .findFirst()
                .orElse(null);
        if (Objects.isNull(sku)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "获取不到对应商品");
        }
        if (0 == sku.getSaleFlag()) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "商品" + sku.getName() + "已下架，无法购买");
        }
        return sku;
    }

    /**
     * 根据SKUID查找收费项
     *
     * @author yuanshuai
     * @date 2023/6/28 14:30
     */
    private List<ChargingItemVo> findItemListBySkuId(String skuId) {
        //根据sku获取收费项
        List<PaymentChargingItem> itemList = paymentChargingItemService.findListBySkuId(skuId);
        if (CollectionUtils.isEmpty(itemList)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "获取不到对应收费项");
        }
        //转换成ItemVo
        return itemList.stream()
                .map(i -> ChargingItemVo.builder()
                        .id(i.getId())
                        .name(i.getName())
                        .price(new BigDecimal(i.getPrice()))
                        .businessTable(i.getBusinessTable())
                        .build())
                .collect(Collectors.toList());
    }

    /**
     * 构建业务订单
     *
     * @author yuanshuai
     * @date 2023/6/21 11:37
     */
    private static BusinessOrder buildBusinessOrder(SysBusinessCate businessCate, SysBusinessService businessService, PaymentOrder paymentOrder, OrderSkuVo skuVo, List<ChargingItemVo> itemVoList) {
        BusinessOrder businessOrder = new BusinessOrder();
        businessOrder.setCateId(businessCate.getId());
        businessOrder.setCateName(businessCate.getCateName());
        businessOrder.setBusinessId(businessService.getId());
        businessOrder.setBusinessName(businessService.getBusinessName());
        businessOrder.setSkuId(skuVo.getSkuId());
        businessOrder.setSkuName(skuVo.getSkuName());
        businessOrder.setUnitPrice(skuVo.getUnitPrice().doubleValue());
        businessOrder.setNum(skuVo.getNum());
        businessOrder.setTotalPrice(skuVo.getTotalPrice().doubleValue());
        businessOrder.setMemberCode(paymentOrder.getMemberCode());
        businessOrder.setOrderCode(paymentOrder.getOrderCode());
        businessOrder.setChargingItemIds(itemVoList.stream()
                .map(ChargingItemVo::getId)
                .collect(Collectors.joining(",")));
        businessOrder.setChargingItemNames(itemVoList.stream()
                .map(ChargingItemVo::getName)
                .collect(Collectors.joining(",")));
        return businessOrder;
    }

    /**
     * 生成订单明细记录
     *
     * @author yuanshuai
     * @date 2023/6/19 16:25
     */
    private void saveOrderDetail(String orderCode, String businessOrderCode, Member member, ChargingItemVo item, Integer refundFlag, Date date) {
        PaymentOrderDetail detail = new PaymentOrderDetail();
        //订单号
        detail.setOrderCode(orderCode);
        //默认值
        detail.setAddTime(date);
        detail.setProccessState("0");
        //收费项信息
        detail.setChargingItemId(item.getId());
        detail.setChargingItemName(item.getName());
        detail.setNum(item.getNum().toString());
        detail.setPrice(item.getPrice().toString());
        detail.setTotalPrice(item.getPrice().multiply(BigDecimal.valueOf(item.getNum())).toString());
        //会员信息
        detail.setMemberCode(member.getMemberCode());
        detail.setMemberName(member.getName());
        //业务信息
        //以前对接用友用时用的字段，用于定义业务
        //新订单逻辑有了商品概念后应不需要此字段，商品中已经有了业务名称的字段
        //徐雪说当时用友的财务系统是张君武干的，具体问题请联系张君武
        if (StringUtils.isBlank(item.getBusinessId())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "业务ID为空");
        }
        SaFinanceSubjectMapping subjectMapping = saFinanceSubjectMappingService.get(item.getBusinessId());
        logger.info("--------------subjectMapping : {}", JSONObjectUtils.toJsonString(subjectMapping));
        if (subjectMapping == null) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, item.getBusinessId() + "业务名字不存在");
        }
        detail.setBusinessId(item.getBusinessId());
        detail.setBusinessName(subjectMapping.getSubjectName());
        //是否可以退款
        //旧订单逻辑是否退款是通过subjectMapping.getBusinessRefund()赋值
        //新订单逻辑中有商品的概念，一个商品下收费项统一赋值是否可以退单，不需要旧的业务信息
        detail.setBusinessRefund(refundFlag.toString());
        // 添加赛事模式字段
        //目前默认为0，后续有赛事订单后需要修改
        //可以根据需求看此字段是否可以优化
        detail.setShowBusinessModelFlag("0");
        //业务表名表ID
        //后续应优化表名和表ID获取方式及存储
        //通过表名自动获取表ID,逗号分隔或者通过关联表存储
        //应避免在表id字段中存储特殊标记
        //仅通过表id查出业务表，再通过业务表中的标记位执行处理逻辑
        if (StringUtils.isNotBlank(item.getBusinessTable()) && StringUtils.isNotBlank(item.getBusinessIds())) {
            detail.setBusinessTable(item.getBusinessTable());
            detail.setBusinessIds(item.getBusinessIds());
        }
        //业务编号
        //不知道干什么用的，猜测是关联订单统一下单之类的作用
        //只看见会员费双12活动用到了这个字段
        //会员费双12活动相关是张君武干的，具体问题请联系张君武
        detail.setBusinessCode(businessOrderCode);
        detail.setBusinessOrderCode(businessOrderCode);
        logger.info("------------------ detail : {}", JSONObjectUtils.toJsonString(detail));
        paymentOrderService.addOrderDetail(detail);
    }

    /**
     * 校验会员是否可以下单
     *
     * @author yuanshuai
     * @date 2023/6/19 14:42
     */
    private void validateMember(Member member) {
        // 校验是否会员已冻结
        if ("1".equals(member.getFreezeFlag())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "会员已被冻结,无法生成订单");
        }
    }

    /**
     * 构建订单信息
     *
     * @author yuanshuai
     * @date 2023/6/19 10:45
     */
    private PaymentOrder buildOrder(Member member, Date addTime, Integer orderSource) {
        PaymentOrder paymentOrder = new PaymentOrder();
        //初始值
        paymentOrder.setLocked("0");
        paymentOrder.setPaymentState("1");
        //添加时间
        paymentOrder.setAddTime(addTime);
        //订单来源
        paymentOrder.setSource(orderSource.toString());
        //生成订单号
        paymentOrder.setOrderCode(SysCodeUtil.getPayRunningNumberWithMemberCode(member.getMemberCode()));
        //会员信息
        paymentOrder.setMemberCode(member.getMemberCode());
        paymentOrder.setMemberName(member.getName());
        //地址信息
        paymentOrder.setReceiveProvince(member.getProvince());
        paymentOrder.setReceiveCity(member.getCity());
        paymentOrder.setReceiveAddress(member.getStreet());
        paymentOrder.setReceivePostCode(member.getPostcode());
        paymentOrder.setReceiveDel(member.getMobile());
        paymentOrder.setReceiver(member.getName());
        return paymentOrder;
    }


    /**
     * 保存订单
     *
     * @author yuanshuai
     * @date 2023/6/19 14:42
     */
    private void saveOrder(PaymentOrder paymentOrder, BigDecimal totalPrice) {
        paymentOrder.setTotalPrice(totalPrice.toString());
        paymentOrderService.addOrder(paymentOrder);
    }

    /**
     * 提交订单并余额支付
     *
     * @author yuanshuai
     * @since 2024/1/26 15:38
     */
    @Transactional
    public OrderCommitResponse commitOrderAndBalancePay(OrderApplyRequest request) {
        //下订单
        OrderCommitResponse resp = commitOrder(request);
        logger.info("commitOrderAndBalancePay ======= resp : {}", JSONObjectUtils.toJsonString(resp));
        //余额支付订单
        PaymentOrder order = paymentOrderService.get(resp.getOrderId());
        logger.info("commitOrderAndBalancePay ======= order : {}", JSONObjectUtils.toJsonString(order));
        if ("1".equals(order.getDelFlag())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该订单已删除，请勿支付");
        }
        if (order.getPaymentState().equals("1")) {
            Member member = memberService.getByMemberCode(request.getMemberCode());
            restOrderService.balPay(order, member);
        }
        return resp;
    }
}
