/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.finance.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import com.cku.logUtil.JSONObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import com.cku.oa.dog.dao.DogVideoAppraisalDao;
import com.cku.oa.dog.entity.DogVideoAppraisal;
import com.cku.oa.finance.dao.PaymentChargingItemDao;
import com.cku.oa.finance.dao.PaymentOrderDetailDao;
import com.cku.oa.finance.dao.PaymentRecordDao;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.finance.entity.PaymentOrder;
import com.cku.oa.finance.entity.PaymentOrderDetail;
import com.cku.oa.finance.entity.PaymentRecord;
import com.cku.oa.kennel.dao.KennelDao;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.sys.dao.user.MemberDao;
import com.cku.oa.sys.entity.user.Member;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;

/**
 * 会员支付流水Service
 * 
 * @author lyy
 * @version 2016-07-18
 */
@Service
@Transactional(readOnly = true)
public class PaymentRecordService extends CrudService<PaymentRecordDao, PaymentRecord> {
	private static final String PAY_WX = "pay@WX";// 微信支付
	private static final String PAY_XCWX = "pay@XCWX";// 现场微信支付
	private static final String PAY_ALI = "pay@ALI";// 支付宝支付
	private static final String PAY_ACCOUNT = "pay@ACCOUNT";// 余额支付
	private static final String PAY_ALLIN = "pay@AALLIN";// 通联支付

	@Autowired
	private PaymentChargingItemDao paymentChargingItemDao;

	@Autowired
	private MemberDao memberDao;

	@Autowired
	private PaymentOrderDetailDao paymentOrderDetailDao;

	@Autowired
	private KennelDao kennelDao;

	@Autowired
	private DogVideoAppraisalDao dogVideoAppraisalDao;

	public PaymentRecord get(String id) {
		return super.get(id);
	}

	public List<PaymentRecord> findList(PaymentRecord paymentRecord) {
		return super.findList(paymentRecord);
	}

	public Page<PaymentRecord> findPage(Page<PaymentRecord> page, PaymentRecord paymentRecord) {
		return super.findPage(page, paymentRecord);
	}

	@Transactional(readOnly = false)
	/**
	 * 后台保存付款记录，记录为oa 该方法不建议使用 fanhuibin
	 */
	public void save(PaymentRecord paymentRecord) {
		paymentRecord.setAlipayS("oa");
		setRecordRegister(paymentRecord);
		super.save(paymentRecord);
	}

	/**
	 * 会员中心使用的方法 通过订单详情，生成流水记录
	 */
	@Transactional(readOnly = false, rollbackFor = Exception.class)
	public PaymentRecord addRecordByOrder(PaymentOrder paymentOrder) {
		boolean updateFlag = true;
		PaymentRecord paymentRecord = dao.findById(paymentOrder.getId());
		if(Objects.isNull(paymentRecord)) {
			updateFlag = false;
			paymentRecord = new PaymentRecord();
		}
		paymentRecord.setMemberCode(paymentOrder.getMemberCode());
		paymentRecord.setPaymentAmount(paymentOrder.getTotalPrice());
		paymentRecord.setPrice(paymentOrder.getTotalPrice());
		paymentRecord.setPaymentWay(paymentOrder.getPaymentWay());
		paymentRecord.setRunningNumber(paymentOrder.getOrderCode());
		paymentRecord.setPaymentState("2");
		paymentRecord.setPaymentTime(new Date());
		paymentRecord.setAlipayS("vip");
		if (paymentOrder.getIsRecharge().equals("1")) {
			// 如果是充值订单，直接标记为充值记录
			paymentRecord.setChargingItemId("0");
		} else {

			// 如果不是充值，统一标记为业务订单支付
			paymentRecord.setChargingItemId("");
			String shortName = "";
			if ("cku.org".equals(paymentOrder.getPaymentWay())) {
				shortName = PAY_ACCOUNT;
			} else if ("24".equals(paymentOrder.getPaymentWay())) {
				shortName = PAY_WX;
			} else if ("1".equals(paymentOrder.getPaymentWay())) {
				shortName = PAY_ALI;
			} else if ("25".equals(paymentOrder.getPaymentWay())) {
				shortName = PAY_ACCOUNT;
			} else if ("28".equals(paymentOrder.getPaymentWay())) {
				shortName = PAY_XCWX;
			} else if ("30".equals(paymentOrder.getPaymentWay())) {
				shortName = PAY_ALLIN;
			}
			PaymentChargingItem paymentChargingItem = paymentChargingItemDao.getByShortName(shortName);
			paymentRecord.setChargingItemId(paymentChargingItem.getId());
		}
		paymentRecord.setChargingItemNum(1);
		
		setRecordRegister(paymentRecord);
		if(updateFlag) {
			paymentRecord.preUpdate();
			dao.update(paymentRecord);
		}else {
			paymentRecord.preInsert();
			paymentRecord.setId(paymentOrder.getId());
			dao.insert(paymentRecord);
		}
		return paymentRecord;
	}

	public void setRecordRegister(PaymentRecord paymentRecord) {
		Member member = memberDao.getByMemberCode(paymentRecord.getMemberCode());
		if (member != null) {
			paymentRecord.setRegisterProvince(member.getRegisterProvince());
			paymentRecord.setRegisterCity(member.getRegisterCity());
			paymentRecord.setBizMemberCode(member.getMemberCode());
		}

		List<PaymentOrderDetail> details = paymentOrderDetailDao.findByOrderCode(paymentRecord.getRunningNumber());
		if (!CollectionUtils.isEmpty(details)) {
			for (PaymentOrderDetail detail : details) {

				if ("sys_member".equals(detail.getBusinessTable()) || "kennel".equals(detail.getBusinessTable())
						|| "dog_video_appraisal".equals(detail.getBusinessTable())) {
					if (paymentRecord.getMemberCode().toUpperCase().startsWith("HZ")
							|| paymentRecord.getMemberCode().toUpperCase().startsWith("MRHZ")
							|| paymentRecord.getMemberCode().toUpperCase().startsWith("CAB")
							|| paymentRecord.getMemberCode().toUpperCase().startsWith("CJH")
							|| paymentRecord.getMemberCode().toUpperCase().startsWith("DQZ")) {
						getBusinessRegister(detail.getBusinessTable(), detail.getBusinessIds(), paymentRecord);
					}
					break;
				}
			}
		}
	}

	private PaymentRecord getBusinessRegister(String business, String id, PaymentRecord paymentRecord) {

		Member member = null;
		if ("sys_member".equals(business)) {
			member = memberDao.get(id);
		} else if ("kennel".equals(business)) {
			Kennel kennel = kennelDao.get(id);
			member = memberDao.getByMemberCode(kennel.getMemberCode());
		} else if ("dog_video_appraisal".equals(business)) {
			DogVideoAppraisal dogVideoAppraisal = dogVideoAppraisalDao.get(id);
			member = memberDao.getByMemberCode(dogVideoAppraisal.getMemberCode());
		}
		if (member != null) {
			paymentRecord.setRegisterProvince(member.getRegisterProvince());
			paymentRecord.setRegisterCity(member.getRegisterCity());
			paymentRecord.setBizMemberCode(member.getMemberCode());
		}
		return paymentRecord;
	}

	/**
	 * @param paymentRecord
	 */
	@Transactional(readOnly = false)
	public void newRecord(PaymentRecord paymentRecord) {
		paymentRecord.preInsert();
		setRecordRegister(paymentRecord);
		dao.newRecord(paymentRecord);
	}

	/**
	 * 根据订单信息生成流水记录
	 */

	@Transactional(readOnly = false)
	public void delete(PaymentRecord paymentRecord) {
		super.delete(paymentRecord);
	}

	public Page<PaymentRecord> findPageCost(Page<PaymentRecord> page, PaymentRecord paymentRecord) {
		int pageNo = page.getPageNo();
		int pageSize = page.getPageSize();
		paymentRecord.setPage(page);
		page.setList(dao.findListCost(paymentRecord));
		if ((pageNo - 1) * pageSize >= page.getCount()) {
			page.setList(new ArrayList<PaymentRecord>(0));
		}
		return page;
	}

	@Transactional(readOnly = false)
	public String getRealMemberByOrder(String orderCode) {
		return dao.getRealMemberByOrder(orderCode);
	}
	

	@Transactional(readOnly = false)
	public PaymentRecord getByRunningNumber(String runningNumber) {
		return dao.getByRunningNumber(runningNumber);
	}


}