package com.cku.restful.v1.dog.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.entity.*;
import com.cku.oa.dog.service.*;
import com.cku.oa.order.vo.GoldCertifiedBusinessVo;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.LoginServerService;
import com.cku.oa.sys.util.DateUtils;
import com.cku.restful.v1.dog.vo.request.GoldCertifiedApplyDetailRequest;
import com.cku.restful.v1.dog.vo.request.GoldCertifiedPageRequest;
import com.cku.restful.v1.dog.vo.response.GoldCertifiedApplyDetailResp;
import com.cku.restful.v1.dog.vo.response.GoldCertifiedPageResp;
import com.cku.restful.v1.sys.utils.BeanUtil;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.modules.sys.entity.Dict;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;

/**
 * 金质证书Service
 *
 * @author yuanshuai
 * @date 2023年6月8日15:56:55
 */
@Service
@Transactional(readOnly = true)
public class RestGoldCertifiedService {

	@Autowired
	private DogService dogService;
	@Autowired
	private DogBirthCertificateService dogBirthCertificateService;
	@Autowired
	private DogChipService dogChipService;
	@Autowired
	private DogDnaArchiveService dogDnaArchiveService;
	@Autowired
	private DogDnaParentsCompareService dogDnaParentsCompareService;
	@Autowired
	private DogTypeService dogTypeService;
	@Autowired
	private DogColorMarkService dogColorMarkService;
	@Autowired
	private LoginServerService loginServerService;

	/**
	 * 分页
	 *
	 * @author yuanshuai
	 * @date 2023/6/8 16:05
	 */
	public Page<GoldCertifiedPageResp> findPage(GoldCertifiedPageRequest request) {
		// 查询
		Member member = UserUtils.getLoginMember();
		Page<GoldCertifiedPageResp> page = dogDnaParentsCompareService.findGoldCertifiedPage(request,
				member.getMemberCode());
		if (!CollectionUtils.isEmpty(page.getList())) {
			List<DogType> dogTypeList = dogTypeService.findAllList();
			List<DogColorMark> colorMarkList = dogColorMarkService.findAllList();
			List<Dict> dogGenderDictList = DictUtils.getDictList("dog_gender");
			page.getList().forEach(r -> {
				// 犬种
				DogType dogType = dogTypeList.stream().filter(dt -> dt.getDogBreedCode().equals(r.getDogBreed()))
						.findFirst().orElse(null);
				if (Objects.nonNull(dogType)) {
					r.setDogBreed(dogType.getBreedCnName());
				}
				// 毛色
				DogColorMark dogColorMark = colorMarkList.stream()
						.filter(cm -> cm.getDogColorMarkCode().equals(r.getColorFlag())).findFirst().orElse(null);
				if (Objects.nonNull(dogColorMark)) {
					r.setColorFlag(dogColorMark.getDogColorMarkCn());
				}
				// 性别
				Dict dogGender = dogGenderDictList.stream().filter(cm -> cm.getValue().equals(r.getGender()))
						.findFirst().orElse(null);
				if (Objects.nonNull(dogGender)) {
					r.setGender(dogGender.getLabel());
				}
			});
		}
		return page;
	}

	/**
	 * 获取申请详情
	 *
	 * @author yuanshuai
	 * @date 2023/6/13 10:32
	 */
	public GoldCertifiedBusinessVo findAndValidateApplyDetailByDogCode(GoldCertifiedApplyDetailRequest request,
			Member member, Boolean validate) {
		if (StringUtils.isBlank(request.getBirthCode()) && StringUtils.isBlank(request.getPedigreeCertified())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "请输入血统证书号或出生纸编号");
		}
		GoldCertifiedBusinessVo vo = new GoldCertifiedBusinessVo();
		// 根据血统证书号查询犬只信息
		if (StringUtils.isNotBlank(request.getPedigreeCertified())) {
			Dog dog = checkDogByPedigreeCertified(request.getPedigreeCertified(), member);
			vo.setDogId(dog.getId());
			vo.setPedigreeCertified(dog.getPedigreeCertifiedCode());
			vo.setOverseasPedigreeCertified(dog.getPedigreeCertifiedEn());
			vo.setBirthCode(dog.getBirthCertificateLevelCode());
			vo.setIdentificationFlag(dog.getIdentificationFlag());
			vo.setNameEn(dog.getNameEn());
			vo.setNickname(dog.getNickname());
			vo.setDogBreed(dog.getDogBreed());
			vo.setColorFlag(dog.getColorFlag());
			vo.setGender(dog.getGender());
			vo.setBirthdate(dog.getBirthdate());
			vo.setMemberCode(dog.getMemberCode());
			vo.setMemberName(dog.getDogOwner());
			vo.setBreederMemberCode(dog.getBreederMemberCode());
			vo.setBreederMemberName(dog.getBreeder());
			vo.setMalePedigreeCertified(dog.getFdogPedigreeCertified());
			vo.setFemalePedigreeCertified(dog.getMdogPedigreeCertified());
			// DNA存档信息
			DogDnaArchive dogDnaArchive = dogDnaArchiveService
					.getByPedigreeCertifiedCode(request.getPedigreeCertified());
			if (Objects.isNull(dogDnaArchive)) {
				vo.setDna(0);
			} else {
				// 判断缴费状态
				if ("1".equals(dogDnaArchive.getPaymentState())) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,
							request.getPedigreeCertified() + "已有未缴费的DNA位点订单，请先处理");
				}
				vo.setDna(1);
			}
		}
		// 根据出生纸编号查询犬只信息
		if (StringUtils.isNotBlank(request.getBirthCode())) {
			DogBirthCertificate dogBirth = checkDogByBirthCode(request.getBirthCode(), member, validate);
			vo.setPedigreeCertified(dogBirth.getPedigreeCertified());
			vo.setBirthCode(dogBirth.getBirthCerRegCode());
			vo.setIdentificationFlag(dogBirth.getIdentificationFlag());
			vo.setNameEn(dogBirth.getDogEnName());
			vo.setDogBreed(dogBirth.getDogBreed());
			vo.setColorFlag(dogBirth.getColorFlag());
			vo.setGender(dogBirth.getDogGender());
			vo.setBirthdate(dogBirth.getDogBirthday());
			vo.setMemberCode(dogBirth.getMemberCode());
			vo.setMemberName(dogBirth.getDogOwenr());
			vo.setBreederMemberCode(dogBirth.getBreederMemberCode());
			vo.setBreederMemberName(dogBirth.getBreeder());
			vo.setMalePedigreeCertified(dogBirth.getFdogPedigreeCertified());
			vo.setFemalePedigreeCertified(dogBirth.getMdogPedigreeCertified());
			// DNA存档信息
			DogDnaArchive dogDnaArchive = dogDnaArchiveService
					.getByPedigreeCertifiedCode(request.getPedigreeCertified());
			if (Objects.isNull(dogDnaArchive)) {
				vo.setDna(0);
			} else {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该出生纸已有DNA存档信息，请通过升级入口申请金质证书");
			}
		}
		// 犬种校验
		// 藏獒和原生犬无法办理此业务
		if (StringUtils.isNotBlank(vo.getDogBreed())
				&& ("230".equals(vo.getDogBreed()) || vo.getDogBreed().startsWith("NB"))) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "西藏獒犬和原生犬无法办理金质证书业务");
		}
		// 父母信息
		if (StringUtils.isBlank(vo.getMalePedigreeCertified()) || StringUtils.isBlank(vo.getFemalePedigreeCertified())
				|| !vo.getMalePedigreeCertified().startsWith("CKU")
				|| !vo.getFemalePedigreeCertified().startsWith("CKU")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父母犬为非CKU犬只，不允许申请");
		}
		// 父犬
		Dog maleDog = dogService.getByPedigreeCertifiedCode(vo.getMalePedigreeCertified());
		if (Objects.isNull(maleDog)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "父犬证书号有误，没有找到相关信息");
		} else {
			vo.setMaleDogId(maleDog.getId());
			vo.setMaleBirthCode(maleDog.getBirthCertificateLevelCode());
			vo.setMaleOverseasPedigreeCertified(maleDog.getPedigreeCertifiedEn());
			vo.setMaleMemberCode(maleDog.getMemberCode());
			vo.setMaleMemberName(maleDog.getDogOwner());
			// DNA存档信息
			DogDnaArchive dogDnaArchive = dogDnaArchiveService
					.getByPedigreeCertifiedCode(vo.getMalePedigreeCertified());
			if (Objects.isNull(dogDnaArchive)) {
				vo.setMaleDna(0);
			} else {
				// 判断缴费状态
				if ("1".equals(dogDnaArchive.getPaymentState())) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,
							"父犬" + vo.getMalePedigreeCertified() + "已有未缴费的DNA位点订单，请先处理");
				}
				vo.setMaleDna(1);
			}
		}
		// 母犬
		Dog femaleDog = dogService.getByPedigreeCertifiedCode(vo.getFemalePedigreeCertified());
		if (Objects.isNull(femaleDog)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "母犬证书号有误，没有找到相关信息");
		} else {
			vo.setFemaleDogId(femaleDog.getId());
			vo.setFemaleBirthCode(femaleDog.getBirthCertificateLevelCode());
			vo.setFemaleOverseasPedigreeCertified(femaleDog.getPedigreeCertifiedEn());
			vo.setFemaleMemberCode(femaleDog.getMemberCode());
			vo.setFemaleMemberName(femaleDog.getDogOwner());
			// DNA存档信息
			DogDnaArchive dogDnaArchive = dogDnaArchiveService
					.getByPedigreeCertifiedCode(vo.getFemalePedigreeCertified());
			if (Objects.isNull(dogDnaArchive)) {
				vo.setFemaleDna(0);
			} else {
				// 判断缴费状态
				if ("1".equals(dogDnaArchive.getPaymentState())) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,
							"母犬" + vo.getFemalePedigreeCertified() + "已有未缴费的DNA位点订单，请先处理");
				}
				vo.setFemaleDna(1);
			}
		}
		return vo;
	}

	/**
	 * 更换字典
	 *
	 * @author yuanshuai
	 * @date 2023/6/18 17:10
	 */
	public GoldCertifiedApplyDetailResp generateDetailResp(GoldCertifiedBusinessVo vo) {
		GoldCertifiedApplyDetailResp resp = GoldCertifiedApplyDetailResp.builder().build();
		BeanUtil.copyProperties(resp, vo, "yyyy-MM-dd HH:mm:ss");
		// 更换字典
		// 犬种
		DogType dogType = dogTypeService.getByDogBreedCode(resp.getDogBreed());
		if (Objects.nonNull(dogType)) {
			resp.setDogBreedCn(dogType.getBreedCnName());
		}
		// 毛色
		DogColorMark colorMark = dogColorMarkService.getByDogColorMarkCode(resp.getColorFlag());
		if (Objects.nonNull(colorMark)) {
			resp.setColorFlag(colorMark.getDogColorMarkCn());
		}
		// 性别
		String dogGenderStr = DictUtils.getDictLabel(resp.getGender(), "dog_gender", null);
		if (StringUtils.isNotBlank(dogGenderStr)) {
			resp.setGender(dogGenderStr);
		}
		return resp;
	}

	/**
	 * 根据登记卡进行参数校验
	 *
	 * @author yuanshuai
	 * @date 2023/6/18 15:38
	 */
	public DogBirthCertificate checkDogByBirthCode(String birthCode, Member member, Boolean validate) {
		if (StringUtils.isBlank(birthCode)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "登记卡编号为空");
		}
		DogBirthCertificate dogBirth = dogBirthCertificateService.getByBirthCerRegCode(birthCode);
		if (Objects.isNull(dogBirth)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "没有查到相关犬只信息");
		}
		if (StringUtils.isNotBlank(dogBirth.getPedigreeCertified())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该登记卡已有血统证书，请通过升级入口申请金质证书");
		}
		// 两岁以上犬只无法办理
		if (!DateUtils.isInXMounth(dogBirth.getDogBirthday(), 24)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬年龄已超过2岁，不允许办理新生犬登记");
		}
		if (validate) {
			DogChip dogChip = dogChipService.getByBirthCode(birthCode);
			if (Objects.nonNull(dogChip)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该登记卡已有血统证书，请通过升级入口申请金质证书");
			}
		}
		if (!"1".equals(dogBirth.getReviewState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只有审核通过登记卡可以办理金质证书");
		}
		if (StringUtils.isBlank(dogBirth.getIdentificationFlag())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该登记卡尚未录入芯片，请先录入芯片号");
		}
		DogDnaParentsCompare dnaParentsCompare = dogDnaParentsCompareService.getByBirthCode(birthCode);
		if (Objects.nonNull(dnaParentsCompare) && !"1".equals(dnaParentsCompare.getIsCanceled())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只已经进行过亲子比对或金质证书申请，无法再次申请");
		}
		// 校验犬只是否能办理当前登录类型的业务
		loginServerService.validBirthDogType(dogBirth, member);
		return dogBirth;
	}

	/**
	 * 根据血统证书进行参数校验
	 *
	 * @author yuanshuai
	 * @date 2023/6/18 15:33
	 */
	public Dog checkDogByPedigreeCertified(String pedigreeCertified, Member member) {
		if (StringUtils.isBlank(pedigreeCertified)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号为空");
		}
		pedigreeCertified = pedigreeCertified.toUpperCase();
		if (!pedigreeCertified.startsWith("CKU")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号应为CKU开头");
		}
		Dog dog = dogService.getByPedigreeCertifiedCode(pedigreeCertified);
		if (Objects.isNull(dog)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "没有查到相关犬只信息");
		}
		if (StringUtils.isBlank(dog.getMemberCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只犬主人信息为空");
		}
		if (!dog.getMemberCode().equals(member.getMemberCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬只需为会员名下犬只");
		}
		DogDnaParentsCompare dnaParentsCompare = dogDnaParentsCompareService.getByPedigreeCertified(pedigreeCertified);
		if (Objects.nonNull(dnaParentsCompare) && !"1".equals(dnaParentsCompare.getIsCanceled())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只已经进行过亲子比对或金质证书申请，无法再次申请");
		}
		return dog;
	}
}