/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.restful.v1.breeder.service;

import com.cku.core.PageBeanResult;
import com.cku.core.RESTResponse;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.breeder.dao.PupDao;
import com.cku.oa.breeder.entity.Pup;
import com.cku.oa.breeder.service.PupService;
import com.cku.oa.breeder.service.StudDogService;
import com.cku.oa.dog.dao.DogBirthCertificateDao;
import com.cku.oa.dog.dao.DogDao;
import com.cku.oa.dog.dao.DogNewbornDao;
import com.cku.oa.dog.entity.Dog;
import com.cku.oa.dog.entity.DogBirthCertificate;
import com.cku.oa.dog.entity.DogNewborn;
import com.cku.oa.dog.service.DogBirthCertificateService;
import com.cku.oa.dog.service.DogColorMarkService;
import com.cku.oa.dog.service.DogService;
import com.cku.oa.dog.service.DogTypeService;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.kennel.service.KennelService;
import com.cku.oa.show.dao.ChampionLoginDao;
import com.cku.oa.show.dao.ShowResultsDao;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.util.ImgUtil;
import com.cku.restful.v1.breeder.model.PupSearchLabelOtherEnum;
import com.cku.util.DateUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thinkgem.jeesite.modules.sys.entity.Dict;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletRequest;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * @description: 幼犬业务
 * @author: laiguanglong
 * @date: 2017/10/16 10:43
 */
@Service
@Transactional(readOnly = true)
public class RestPupService {

    @Autowired
    private PupDao pupDao;
    @Autowired
    private DogBirthCertificateDao dogBirthCertificateDao;
    @Autowired
    private DogNewbornDao dogNewbornDao;
    @Autowired
    private DogDao dogDao;
    @Autowired
    private ShowResultsDao showResultsDao;
    @Autowired
    private ChampionLoginDao championLoginDao;
    @Autowired
    private StudDogService studDogService;
    @Autowired
    private KennelService kennelService;
    @Autowired
    private PupService pupService;
    @Autowired
    private RestCityService restCityService;
    @Autowired
    private DogTypeService dogTypeService;
    @Autowired
    private DogService dogService;
    @Autowired
    private DogColorMarkService dogColorMarkService;
    @Autowired
    private DogBirthCertificateService dogBirthCertificateService;


    public RESTResponse getMyPup(HttpServletRequest request) {
        String pageNo = request.getParameter("pageNo");
        String pageSize = request.getParameter("pageSize");
        // 参数校验
        int pageNoInt = 1;
        int pageSizeInt = 10;
        try {
            if (StringUtils.isNotBlank(pageNo)) {
                pageNoInt = Integer.parseInt(pageNo);
            }
            if (StringUtils.isNotBlank(pageSize)) {
                pageSizeInt = Integer.parseInt(pageSize);
            }
        } catch (Exception e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
        }
        String memberCode = UserUtils.getLoginMember().getMemberCode();
        List<Pup> pupList = pupDao.getByMemberCode(memberCode, (pageNoInt - 1) * pageSizeInt, pageSizeInt);
        List<JSONObject> list = buildPupList(pupList, request);
        return new RESTResponse("list", new PageBeanResult<>(list, pupDao.getCountByMemberCode(memberCode)));
    }

    @Transactional(readOnly = false)
    public RESTResponse post(String json) {
        // 获取参数
        JSONObject jo = JSONObject.fromObject(json);
        //参数校验
        postValidateParameter(jo);
        //基本数据校验
        DogBirthCertificate dogBirthCertificate = dogBirthCertificateDao.getByBirthCerRegCode(jo.getString("birthCode"));
        if (dogBirthCertificate == null || !dogBirthCertificate.getBreederMemberCode().equals(UserUtils.getLoginMember().getMemberCode())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "birthCode有误");
        }
        //构造实体类
        Pup po = buildPo(jo);
        //业务校验
        pupService.validateBusiness(po);
        Pup example = new Pup();
        example.setBirthCode(po.getBirthCode());
        List<Pup> list = pupDao.findList(example);
        if (list.size() > 0) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该幼犬已经添加到幼犬列表中");
        }
        //写入业务表
        pupDao.insert(po);
        return new RESTResponse("data", po.getId());
    }

    @Transactional(readOnly = false)
    public RESTResponse put(String json) {
        Member member = UserUtils.getLoginMember();
        // 获取参数
        JSONObject jo = JSONObject.fromObject(json);
        //参数校验
        putValidateParameter(jo);
        //基本数据校验
        Pup po = pupDao.get(jo.getString("id"));
        if (po == null || !po.getMemberCode().equals(member.getMemberCode())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
        }
        //业务校验
        pupService.validateBusiness(po);
        po.setNickname(jo.getString("nickname"));
        po.setMainLabel(getMainLabel(po.getBirthCode()));
        po.setLabel(jo.getString("label"));
        po.setRecommendInfo(jo.getString("recommendInfo"));
        po.setCoverImgId(jo.getString("coverImgId"));
        po.setOtherImgId(jo.getString("otherImgId"));
        po.setCity(jo.getString("city"));
        po.setAddTime(new Date());
        //修改后立刻上架
        po.setState("1");
        po.preUpdate();
        pupDao.update(po);
        return new RESTResponse();
    }

    @Transactional(readOnly = false)
    public RESTResponse pupState(String json) {
        Member member = UserUtils.getLoginMember();
        JSONObject jo = JSONObject.fromObject(json);
        if (!jo.containsKey("id") || StringUtils.isBlank(jo.getString("id"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id为空");
        }
        if (!jo.containsKey("state") || StringUtils.isBlank(jo.getString("state"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "state为空");
        }
        if (!"0".equals(jo.getString("state")) && !"1".equals(jo.getString("state"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "state只能为0或1");
        }
        Pup po = pupDao.get(jo.getString("id"));
        if (po == null || !member.getMemberCode().equals(po.getMemberCode())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
        }
        //业务校验
        pupService.validateBusiness(po);
        po.setState(jo.getString("state"));
        if ("1".equals(jo.getString("state"))) {
            po.setAddTime(new Date());
            po.setMainLabel(getMainLabel(po.getBirthCode()));
        }
        po.preUpdate();
        pupDao.update(po);
        return new RESTResponse();
    }

    public RESTResponse pupSearch(HttpServletRequest request) {
        String pageNo = request.getParameter("pageNo");
        String pageSize = request.getParameter("pageSize");
        // 参数校验
        int pageNoInt = 1;
        int pageSizeInt = 10;
        try {
            if (StringUtils.isNotBlank(pageNo)) {
                pageNoInt = Integer.parseInt(pageNo);
            }
            if (StringUtils.isNotBlank(pageSize)) {
                pageSizeInt = Integer.parseInt(pageSize);
            }
        } catch (Exception e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
        }
        Map map = Maps.newHashMap();
        map.put("pageNo", (pageNoInt - 1) * pageSizeInt);
        map.put("pageSize", pageSizeInt);
        map.put("dogBreed", request.getParameter("dogBreed"));
        map.put("city", request.getParameter("city"));
        String other = request.getParameter("other");
        if (StringUtils.isNotBlank(other)) {
            for (String value : other.split(",")) {
                PupSearchLabelOtherEnum e = PupSearchLabelOtherEnum.getByValue(value);
                if (e != null) {
                    map.put(e.getParameter(), "1");
                }
            }
        }
        List<Pup> pupList = pupDao.search(map);
        List<JSONObject> list = buildPupList(pupList, request);
        return new RESTResponse("list", new PageBeanResult<>(list, pupDao.getSearchCount(map)));
    }

    private void postValidateParameter(JSONObject jo) {
        if (!jo.containsKey("birthCode") || StringUtils.isBlank(jo.getString("birthCode"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "birthCode为空");
        }
        basicValidateParameter(jo);
    }

    private void putValidateParameter(JSONObject jo) {
        if (!jo.containsKey("id") || StringUtils.isBlank(jo.getString("id"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id为空");
        }
        basicValidateParameter(jo);
    }

    private void basicValidateParameter(JSONObject jo) {
        if (!jo.containsKey("nickname") || StringUtils.isBlank(jo.getString("nickname"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "nickname为空");
        }
        if (!jo.containsKey("coverImgId") || StringUtils.isBlank(jo.getString("coverImgId"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "coverImgId为空");
        }
        if (!jo.containsKey("otherImgId")) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "otherImgId为空");
        }
        if (!jo.containsKey("recommendInfo")) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "recommendInfo为空");
        }
        if (!jo.containsKey("city") || StringUtils.isBlank(jo.getString("city"))) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "city为空");
        }
    }

    @Transactional(readOnly = false)
    public RESTResponse get(String id, HttpServletRequest request) {
        Pup po = pupDao.get(id);
        if (po == null) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
        }
        DogBirthCertificate dogBirthCertificate = dogBirthCertificateDao.getByBirthCerRegCode(po.getBirthCode());
        JSONObject data = new JSONObject();
        data.put("id", getString(po.getId()));
        data.put("birthCode", getString(po.getBirthCode()));
        //犬种名
        data.put("dogBreed", getString(dogTypeService.getBreedCnName(dogBirthCertificate.getDogBreed())));
        //犬舍名
        Kennel kennel = kennelService.getKennelByMemberCode(dogBirthCertificate.getBreederMemberCode());
        String kennelName = "";
        if (kennel != null) {
            kennelName = kennel.getName();
        }
        data.put("kennelName", getString(kennelName));
        //毛色
        data.put("colorFlag", getString(dogColorMarkService.getDogColorMarkCn(dogBirthCertificate.getColorFlag())));
        //年龄
        data.put("age", getString(dogService.getAge(po.getBirthdate())));
        //标签
        JSONArray label = new JSONArray();
        if (StringUtils.isNotBlank(po.getMainLabel())) {
            for (String s : po.getMainLabel().split(",")) {
                label.add(DictUtils.getDictLabel(s, "pup_label", ""));
            }
        }
        if (StringUtils.isNotBlank(po.getLabel())) {
            for (String s : po.getLabel().split(",")) {
                label.add(DictUtils.getDictLabel(s, "pup_label", ""));
            }
        }
        data.put("label", label);
        //直系亲属
        DogNewborn dogNewborn = dogNewbornDao.getByBreedCertifiedCode(dogBirthCertificate.getFmdogBreedCertifiedCode());
        dogNewborn.getMalePedigreeCertified();
        data.put("fatherPedigreeCertified", dogNewborn.getMalePedigreeCertified());
        JSONArray fatherLabel = new JSONArray();
        String[] fatherLoginTypeArray = studDogService.getloginTypeArray(dogNewborn.getMalePedigreeCertified());
        for (int i = 0; i < fatherLoginTypeArray.length; i++) {
            String studDogLabel = DictUtils.getDictLabel(fatherLoginTypeArray[i], "stud_dog_label", "");
            if (StringUtils.isNotBlank(studDogLabel)) {
                fatherLabel.add(studDogLabel);
            }
        }
        data.put("fatherLabel", fatherLabel);
        dogNewborn.getFemalePedigreeCertified();
        data.put("motherPedigreeCertified", dogNewborn.getFemalePedigreeCertified());
        JSONArray motherLabel = new JSONArray();
        String[] motherLoginTypeArray = studDogService.getloginTypeArray(dogNewborn.getFemalePedigreeCertified());
        for (int i = 0; i < motherLoginTypeArray.length; i++) {
            String studDogLabel = DictUtils.getDictLabel(motherLoginTypeArray[i], "stud_dog_label", "");
            if (StringUtils.isNotBlank(studDogLabel)) {
                motherLabel.add(studDogLabel);
            }
        }
        data.put("motherLabel", motherLabel);
        //城市
        data.put("city", getString(po.getCity()));
        //犬主推荐
        data.put("recommendInfo", getString(po.getRecommendInfo()));
        //风采展示
        JSONArray img = new JSONArray();
        if (StringUtils.isNotBlank(po.getCoverImgId())) {
            img.add(getString(ImgUtil.getImgUrl(request, po.getCoverImgId())));
        }
        if (StringUtils.isNotBlank(po.getOtherImgId())) {
            for (String otherImgId : po.getOtherImgId().split(",")) {
                if (StringUtils.isNotBlank(otherImgId)) {
                    img.add(getString(ImgUtil.getImgUrl(request, otherImgId)));
                }
            }
        }
        data.put("img", img);
        //犬呼名
        data.put("nickname", getString(po.getNickname()));
        //犬名
        data.put("dogEnName", getString(dogBirthCertificate.getDogEnName()));
        //性别
        data.put("dogGender", DictUtils.getDictLabel(dogBirthCertificate.getDogGender(), "dog_gender", ""));
        //出生日期
        data.put("dogBirthday", DateUtils.dateToStr(po.getBirthdate()));
        //点击次数加一
        pupDao.addViewCount(po.getId());
        //幼犬年龄大于2岁,标记为下架
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(po.getBirthdate());
        calendar.add(Calendar.YEAR, 2);
        if (calendar.getTimeInMillis() < System.currentTimeMillis()) {
            po.setState("0");
            po.preUpdate();
            pupDao.update(po);
        }
        return new RESTResponse("data", data);
    }

    private Pup buildPo(JSONObject jo) {
        Member member = UserUtils.getLoginMember();
        String memberCode = member.getMemberCode();
        DogBirthCertificate dogBirthCertificate = dogBirthCertificateDao.getByBirthCerRegCode(jo.getString("birthCode"));
        if (dogBirthCertificate == null || !memberCode.equals(dogBirthCertificate.getBreederMemberCode())) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "新生犬登记卡编号有误");
        }
        //待打印、待邮寄、已寄出
        String businessState = dogBirthCertificateService.getBusinessState(dogBirthCertificate);
        if (!Lists.newArrayList("待打印", "待邮寄", "已寄出").contains(businessState)) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只有新生犬身份识别卡的业务状态为待打印或待邮寄或已寄出的状态下可以申请幼犬登记");
        }
        Pup po = new Pup();
        po.setBirthCode(dogBirthCertificate.getBirthCerRegCode());
        po.setNickname(jo.getString("nickname"));
        po.setState("1");
        po.setDogBreed(dogBirthCertificate.getDogBreed());
        po.setMemberCode(memberCode);
        po.setMainLabel(getMainLabel(dogBirthCertificate.getBirthCerRegCode()));
        po.setLabel(jo.getString("label"));
        po.setRecommendInfo(jo.getString("recommendInfo"));
        po.setCoverImgId(jo.getString("coverImgId"));
        po.setOtherImgId(jo.getString("otherImgId"));
        String city = restCityService.getCity(member);
        if (jo.containsKey("city") && StringUtils.isNotBlank(jo.getString("city"))) {
            city = jo.getString("city");
        }
        po.setCity(city);
        po.setBirthdate(dogBirthCertificate.getDogBirthday());
        po.setGender(dogBirthCertificate.getDogGender());
        po.setAddTime(new Date());
        po.preInsert();
        return po;
    }

    public RESTResponse getCanApplyPupList(HttpServletRequest request) {
        String dogBreed = request.getParameter("dogBreed");
        String gender = request.getParameter("gender");
        String order = request.getParameter("order");
        String pageNo = request.getParameter("pageNo");
        String pageSize = request.getParameter("pageSize");
        // 参数校验
        int pageNoInt = 1;
        int pageSizeInt = 10;
        try {
            if (StringUtils.isNotBlank(pageNo)) {
                pageNoInt = Integer.parseInt(pageNo);
            }
            if (StringUtils.isNotBlank(pageSize)) {
                pageSizeInt = Integer.parseInt(pageSize);
            }
        } catch (Exception e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
        }
        Member member = UserUtils.getLoginMember();
        List<DogBirthCertificate> dogBirthCertificateList = dogBirthCertificateDao.getCanApplyPupList(member.getMemberCode(), dogBreed, gender, order, (pageNoInt - 1) * pageSizeInt, pageSizeInt);
        List<JSONObject> list = Lists.newArrayList();
        for (DogBirthCertificate dogBirthCertificate : dogBirthCertificateList) {
            JSONObject jo = new JSONObject();
            jo.put("id", getString(dogBirthCertificate.getId()));
            jo.put("birthCerRegCode", getString(dogBirthCertificate.getBirthCerRegCode()));
            jo.put("dogBirthday", getString(DateUtils.dateToStr(dogBirthCertificate.getDogBirthday())));
            jo.put("identificationFlag", getString(dogBirthCertificate.getIdentificationFlag()));
            jo.put("colorFlag", getString(dogColorMarkService.getDogColorMarkCn(dogBirthCertificate.getColorFlag())));
            jo.put("dogBreed", getString(dogTypeService.getBreedCnName(dogBirthCertificate.getDogBreed())));
            jo.put("dogGender", getString(DictUtils.getDictLabel(dogBirthCertificate.getDogGender(), "dog_gender", "")));
            jo.put("dogEnName", getString(dogBirthCertificate.getDogEnName()));
            jo.put("city", restCityService.getCity(member));
            list.add(jo);
        }
        return new RESTResponse("list", new PageBeanResult<>(list, dogBirthCertificateDao.getCanApplyPupCount(member.getMemberCode(), dogBreed, gender, order)));
    }

    public RESTResponse getCanApplyPupDogBreedList() {
        List<Map<String, String>> mapList = dogBirthCertificateDao.getCanApplyPupDogBreed(UserUtils.getLoginMember().getMemberCode());
        List<JSONObject> list = Lists.newArrayList();
        for (Map<String, String> map : mapList) {
            JSONObject jo = new JSONObject();
            jo.put("dogBreed", map.get("dogBreed"));
            jo.put("breedCnName", map.get("breedCnName"));
            list.add(jo);
        }
        return new RESTResponse("list", new PageBeanResult<>(list));
    }

    public RESTResponse label() {
        List<JSONObject> list = Lists.newArrayList();
        List<Dict> dictList = DictUtils.getDictList("pup_label");
        for (Dict dict : dictList) {
            //排序大于20的是可编辑标签
            if (dict.getSort() > 20) {
                JSONObject jo = new JSONObject();
                jo.put("label", getString(dict.getLabel()));
                jo.put("value", getString(dict.getValue()));
                list.add(jo);
            }
        }
        return new RESTResponse("list", new PageBeanResult<>(list, new Long(list.size())));
    }

    public RESTResponse pupSearchLabelOther() {
        List<JSONObject> list = Lists.newArrayList();
        for (PupSearchLabelOtherEnum pupSearchLabelOtherEnum : PupSearchLabelOtherEnum.values()) {
            JSONObject jo = new JSONObject();
            jo.put("label", pupSearchLabelOtherEnum.getLabel());
            jo.put("value", pupSearchLabelOtherEnum.getValue());
            list.add(jo);
        }
        return new RESTResponse("list", new PageBeanResult<>(list, new Long(list.size())));
    }

    private List<JSONObject> buildPupList(List<Pup> pupList, HttpServletRequest request) {
        List<JSONObject> list = Lists.newArrayList();
        for (Pup pup : pupList) {
            JSONObject jo = new JSONObject();
            jo.put("id", getString(pup.getId()));
            String state = "";
            if ("1".equals(pup.getState())) {
                state = "已发布";
            }
            jo.put("state", getString(state));
            jo.put("coverImgId", getString(ImgUtil.getImgUrl(request, pup.getCoverImgId())));
            jo.put("nickname", getString(pup.getNickname()));
            jo.put("dogBreed", getString(dogTypeService.getBreedCnName(pup.getDogBreed())));
            String labels = StringUtils.join(pup.getMainLabel(), ",", pup.getLabel());
            JSONArray ja = new JSONArray();
            for (String label : labels.split(",")) {
                if (StringUtils.isNotBlank(label)) {
                    ja.add(DictUtils.getDictLabel(label, "pup_label", ""));
                    if (ja.size() >= 4) {
                        break;
                    }
                }
            }
            jo.put("label", ja);
            jo.put("age", getString(studDogService.getAge(pup.getBirthdate())));
            jo.put("city", getString(pup.getCity()));
            jo.put("gender", getString(pup.getGender()));
            jo.put("viewCount", getString(pup.getViewCount()));
            list.add(jo);
        }
        return list;
    }

    public RESTResponse pupTopList(HttpServletRequest request) {
        String pageSize = request.getParameter("pageSize");
        // 参数校验
        int pageSizeInt = 10;
        try {
            if (StringUtils.isNotBlank(pageSize)) {
                pageSizeInt = Integer.parseInt(pageSize);
            }
        } catch (Exception e) {
            throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageSize必须为数字");
        }
        List<Pup> studDogList = pupDao.topList(pageSizeInt);
        List<JSONObject> list = buildPupList(studDogList, request);
        return new RESTResponse("list", new PageBeanResult<>(list, new Long(list.size())));
    }

    /**
     * @description: 获取标签
     * @author: laiguanglong
     * @date: 2017/10/16 10:44
     */
    private String getMainLabel(String birthCode) {
        List<String> label = Lists.newArrayList();
        Dog dog = dogDao.getByBirthCertificateLevelCode(birthCode);
        if (dog != null && StringUtils.isNotBlank(dog.getPedigreeCertifiedCode())) {
            String pedigreeCertified = dog.getPedigreeCertifiedCode();
            String[] loginTypeArray = studDogService.getloginTypeArray(pedigreeCertified);
            if (StringUtils.isNotBlank(loginTypeArray[0])) {
                label.add("2" + loginTypeArray[0]);
            }
            if (StringUtils.isNotBlank(loginTypeArray[1])) {
                label.add("2" + loginTypeArray[1]);
            }
            String label3 = null;
            int bisCount = showResultsDao.getBisCountByPedigreeCertified(pedigreeCertified);
            if (bisCount > 0) {
                label3 = "11";
            }
            if (StringUtils.isBlank(label3)) {
                int bigCount = showResultsDao.getBigCountByPedigreeCertified(pedigreeCertified);
                if (bigCount > 0) {
                    label3 = "12";
                }
            }
            if (StringUtils.isBlank(label3)) {
                int bobCount = showResultsDao.getBobCountByPedigreeCertified(pedigreeCertified);
                if (bobCount > 0) {
                    label3 = "13";
                }
            }
            if (StringUtils.isNotBlank(label3)) {
                label.add(label3);
            }
        }
        String label2 = null;
        DogBirthCertificate dogBirthCertificate = dogBirthCertificateDao.getByBirthCerRegCode(birthCode);
        DogNewborn dogNewborn = dogNewbornDao.getByBreedCertifiedCode(dogBirthCertificate.getFmdogBreedCertifiedCode());
        int fatherLoginSize = championLoginDao.getLoginTypeByPedigreeCertified(dogNewborn.getMalePedigreeCertified()).size();
        int motherLoginSize = championLoginDao.getLoginTypeByPedigreeCertified(dogNewborn.getFemalePedigreeCertified()).size();
        if (fatherLoginSize > 0 || motherLoginSize > 0) {
            label2 = "16";
            if (fatherLoginSize > 0 && motherLoginSize > 0) {
                label2 = "15";
            }
        }
        if (StringUtils.isNotBlank(label2)) {
            label.add(label2);
        }
        return StringUtils.join(label, ",");
    }

    private String getString(String str) {
        return str == null ? "" : str;
    }
}