package com.cku.restful.v1.appreciation.service;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.ListResultDto;
import com.cku.core.RESTResponse;
import com.cku.core.ResultDto;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.appreciation.entity.AppreciationApply;
import com.cku.oa.appreciation.entity.AppreciationCourses;
import com.cku.oa.appreciation.entity.AppreciationCourseware;
import com.cku.oa.appreciation.service.AppreciationApplyService;
import com.cku.oa.appreciation.service.AppreciationCoursesService;
import com.cku.oa.appreciation.service.AppreciationCoursewareService;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.finance.entity.PaymentOrder;
import com.cku.oa.finance.service.PaymentChargingItemService;
import com.cku.oa.finance.service.PaymentOrderService;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.SMSRemindService;
import com.cku.oa.sys.util.ValidateUtil;
import com.cku.restful.v1.appreciation.model.RestAppreciationApply;
import com.cku.restful.v1.appreciation.model.RestAppreciationCourse;
import com.cku.restful.v1.appreciation.model.RestAppreciationCourseware;
import com.cku.restful.v1.appreciation.model.RestAppreciationLogin;
import com.cku.restful.v1.appreciation.model.RestAppreciationUser;
import com.cku.restful.v1.appreciation.model.RestAppreciationValidCode;
import com.cku.restful.v1.appreciation.model.RestAppreciationValidCodeEnum;
import com.cku.restful.v1.appreciation.profile.AppreciationUserProfile;
import com.cku.util.TokenUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.subscription.restful.service.WXpayServiceImpl;
import com.thinkgem.jeesite.common.utils.JedisUtils;

@Service
@Transactional(readOnly = true)
public class RestAppreciationService {

	@Autowired
	private AppreciationApplyService appreciationApplyService;

	@Autowired
	private AppreciationCoursesService appreciationCoursesService;

	@Autowired
	private AppreciationCoursewareService appreciationCoursewareService;

	@Autowired
	private PaymentOrderService paymentOrderService;

	@Autowired
	private PaymentChargingItemService paymentChargingItemService;

	@Autowired
	private SMSRemindService smsRemindService;

	@Autowired
	private WXpayServiceImpl wxpayServiceImpl;

	public ListResultDto<RestAppreciationCourse> findCourseList(String mobile) {
		List<RestAppreciationCourse> respList = null;
		List<AppreciationCourses> courseList = appreciationCoursesService.findListByApplyMobile(mobile);
		if (CollectionUtils.isNotEmpty(courseList)) {
			respList = courseList.stream().map(c -> {
				return RestAppreciationCourse.builder().id(c.getId()).courseCode(c.getCourseCode())
						.courseName(c.getCourseName()).courseLocation(c.getCourseLocation()).startDate(c.getStartDate())
						.endDate(c.getEndDate()).teacher(c.getTeacher())
						.lastCoursewareUpdateDate(c.getLastCoursewareUpdateDate()).build();
			}).collect(Collectors.toList());
		}
		return ListResultDto.success(respList);
	}

	public ListResultDto<RestAppreciationCourseware> findCoursewareList(String courseId) {

		List<RestAppreciationCourseware> respList = null;
		List<AppreciationCourseware> coursewareList = appreciationCoursewareService.findListByCourseId(courseId);
		if (CollectionUtils.isNotEmpty(coursewareList)) {
			respList = coursewareList.stream().map(cw -> {
				return RestAppreciationCourseware.builder().coursewareName(cw.getCoursewareName())
						.coursewareUrl(cw.getUrl()).updateDate(cw.getUpdateDate()).build();
			}).collect(Collectors.toList());
		}
		return ListResultDto.success(respList);
	}

	@Transactional(readOnly = false)
	public ResultDto<Boolean> sendCodeMessage(RestAppreciationValidCode appreciationValidCode) throws Exception {
		if (RestAppreciationValidCodeEnum.LOGIN.code().equals(appreciationValidCode.getCodeType())) {
			if (StringUtils.isBlank(appreciationValidCode.getMobile())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "手机号为空");
			}
			if (!ValidateUtil.phoneMobile(appreciationValidCode.getMobile())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "手机号格式不正确");
			}
			// 验证用户存在报名信息且缴费状态为已缴费的用户可登录成功
			List<AppreciationApply> applyList = appreciationApplyService
					.findPaidByMobile(appreciationValidCode.getMobile());
			if (CollectionUtils.isEmpty(applyList)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "当前手机账号没有课程报名信息，无法登录！");
			}
		}
		smsRemindService.sendAppreciationApplySMSCode(appreciationValidCode.getMobile(),
				appreciationValidCode.getCodeType());
		return ResultDto.success(true);
	}

	@Transactional(readOnly = false)
	public ResultDto<RestAppreciationUser> login(RestAppreciationLogin appreciationLogin) throws Exception {
		// 验证手机
		if (StringUtils.isBlank(appreciationLogin.getMobile())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "手机号为空");
		}
		if (!ValidateUtil.phoneMobile(appreciationLogin.getMobile())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "手机号格式不正确");
		}
		// 验证手机验证码
		smsRemindService.checkAppreciationApplySmsCode(appreciationLogin.getMobile(), appreciationLogin.getSmsCode(),
				RestAppreciationValidCodeEnum.LOGIN.code());
		// 验证用户存在报名信息且缴费状态为已缴费的用户可登录成功
		List<AppreciationApply> applyList = appreciationApplyService.findPaidByMobile(appreciationLogin.getMobile());
		if (CollectionUtils.isEmpty(applyList)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "当前手机账号没有课程报名信息，无法登录！");
		}
		if (applyList.stream().collect(Collectors.groupingBy(AppreciationApply::getMemberCode)).size() > 1) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "当前手机账号关联多会员的报名信息，无法登录，请联系客服，致电400-660-7000！");
		}
		String memberCode = applyList.stream().findAny().get().getMemberCode();
		String token = TokenUtil.createJwt(
				new ObjectMapper()
						.writeValueAsString(AppreciationUserProfile.builder().mobile(appreciationLogin.getMobile())
								.memberCode(applyList.stream().findAny().get().getMemberCode()).build()),
				1 * 24 * 60 * 60 * 1000);
		JedisUtils.set(generateLoginRdsKey(appreciationLogin.getMobile(), memberCode), token, 1 * 24 * 60 * 60);
		return ResultDto.success(RestAppreciationUser.builder().mobile(appreciationLogin.getMobile())
				.memberCode(memberCode).token(token).build());
	}

	public String generateLoginRdsKey(String mobile, String memberCode) {
		return "appreciation_member_code_" + memberCode + "_" + "system" + "_" + mobile;
	}

	@Transactional(readOnly = false, rollbackFor = Exception.class)
	public RESTResponse apply(RestAppreciationApply request, String ip) throws Exception {
		// 参数校验
		validateParameter(request);
		// 新建或者查询会员
		Member member = appreciationApplyService.createMemberWithAppreciation(request);
		// 保存业务数据
		AppreciationApply apply = buildPo(request, member);
		// 0正常 1试听
		String shortName = "0".equals(request.getApplyType()) ? "appreciation_apply@formal" : "appreciation_apply@free";
		// 建订单
		PaymentOrder order = paymentOrderService.saveOrderForOneDetail(member, shortName, apply.getId(),
				"appreciation_apply", "gzh", "24", "1203");
		apply.setRunningNumber(order.getOrderCode());
		appreciationApplyService.save(apply);
		// 微信预下单
		return wxpayServiceImpl.createPayOrder(order.getId(), ip, request.getOpenId());
	}

	private void validateParameter(RestAppreciationApply vo) {
		if (!("1".equals(vo.getApplyType()) || "0".equals(vo.getApplyType()))) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "报名方式参数错误！");
		}
		// 验证码
		smsRemindService.checkAppreciationApplySmsCode(vo.getMobile(), vo.getSmsCode(),
				RestAppreciationValidCodeEnum.APPLY.code());
	}

	private AppreciationApply buildPo(RestAppreciationApply request, Member member) {
		// 保存业务数据
		AppreciationApply apply = new AppreciationApply();
		apply.setApplyChannel("个人");
		apply.setApplyCode(appreciationApplyService.getMaxApplyCode());// 六位从000001递增
		apply.setMemberName(request.getRealName());
		apply.setMemberCode(member.getMemberCode());
		apply.setMobile(request.getMobile());
		apply.setShopName(request.getShopName());
		apply.setProvince(request.getProvince());
		apply.setCity(request.getCity());
		apply.setArea(request.getArea());
		apply.setApplyType(request.getApplyType());
		appreciationApplyService.save(apply);
		return apply;
	}

	@Transactional(readOnly = false)
	public ResultDto<String> getPrice(String applyType) throws Exception {
		// 0正常 1试听
		String shortName = "0".equals(applyType) ? "appreciation_apply@formal"
				: ("1".equals(applyType) ? "appreciation_apply@free" : null);
		if (shortName == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "报名方式必传！");
		}
		PaymentChargingItem item = paymentChargingItemService.getByShortName(shortName);
		if (item == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "收费项没有找到！");
		}
		return ResultDto.success(item.getPrice());
	}

}
