/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.show.web;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.shiro.SecurityUtils;
import org.apache.shiro.authz.UnauthenticatedException;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.cku.oa.dog.service.DogDnaService;
import com.cku.oa.finance.service.PaymentOrderService;
import com.cku.oa.show.entity.ChampionLogin;
import com.cku.oa.show.service.ChampionLoginService;
import com.cku.oa.show.vo.ChampionLoginVO;
import com.thinkgem.jeesite.common.config.Global;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.common.web.BaseController;

/**
 * 冠军登录Controller
 * 
 * @author zhuoHeng
 * @version 2016-07-14
 */
@Controller
@RequestMapping(value = "${adminPath}/championlogin/championLogin")
public class ChampionLoginController extends BaseController {

	@Autowired
	private ChampionLoginService championLoginService;
	@Autowired
	private DogDnaService dogDnaService;

	private final static String AUTHORITY_PRDFIX = "championlogin:championLogin:";

	@ModelAttribute
	public ChampionLogin get(@RequestParam(required = false) String id) {
		ChampionLogin entity = null;
		if (StringUtils.isNotBlank(id)) {
			entity = championLoginService.get(id);
		}
		if (entity == null) {
			entity = new ChampionLogin();
		}
		return entity;
	}

	@Autowired
	private PaymentOrderService paymentOrderService;

	/**
	 *
	 * @Description：冠军登录列表查询
	 * @author: zhuoHeng
	 * @version: 2016年7月14日 下午3:29:51
	 */
	@RequiresPermissions(AUTHORITY_PRDFIX + "view")
	@RequestMapping(value = { "list", "" })
	public String list(ChampionLogin championLogin, HttpServletRequest request, HttpServletResponse response,
			Model model, @ModelAttribute("importError") String importError) {
		Page<ChampionLogin> page = championLoginService.findPage(new Page<ChampionLogin>(request, response),
				championLogin);
		for (ChampionLogin vo : page.getList()) {
			vo.setIsHaveDogDna(dogDnaService.isHaveDogDna(vo.getDog().getId()) ? "是" : "否");
		}
		// 查询有没有退款或退款中
		Map<String, String> map = paymentOrderService
				.getRefundStateByBusinessId(page.getList().stream().map(a -> a.getId()).collect(Collectors.toSet()));
		List<ChampionLogin> list = page.getList().stream().map(a -> {
			ChampionLoginVO b = new ChampionLoginVO();
			BeanUtils.copyProperties(a, b);
			b.setBusinessRefundState(map.keySet().contains(a.getId()));
			return b;
		}).collect(Collectors.toList());
		page.setList(list);
		model.addAttribute("page", page);

		model.addAttribute("importError", importError);
		return "oa/show/championlogin/championLoginList";
	}

	/**
	 *
	 * @Description：进入冠军登录编辑界面
	 * @author: zhuoHeng
	 * @version: 2016年7月14日 下午3:30:16
	 */
	@RequiresPermissions(AUTHORITY_PRDFIX + "view")
	@RequestMapping(value = "form")
	public String form(ChampionLogin championLogin, Model model) {
		model.addAttribute("championLogin", championLogin);
		return "oa/show/championlogin/championLoginForm";
	}

	/**
	 *
	 * @Description：冠军登录业务添加页
	 * @author: zhuoHeng
	 * @version: 2016年9月5日 下午7:53:21
	 */
	@RequiresPermissions("finance:paymentOrder:orderItem")
	@RequestMapping(value = "orderAdd")
	public String orderAdd(ChampionLogin championLogin, Model model) {
		model.addAttribute("championLogin", championLogin);
		return "oa/show/championlogin/championLoginOrderAdd";
	}

	/**
	 *
	 * @Description：冠军登录业务信息保存
	 * @author: zhuoHeng
	 * @version: 2016年9月5日 下午7:52:59
	 */
	@RequiresPermissions("finance:paymentOrder:orderItem")
	@RequestMapping(value = "orderSave")
	@ResponseBody
	public Object orderSave(ChampionLogin championLogin, Model model) {
		HashMap<String, String> map = new HashMap<String, String>();
		map.put("rc", "0");
		try {
			championLoginService.orderSave(championLogin);
		} catch (Exception e) {
			map.put("rc", "1");
			map.put("msg", e.getMessage());
		}
		return map;
	}

	/**
	 *
	 * @Description：冠军登录订单列表
	 * @author: zhuoHeng
	 * @version: 2016年9月5日 下午7:52:44
	 */
	@RequiresPermissions("finance:paymentOrder:orderItem")
	@RequestMapping(value = "orderList")
	public String orderList(ChampionLogin championLogin, Model model) {
		List<ChampionLogin> list = championLoginService.findList(championLogin);
		// 查询有没有退款或退款中
		Map<String, String> map = paymentOrderService
				.getRefundStateByBusinessId(list.stream().map(a -> a.getId()).collect(Collectors.toSet()));
		List<ChampionLoginVO> list2 = list.stream().map(a -> {
			ChampionLoginVO b = new ChampionLoginVO();
			BeanUtils.copyProperties(a, b);
			b.setBusinessRefundState(map.keySet().contains(a.getId()));
			return b;
		}).collect(Collectors.toList());
		model.addAttribute("list", list2);
		return "oa/show/championlogin/championLoginOrderList";
	}

	/**
	 *
	 * @Description：保存冠军登录信息
	 * @author: zhuoHeng
	 * @version: 2016年7月14日 下午3:30:34
	 */
	@RequiresPermissions(AUTHORITY_PRDFIX + "view")
	@RequestMapping(value = "save")
	public String save(ChampionLogin championLogin, Model model, RedirectAttributes redirectAttributes) {
		boolean hasPermission = false;
		if (!StringUtils.isEmpty(championLogin.getId())) {
			hasPermission = SecurityUtils.getSubject().isPermitted(AUTHORITY_PRDFIX + "edit");
		} else {
			hasPermission = SecurityUtils.getSubject().isPermitted(AUTHORITY_PRDFIX + "add");
		}
		if (hasPermission) {
			if (!beanValidator(model, championLogin)) {
				return form(championLogin, model);
			}
			try {
				championLoginService.save(championLogin);
				addMessage(redirectAttributes, "保存冠军登录信息成功");
			} catch (Exception e) {
				addMessage(redirectAttributes, "审核失败:", e.getMessage());
			}
		} else {
			throw new UnauthenticatedException();
		}
		return "redirect:" + Global.getAdminPath() + "/championlogin/championLogin/?repage";
	}

	/**
	 *
	 * @Description：订单业务删除
	 * @author: zhuoHeng
	 * @version: 2016年9月6日 上午11:33:50
	 */
	@RequiresPermissions("finance:paymentOrder:orderItem")
	@RequestMapping(value = "orderDelete")
	public String orderDelete(ChampionLogin championLogin, RedirectAttributes redirectAttributes) {
		championLoginService.delete(championLogin);
		String runningNumber = championLogin.getRunningNumber();
		addMessage(redirectAttributes, "删除冠军登录信息成功");
		return "redirect:" + Global.getAdminPath() + "/championlogin/championLogin/orderList?runningNumber="
				+ runningNumber + "&repage";
	}

	/**
	 *
	 * @Description：删除冠军登录信息
	 * @author: zhuoHeng
	 * @version: 2016年7月14日 下午3:30:47
	 */
	@RequiresPermissions(AUTHORITY_PRDFIX + "delete")
	@RequestMapping(value = "delete")
	public String delete(ChampionLogin championLogin, RedirectAttributes redirectAttributes) {
		championLoginService.delete(championLogin);
		addMessage(redirectAttributes, "删除冠军登录信息成功");
		return "redirect:" + Global.getAdminPath() + "/championlogin/championLogin/?repage";
	}

	/**
	 *
	 * @Description：根据血统证书号获取犬只信息和比赛成绩
	 * @author: zhuoHeng
	 * @version: 2016年8月26日 上午9:55:14
	 */
	@RequestMapping(value = "/getDogMessage")
	public void getDogMessage(HttpServletRequest request, HttpServletResponse response) {
		String pedigreeCertified = request.getParameter("pedigreeCertified");
		String str = championLoginService.getDogMessage(pedigreeCertified);
		try {
			response.setContentType("text/Xml;charset=utf-8");
			response.getWriter().write(str);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 *
	 * @Description：冠军登录打印
	 * @author: zhuoHeng
	 * @version: 2016年8月26日 上午10:14:09
	 */
	@RequiresPermissions(AUTHORITY_PRDFIX + "print")
	@RequestMapping(value = "print")
	public String print(ChampionLogin championLogin, HttpServletRequest request, HttpServletResponse response,
			Model model) {
		model.addAttribute("championLogin", championLogin);
		championLoginService.print(championLogin);
		return "oa/show/championlogin/championPrint";
	}

	/**
	 *
	 * @Description：重置冠军登录打印状态
	 * @author: zhuoHeng
	 * @version: 2016年8月26日 上午9:56:42
	 */
	@RequiresPermissions(AUTHORITY_PRDFIX + "resetPrint")
	@RequestMapping(value = "resetPrint")
	public String resetPrint(ChampionLogin championLogin, RedirectAttributes redirectAttributes) {
		championLoginService.resetPrint(championLogin);
		addMessage(redirectAttributes, "重置冠军登录打印成功");
		return "redirect:" + Global.getAdminPath() + "/championlogin/championLogin/?repage";
	}

	/**
	 *
	 * @Description：冠军登录查看页
	 * @author: zhuoHeng
	 * @version: 2016年8月27日 下午5:15:42
	 */
	@RequiresPermissions(AUTHORITY_PRDFIX + "view")
	@RequestMapping(value = "view")
	public String view(ChampionLogin championLogin, Model model) {
		model.addAttribute("championLogin", championLogin);
		return "oa/show/championlogin/championLoginView";
	}

	/**
	 *
	 * @Description：是否符合申请条件验证
	 * @author: zhuoHeng
	 * @version: 2016年8月29日 上午10:43:37
	 */
	@RequestMapping(value = "validateLoginType")
	@ResponseBody
	public String validateLoginType(HttpServletRequest request) {
		String loginType = request.getParameter("loginType");
		String pedigreeCertified = request.getParameter("pedigreeCertified");
		return championLoginService.validateLoginType(loginType, pedigreeCertified);
	}

	/**
	 *
	 * @Description：登录申请是否重复
	 * @author: zhuoHeng
	 * @version: 2016年8月29日 下午1:58:23
	 */
	@RequestMapping(value = "validateLogin")
	@ResponseBody
	public String validateLogin(HttpServletRequest request) {
		String loginType = request.getParameter("loginType");
		String pedigreeCertified = request.getParameter("pedigreeCertified");
		return championLoginService.validateLogin(loginType, pedigreeCertified);
	}

	/**
	 * @Description： 导入登录信息的模板
	 * 
	 * @author: yuanshuai
	 * @date: 2019/6/14 11:23
	 */
	@RequiresPermissions(AUTHORITY_PRDFIX + "import")
	@RequestMapping(value = "import/template")
	public String importFileTemplate(HttpServletResponse response, RedirectAttributes redirectAttributes) {
		try {
			return championLoginService.importChampionLoginTemplate(response);
		} catch (Exception e) {
			addMessage(redirectAttributes, "导入模板下载失败！失败信息：" + e.getMessage());
		}
		return "redirect:" + Global.getAdminPath() + "/championlogin/championLogin/?repage";
	}

	/**
	 * @Description： 导入登录信息
	 * 
	 * @author: yuanshuai
	 * @date: 2019/6/14 11:18
	 */
	@RequiresPermissions(AUTHORITY_PRDFIX + "import")
	@RequestMapping(value = "import", method = RequestMethod.POST)
	public String importFile(MultipartFile file, RedirectAttributes redirectAttributes) {
		try {
			// 会员信息导入
			championLoginService.importChampionLogin(file);
			addMessage(redirectAttributes, "导入登录信息成功！");
		} catch (Exception e) {
			redirectAttributes.addFlashAttribute("importError", "导入登录信息失败！失败信息：##" + e.getMessage());
		}
		return "redirect:" + Global.getAdminPath() + "/championlogin/championLogin/?repage";
	}

}