package com.cku.catx.dog.service;

import com.cku.catx.dog.vo.DogOwnerChangeVo;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.entity.Dog;
import com.cku.oa.dog.entity.DogChipMemberLock;
import com.cku.oa.dog.entity.DogOwnerChange;
import com.cku.oa.dog.service.DogChipMemberLockService;
import com.cku.oa.dog.service.DogOwnerChangeService;
import com.cku.oa.dog.service.DogService;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.oa.sys.util.DateUtils;
import com.cku.oa.sys.util.MemberUtil;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.utils.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

/**
 * @author yuanshuai
 * @date 2021/3/3 10:52
 */
@Service
@Transactional(readOnly = true)
public class CATXDogOwnerChangeService {

	@Autowired
	private DogService dogService;
	@Autowired
	private DogOwnerChangeService dogOwnerChangeService;
	@Autowired
	private DogChipMemberLockService dogChipMemberLockService;
	@Autowired
	private MemberService memberService;

	public DogOwnerChange get(String id) {
		return dogOwnerChangeService.get(id);
	}

	public Page<DogOwnerChange> findPage(Page<DogOwnerChange> page, DogOwnerChange dogOwnerChange) {
		Page<DogOwnerChange> resultPage = dogOwnerChangeService.findPage(page, dogOwnerChange);
		for (DogOwnerChange change : resultPage.getList()) {
			Member newMember = memberService.getByMemberCode(change.getNewOwnerMemberCode());
			if (!MemberUtil.isActiveMember(newMember)) {
				change.setNewOwnerMemberId(newMember.getId());
			}
			change.setItems(dogOwnerChangeService.getPaymentChargingItemShortName(change.getDog().getDogBreed()));
			if (StringUtils.isNotBlank(change.getDog().getBreederMemberCode())
					&& change.getDog().getBreederMemberCode().equals(change.getDog().getMemberCode())
					&& DateUtils.isInXMounth(change.getDog().getBirthdate(), 12)) {
				change.setItems(change.getItems() + "," + dogOwnerChangeService.getPaymentChargingItemShortNameSale());
			}
		}
		return resultPage;
	}

	@Transactional(readOnly = false)
	public void delete(DogOwnerChange dogOwnerChange) {
		if ("2".equals(dogOwnerChange.getPaymentState())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "已缴费订单不允许删除");
		}
		dogOwnerChangeService.delete(dogOwnerChange);
	}

	/**
	 * 保存犬主变更信息
	 *
	 * @author yuanshuai
	 * @date 2021/3/3 10:56
	 */
	@Transactional(readOnly = false)
	public void save(DogOwnerChange dogOwnerChange) {
		Dog dog = dogService.get(dogOwnerChange.getDogId());
		buildPo(dogOwnerChange, dog);
		validateBusiness(dogOwnerChange, dog);
		dogOwnerChangeService.save(dogOwnerChange);
	}

	private void validateBusiness(DogOwnerChange dogOwnerChange, Dog dog) {
		//血统证书号必须以CKU开头
		if (!dog.getPedigreeCertifiedCode().toUpperCase().startsWith("CKU")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号必须为CKU证书号");
		}

		//校验会员是否为犬只芯片绑定会员
		dogChipMemberLockService.checkDogLocked(dogOwnerChange.getNewOwnerMemberCode(), dog.getIdentificationFlag());

		//血统证书无未处理的变更信息(初始化订单为未处理，故已包括订单重复校验)
		DogOwnerChange example = new DogOwnerChange();
		example.setDogId(dogOwnerChange.getDogId());
		example.setProcessState("0");
		example.setDelFlag("0");
		List<DogOwnerChange> dogOwnerChangeList = dogOwnerChangeService.findList(example);
		//排除已注销未通过的记录
		dogOwnerChangeList.removeIf(temp -> "3".equals(dogOwnerChange.getReviewState()) || "2".equals(dogOwnerChange.getReviewState()));
		if (dogOwnerChangeList.size() != 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该犬只有未处理的犬主变更申请，不允许申请犬主变更");
		}
	}

	private void buildPo(DogOwnerChange dogOwnerChange, Dog dog) {
		BigDecimal totalPrice;
		PaymentChargingItem item = dogOwnerChangeService.getPaymentChargingItem(dog.getDogBreed());
		totalPrice = BigDecimal.valueOf(Double.parseDouble(item.getPrice()));
		//原犬主为繁殖人且犬只小于1岁则有见面
		if (StringUtils.isNotBlank(dog.getBreederMemberCode())
				&& dog.getBreederMemberCode().equals(dog.getMemberCode())
				&& DateUtils.isInXMounth(dog.getBirthdate(), 12)) {
			PaymentChargingItem itemSale = dogOwnerChangeService.getPaymentChargingItemSale();
			totalPrice = totalPrice.subtract(BigDecimal.valueOf(Double.parseDouble(itemSale.getPrice())));
		}
		dogOwnerChange.setPrice(totalPrice.toString());
		//初始化
		dogOwnerChange.setReviewState("0");
		dogOwnerChange.setPaymentState("1");
		dogOwnerChange.setProcessState("0");
		dogOwnerChange.setPaymentMemberCode(dogOwnerChange.getOriginOwnerMemberCode());
	}

	/**
	 * 根据证书号查询犬只信息
	 *
	 * @author yuanshuai
	 * @date 2021/3/3 11:29
	 */
	public DogOwnerChangeVo getDogByIdentificationFlag(String identificationFlag) {
		//查询犬只信息
		Dog dog = dogService.getByIdentificationFlag(identificationFlag);
		if (dog == null || StringUtils.isBlank(dog.getId())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "芯片号有误，查询不到犬只");
		}
		DogChipMemberLock dogChipMemberLock = dogChipMemberLockService.getByChipCode(identificationFlag,"1");
		if (dogChipMemberLock == null || StringUtils.isBlank(dogChipMemberLock.getId())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "此犬只没有锁定芯片-会员信息，如需进行犬主变更，请先进行锁定");
		}
		if (dog.getMemberCode().equals(dogChipMemberLock.getMemberCode())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "原犬主与新犬主不能为同一人！");
		}
		DogOwnerChangeVo vo = new DogOwnerChangeVo();
		BeanUtils.copyProperties(dog, vo, "yyyy-MM-dd");
		// 查询原犬主
		Member member = memberService.getByMemberCode(dog.getMemberCode());
		if (member == null || StringUtils.isBlank(member.getId())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "犬主信息有误，请联系CKU处理");
		}
		vo.setMemberName(member.getName());
		//查询新犬主
		Member newMember = memberService.getByMemberCode(dogChipMemberLock.getMemberCode());
		if (newMember == null || StringUtils.isBlank(newMember.getId())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "新犬主信息有误，请联系CKU处理");
		}
		vo.setNewMemberCode(dogChipMemberLock.getMemberCode());
		vo.setNewMemberName(newMember.getName());
		return vo;
	}
}
