package com.cku.oa.sys.service;

import com.cku.core.RESTResponse;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.sys.code.SysCodeUtil;
import com.cku.oa.sys.coupon.entity.SysCoupon;
import com.cku.oa.sys.coupon.service.SysCouponService;
import com.cku.oa.sys.dao.OrgDao;
import com.cku.oa.sys.dao.user.MemberDao;
import com.cku.oa.sys.entity.Org;
import com.cku.oa.sys.entity.OrgChip;
import com.cku.oa.sys.entity.OrgDogTypeLimit;
import com.cku.oa.sys.entity.SysOrgPrint;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.entity.user.MemberType;
import com.cku.oa.sys.util.PinYinUtil;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.service.SystemService;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletRequest;
import java.util.*;

/**
 * 合作机构、合作医院Service
 *
 * @author fanhuibin
 * @version 2016-08-22
 */
@Service
@Transactional(readOnly = true)
public class OrgService extends CrudService<OrgDao, Org> {

	@Autowired
	private SystemService systemService;
	@Autowired
	private MemberDao memberDao;
	@Autowired
	private OrgChipService orgChipService;
	@Autowired
	private SysOrgPrintService sysOrgPrintService;
	@Autowired
	private OrgDogTypeLimitService orgDogTypeLimitService;
	@Autowired
	private SysCouponService sysCouponService;

	public Org get(String id) {
		return super.get(id);
	}

	public List<Org> findList(Org org) {
		return super.findList(org);
	}

	public Page<Org> findPage(Page<Org> page, Org org) {
		String chip = org.getChip();
		if (StringUtils.isNotBlank(chip)) {
			List<OrgChip> orgChip = orgChipService.getByChip(chip);
			if (orgChip.size() > 0) {
				String id = orgChip.get(0).getOrgId();
				org.setId(id);
			}
			// 如果在合作机构列表输入的芯片号查询不到记录，则列表无记录
			else {
				org.setId(" ");
			}
		}
		return super.findPage(page, org);
	}

	public Page<Org> findPageBySingleType(Page<Org> page, Org search) {
		search.setOrgType("singleTypeClub");
		Page<Org> orgPage = super.findPage(page, search);
		for (Org org : orgPage.getList()) {
			buildSingleTypeOrgData(org);
		}
		return orgPage;
	}

	/**
	 * @Description： 为单犬种俱乐部添加特殊数据
	 *
	 * @author: yuanshuai
	 * @date: 2019/10/16 16:28
	 */
	public void buildSingleTypeOrgData(Org org) {
		org.setOrgDogType(StringUtils.join(orgDogTypeLimitService.findTypeByOrgId(org.getId()), ","));
		org.setMemberCouponNum(sysCouponService.findCouponCount(org.getMemberCode(), null));
		org.setMemberCouponNumCanUse(sysCouponService.findCouponCount(org.getMemberCode(), "0"));
		org.setMemberCouponNumUsed(sysCouponService.findCouponCount(org.getMemberCode(), "2"));
	}

	public Org getByMemberCode(String memberCode) {
		return dao.getOrgByMemberCode(memberCode);
	}

	@Transactional(readOnly = false)
	public void save(Org org) {
		// 机构负责人不为空
		if (StringUtils.isNotEmpty(org.getAdminCode())) {
			Member member = memberDao.getByMemberCode(org.getAdminCode());
			if (member == null) {
				throw new ZAException(ZAErrorCode.ZA_ERC_INVALID_PARAMETER, "cku会员编号不存在");
			}
		}
		if (org.getIsNewRecord()) {
			if (UserUtils.getByLoginNameAndUserType(org.getUser().getLoginName(), "3") != null) {
				throw new ZAException(ZAErrorCode.ZA_ERC_INVALID_PARAMETER, "用户帐号已经存在");
			}
			String memberCode = SysCodeUtil.getHZMemberCode(org);
			// 同步创建用户登录帐号
			User user = buildOrgUser(org);
			// 如果是合作机构的化，还需要同步创建会员帐号
			if ("club".equals(org.getOrgType()) || "chongjihemendian".equals(org.getOrgType())
					|| "catRenZheng".equals(org.getOrgType()) || "singleTypeClub".equals(org.getOrgType())) {
				buildOrgMember(org, memberCode, user);
			}
			// 保存俱乐部信息
			org.setAddress(org.getProvince() + org.getCity() + org.getStreet());
			org.setMemberCode(memberCode);
			org.setOrgSite(StringUtils.join(org.getOrgSiteArr(), ","));
			org.preInsert();
			dao.insert(org);
			// 是俱乐部club/catRenZheng类型则增加重置打印对象
			if ("club".equals(org.getOrgType()) || "catRenZheng".equals(org.getOrgType())
					|| "singleTypeClub".equals(org.getOrgType())) {
				restOrgPrint(org, memberCode);
			}
			// 单犬种俱乐部保存会员优惠券及经营犬种
			if ("singleTypeClub".equals(org.getOrgType())) {
				saveOperatingDogType(org);
				buildMemberCoupon(org);
			}
		} else {
			// 同步修改用户信息
			User user = systemService.getUser(org.getUser().getId());
			user.setName(org.getName());
			// 修改密码
			if (!StringUtils.isEmpty(org.getUser().getPassword())) {
				user.setPassword(systemService.entryptPassword(org.getUser().getPassword()));
			}
			systemService.saveUser(user);

			// 如果是合作机构的化，还需要同步修改会员帐号
			if ("club".equals(org.getOrgType())) {
				Member m = memberDao.getByMemberCode(org.getMemberCode());
				m.setName(org.getName());
				memberDao.update(m);
			}
			// 单犬种俱乐部保存经营犬种
			if ("singleTypeClub".equals(org.getOrgType())) {
				saveOperatingDogType(org);
			}
			// 更新俱乐部信息
			org.setAddress(org.getProvince() + org.getCity() + org.getStreet());
			org.setOrgSite(StringUtils.join(org.getOrgSiteArr(), ","));
			org.preUpdate();
			dao.update(org);
		}
	}

	/**
	 * @Description：
	 *
	 * @author: yuanshuai
	 * @date: 2019/10/16 14:30
	 * @param org
	 */
	private void buildMemberCoupon(Org org) {
		int memberCouponNum;
		try {
			memberCouponNum = Integer.parseInt(org.getMemberCouponNum());
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "优惠券数量有误");
		}
		sysCouponService.saveCoupon(org.getMemberCode(), "单犬种俱乐部会员费代金券", "120", org.getEndDate(), new Date(), "member",
				null, "该代金券价值￥120.00（人民币壹佰贰拾元整），仅限抵扣会员费之用（基础会员、专业会员皆可）。", memberCouponNum, "1");
	}

	/**
	 * @Description： 保存经营犬种s
	 *
	 * @author: yuanshuai
	 * @date: 2019/10/16 13:45
	 * @param org
	 */
	@Transactional(readOnly = false)
	private void saveOperatingDogType(Org org) {
		if (StringUtils.isBlank(org.getOrgDogType())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "请选择经营犬种");
		}
		// 查找已有犬种数据
		List<String> originalList = orgDogTypeLimitService.findTypeByOrgId(org.getId());
		List<String> newList = new ArrayList<>(Arrays.asList(org.getOrgDogType().split(",")));
		// 数据去重
		List<String> retainList = new ArrayList<>(orgDogTypeLimitService.findTypeByOrgId(org.getId()));
		retainList.retainAll(newList);
		originalList.removeAll(retainList);
		newList.removeAll(retainList);
		// 删除剩余原有数据
		if (originalList.size() != 0) {
			for (String type : originalList) {
				orgDogTypeLimitService.delete(new OrgDogTypeLimit(org.getId(), type));
			}
		}
		// 保存剩余新数据
		if (newList.size() != 0) {
			for (String type : newList) {
				orgDogTypeLimitService.save(new OrgDogTypeLimit(org.getId(), type));
			}
		}
	}

	/**
	 * @Description： 重置俱乐部打印
	 *
	 * @author: yuanshuai
	 * @date: 2019/10/16 13:40
	 */
	@Transactional(readOnly = false)
	private void restOrgPrint(Org org, String memberCode) {
		SysOrgPrint sysOrgPrint = new SysOrgPrint();
		sysOrgPrint.setOrgId(org.getId());
		sysOrgPrint.setValidDate(new Date());
		sysOrgPrint.setOrgCode(memberCode);
		sysOrgPrint.setRestPrintCode("3");
		sysOrgPrintService.save(sysOrgPrint);
	}

	/**
	 * @Description： 构建登录账号
	 *
	 * @author: yuanshuai
	 * @date: 2019/10/16 13:41
	 */
	@Transactional(readOnly = false)
	private User buildOrgUser(Org org) {
		User user = org.getUser();
		user.setUserType("3"); // 3代表俱乐部系统
		user.setName(org.getName());
		user.setPassword(systemService.entryptPassword(user.getPassword()));
		systemService.saveUser(user);
		return user;
	}

	/**
	 * @Description： 构建俱乐部会员
	 *
	 * @author: yuanshuai
	 * @date: 2019/10/16 13:34
	 */
	@Transactional(readOnly = false)
	private void buildOrgMember(Org org, String memberCode, User user) {
		Member member = new Member();
		member.setUser(user);
		member.setName(user.getName());
		member.setNameEn(PinYinUtil.getFullSpell(member.getName()));
		member.setMemberType(MemberType.basic.toString());// 基础会员
		member.setGender("male");// 默认初始化为male
		member.setPaymentState("1"); // 未付费
		member.setMemberEndDate(null);
		member.setKuFlag(null);
		member.setKuEndDate(null);
		member.setGroomerType(null);
		member.setGroomerEndDate(null);
		member.setDgMemberType(null);
		member.setDgEndDate(null);
		member.setAccountBalance("0");
		member.setMemberCode(memberCode);
		member.setProvince(org.getProvince());
		member.setCity(org.getCity());
		member.setStreet(org.getStreet());
		member.preInsert();
		memberDao.insert(member);
	}

	@Transactional(readOnly = false)
	public void singleTypeOrgsave(Org org) {

	}

	@Transactional(readOnly = false)
	public void delete(Org org) {
		super.delete(org);
	}

	@Transactional(readOnly = false)
	public void update(Org org) {
		org.preUpdate();
		dao.updateAddress(org);
	}

	@Transactional(readOnly = false)
	public void updateAdminCode(Org org) {
		org.preUpdate();
		dao.updateAdminCode(org);
	}

	/**
	 *
	 * @Description：根据芯片号获取上传该芯片的合作机构名称
	 * @author: zhuoHeng
	 * @version: 2017年1月4日 下午3:12:35
	 */
	public RESTResponse getClubNameByChip(String chip) {
		List<Org> org = dao.getClubNameByChip(chip);
		Map<String, String> map = new HashMap<>();
		if (org.size() > 0) {
			map.put("name", org.get(0).getName());
			map.put("memberCode", org.get(0).getMemberCode());
		} else {
			map.put("name", "");
			map.put("memberCode", "");
		}
		return new RESTResponse("data", map);
	}

	/**
	 * @Description: 查询合作机构省份
	 * @author: zhangxiang
	 * @date: 2019/7/12 12:08
	 */
	public List<Org> findProvince(String orgType) {
		return dao.findProvince(orgType);
	}

	/**
	 * @Description： 修改俱乐部优惠券数量
	 *
	 * @author: yuanshuai
	 * @date: 2019/10/18 15:54
	 */
	@Transactional(readOnly = false)
	public RESTResponse changeCouponCount(HttpServletRequest request) {
		String id = request.getParameter("id");
		String type = request.getParameter("type");
		String countStr = request.getParameter("count");
		if (StringUtils.isBlank(type)) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "修改优惠券类型有误，请重试");
		}
		if (StringUtils.isBlank(id)) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "俱乐部信息有误，请重试");
		}
		if (StringUtils.isBlank(countStr)) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "优惠券变更数量有误，请重试");
		}
		int count;
		try {
			count = Integer.parseInt(countStr);
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "优惠券变更数量有误，请重试");
		}
		if (count <= 0) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "优惠券变更数量有误，请重试");
		}
		Org org = get(id);
		if (org == null) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "俱乐部信息有误，请重试");
		}
		if ("up".equals(type)) {
			sysCouponService.saveCoupon(org.getMemberCode(), "单犬种俱乐部会员优惠券", "120", org.getEndDate(), new Date(),
					"member", null, "限" + org.getName() + "单犬种俱乐部发展会员优惠券", count, "3");
		} else if ("down".equals(type)) {
			List<SysCoupon> couponList = sysCouponService.findCouponList(org.getMemberCode(), "0");
			if (count > couponList.size()) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "优惠券删除数量不能大于当前可用优惠券数量");
			}
			int i = 0;
			for (SysCoupon coupon : couponList) {
				if (i >= count) {
					break;
				}
				sysCouponService.delete(coupon);
				i++;
			}
		}
		return new RESTResponse();
	}

	public List<Org> findAll() {
		List<Org> orgList = dao.findAll();
		return orgList;
	}

	@Transactional(readOnly = false)
	public void dogNoseEdit(String memberCode, String dogNoseRepeatedDetection, String dogNoseLivenessDetection) {
		Org member = this.getByMemberCode(memberCode);
		if (member == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "输入的会员信息有误,修改失败");
		}
		dao.updateDogNoseEdit(memberCode, dogNoseRepeatedDetection, dogNoseLivenessDetection);
	}

	@Transactional(readOnly = false)
	public void updatePasswordById(String id, String newPassword) {
		User user = systemService.getUser(id);
		// 修改密码
		if (!StringUtils.isEmpty(newPassword)) {
			user.setPassword(systemService.entryptPassword(newPassword));
		}
		systemService.saveUser(user);
	}
}