package com.cku.oa.appreciation.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.appreciation.dao.AppreciationCoursesDao;
import com.cku.oa.appreciation.entity.AppreciationCourses;
import com.cku.oa.appreciation.vo.AppreciationCoursesExport;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.excel.ExportExcel;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 鉴赏课Service
 *
 * @author xx
 * @version 2022-02-17
 */
@Service
@Transactional(readOnly = true)
public class AppreciationCoursesService extends CrudService<AppreciationCoursesDao, AppreciationCourses> {

	@Autowired
	private AppreciationApplyService appreciationApplyService;

	public AppreciationCourses get(String id) {
		return super.get(id);
	}

	public List<AppreciationCourses> findList(AppreciationCourses appreciationCourses) {
		return super.findList(appreciationCourses);
	}

	public List<AppreciationCourses> findListByApplyMobile(String mobile) {
		return dao.findListByApplyMobile(mobile);
	}

	public Page<AppreciationCourses> findPage(Page<AppreciationCourses> page, AppreciationCourses appreciationCourses) {
		return super.findPage(page, appreciationCourses);
	}

	@Transactional(readOnly = false)
	public void save(AppreciationCourses appreciationCourses) {
		super.save(appreciationCourses);
	}

	@Transactional(readOnly = false)
	public void delete(AppreciationCourses appreciationCourses) {
		super.delete(appreciationCourses);
	}

	/**
	 * 添加课程
	 *
	 * @author yuanshuai
	 * @date 2022/2/18 11:36
	 */
	@Transactional(readOnly = false)
	public void addCourses(AppreciationCourses courses) {
		// 默认报名中
		courses.setStates("0");
		// 保存
		save(courses);
	}

	/**
	 * 结束课程
	 *
	 * @author yuanshuai
	 * @date 2022/2/18 13:52
	 */
	@Transactional(readOnly = false)
	public void endCourses(AppreciationCourses courses) {
		// 判断是否有人报名
		if (0 == courses.getCourseApplyCount()) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "课程没有报名信息，不可结束课程");
		}
		// 判断是否大于截止日期
		if (courses.getEndDate().compareTo(new Date()) >= 0) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "未到结课日期，不可结束课程");
		}
		//创建证书号
		appreciationApplyService.createCertificateCodeByCourses(courses.getId());
		// 修改课程状态
		dao.endCourses(courses);
	}

	/**
	 * 校验参数
	 *
	 * @author yuanshuai
	 * @date 2022/2/18 14:09
	 */
	public void validateParam(AppreciationCourses courses) {
		if (!courses.getCourseCode().matches("^[0-9A-Za-z]{8}$")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "编号只能为8位数字或字母");
		}
		if (courses.getStartDate().compareTo(courses.getEndDate()) > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "结束日期不可小于开始日期");
		}
	}

	/**
	 * 业务校验
	 * 
	 * @author yuanshuai
	 * @date 2022/2/24 17:24
	 */
	public void validateBusiness(AppreciationCourses courses) {
		if (Objects.nonNull(courses.getCourseApplyCount()) && courses.getMaxNum() < courses.getCourseApplyCount()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "报名上限不可低于已报名人数");
		}
		if ("2".equals(courses.getStates())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "课程结束后不可编辑");
		}
	}

	/**
	 * 导出课程信息
	 *
	 * @author yuanshuai
	 * @date 2022/2/18 15:20
	 */
	public void export(AppreciationCourses courses, HttpServletResponse response) throws IOException {
		List<AppreciationCourses> list = dao.findList(courses);
		List<AppreciationCoursesExport> voList = list.stream().map(o -> {
			AppreciationCoursesExport vo = new AppreciationCoursesExport();
			BeanUtils.copyProperties(o, vo);
			vo.setStates(DictUtils.getDictLabel(vo.getStates(), "appreciation_courses_state", null));
			return vo;
		}).collect(Collectors.toList());
		// 导出
		new ExportExcel("鉴赏课程", AppreciationCoursesExport.class, 2).setDataList(voList).write(response, "鉴赏课程.xlsx")
				.dispose();
	}

	/**
	 * 删除课程
	 * 
	 * @author yuanshuai
	 * @date 2022/2/18 16:57
	 */
	@Transactional(readOnly = false)
	public void deleteCourses(AppreciationCourses courses) {
		if (Objects.nonNull(courses.getCourseApplyCount()) && courses.getCourseApplyCount() > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "已报名课程无法删除");
		}
		delete(courses);
	}

}