/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.thinkgem.jeesite.common.service;

import com.thinkgem.jeesite.common.persistence.CrudDao;
import com.thinkgem.jeesite.common.persistence.DataEntity;
import com.thinkgem.jeesite.common.persistence.Page;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * Service基类
 * @author ThinkGem
 * @version 2014-05-16
 */
@Transactional(readOnly = true)
public abstract class CrudService<D extends CrudDao<T>, T extends DataEntity<T>> extends BaseService {
	
	/**
	 * 持久层对象
	 */
	@Autowired
	protected D dao;
	
	/**
	 * 获取单条数据
	 * @param id
	 * @return
	 */
	public T get(String id) {
		return dao.get(id);
	}
	
	/**
	 * 获取单条数据
	 * @param entity
	 * @return
	 */
	public T get(T entity) {
		return dao.get(entity);
	}
	
	/**
	 * 查询列表数据
	 * @param entity
	 * @return
	 */
	public List<T> findList(T entity) {
		return dao.findList(entity);
	}
	
	/**
	 * 查询分页数据
	 * @param page 分页对象
	 * @param entity
	 * @return
	 */
	public Page<T> findPage(Page<T> page, T entity) {
		int pageNo = page.getPageNo();
		int pageSize = page.getPageSize();
		entity.setPage(page);
		page.setList(dao.findList(entity));
		if((pageNo-1)*pageSize>=page.getCount()){
			page.setList(new ArrayList<T>(0));
		}
		return page;
	}
	public Page<T> findListApp(Page<T> page, T entity) {
		int pageNo = page.getPageNo();
		int pageSize = page.getPageSize();
		entity.setPage(page);
		page.setList(dao.findListApp(entity));
		if((pageNo-1)*pageSize>=page.getCount()){
			page.setList(new ArrayList<T>(0));
		}
		return page;
	}

	public Page<T> findPages(Page<T> page, T entity) {
		int pageNo = page.getPageNo();
		int pageSize = page.getPageSize();
		entity.setPage(page);
		page.setList(dao.findLists(entity));
		if((pageNo-1)*pageSize>=page.getCount()){
			page.setList(new ArrayList<T>(0));
		}
		return page;
	}
	
	
	/**
	 * 查询活动能报名的分页数据
	 * @Author chaixueteng
	 * @2016年7月13日上午11:03:30
	 */
	public Page<T> findRegisterPage(Page<T> page, T entity){
		entity.setPage(page);
		page.setList(dao.findRegisterList(entity));
		return page;
	}
	
	
	
	
	/**
	 * 保存数据（插入或更新）
	 * @param entity
	 */
	@Transactional(readOnly = false)
	public void save(T entity) {
		if (entity.getIsNewRecord()){
			entity.preInsert();
			dao.insert(entity);
		}else{
			entity.preUpdate();
			dao.update(entity);
		}
	}
	
	/**
	 * 删除数据
	 * @param entity
	 */
	@Transactional(readOnly = false)
	public void delete(T entity) {
		entity.preUpdate();
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		entity.setRemarks("["+entity.getUpdateBy()+":"+format.format(entity.getUpdateDate())+"]");
		dao.delete(entity);
	}
}
