/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.restful.v1.kennel.service;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.RESTResponse;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.finance.dao.PaymentOrderDao;
import com.cku.oa.finance.dao.PaymentOrderDetailDao;
import com.cku.oa.finance.entity.PaymentCart;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.finance.entity.PaymentOrder;
import com.cku.oa.finance.entity.PaymentOrderDetail;
import com.cku.oa.finance.service.PaymentCartService;
import com.cku.oa.finance.service.PaymentChargingItemGroupService;
import com.cku.oa.finance.service.PaymentChargingItemService;
import com.cku.oa.kennel.dao.KennelDao;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.kennel.service.KennelService;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.util.DateUtils;
import com.cku.oa.sys.util.MemberUtil;
import com.cku.restful.v1.finance.service.KennelRegisterDeleteService;
import com.cku.restful.v1.sys.model.RestKennel;
import com.cku.restful.v1.sys.utils.BeanUtil;
import com.thinkgem.jeesite.common.utils.IdGen;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import net.sf.json.JSONObject;

/**
 * 犬舍基本信息Service
 * @author fanhuibin
 * @version 2016-07-13
 */
@Service
@Transactional(readOnly = true)
public class RestKennelService {

    public static final String KENNEL_ZC = "kennel@ZC";//犬舍名称初次注册费
    public static final String KENNEL_ND = "kennel@ND";//犬舍年度登记费
    public static final String KENNEL_YJ = "kennel@YJ";//永久犬舍名称登记费

	@Autowired
	private KennelService kennelService;
	@Autowired
	private PaymentChargingItemService paymentChargingItemService;
	@Autowired
	private PaymentCartService paymentCartService;
	@Autowired
	public PaymentOrderDao paymentOrderDao;
	@Autowired
	public PaymentOrderDetailDao paymentOrderDetailDao;
	@Autowired
	public KennelRegisterDeleteService kennelRegisterDeleteService;
	@Autowired
	public PaymentChargingItemGroupService paymentChargingItemGroupService;

	public RESTResponse get(){
		Member member = UserUtils.getLoginMember();
		Kennel kennel = kennelService.getKennelByMemberCode(member.getMemberCode());
		RestKennel vo = null;
		if(kennel!=null){
			vo = new RestKennel();
			BeanUtil.copyProperties(vo, kennel, "yyyy-MM-dd");
			vo.setMemberName(member.getName());
			String memberType = "";
			if("basic".equals(member.getMemberType())){
				memberType = "基础会员";
			}else if("professional".equals(member.getMemberType())){
				memberType = "专业会员";
			}
			vo.setMemberType(memberType);
			String memberEndTime = "";
			if(member.getMemberEndDate()!=null){
				memberEndTime = new SimpleDateFormat("yyyy-MM-dd").format(member.getMemberEndDate());
			}
			vo.setMemberEndTime(memberEndTime);
			PaymentChargingItem paymentChargingItem = paymentChargingItemService.getByShortName("kennel_rename@BG");
			vo.setPrice(paymentChargingItem.getPrice());
		}
		return new RESTResponse("data",vo);
	}

	public RESTResponse checkKennelName(Member member,Kennel vo){
		//获取参数
		if(StringUtils.isBlank(vo.getIsCheckType())){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍校验类型不能为空");
		}
		if("name".equals(vo.getIsCheckType())){
			checkKennelName(vo,member,false);
		}else if("nameEn".equals(vo.getIsCheckType())){
			checkKennelNameEn(vo, member,false);
		}else if("fciName".equals(vo.getIsCheckType())){
			checkKennelFCIName(vo, member,false);
		}else if("fciNameBackup".equals(vo.getIsCheckType())){
			checkKennelFCIBack(vo, member,false);
		}
		//业务校验
		return new RESTResponse("data","true");
	}
	
	/**
	 * 校验犬舍名称
	 * @return
     */
	public RESTResponse checkKennelName(Kennel vo){
		//获取参数
		Member member = UserUtils.getLoginMember();

		if(StringUtils.isBlank(vo.getIsCheckType())){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍校验类型不能为空");
		}
		if("name".equals(vo.getIsCheckType())){
			checkKennelName(vo,member,false);
		}else if("nameEn".equals(vo.getIsCheckType())){
			checkKennelNameEn(vo, member,false);
		}else if("fciName".equals(vo.getIsCheckType())){
			checkKennelFCIName(vo, member,false);
		}else if("fciNameBackup".equals(vo.getIsCheckType())){
			checkKennelFCIBack(vo, member,false);
		}
		//业务校验
		return new RESTResponse("data","true");
	}
	/**
	 * 校验犬舍名称
	 * @return
     */
	public RESTResponse checkKennelName(String json){
		//获取参数
		Kennel vo = BeanUtil.toBean(json, Kennel.class);
		Member member = UserUtils.getLoginMember();

		if(StringUtils.isBlank(vo.getIsCheckType())){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍校验类型不能为空");
		}
		if("name".equals(vo.getIsCheckType())){
			checkKennelName(vo,member,false);
		}else if("nameEn".equals(vo.getIsCheckType())){
			checkKennelNameEn(vo, member,false);
		}else if("fciName".equals(vo.getIsCheckType())){
			checkKennelFCIName(vo, member,false);
		}else if("fciNameBackup".equals(vo.getIsCheckType())){
			checkKennelFCIBack(vo, member,false);
		}
		//业务校验
		return new RESTResponse("data","true");
	}

	/**
	 * 判断犬舍FCI备用名是否符合规则
	 * @author yuanshuai
	 * @data 2017年3月27日 下午4:16:19
	 */
	private void checkKennelFCIBack(Kennel vo, Member member,boolean notNull) {
		vo.setFciNameBackup(vo.getFciNameBackup().trim());
		//空
		if(!notNull){
			if(StringUtils.isBlank(vo.getFciNameBackup())){
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"FCI英文备用名不能为空");
			}
		}
		//只能为英文
		String str = vo.getFciNameBackup();
		boolean isWord=str.matches("^[a-zA-Z ]+$");
		if(!isWord){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"FCI英文备用名称只能为英文");
		}
		//重复
		if(kennelDao.countByName("fciNameBackup",vo.getFciNameBackup(),member.getMemberCode())>0){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "FCI英文备用名重复");
		}
	}

	/**
	 * 判断犬舍FCI名是否符合规则
	 * @author yuanshuai
	 * @data 2017年3月27日 下午4:15:23
	 */
	private void checkKennelFCIName(Kennel vo, Member member,boolean notNull) {
		vo.setFciName(vo.getFciName().trim());
		//空
		if(!notNull){
			if(StringUtils.isBlank(vo.getFciName())){
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"FCI犬舍名不能为空");
			}
		}
		//只能为英文
		String str = vo.getFciName();
		boolean isWord=str.matches("^[a-zA-Z ]+$");
		if(!isWord){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"FCI犬舍名只能为英文");
		}
		//重复
		if(kennelDao.countByName("fciName",vo.getFciName(),member.getMemberCode())>0){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "FCI犬舍名重复");
		}
	}

	/**
	 * 判断犬舍英文名是否符合规则
	 * @author yuanshuai
	 * @data 2017年3月27日 下午4:12:32
	 */
	private void checkKennelNameEn(Kennel vo, Member member,boolean notNull) {
		vo.setNameEn(vo.getNameEn().trim());
		//空
		if(!notNull){
			if(StringUtils.isBlank(vo.getNameEn())){
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍英文名称不能为空");
			}
		}
		//只能为英文
		String str = vo.getNameEn();
		boolean isWord=str.matches("^[a-zA-Z ]+$");
		if(!isWord){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍英文名称只能为英文");
		}
		//重复
		if(kennelDao.countByName("nameEn",vo.getNameEn(),member.getMemberCode())>0){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍英文名重复");
		}
	}

	/**
	 * 判断犬舍中文名是否符合规则
	 * @author yuanshuai
	 * @data 2017年3月27日 下午4:03:03
	 */
	private void checkKennelName(Kennel vo,Member member,boolean notNull) {
		vo.setName(vo.getName().trim());
		//空
		String str = vo.getName();
		if(!notNull){
			if(StringUtils.isBlank(str)){
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍名称不能为空");
			}
		}
		// 不能有空格
		if (str.contains(" ")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中文名不能有空格");
		}
		// 只能中文
		String regex = "[\\u4E00-\\u9FA5]+";
		if (!str.matches(regex)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中文名只能是中文");
		}
		// 只能2-4个字
		if (str.length() < 2 || str.length() > 4) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中文名只能是2-4个字");
		}
		// 包含犬舍二字
		if (str.lastIndexOf("犬舍") != -1) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中文名不可包含犬舍二字");
		}
		// 繁体
		String encode = "GB2312";
		String nowStr = "";
		try {
			nowStr = new String(str.getBytes(encode), encode);
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中文名不能有繁体字");
		}
		if (!str.equals(nowStr)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中文名不能有繁体字");
		}
		//重复
		if(kennelDao.countByName("name",str+"犬舍",member.getMemberCode())>0){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中文名重复");
		}
	}


	@Transactional(readOnly = false)
	public RESTResponse post(String json){
		Member member = UserUtils.getLoginMember();
		//获取参数
		Kennel vo = BeanUtil.toBean(json, Kennel.class);
		//参数校验
		if(StringUtils.isBlank(vo.getType())){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍类型为空");
		}
		if(!"year".equals(vo.getType())&&!"permanent".equals(vo.getType())){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍类型有误");
		}
		//校验犬舍中文名
		checkKennelName(vo,member,false);
		//校验犬舍英文名
		checkKennelNameEn(vo, member,false);
		//永久犬舍判断fci名
		if("permanent".equals(vo.getType())){
			//校验犬舍fci名
			checkKennelFCIName(vo, member,false);
			//校验犬舍fci备用名
			checkKennelFCIBack(vo, member,false);
		}

		Kennel po = buildPo(vo);
		//2017-05-05增加逻辑
		//待柴雪腾的微信支付和阿里支付增加订单超时时间功能上线后去掉
		Kennel kennel = kennelDao.getDeleteRegisterKennelByMemberCode(po.getMemberCode());
		if(kennel!=null){
			//如果有标记为已删除的犬舍信息，物理删除该记录
			kennelRegisterDeleteService.deleteKennelRegister(kennel.getId());
		}
		//业务校验
		kennelService.validateBusiness(po);
		//写入业务表
		kennelService.save(po);
		//犬舍注册加入购物车
		String cartIds = registerAddCart(po);
		return new RESTResponse("data",cartIds);
	}
	/**
	 *
	 * @description: 犬舍注册加入购物车
	 * @author: laiguanglong
	 * @date: 2016年11月1日 下午1:17:18
	 */
	private String registerAddCart(Kennel po){
		//业务批次号
		String businessOrderCode = IdGen.uuid();
		//年度或永久费
		String shortName = null;
		if("year".equals(po.getType())){
			shortName = KENNEL_ND;
		}else if("permanent".equals(po.getType())){
			shortName = KENNEL_YJ;
		}
		String businessId = KENNEL_YJ.equals(shortName) ? "1023" : "1022";
		//初次注册费
		PaymentCart registerPaymentCart = buildPaymentCart(po, KENNEL_ZC, businessOrderCode);
		paymentCartService.addToCart(registerPaymentCart, businessId);
		
		PaymentCart typePaymentCart = buildPaymentCart(po,shortName,businessOrderCode);
		paymentCartService.addToCart(typePaymentCart, businessId);
		return registerPaymentCart.getId()+","+typePaymentCart.getId();
	}

	@Transactional(readOnly = false)
	public RESTResponse put(String json) {
		Member member = UserUtils.getLoginMember();
		// json转换为对象
		Kennel vo = BeanUtil.toBean(json, Kennel.class);
		// 业务校验
		Kennel kennel = kennelService.getKennelByMemberCode(member.getMemberCode());

		if(StringUtils.isNotBlank(vo.getName())&&StringUtils.isNotBlank(vo.getNameEn())){
			if ("0".equals(kennel.getReviewState())||"2".equals(kennel.getReviewState())) {
				//中文名校验
				checkKennelName(vo,member,true);
				kennel.setName(vo.getName() + "犬舍");
				//英文名校验
				checkKennelNameEn(vo, member,true);
				kennel.setNameEn(vo.getNameEn());
				//审核不通过的修改后改为未审核
				if("2".equals(kennel.getReviewState())){
					kennel.setReviewState("0");
				}
			}else if ("1".equals(kennel.getReviewState())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中英文名已审核通过，不允许修改");
			}
		}

		if(StringUtils.isNotBlank(vo.getFciName())&&StringUtils.isNotBlank(vo.getFciNameBackup())){
			//永久犬舍FCI名校验
			if ("0".equals(kennel.getFciReviewState())||"2".equals(kennel.getFciReviewState())) {
				//FCI名校验
				checkKennelFCIName(vo, member,true);
				kennel.setFciName(vo.getFciName());
				//FCI备用名校验
				checkKennelFCIBack(vo, member,true);
				kennel.setFciNameBackup(vo.getFciNameBackup());
				//审核不通过的修改后改为未审核
				if("2".equals(kennel.getFciReviewState())){
					kennel.setFciReviewState("0");
				}
			}else if ("1".equals(kennel.getFciReviewState())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍FCI名和FCI备用名已审核通过，不允许修改");
			}
		}

		// 更新,并且更新后改为待审核
		kennelService.update(kennel);
		return new RESTResponse();
	}

	/**
	 *
	 * @description: 参数校验
	 * @author: laiguanglong
	 * @date: 2016年9月12日 上午10:49:07
	 */
	private void validateParameter(Kennel vo){
		if(StringUtils.isBlank(vo.getName())){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍中文名为空");
		}
		if(StringUtils.isBlank(vo.getNameEn())){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍英文名为空");
		}
		//如果是永久犬舍，fciName不能为空
		if("permanent".equals(vo.getType())){
			if(StringUtils.isBlank(vo.getFciName())){
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍FCI名为空");
			}
			if(StringUtils.isBlank(vo.getFciNameBackup())){
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍FCI备用名为空");
			}
		}
	}


	private Kennel buildPo(Kennel vo){
		Member member = UserUtils.getLoginMember();
		Kennel po = new Kennel();
		po.setMemberCode(member.getMemberCode());
		po.setKennelCode(member.getMemberCode());
		po.setName(vo.getName());
		po.setNameEn(vo.getNameEn());
		String fciName = "";
		String fciNameBackup = "";
		//只有年度犬舍注册时写入fci名
		if("permanent".equals(vo.getType())){
			fciName = vo.getFciName();
			fciNameBackup = vo.getFciNameBackup();
		}
		po.setFciName(fciName);
		po.setFciNameBackup(fciNameBackup);
		po.setReviewState("0");
		po.setFciReviewState("0");
		po.setPrintState("0");
		po.setFciPrintState("0");
		po.setCalculationFlag("0");  //自17年2月1日之后注册的 都需参与核算
		po.setCalculationState("0"); //默认待核算
		po.setToken(vo.getToken());
		if("permanent".equals(vo.getType())){
			po.setType("permanent");
		}else if("year".equals(vo.getType())){
			po.setType("year");
		}

		return po;
	}

	private PaymentCart buildPaymentCart(Kennel po,String shortName,String businessOrderCode) {
		PaymentCart paymentCart = new PaymentCart();
		PaymentChargingItem paymentChargingItem = paymentChargingItemService.getByShortName(shortName);
		paymentCart.setChargingItemId(paymentChargingItem.getId());
		paymentCart.setChargingItemName(paymentChargingItem.getName());
		paymentCart.setNum("1");
		paymentCart.setPrice(paymentChargingItem.getPrice());
		paymentCart.setTotalPrice(paymentChargingItem.getPrice());
		paymentCart.setBusinessTable("kennel");
		paymentCart.setBusinessIds(po.getId());
		paymentCart.setBusinessOrderCode(businessOrderCode);
		paymentCart.setToken(po.getToken());
		return paymentCart;
	}

	@Transactional(readOnly = false)
	public RESTResponse renew(String json){
		Member member = UserUtils.getLoginMember();
		Kennel po = kennelService.getKennelByMemberCode(member.getMemberCode());
		StringBuffer sb = new StringBuffer();
		if(po==null){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"没有注册犬舍，不能进行续费");
		}
		if(po.getEndTime()==null){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"犬舍截止日期为空，不能进行续费");
		}
		if(!MemberUtil.isActiveMember(member)){
			throw new ZAException(ZAErrorCode.ZA_ERC_MEMBER_INVALID,"该业务仅限有效专业会员办理");
		}
		if("3".equals(po.getReviewState())){
			throw new ZAException(ZAErrorCode.ZA_ERC_MEMBER_INVALID,"该业务仅限未冻结的犬舍办理");
		}
		//参与核算且核算状态为已通过的，且fci名为空的需要写入fci名
		if(isNeedFciInfo(po)){
			setFciInfo(json,member,po);
		}
		//业务查重
		List<PaymentCart> paymentCartList =  paymentCartService.getMyCart();
		PaymentChargingItem paymentChargingItem = paymentChargingItemService.getByShortName(KENNEL_ND);
		for(PaymentCart paymentCart:paymentCartList){
			if(paymentChargingItem.getId().equals(paymentCart.getChargingItemId())){
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"购物车已有犬舍续费订单，请勿重复提交");
			}
		}
		//业务批次号
		String businessOrderCode = IdGen.uuid();
		//续费
		PaymentCart typePaymentCart = buildPaymentCart(po,KENNEL_ND,businessOrderCode);
		paymentCartService.addToCart(typePaymentCart, "1022");
		sb.append(typePaymentCart.getId());
		Date now = new Date();
		//过期超过一个月收初次注册费
		if(now.after(po.getEndTime())&&!DateUtils.isInXMounth(po.getEndTime(),1)){
			//初次注册费
			PaymentCart registerPaymentCart = buildPaymentCart(po,KENNEL_ZC,businessOrderCode);
			paymentCartService.addToCart(registerPaymentCart, "1022");
			sb.append(","+registerPaymentCart.getId());
		}
		return new RESTResponse("data",sb.toString());
	}

	@Transactional(readOnly = false)
	public RESTResponse upgrade(String json){
		Member member = UserUtils.getLoginMember();
		Kennel po = kennelService.getKennelByMemberCode(member.getMemberCode());
		StringBuffer sb = new StringBuffer();
		if(po==null){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"没有注册犬舍，不能进行升级");
		}
		if("1".equals(member.getRegisterFlag())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"繁育专业会员，不能进行升级");
		}
		if(!"year".equals(po.getType())){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"只有年度犬舍才能进行升级");
		}
		if(po.getEndTime()==null){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"新注册的年度犬舍尚未缴费，无法升级");
		}
		if(!MemberUtil.isActiveMember(member)){
			throw new ZAException(ZAErrorCode.ZA_ERC_MEMBER_INVALID,"该业务仅限有效专业会员办理");
		}
		if("3".equals(po.getReviewState())){
			throw new ZAException(ZAErrorCode.ZA_ERC_MEMBER_INVALID,"该业务仅限未冻结的犬舍办理");
		}
		//只要是犬舍升级成永久犬舍,
		// 且FCI信息都不为空（年度犬舍，续费填写FCI信息，然后在进行升级，此时已有FCI信息，所以不允许在升级的时候修改FCI信息）
		// 就必须输入fci信息
		if(StringUtils.isBlank(po.getFciName())||StringUtils.isBlank(po.getFciNameBackup())){
			setFciInfo(json,member,po);
		}
		//业务查重
		List<PaymentCart> paymentCartList =  paymentCartService.getMyCart();
		PaymentChargingItem paymentChargingItem = paymentChargingItemService.getByShortName(KENNEL_YJ);
		for(PaymentCart paymentCart:paymentCartList){
			if(paymentChargingItem.getId().equals(paymentCart.getChargingItemId())){
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"购物车已有犬舍升级订单，请勿重复提交");
			}
		}
		//订单查重
		PaymentOrder paymentOrderExample = new PaymentOrder();
		paymentOrderExample.setMemberCode(member.getMemberCode());
		paymentOrderExample.setPaymentState("1");
		List<PaymentOrder> paymentOrderList = paymentOrderDao.findList(paymentOrderExample);
		for(PaymentOrder paymentOrder:paymentOrderList){
			List<PaymentOrderDetail> paymentOrderDetailList = paymentOrderDetailDao.getOrderByRunningNum(paymentOrder.getOrderCode(),paymentOrder.getMemberCode());
			for(PaymentOrderDetail paymentOrderDetail:paymentOrderDetailList){
				if(paymentChargingItem.getId().equals(paymentOrderDetail.getChargingItemId())){
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"订单中已有犬舍升级订单，请勿重复提交");
				}
			}
		}
		//业务批次号
		String businessOrderCode = IdGen.uuid();
		// 写入购物车
		PaymentCart paymentCart = buildPaymentCart(po,KENNEL_YJ,businessOrderCode);
		paymentCartService.addToCart(paymentCart, "1023");
		sb.append(paymentCart.getId());

		//过期超过一个月收初次注册费
		if(po.getEndTime()==null||(new Date().after(po.getEndTime())&&!DateUtils.isInXMounth(po.getEndTime(),1))){
			//初次注册费
			PaymentCart registerPaymentCart = buildPaymentCart(po,KENNEL_ZC,businessOrderCode);
			paymentCartService.addToCart(registerPaymentCart, "1023");
			sb.append(","+registerPaymentCart.getId());
		}
		return new RESTResponse("data",sb.toString());
	}

	@Autowired
	private KennelDao kennelDao;
	
	/**
	 * @description: CKU官网及APP首页 根据犬舍名称查询犬舍信息的接口
	 */
	public RESTResponse getByName(HttpServletRequest request,HttpServletResponse response){
		/** 自 2021-06-09 将官网及APP首页犬舍查询，统一接口逻辑 **/
		return kennelService.kennelList(request,response,Boolean.FALSE);
	}

	@Transactional(readOnly = false)
	public RESTResponse addCart(){
		Member member = UserUtils.getLoginMember();
		if(!MemberUtil.isActiveMember(member)){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该业务仅限有效专业会员办理");
		}
		Kennel kennel = kennelService.getKennelByMemberCode(member.getMemberCode());
		if("1".equals(getAddCarFlag(member, kennel))){
			//犬舍注册加入购物车
			registerAddCart(kennel);
		}
		return new RESTResponse();
	}

	/**
	 *
	 * @description: 犬舍注册标记位
	 * @author: laiguanglong
	 * @date: 2016年11月1日 下午12:32:57
	 */
	public String getAddCarFlag(Member member,Kennel kennel){
		//能否加入购物车标志
		String addCarFlag = "0";
		if(kennel.getEndTime()==null){
			addCarFlag = "1";
			//截至日期为空表示未缴费
			//判断购物车和订单里是否有续费订单，如果没有就标记为1表示可以加入购物车
			List<PaymentCart> carts = paymentCartService.getMyCart(false);
			for(PaymentCart cart:carts){
				//收费项为5的是犬舍名称初次注册费
				if("5".equals(cart.getChargingItemId())){
					addCarFlag = "0";
					break;
				}
			}
			if("1".equals(addCarFlag)){
				//查询未缴费订单
				PaymentOrder example = new PaymentOrder();
				example.setMemberCode(member.getMemberCode());
				example.setPaymentState("1");
				example.setClientFlag("true");
				List<PaymentOrder> paymentOrderList = paymentOrderDao.findList(example);
				for(PaymentOrder paymentOrder:paymentOrderList){
					PaymentOrder order = new PaymentOrder();
					order.setMemberCode(member.getMemberCode());
					order.setOrderCode(paymentOrder.getOrderCode());
					List<PaymentOrderDetail> orderDetailList = paymentOrderDetailDao.findList(new PaymentOrderDetail(order));
					for(PaymentOrderDetail detail:orderDetailList){
						if("5".equals(detail.getChargingItemId())){
							addCarFlag = "0";
							break;
						}
					}
					if("0".equals(addCarFlag)){
						break;
					}
				}
			}
		}
		return addCarFlag;
	}

	/**
	 *
	 * @Description：犬舍名称校验
	 * @author: zhuoHeng
	 * @version: 2016年12月2日 下午3:57:05
	 */
	public RESTResponse kennelValidata(HttpServletRequest request) {
		String name = request.getParameter("name");
		String nameEn = request.getParameter("nameEn");
		String fciName = request.getParameter("fciName");
		String fciNameBackup = request.getParameter("fciNameBackup");
		Member member = UserUtils.getLoginMember();
		if(StringUtils.isNotBlank(name)&&kennelDao.countByName("name",name + "犬舍",member.getMemberCode())>0){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中文名重复");
		}
		if(StringUtils.isNotBlank(nameEn)&&kennelDao.countByName("nameEn",nameEn,member.getMemberCode())>0){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍英文名重复");
		}
		if(StringUtils.isNotBlank(fciName)&&kennelDao.countByName("fciName",fciName,member.getMemberCode())>0){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "fci国际犬舍名重复");
		}
		if(StringUtils.isNotBlank(fciNameBackup)&&kennelDao.countByName("fciNameBackup",fciNameBackup,member.getMemberCode())>0){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍备用名重复");
		}
		return new RESTResponse();
	}

	/**
	 *
	 * @Description：根据shortName获取价格
	 * @author: zhuoHeng
	 * @version: 2016年12月2日 下午4:23:16
	 */
	public RESTResponse getPrice(HttpServletRequest request) {
		// 获取参数
		String shortName = request.getParameter("shortName");
		if (StringUtils.isBlank(shortName)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "参数为空");
		}
		// 获取价格
		PaymentChargingItem paymentChargingItem = paymentChargingItemService
				.getByShortName(shortName);
		JSONObject jo = new JSONObject();
		jo.put("price", paymentChargingItem.getPrice());
		jo.put("name", paymentChargingItem.getName());
		return new RESTResponse("data", jo);
	}


	/**
	 *
	 * @Description：订单详细中业务数据查询
	 * @author: zhuoHeng
	 * @version: 2017年1月12日 上午10:41:29
	 */
	public RESTResponse orderDetail(String id) {
		Kennel model = kennelService.get(id);
		Map<String, String> map = new HashMap<>();
		map.put("key1",model.getKennelCode());
		map.put("key2",model.getName());
		map.put("key3",model.getFciName());
		map.put("key4",model.getFciNameBackup());
		map.put("key5",model.getNameEn());
		return new RESTResponse("data",map);
	}

	/**
	 * @Description：设置FCI信息
	 * @author: laiguanglong
	 * @version: 2017年3月14日
	 */
	private void setFciInfo(String json,Member member,Kennel po){
		JSONObject jo = JSONObject.fromObject(json);
		String fciName = jo.getString("fciName");
		String fciNameBackup = jo.getString("fciNameBackup");
		Kennel vo = new Kennel();
		vo.setFciName(fciName);
		vo.setFciNameBackup(fciNameBackup);
		//校验犬舍fci名
		checkKennelFCIName(vo, member,false);
		po.setFciName(fciName);
		//校验犬舍fci备用名
		checkKennelFCIBack(vo, member,false);
		po.setFciNameBackup(fciNameBackup);
		kennelService.update(po);
	}

	/**
	 * @description: 是否需要填写FCI信息，需要返回true
	 * @author: laiguanglong
	 * @date: 2017/3/20 15:12
	 */
	public boolean isNeedFciInfo(Kennel kennel){
		boolean flag = false;
		//只要FCI信息有空的，就需要填写FCI信息
		if(StringUtils.isBlank(kennel.getFciName())||StringUtils.isBlank(kennel.getFciNameBackup())){
			flag = true;
		}
		return flag;
	}

	public RESTResponse getById(String id) {
		Kennel kennel = kennelDao.get(id);
		RestKennel vo = null;
		if(kennel!=null){
			vo = new RestKennel();
			BeanUtil.copyProperties(vo, kennel, "yyyy-MM-dd");
			String type = "";
			if("year".equals(vo.getType())){
				type = "年度";
			}else if("permanent".equals(vo.getType())){
				type = "永久";
			}
			vo.setType(type);
			//姓名长度大于3，隐藏头两个字，其余隐藏头一个字
			String memberName = kennel.getMember().getName();
			if(StringUtils.isNotBlank(memberName)){
				if(memberName.length()>3){
					memberName = "**"+memberName.substring(2);
				}else{
					memberName = "*"+memberName.substring(1);
				}
				vo.setMemberName(memberName);
			}
		}
		return new RESTResponse("data",vo);
	}
}