/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.restful.v1.breeder.service;

import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import com.cku.core.PageBeanResult;
import com.cku.core.RESTResponse;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.breeder.service.StudDogService;
import com.cku.oa.dog.dao.DogDao;
import com.cku.oa.dog.dao.DogTypeDao;
import com.cku.oa.dog.entity.Dog;
import com.cku.oa.dog.entity.DogBirthCertificate;
import com.cku.oa.dog.entity.DogNose;
import com.cku.oa.dog.entity.DogType;
import com.cku.oa.dog.service.DogBirthCertificateService;
import com.cku.oa.dog.service.DogColorMarkService;
import com.cku.oa.dog.service.DogNoseService;
import com.cku.oa.dog.service.DogTypeService;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.util.ImgUtil;
import com.cku.util.DateUtils;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

/**
 * @description: 专业繁殖人相关业务
 * @author: laiguanglong
 * @date: 2017/11/14 15:21
 */
@Service
@Transactional(readOnly = true)
public class RestBreederService {

	@Autowired
	private DogDao dogDao;
	@Autowired
	private DogTypeDao dogTypeDao;
	@Autowired
	private DogTypeService dogTypeService;
	@Autowired
	private DogColorMarkService dogColorMarkService;
	@Autowired
	private StudDogService studDogService;
	@Autowired
	private RestCityService restCityService;
	@Autowired
	private DogNoseService dogNoseService;
	@Autowired
	private DogBirthCertificateService dogBirthCertificateService;

	public RESTResponse get(HttpServletRequest request) {
		String pedigreeCertified = request.getParameter("pedigreeCertified");
		if (StringUtils.isBlank(pedigreeCertified)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号为空");
		}

		return new RESTResponse();
	}

	public RESTResponse getMyDog(HttpServletRequest request) {
		String pageNo = request.getParameter("pageNo");
		String pageSize = request.getParameter("pageSize");
		// 参数校验
		int pageNoInt = 1;
		int pageSizeInt = 10;
		try {
			if (StringUtils.isNotBlank(pageNo)) {
				pageNoInt = Integer.parseInt(pageNo);
			}
			if (StringUtils.isNotBlank(pageSize)) {
				pageSizeInt = Integer.parseInt(pageSize);
			}
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
		}
		String memberCode = UserUtils.getLoginMember().getMemberCode();
		List<Dog> dogList = dogDao.getMyDogByMemberCode(memberCode, (pageNoInt - 1) * pageSizeInt, pageSizeInt);
		List<JSONObject> list = buildDogList(dogList, null);
		return new RESTResponse("list", new PageBeanResult<>(list, dogDao.getMyDogCountByMemberCode(memberCode)));
	}

	public RESTResponse getMyDogList(HttpServletRequest request) {
		String pageNo = request.getParameter("pageNo");
		String pageSize = request.getParameter("pageSize");
		// 参数校验
		int pageNoInt = 1;
		int pageSizeInt = 10;
		try {
			if (StringUtils.isNotBlank(pageNo)) {
				pageNoInt = Integer.parseInt(pageNo);
			}
			if (StringUtils.isNotBlank(pageSize)) {
				pageSizeInt = Integer.parseInt(pageSize);
			}
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
		}
		String type = request.getParameter("type");
		if (StringUtils.isBlank(type)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "type参数为空");
		}
		if (!"sold".equals(type) && !"unsold".equals(type) && !"all".equals(type)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "type参数只能为sold或者unsold或者all");
		}
		// add by zhangjunwu 新增 查询条件，按犬名或血统证书号查询
		String searchKey = request.getParameter("searchKey");
		Member member = UserUtils.getLoginMember();
		List<Dog> dogList = Lists.newArrayList();
		Long count = 0L;
		if ("unsold".equals(type)) {
			dogList = dogDao.getMyUnsoldDog(member.getMemberCode(), searchKey, (pageNoInt - 1) * pageSizeInt,
					pageSizeInt);
			count = dogDao.getMyUnsoldDogCount(member.getMemberCode(), searchKey);
		} else if ("sold".equals(type)) {
			dogList = dogDao.getMySoldDog(member.getMemberCode(), searchKey, (pageNoInt - 1) * pageSizeInt,
					pageSizeInt);
			count = dogDao.getMySoldDogCount(member.getMemberCode(), searchKey);
		} else if ("all".equals(type)) {
			dogList = dogDao.getMyAllDog(member.getMemberCode(), searchKey, (pageNoInt - 1) * pageSizeInt, pageSizeInt);
			count = dogDao.getMyAllDogCount(member.getMemberCode(), searchKey);
		}
		List<JSONObject> list = Lists.newArrayList();
		for (Dog dog : dogList) {
			JSONObject jo = new JSONObject();
			jo.put("id", getString(dog.getId()));
			jo.put("pedigreeCertifiedCode", getString(dog.getPedigreeCertifiedCode()));
			jo.put("dogBreed", getString(dogTypeService.getBreedCnName(dog.getDogBreed())));
			jo.put("gender", getString(dog.getGender()));
			jo.put("nickname", getString(dog.getNickname()));
			// add by zhangjunwu 新增犬名 呼名存在取呼名，不存在取英文名
			jo.put("dogName", getString(StringUtils.isBlank(dog.getCallName()) ? dog.getNameEn() : dog.getCallName()));
			jo.put("age", getString(studDogService.getAge(dog.getBirthdate())));
			jo.put("studDogId", StringUtils.isBlank(dog.getStudDogId()) ? "" : dog.getStudDogId());
			String state = "";
			if ("1".equals(dog.getStudDogState())) {
				state = "已发布";
			}
			jo.put("studDogState", getString(state));
			jo.put("studDogCoverImg", getString(ImgUtil.getImgUrl(request, dog.getStudDogCoverImgId())));
			jo.put("ageOneToTen", studDogService.isAgeOneToTen(dog.getBirthdate()) ? "1" : "0");
			jo.put("city", restCityService.getCity(member));
			DogNose dogNose = new DogNose();

			DogBirthCertificate dogBirthCertificate = dogBirthCertificateService
					.getByBirthCerRegCode(dog.getBirthCertificateLevelCode());
			dogNose.setDogBirthCertificateId(dogBirthCertificate == null ? "" : dogBirthCertificate.getId());
			dogNose.setDelFlag("0");
			dogNose.setDogId(dog.getId());
			if (StringUtils.isNotBlank(dogNose.getDogBirthCertificateId())
					&& StringUtils.isNotBlank(dogNose.getDogId())) {
				dogNose.setRelationFlag("1");
			}
			List<DogNose> ns = dogNoseService.findList(dogNose);
			if (!CollectionUtils.isEmpty(ns)) {
				DogNose n = ns.get(0);
				jo.put("noseCode", n.getNoseCode());
			} else {
				jo.put("noseCode", "");
			}

			list.add(jo);
		}
		return new RESTResponse("list", new PageBeanResult<>(list, count));
	}

	@Transactional(readOnly = false)
	public RESTResponse putMyDog(String json) {
		String memberCode = UserUtils.getLoginMember().getMemberCode();
		JSONObject jo = JSONObject.fromObject(json);
		if (!jo.containsKey("id") || jo.getJSONArray("id") == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id为空");
		}
		JSONArray ids = jo.getJSONArray("id");
		for (int i = 0, len = ids.size(); i < len; i++) {
			Dog dog = dogDao.get(ids.getString(i));
			if (dog == null || !memberCode.equals(dog.getMemberCode())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
			}
			// 已售
			if ("sold".equals(jo.getString("state"))) {
				dogDao.saveSoldDog(dog.getId(), memberCode);
			}
			// 未售
			if ("unsold".equals(jo.getString("state"))) {
				dogDao.deleteSoldDog(dog.getId(), memberCode);
			}
		}
		return new RESTResponse();
	}

	/**
	 * @description: 构造犬只信息列表
	 * @author: laiguanglong
	 * @date: 2017/11/14 15:21
	 */
	public List<JSONObject> buildDogList(List<Dog> dogList, Member member) {
		List<JSONObject> list = Lists.newArrayList();
		for (Dog dog : dogList) {
			JSONObject jo = new JSONObject();
			jo.put("id", getString(dog.getId()));
			jo.put("pedigreeCertifiedCode", getString(dog.getPedigreeCertifiedCode()));
			jo.put("nameEn", getString(dog.getNameEn()));
			jo.put("dogBreed", getString(dogTypeService.getBreedCnName(dog.getDogBreed())));
			jo.put("gender", getString(DictUtils.getDictLabel(dog.getGender(), "dog_gender", "")));
			jo.put("colorFlag", getString(dogColorMarkService.getDogColorMarkCn(dog.getColorFlag())));
			jo.put("identificationFlag", getString(dog.getIdentificationFlag()));
			String birthdate = "";
			if (dog.getBirthdate() != null) {
				birthdate = DateUtils.dateToStr(dog.getBirthdate());
			}
			jo.put("birthdate", getString(birthdate));
			jo.put("nickname", getString(dog.getNickname()));
			jo.put("ageOneToTen", studDogService.isAgeOneToTen(dog.getBirthdate()) ? "1" : "0");
			jo.put("studDogId", StringUtils.isBlank(dog.getStudDogId()) ? "" : dog.getStudDogId());
			if (member != null) {
				jo.put("city", restCityService.getCity(member));
			}
			list.add(jo);
		}
		return list;
	}

	public RESTResponse getDogBreedList() {
		List<JSONObject> list = Lists.newArrayList();
		List<String> dogBreedList = dogDao.getDogBreedListByMemberCode(UserUtils.getLoginMember().getMemberCode());
		for (String dogBreed : dogBreedList) {
			if (StringUtils.isNotBlank(dogBreed)) {
				DogType dogType = dogTypeDao.getByDogBreedCode(dogBreed);
				if (dogType != null) {
					JSONObject jo = new JSONObject();
					jo.put("dogBreedCode", getString(dogType.getDogBreedCode()));
					jo.put("breedCnName", getString(dogType.getBreedCnName()));
					list.add(jo);
				}
			}
		}
		return new RESTResponse("list", new PageBeanResult<>(list, new Long(list.size())));
	}

	private String getString(String str) {
		return str == null ? "" : str;
	}

	@Transactional(readOnly = false)
	public RESTResponse addDogNose(String json) {
		UserUtils.getLoginMember().getMemberCode();
		JSONObject jo = JSONObject.fromObject(json);
		if (!jo.containsKey("id") || jo.getString("id") == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id为空");
		}
		if (!jo.containsKey("noseId") || jo.getString("noseId") == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "noseId为空");
		}
		if (dogDao.get(jo.getString("id")) == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id不正确");
		}
		String flag = dogNoseService.saveNoseId(jo.getString("noseId"), null, jo.getString("id"));
		if (flag == null)
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "鼻纹标识与库中信息重复！");
		return new RESTResponse();
	}

}