package com.cku.oa.sampling.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.sampling.dao.SamplingFlowLogDao;
import com.cku.oa.sampling.entity.Sampling;
import com.cku.oa.sampling.entity.SamplingFlowLog;
import com.cku.oa.sampling.enums.SamplingFlowStateEnum;
import com.cku.oa.sampling.enums.SamplingLogFlowTypeEnum;
import com.cku.util.DateUtils;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;

/**
 * 采样包流程日志Service
 *
 * @author yuanshuai
 * @version 2023-05-22
 */
@Service
@Transactional(readOnly = true)
public class SamplingFlowLogService extends CrudService<SamplingFlowLogDao, SamplingFlowLog> {

	public SamplingFlowLog get(String id) {
		return super.get(id);
	}

	public List<SamplingFlowLog> findList(SamplingFlowLog samplingFlowLog) {
		return super.findList(samplingFlowLog);
	}

	public Page<SamplingFlowLog> findPage(Page<SamplingFlowLog> page, SamplingFlowLog samplingFlowLog) {
		return super.findPage(page, samplingFlowLog);
	}

	@Transactional(readOnly = false)
	public void batchInsert(List<SamplingFlowLog> logList) {
		if (CollectionUtils.isEmpty(logList)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "采样包记录数据为空");
		}
		dao.batchInsert(logList);
	}

	/**
	 * 添加采样包操作日志
	 *
	 * @param flowType 使用com.cku.oa.sampling.enums.SamplingLogFlowTypeEnum
	 * @param operData 根据流程状态记录数据。如犬只证书号、会员号、基因编码等
	 * @author yuanshuai
	 * @date 2023/5/22 16:11
	 */
	@Transactional(readOnly = false)
	public void add(String samplingCode, SamplingLogFlowTypeEnum flowType, String operData) {
		SamplingFlowLog log = new SamplingFlowLog();
		log.setSamplingCode(samplingCode);
		log.setFlowType(flowType.code());
		log.setOperData(operData);
		log.setOperTime(DateUtils.getNow());
		User user = UserUtils.getLoginUser();
		String userId = "1";
		if (user != null && StringUtils.isBlank(user.getId())) {
			userId = user.getId();
		}
		log.setOperBy(userId);
		save(log);
	}

	@Transactional(readOnly = false)
	public void batchAdd(List<Sampling> samplingList, SamplingLogFlowTypeEnum flowType, Date operTime, String operBy) {
		List<SamplingFlowLog> logList = Lists.newArrayList();
		samplingList.forEach(s ->
				logList.add(initLog(s.getSamplingCode(), flowType, null, operBy, operTime)));
		batchInsert(logList);
	}

	/**
	 * 初始化日志对象
	 *
	 * @author yuanshuai
	 * @date 2023/5/24 11:36
	 */
	public SamplingFlowLog initLog(String samplingCode, SamplingLogFlowTypeEnum flowType, String operData, String operBy, Date operTime) {
		SamplingFlowLog log = new SamplingFlowLog();
		log.setSamplingCode(samplingCode);
		log.setFlowType(flowType.code());
		log.setOperData(operData);
		log.setOperTime(operTime);
		log.setOperBy(operBy);
		log.preInsert();
		return log;
	}

	/**
	 * 根据采样包初始化添加日志对象
	 *
	 * @author yuanshuai
	 * @date 2023/5/23 11:24
	 */
	public SamplingFlowLog initAddLog(Sampling sampling) {
		SamplingFlowLog log = new SamplingFlowLog();
		log.setSamplingCode(sampling.getSamplingCode());
		log.setFlowType(SamplingLogFlowTypeEnum.ADD.code());
		log.setOperTime(sampling.getCreateDate());
		log.setOperBy(sampling.getCreateBy().getId());
		log.preInsert();
		return log;
	}

	@Transactional(readOnly = false)
	public void save(SamplingFlowLog samplingFlowLog) {
		super.save(samplingFlowLog);
	}

	@Transactional(readOnly = false)
	public void delete(SamplingFlowLog samplingFlowLog) {
		super.delete(samplingFlowLog);
	}

	/**
	 * 把采样包流程状态转成采样包日志流程类型
	 *
	 * @author yuanshuai
	 * @date 2023/5/23 16:49
	 */
	public SamplingLogFlowTypeEnum flowState2LogFlowType(SamplingFlowStateEnum flowState) {
		switch (flowState.code()) {
			case 1:
				return SamplingLogFlowTypeEnum.POST;
			case 2:
				return SamplingLogFlowTypeEnum.SEND;
			case 3:
				return SamplingLogFlowTypeEnum.RESULT;
			case 4:
				return SamplingLogFlowTypeEnum.RESULT;
			case 5:
				return SamplingLogFlowTypeEnum.LOGOUT;
			default:
				return null;
		}
	}

}