package com.cku.oa.project.web;

import java.util.Calendar;
import java.util.List;
import java.util.Objects;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.cku.oa.project.entity.CkuProject;
import com.cku.oa.project.service.CkuProjectService;
import com.cku.oa.project.util.WeekUtil;
import com.thinkgem.jeesite.common.config.Global;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.utils.DateUtils;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.common.web.BaseController;

/**
 * 技术项目表Controller
 * 
 * @author Sunny
 * @version 2021-12-07
 */
@Controller
@RequestMapping(value = "${adminPath}/project/ckuProject")
public class CkuProjectController extends BaseController {

	@Autowired
	private CkuProjectService ckuProjectService;

	@ModelAttribute
	public CkuProject get(@RequestParam(required = false) String id) {
		CkuProject entity = null;
		if (StringUtils.isNotBlank(id)) {
			entity = ckuProjectService.get(id);
		}
		if (entity == null) {
			entity = new CkuProject();
		}
		return entity;
	}

	@RequiresPermissions("project:ckuProject:view")
	@RequestMapping(value = { "list", "" })
	public String list(CkuProject ckuProject, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<CkuProject> page = ckuProjectService.findPage(new Page<CkuProject>(request, response), ckuProject);

		if (!CollectionUtils.isEmpty(page.getList())) {
			page.getList().forEach(p -> {
				p.setStartWeek(WeekUtil.getWeekOfYear(p.getStartTime()));
				p.setEndWeek(WeekUtil.getWeekOfYear(p.getEndTime()));
			});
		}
		model.addAttribute("page", page);
		return "oa/project/ckuProjectList";
	}

	@RequiresPermissions("project:ckuProject:view")
	@RequestMapping(value = "view")
	public String view(CkuProject ckuProject, Model model) {
		model.addAttribute("ckuProject", ckuProject);
		return "oa/project/ckuProjectView";
	}

	@RequiresPermissions("project:ckuProject:view")
	@RequestMapping(value = "form")
	public String form(CkuProject ckuProject, Model model) {
		model.addAttribute("ckuProject", ckuProject);
		return "oa/project/ckuProjectForm";
	}

	@RequiresPermissions("project:ckuProject:edit")
	@RequestMapping(value = "save")
	public String save(CkuProject ckuProject, Model model, RedirectAttributes redirectAttributes) {
		if (!beanValidator(model, ckuProject)) {
			return form(ckuProject, model);
		}
		ckuProjectService.save(ckuProject);
		addMessage(redirectAttributes, "保存技术项目表成功");
		return "redirect:" + Global.getAdminPath() + "/project/ckuProject/?repage";
	}

	@RequiresPermissions("project:ckuProject:edit")
	@RequestMapping(value = "delete")
	public String delete(CkuProject ckuProject, RedirectAttributes redirectAttributes) {
		ckuProjectService.delete(ckuProject);
		addMessage(redirectAttributes, "删除技术项目表成功");
		return "redirect:" + Global.getAdminPath() + "/project/ckuProject/?repage";
	}

	@RequiresPermissions("project:ckuProject:viewTable")
	@RequestMapping(value = "viewTable")
	public String viewTable(CkuProject ckuProject, Model model) {
		final Integer queryYear = Objects.isNull(ckuProject.getQueryYear()) ? Calendar.getInstance().get(Calendar.YEAR)
				: ckuProject.getQueryYear();
		List<CkuProject> projectList = ckuProjectService.findListByYear(queryYear);
		if (!CollectionUtils.isEmpty(projectList)) {
			projectList.forEach(p -> {
				p.setName(String.format("%s%s%s%s%s%s", p.getName(), " (",
						DateUtils.formatDate(p.getStartTime(), "MM月dd日"), "-",
						DateUtils.formatDate(p.getEndTime(), "MM月dd日"), ")"));
				p.setStartWeek(WeekUtil.getWeekOfYear(queryYear, p.getStartTime()));
				p.setEndWeek(WeekUtil.getWeekOfYear(queryYear, p.getEndTime()));
			});
		}
		model.addAttribute("currentMonth", Calendar.getInstance().get(Calendar.MONTH) + 1);
		model.addAttribute("currentYear", Calendar.getInstance().get(Calendar.YEAR));
		model.addAttribute("projectList", projectList);
		model.addAttribute("monthWeek", WeekUtil.getYearMonthWeek(queryYear));
		return "oa/project/ckuProjectViewTable";
	}

}