/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.kennel.service;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import com.cku.core.PageBeanResult;
import com.cku.core.RESTResponse;
import com.cku.core.ResultDto;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.activity.service.SysMemberActivityRecordService;
import com.cku.oa.dog.dao.DogBirthCertificateDao;
import com.cku.oa.kennel.dao.KennelDao;
import com.cku.oa.kennel.dao.KennelImageDao;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.kennel.entity.KennelImage;
import com.cku.oa.sys.dao.SysBussinessLogsDao;
import com.cku.oa.sys.dao.user.MemberDao;
import com.cku.oa.sys.entity.SysBussinessLogs;
import com.cku.oa.sys.entity.user.KennelType;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.entity.user.MemberCardTypeEnum;
import com.cku.oa.sys.entity.user.MemberType;
import com.cku.oa.sys.service.ExpressQueueService;
import com.cku.oa.sys.service.LoginServerService;
import com.cku.oa.sys.service.RemindService;
import com.cku.oa.sys.util.ConstantUtils;
import com.cku.oa.sys.util.MemberUtil;
import com.cku.oa.sys.zalog.SysContext;
import com.cku.restful.v1.sys.model.RestOrg;
import com.cku.util.ServletUtils;
import com.thinkgem.jeesite.common.config.Global;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.DateUtils;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.common.utils.excel.ExportExcel;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import net.sf.json.JsonConfig;
import net.sf.json.util.CycleDetectionStrategy;
import okhttp3.OkHttpClient;
import okhttp3.Request;

/**
 * 犬舍基本信息Service
 * 
 * @author fanhuibin
 * @version 2016-07-13
 */
@Service
@Transactional(readOnly = true)
public class KennelService extends CrudService<KennelDao, Kennel> {

	private static final String KENNEL = "犬舍";
	private static final String KENNEL_START_URL = Global.getConfig("petpricingUrl") + "/kennelStar/getByCode?code=";

	@Autowired
	private RemindService remindService;
	@Autowired
	private DogBirthCertificateDao dogBirthCertificateDao;
	@Autowired
	private ExpressQueueService expressQueueService;
	@Autowired
	private LoginServerService loginServerService;
	@Autowired
	private MemberDao memberDao;
	@Autowired
	private SysBussinessLogsDao sysBussinessLogsDao;
	@Autowired
	private KennelDao kennelDao;
	@Autowired
	private KennelImageDao kennelImageDao;
	@Autowired
	private SysMemberActivityRecordService sysMemberActivityRecordService;

	public Kennel get(String id) {
		return super.get(id);
	}

	public List<Kennel> findList(Kennel kennel) {
		kennel.getSqlMap().put("memberSiteMap", loginServerService.getMemberSiteByAlike());
		return super.findList(kennel);
	}

	public Page<Kennel> findPage(Page<Kennel> page, Kennel kennel) {
		kennel.getSqlMap().put("memberSiteMap", loginServerService.getMemberSiteByAlike());
		return super.findPage(page, kennel);
	}

	@Transactional(readOnly = false)
	public String nameCheck(String name, String id) {
		List<Kennel> list = dao.nameCheck(name.trim());
		if (list.size() > 0) {
			String id2 = list.get(0).getId();
			if (id == null) {
				return "{\"rc\":false,\"msg\":\"犬舍中文名重复\"}";
			} else {
				if (!id.equals(id2)) {
					return "{\"rc\":false,\"msg\":\"犬舍中文名重复\"}";
				}
			}
		}
		return "{\"rc\":true}";
	}

	public Boolean nameCnCheck(String kennelName, String memberCode) {
		// 空
		if (StringUtils.isBlank(kennelName)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称不能为空");
		}
		// 不能有空格
		if (kennelName.contains(" ")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称不能有空格");
		}
		// 只能中文
		String regex = "[\u4e00-\u9fa5]+";
		if (!kennelName.matches(regex)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称只能是中文");
		}
		// 包含犬舍二字
		if (kennelName.lastIndexOf("犬舍") > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称不可包含犬舍二字");
		}
		// 只能2-4个字
		if (kennelName.length() < 2 || kennelName.length() > 4) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称只能是2-4个字");
		}
		// 繁体
		String encode = "GB2312";
		String nowStr = "";
		try {
			nowStr = new String(kennelName.getBytes(encode), encode);
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称不能有繁体字");
		}
		if (!kennelName.equals(nowStr)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称不能有繁体字");
		}
		if (kennelDao.countCnName(kennelName.endsWith("犬舍") ? kennelName : kennelName + "犬舍", memberCode) > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称重复");
		}
		return Boolean.TRUE;
	}

	public Boolean nameEnCheck(String kennelNameEn, Boolean isValidBlank, String memberCode) {
		if (!isValidBlank && StringUtils.isBlank(kennelNameEn)) {
			return Boolean.TRUE;
		}
		// 空
		if (StringUtils.isBlank(kennelNameEn)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称不能为空");
		}
		// 只能为英文
		boolean isWord = kennelNameEn.matches("^[a-zA-Z ]+$");
		if (!isWord) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只能为英文");
		}
		// 重复
		if (kennelDao.countEnName(kennelNameEn, memberCode) > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "名称重复");
		}
		return Boolean.TRUE;
	}

	@Transactional(readOnly = false)
	public void update(Kennel kennel) {
		kennel.preUpdate();
		dao.update(kennel);
	}

	@Transactional(readOnly = false)
	public void save(Kennel kennel) {
		if (StringUtils.isEmpty(kennel.getId())) {
			if (!kennel.getName().contains("犬舍")) {
				kennel.setName(kennel.getName() + "犬舍");
			}
			kennel.setKennelCode(kennel.getMemberCode());
			kennel.setReviewState("0");// 新申请犬舍初始审核状态
			kennel.setFciReviewState("0");// 新申请犬舍初始审核状态
			kennel.setPrintState("0");// 新申请犬舍初始审核状态
			kennel.setFciPrintState("0");// 新申请犬舍初始审核状态
		}
		super.save(kennel);
		if (!StringUtils.isEmpty(kennel.getId())) {
			String date = new SimpleDateFormat("yyyy年MM月dd日HH时mm分").format(kennel.getCreateDate());
			String type = "";
			if ("year".equals(kennel.getType())) {
				type = DateUtils.formatDate(kennel.getCreateDate(), "yyyy") + "年度";
			} else if ("permanent".equals(kennel.getType())) {
				type = "永久";
			}
			StringBuffer sb = new StringBuffer();
			sb.append("您于").append(date).append("提交了").append(type).append("犬舍注册申请，请您务必于当日24时前完成缴费，过期订单失效哦。");
			remindService.addRemind(sb.toString(), kennel.getMemberCode());
			System.out.println(sb);
		}

	}

	@Transactional(readOnly = false)
	public ResultDto<String> saveExtra(Kennel kennel) {

		if (Objects.nonNull(kennel.getBusinessDaysArr())) {
			kennel.setBusinessDays(String.join(",", kennel.getBusinessDaysArr()));
		}
		List<KennelImage> images = new ArrayList<>();
		// 将图片的第一张设置为主图
		if (StringUtils.isNotBlank(kennel.getKennelImages())) {
			String[] imageUrls = kennel.getKennelImages().split(",");
			for (int i = 0; i < imageUrls.length; i++) {
				KennelImage img = new KennelImage();
				img.preInsert();
				img.setKennelId(kennel.getId());
				img.setImageType(i == 0 ? "1" : "2");
				img.setImageUrl(imageUrls[i]);
				img.setSort(i);
				images.add(img);
			}
			kennel.setMainImage(images.get(0).getImageUrl());
		}
		//处理文本转义
		if(StringUtils.isNotBlank(kennel.getIntroduction())) {
			kennel.setIntroduction(StringEscapeUtils.unescapeHtml4(kennel.getIntroduction()));
		}
		this.kennelDao.updateKennelExtra(kennel);
		// 删除犬舍的所有图片
		this.kennelImageDao.deleteByKennelId(kennel.getId());
		if (!CollectionUtils.isEmpty(images)) {
			// 从新插入图片
			this.kennelImageDao.insertBatch(images);
		}
		return ResultDto.success("保存成功！", "");

	}

	public List<KennelImage> findImageList(String kennelId) {
		return this.kennelImageDao.selectByKennelId(kennelId);
	}

	@Transactional(readOnly = false)
	public void delete(Kennel kennel) {
		super.delete(kennel);
	}

	/**
	 *
	 * @description: 根据会员号查询犬舍
	 * @author: laiguanglong
	 * @date: 2016年8月17日 下午12:04:26
	 */
	public Kennel getKennelByMemberCode(String memberCode) {
		return dao.getKennelByMemberCode(memberCode);
	}

	@Transactional(readOnly = false)
	public Kennel print(Kennel kennel) throws ZAException {
		kennel.setPrintState("1");
		SysContext sysContext = new SysContext(null, null, null);
		sysContext.noStateExecute(sysContext, KENNEL + "打印", kennel.getMemberCode(), "kennel", kennel.getId(), "",
				(str) -> {
				});
		super.save(kennel);
		// 添加待邮寄物品
		expressQueueService.push(kennel.getMemberCode(), null, "中文犬舍证书", kennel.getName(), "6"); // GYH1 modified
		return kennel;
	}

	@Transactional(readOnly = false)
	public void resetPrint(Kennel kennel) {
		kennel.setPrintState("0");
		super.save(kennel);
	}

	@Transactional(readOnly = false)
	public Kennel fciPrint(Kennel kennel) throws ZAException {
		kennel.setFciPrintState("1");
		SysContext sysContext = new SysContext(null, null, null);
		sysContext.noStateExecute(sysContext, KENNEL + "FCI打印", kennel.getMemberCode(), "kennel", kennel.getId(), "",
				(str) -> {
				});
		super.save(kennel);
		// 添加待邮寄物品
		expressQueueService.push(kennel.getMemberCode(), null, "FCI犬舍证书", kennel.getFciName(), "115"); // GYH1 modified
		return kennel;
	}

	@Transactional(readOnly = false)
	public void fciResetPrint(Kennel kennel) {
		kennel.setFciPrintState("0");
		super.save(kennel);
	}

	public String getPayItem(Kennel kennel) {
		if (kennel.getEndTime() == null) {
			if ("year".equals(kennel.getType())) {
				return "kennel@ZC,kennel@ND";
			} else if ("permanent".equals(kennel.getType())) {
				return "kennel@ZC,kennel@YJ";
			} else {
				return "";
			}
		} else {
			Date date = kennel.getEndTime();
			Calendar calendar = Calendar.getInstance();// 日历对象
			calendar.setTime(date);// 设置当前日期
			calendar.add(Calendar.MONTH, 1);// 月份+1
			if (calendar.getTime().getTime() > System.currentTimeMillis()) {// 用户过期小于一个月，可以直接补交年度费或申请永久犬舍
				if ("year".equals(kennel.getType())) {
					return "kennel@ND";
				} else if ("permanent".equals(kennel.getType())) {
					return "kennel@YJ";
				} else {
					return "";
				}
			} else {
				if ("year".equals(kennel.getType())) {
					return "kennel@ZC,kennel@ND";
				} else if ("permanent".equals(kennel.getType())) {
					return "kennel@ZC,kennel@YJ";
				} else {
					return "";
				}
			}
		}
	}

	@Transactional(readOnly = false)
	public void kennelGuoQiBeforeTwoWeek() {
		List<Kennel> list = dao.getList(14);
		for (Kennel kennel : list) {
			String date = DateFormatUtils.format(kennel.getEndTime(), "yyyy-MM-dd");
			StringBuffer sb = new StringBuffer();
			sb.append("您的").append(kennel.getName()).append("将于").append(date)
					.append("过期，为了不影响你的业务的正常使用，请及时续费，续费期为有效期截止后的一个月内，超过续费期将按照新注册犬舍标准收费。");
			remindService.addRemind(sb.toString(), kennel.getMemberCode());
			System.out.println(sb);
		}
	}

	/**
	 * 校验犬舍是否 过期
	 * 
	 * @Author chaixueteng
	 * @2016年9月21日下午3:09:34
	 */
	// public void validKennel(String memberCode){
	// Kennel kennel = dao.getKennelByMemberCode(memberCode);
	// boolean flag = false;
	// if (kennel!=null) {
	// Date endTime = kennel.getEndTime();
	// if (endTime!=null) {
	// if ((endTime.getTime()+86399000)>new Date().getTime()) {//有效
	// flag = true;
	// }
	// }
	// }
	// if(!flag){
	// throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"母犬主犬舍无效");
	// }
	// if(!"1".equals(kennel.getReviewState())){
	// throw new ZAException(ZAErrorCode.ZA_VALID_FAILED,"母犬主犬舍必须审核通过才能申请配种证明");
	// }
	// }

	/**
	 * 首页--得到未审核的年度犬舍
	 * 
	 * @Author chaixueteng
	 * @2016年9月6日下午2:04:48
	 */
	public int getKennelCount() {
		return dao.getKennelCount("0");
	}

	/**
	 * 首页--得到未审核的永久犬舍
	 * 
	 * @Author chaixueteng
	 * @2016年9月6日下午2:04:48
	 */
	public int getProfessionalCount() {
		return dao.getProfessionalCount("0");
	}

	/**
	 * 编辑
	 * 
	 * @Author chaixueteng
	 * @2016年9月7日下午2:54:25
	 */
	@Transactional(readOnly = false)
	public void edit(Kennel kennel) throws ZAException {
		try {
			String reviewState = kennel.getReviewState();
			String fciReviewState = kennel.getFciReviewState();
			// 原来状态
			Kennel kennelOrg = super.get(kennel.getId());
			String reviewStateOrg = kennelOrg.getReviewState();
			String fciReviewStateOrg = kennelOrg.getFciReviewState();
			// 原来支付状态
			String paymentState;
			if (kennelOrg.getEndTime() != null && kennelOrg.getEndTime().getTime() > new Date().getTime()) {
				paymentState = ConstantUtils.PAYMENT_PAID;
			} else {
				paymentState = ConstantUtils.PAYMENT_UNPAID;
			}

			if (reviewState != null && "0".equals(reviewStateOrg)) {
				kennel.setReviewDate(new Date());
				SysContext sysContext = new SysContext(paymentState, reviewStateOrg, "0");
				if ("1".equals(reviewState)) {
					sysContext.confirmSuccess(sysContext, "犬舍中文审核", kennel.getMemberCode(), "kennel", kennel.getId(),
							"", (str) -> {
							});
				} else if ("2".equals(reviewState)) {
					sysContext.confirmFail(sysContext, "犬舍中文审核", kennel.getReviewRemark(), kennel.getMemberCode(),
							"kennel", kennel.getId(), "", (str) -> {
							});
				}
			}
			if (fciReviewState != null && "0".equals(fciReviewStateOrg)) {
				SysContext sysContext = new SysContext(paymentState, fciReviewStateOrg, "0");
				if ("1".equals(fciReviewState)) {
					kennel.setFciReviewDate(new Date());// fci犬舍审核通过才有签发日期
					sysContext.confirmSuccess(sysContext, "犬舍fci审核", kennel.getMemberCode(), "kennel", kennel.getId(),
							"", (str) -> {
							});
				} else if ("2".equals(fciReviewState)) {
					sysContext.confirmFail(sysContext, "犬舍fci审核", kennel.getFciReviewRemark(), kennel.getMemberCode(),
							"kennel", kennel.getId(), "", (str) -> {
							});
				}
			}
			// 犬舍有效的情况下不能修改犬舍类型，无效可修改
			if (kennelOrg.getEndTime() != null && kennelOrg.getEndTime().getTime() > new Date().getTime()) {
				kennel.setType("");
			}
			if ("1".equals(fciReviewState)) {// 英文名重新审核也是重置为最新时间
				if (kennelOrg.getFciReviewDate() == null) {// 第一次审核通过才修改时间
					kennel.setFciReviewDate(new Date());// fci犬舍审核通过才有签发日期
				}
			}

			// 如果犬舍修改为冻结状态，则修改会员类型
			if ("3".equals(reviewState)) {
				memberDao.updateMemberType(kennel.getMemberCode(), MemberType.professional.toString(),
						MemberCardTypeEnum.GOLDEN_CARD.code());
			} else {
				memberDao.updateMemberType(kennel.getMemberCode(), MemberType.professional.toString(),
						KennelType.permanent.toString().equals(kennel.getType()) ? MemberCardTypeEnum.PURPLE_CARD.code()
								: MemberCardTypeEnum.PLATINUM_CARD.code());
			}

			kennel.preUpdate();
			dao.update(kennel);
		} catch (ZAException e) {
			throw e;
		}
	}

	// 获取一周内 新注册的犬舍
	public int getNewKennel() {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
		Date newDate = new Date();
		Calendar cal = Calendar.getInstance();
		cal.setTime(newDate);
		cal.add(Calendar.DAY_OF_MONTH, -6);
		Date time = cal.getTime();
		String beginDate = sdf.format(time);
		System.out.println(beginDate);
		return dao.getNewKennel(beginDate);
	}

	/**
	 *
	 * @description: 犬舍注册业务
	 * @author: laiguanglong
	 * @date: 2016年9月12日 上午10:57:06
	 */
	public void validateBusiness(Kennel kennel) {
		Member member = UserUtils.getLoginMember();
		if (dao.getKennelByMemberCode(member.getMemberCode()) != null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "您已经注册过犬舍，不能重复注册");
		}
		if (!MemberUtil.isActiveMember(member)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该业务仅限有效专业会员办理");
		}
		if (!MemberUtil.isProfessionalMember(member)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "只有专业会员才能申请注册犬舍");
		}
		if (dao.countByName("name", kennel.getName(), member.getMemberCode()) > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍中文名重复");
		}
		if (dao.countByName("nameEn", kennel.getNameEn(), member.getMemberCode()) > 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍英文名重复");
		}
		// 如果是永久犬舍，fciName查重
		if ("permanent".equals(kennel.getType())) {
			if (dao.countByName("fciName", kennel.getFciName(), member.getMemberCode()) > 0) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍英文名重复");
			}
			if (dao.countByName("fciNameBackup", kennel.getFciNameBackup(), member.getMemberCode()) > 0) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍备用名重复");
			}
		}
	}

	/**
	 * @Author chaixueteng
	 * @2016年9月13日下午4:11:18
	 */
	@Transactional(readOnly = false)
	public void updateKennelRname(Kennel kennel2) {
		dao.update(kennel2);
	}

	/**
	 *
	 * @description: 犬舍注册
	 * @author: zhangjunwu
	 * @date: 2020年06月03日 下午3:03:50
	 */
	public void kennelFirstSql(StringBuffer buffer, String businessId) {
		// //初次费用暂不做任何更新
		// String date = DateUtils.formatDate(new Date(), "yyyy-MM-dd");
		// buffer.append("UPDATE kennel SET ").append("end_time =
		// '").append(date).append("' ")
		// .append("WHERE id='").append(businessId).append("'");
	}

	/**
	 *
	 * @description: 年度犬舍
	 * @author: zhangjunwu
	 * @date: 2020年06月03日 下午3:03:50
	 */
	public void kennelYearSql(StringBuffer buffer, String businessId) {
		Kennel kennel = get(businessId);
		// 日期向后延伸一个月，过期未超过一个月 算有效
		Calendar calendar = Calendar.getInstance();// 日历对象
		calendar.add(Calendar.MONTH, -1);// 月份 -1
		String date = "";
		// 如犬舍过期时间为空 或 犬舍过期超过一个月，从当前日期开始计算
		if (kennel.getEndTime() == null || kennel.getEndTime().before(calendar.getTime())) {
			date = DateUtils.get1YearLater(new Date());
		}
		// 如犬舍过期时间在有效期内 或 犬舍过期未超过一个月，则从有效期开始计算
		else {
			date = DateUtils.get1YearLater(kennel.getEndTime());
		}
		buffer.append("UPDATE kennel SET type = 'year',").append("end_time = '").append(date).append("' ")
				.append("WHERE id='").append(businessId).append("'");

	}

	/**
	 *
	 * @description: 永久犬舍
	 * @author: zhangjunwu
	 * @date: 2020年06月03日 下午17:05:02 永久犬舍 日期延后 100 年
	 */
	public void kennelPermanentSql(StringBuffer buffer, String businessId) {

		Kennel kennel = get(businessId);
		// 截止日期
		Calendar calendar = Calendar.getInstance();// 日历对象
		calendar.add(Calendar.MONTH, -1);// 月份 -1
		if (kennel.getEndTime() != null && kennel.getEndTime().after(calendar.getTime())) {
			calendar.setTime(kennel.getEndTime());
		} else {
			calendar.setTime(new Date());
		}
		calendar.add(Calendar.YEAR, 100);
		String date = DateUtils.formatDate(calendar.getTime(), "yyyy-MM-dd");
		buffer.append("UPDATE sys_member SET member_card_type='").append(MemberCardTypeEnum.PURPLE_CARD.code())
				.append("'").append(" WHERE member_code='").append(kennel.getMemberCode()).append("';");
		buffer.append("UPDATE kennel SET type = 'permanent',").append("end_time = '").append(date).append("' ")
				.append("WHERE id='").append(businessId).append("'");
	}

	/**
	 * 计算月龄
	 *
	 * @param date
	 * @param birthdate
	 * @return
	 */
	private int getMonthAge(Date date, Date birthdate) {
		Calendar dateCalendar = Calendar.getInstance();
		Calendar birthdateCalendar = Calendar.getInstance();
		dateCalendar.setTime(date);
		birthdateCalendar.setTime(birthdate);
		int monthAge = dateCalendar.get(Calendar.MONTH) - birthdateCalendar.get(Calendar.MONTH);
		if (dateCalendar.get(Calendar.YEAR) != birthdateCalendar.get(Calendar.YEAR)) {
			monthAge = (dateCalendar.get(Calendar.YEAR) - birthdateCalendar.get(Calendar.YEAR)) * 12 + monthAge;
		}
		birthdateCalendar.add(Calendar.MONTH, monthAge);
		if (birthdateCalendar.getTime().after(dateCalendar.getTime())) {
			monthAge = monthAge - 1;
		}
		return monthAge;
	}

	/**
	 * 根据犬舍中文名模糊查询相同的信息
	 * 
	 * @Author chaixueteng
	 * @2016年10月29日下午3:22:07
	 */
	public String getkennel(String name) {
		JSONArray json = null;
		if (name.trim().contains("犬舍")) {
			String name2 = name.replaceAll("犬舍", "");
			List<Kennel> kennelMsg = dao.getKennel(name2);
			JsonConfig jsonConfig = new JsonConfig();
			jsonConfig.setCycleDetectionStrategy(CycleDetectionStrategy.LENIENT);
			json = JSONArray.fromObject(kennelMsg, jsonConfig);
		} else {
			List<Kennel> kennelMsg = dao.getKennel(name.trim());
			JsonConfig jsonConfig = new JsonConfig();
			jsonConfig.setCycleDetectionStrategy(CycleDetectionStrategy.LENIENT);
			json = JSONArray.fromObject(kennelMsg, jsonConfig);
		}

		return json.toString();
	}

	/**
	 * 根据犬舍英文名模糊查询相同的信息
	 * 
	 * @Author chaixueteng
	 * @2016年10月29日下午3:22:45
	 */
	public String getKennelFci(String fciName) {
		JSONArray json = null;
		if (fciName.trim().contains("’")) {
			String[] split = fciName.split("’");
			String string = split[0];
			List<Kennel> kennelMsg = dao.getKennelFci(string);
			JsonConfig jsonConfig = new JsonConfig();
			jsonConfig.setCycleDetectionStrategy(CycleDetectionStrategy.LENIENT);
			json = JSONArray.fromObject(kennelMsg, jsonConfig);
		} else {
			List<Kennel> kennelMsg = dao.getKennelFci(fciName.trim());
			JsonConfig jsonConfig = new JsonConfig();
			jsonConfig.setCycleDetectionStrategy(CycleDetectionStrategy.LENIENT);
			json = JSONArray.fromObject(kennelMsg, jsonConfig);
		}

		return json.toString();
	}

	public int kennelTypeChange(Kennel kennel) {
		kennel.preUpdate();
		return dao.kennelTypeChange(kennel);
	}

	public int kennelEndDatecharge(Kennel kennel) {
		kennel.preUpdate();
		return dao.kennelEndDatecharge(kennel);
	}

	/**
	 * 当前月新注册的犬舍(已缴费 未审核 )
	 * 
	 * @Author chaixueteng
	 * @2017年2月6日下午2:05:58
	 */
	public int getKennelByMouth(String type) {
		Calendar c = Calendar.getInstance();
		c.add(Calendar.MONTH, 0);
		c.set(Calendar.DAY_OF_MONTH, 1);// 设置为1号,当前日期既为本月第一天
		Date beginTime = c.getTime();
		if ("fci".equals(type)) {
			return dao.getNewFciKennelByMouth(beginTime);
		} else {
			return dao.getNewKennelByMouth(beginTime, type);

		}
	}

	/**
	 * FCI犬舍信息导出
	 * 
	 * @throws IOException
	 * @Author chaixueteng
	 * @2017年2月8日上午11:16:45
	 */
	public void export(Kennel kennel, HttpServletResponse response) throws IOException {
		List<Kennel> list = findList(kennel);
		for (Kennel temp : list) {
			// 会员信息
			Member member = temp.getMember();
			if (member != null) {
				temp.setMemberNameCn(member.getName());
				temp.setMemberAddress(member.getProvince() + member.getCity() + member.getStreet());
				// 拆分姓名英文
				if (!StringUtils.isBlank(member.getNameEn())) {
					String[] nameEns = member.getNameEn().split(" ");
					StringBuilder firstName = new StringBuilder();
					for (int i = 1; i < nameEns.length; i++) {
						firstName.append(nameEns[i]);
					}
					temp.setMemberNameEnFirst(firstName.toString());
					temp.setMemberNameEnLast(nameEns[0]);
				}
				// 查询主营犬种
				temp.setMainBreed(dogBirthCertificateDao.getMainBreedByMemberCode(temp.getMemberCode()));
			}
			// 审核状态字典
			temp.setReviewState(DictUtils.getDictLabel(temp.getReviewState(), "kennel_review_state", ""));
			temp.setFciReviewState(DictUtils.getDictLabel(temp.getFciReviewState(), "fci_review_state", ""));
		}
		new ExportExcel("犬舍信息", Kennel.class, 1).setDataList(list).write(response, "犬舍信息表.xlsx").dispose();
	}

	// 犬舍审核关闭 2019年2月22日17:15:09
	/**
	 * 犬舍核算保留操作
	 * 
	 * @Author chaixueteng
	 * @2017年2月14日上午10:05:35
	 */
	// @Transactional(readOnly = false)
	// public void calculteSql(StringBuffer buffer, String businessId) {
	// Kennel kennel = get(businessId);
	// String date = "";
	// Calendar calendar = Calendar.getInstance();//日历对象
	// SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
	// if(kennel.getCalculationDate()!=null){//核算日期为空不可更新核算
	// if(format.format(kennel.getCalculationDate()).endsWith("-02-29")){
	// Calendar c = Calendar.getInstance();
	// c.setTime(kennel.getCalculationDate());
	// c.add(Calendar.YEAR, 1);//年份+1
	// c.add(Calendar.DATE, 1);//日期+1
	// date =format.format(c.getTime());
	// date=date+" 23:59:59";
	// }else{
	// calendar.setTime(kennel.getCalculationDate());//设置当前日期
	// calendar.add(Calendar.YEAR, 1);//年份+1
	// date =format.format(calendar.getTime());
	// date=date+" 23:59:59";
	// }
	// //start 冻结后缴纳保留费 恢复待审核状态,及犬舍名恢复 20180131chaixueteng ;20180227王梓取消审核状态的恢复
	// 保留之前审核状态
	// if(StringUtils.isNotBlank(kennel.getReviewStateBack())&&!"end".equals(kennel.getReviewStateBack())){
	// String[] state =kennel.getReviewStateBack().split("-");
	// kennel.setReviewState(state[0]);
	// kennel.setFciReviewState(state[1]);
	// kennel.setReviewStateBack("end");
	// }
	// dao.update(kennel);
	// //end
	// buffer.append("UPDATE kennel SET ").append("calculation_date =
	// '").append(date).append("' ").append(",calculation_state
	// ='0'").append(",calculation_flag ='0'")
	// .append("WHERE id='").append(businessId).append("'");
	// sysCouponService.saveCoupon(kennel.getKennelCode(),"登记卡抵扣劵","50",null,"dogBirthCertificate","Y",10);
	// }else{
	// throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "核算日期为空不可缴纳保留费");
	// }
	// }

	/**
	 * 根据核算日期得到需要更改核算状态的列表 1.状态为待核算 2.核算flag为0
	 * 
	 * @Author chaixueteng
	 * @2017年2月14日下午2:55:25
	 */
	// @Transactional(readOnly = false)
	// public int kennelUpdateCalculateState(){
	// //计划任务修改为凌晨1点执行前一天需要核算的犬舍
	// Calendar nowDate = Calendar.getInstance();// 得到一个Calendar的实例
	// nowDate.setTime(new Date());
	// nowDate.add(Calendar.DATE, -1);
	// Date resultDate = nowDate.getTime();
	// List<Kennel> l=dao.findKennlCalculationList(resultDate);
	// int count = 0;
	// for (Kennel kennel : l) {
	// Date calculationDate = kennel.getCalculationDate();
	// Calendar ca = Calendar.getInstance();// 得到一个Calendar的实例
	// ca.setTime(calculationDate);// 月份是从0开始的，所以11表示12月
	// ca.add(Calendar.YEAR, -1); // 年份减1
	// Date oneYearAge = ca.getTime();//得到一年前的日期
	// Calendar c = Calendar.getInstance();// 得到一个Calendar的实例
	// c.setTime(calculationDate);// 月份是从0开始的，所以11表示12月
	// c.add(Calendar.YEAR, 1); // 年份加1
	// Date oneYear = c.getTime();
	// String memberCode = kennel.getMemberCode();
	// String name = kennel.getName();
	// if(StringUtils.isNotBlank(memberCode)){
	// List findBirthByBreeder =
	// dogBirthCertificateDao.findBirthByBreeder(memberCode,
	// calculationDate,oneYearAge);
	// if(findBirthByBreeder.size()<10){ // 年度申请出生纸不足10张的 注销该犬舍并且会员不可再申请犬舍
	// 犬舍名称可以再次使用
	// kennel.setReviewStateBack(kennel.getReviewState()+"-"+kennel.getFciReviewState());
	// kennel.setReviewState("3"); //注销
	// kennel.setFciReviewState("4"); //注销
	// kennel.setCalculationFlag("1");//不参与核算
	// kennel.setCalculationState("2");//核算不通过
	// Date date = new Date();
	// SimpleDateFormat sdf = new SimpleDateFormat("yyyy年M月d日 HH:mm:ss");
	// if(StringUtils.isNotBlank(kennel.getRemarks())){
	// kennel.setRemarks(kennel.getRemarks()+"该犬舍于"+sdf.format(date)+",自动核算未通过,已冻结");
	// }else{
	// kennel.setRemarks("该犬舍于"+sdf.format(date)+",自动核算未通过,已冻结");
	// }
	// remindService.addKennelRemind("您名下<"+name+">核算不合格已经冻结，如有疑问请联系协会",
	// kennel.getMemberCode()); //会员中心推送注销消息
	// }else{//大于10张 核算通过 核算日期延长一年
	// kennel.setCalculationFlag("0");
	// kennel.setCalculationState("1");//通过
	// kennel.setCalculationDate(oneYear);//核算日期延长一年
	// }
	// dao.update(kennel);
	// count=count+1;
	// }
	// }
	// return count;
	// }

	/**
	 * @description: 犬舍注册付款后业务
	 * @author: laiguanglong
	 * @date: 2017/2/8 9:43
	 */
	// public void kennelZCAfterPay(String id){
	// Kennel kennel = get(id);
	// //设置核算状态为0:待核算;核算日期为一年后;是否参与核算标记位为0: 是
	// Date now = new Date();
	// //如果是2月29日，计算一年后为2月28日；所以加一天后计算一年后则为3月1日
	// if(new SimpleDateFormat("yyyy-MM-dd").format(now).endsWith("-02-29")){
	// Calendar c = Calendar.getInstance();
	// c.setTime(now);
	// c.add(Calendar.DATE, 1);//日期+1
	// now = c.getTime();
	// }
	// String oneYearLaterDateString = DateUtils.get1YearLater(now);
	// Date calculationDate = null;
	// try {
	// calculationDate = new SimpleDateFormat("yyyy-MM-dd
	// HH:mm:ss").parse(oneYearLaterDateString+" 23:59:59");
	// }catch (Exception e){
	// throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "格式化犬舍核算日期出错");
	// }
	// kennel.setCalculationDate(calculationDate);
	// dao.update(kennel);
	// }
	/**
	 * @description: 犬舍续费升级付款后业务(核算)
	 * @author: chaixueteng
	 * @date: 2017/2/8 9:43
	 */

	// public void kennelXFAfterPay(String id){
	// Kennel kennel = get(id);
	// if(kennel.getCalculationDate()==null&&"0".equals(kennel.getCalculationFlag())&&!"3".equals(kennel.getCalculationState())){//只有核算日期为空参与核算并且非免核算的的才进行操作
	// Date endTime=kennel.getEndTime();
	// if(endTime==null){
	// endTime = new Date();
	// }
	// Calendar c = Calendar.getInstance();
	// int year =c.get(Calendar.YEAR)+1;
	// c.setTime(endTime);
	// c.set(Calendar.YEAR,year);// 年
	// c.set(Calendar.HOUR_OF_DAY, 23);//时
	// c.set(Calendar.MINUTE, 59);//分
	// c.set(Calendar.SECOND, 59);//秒
	// Date now = c.getTime();
	// kennel.setCalculationDate(now);
	// dao.update(kennel);
	// }
	// }
	/**
	 * @Description： 犬舍转杜高马犬
	 * 
	 * @author: zhangxiang
	 * @date: 2019/07/22 11:06
	 */
	@Transactional(readOnly = false)
	public void editMemberType(String searchMemberCode, String memberCode, String remarklogs) {

		Member member = memberDao.getByMemberCode(searchMemberCode);
		Member yOldMember = memberDao.getByMemberCode(memberCode);
		if (member != null) {
			if (!(MemberUtil.isProfessionalMember(member) && MemberUtil.isActiveMember(member))) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该会员不是有效的专业会员");
			}
		} else {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该会员信息有误,修改失败");
		}
		Kennel kennel = this.getKennelByMemberCode(memberCode);
		if (kennel == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍信息有误");
		}
		// 修改犬舍
		kennel.preUpdate();
		kennel.setMemberCode(searchMemberCode);
		kennel.setKennelCode(searchMemberCode);
		try {
			kennelDao.updateKennel(kennel);
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "修改犬舍类型错误");
		}
		// 记录会员变动日志
		// 转之前会员
		Member yNewMember = memberDao.getByMemberCode(memberCode);
		sysMemberActivityRecordService.saveMemberRecordWoOrder(yOldMember, yNewMember, UserUtils.getUser(), null);
		// 转之后会员
		Member xNewMember = memberDao.getByMemberCode(searchMemberCode);
		sysMemberActivityRecordService.saveMemberRecordWoOrder(member, xNewMember, UserUtils.getUser(), null);
		// 插入日志
		SysBussinessLogs bussinessLogs = new SysBussinessLogs();
		bussinessLogs.preInsert();
		bussinessLogs.setType("2");
		bussinessLogs.setParamsA(memberCode);// 修改前犬舍号
		bussinessLogs.setParamsB(searchMemberCode);// 修改前犬舍号
		bussinessLogs.setParamsD(kennel.getId());// 修改犬舍id
		bussinessLogs.setRemarks(remarklogs);
		try {
			sysBussinessLogsDao.insert(bussinessLogs);
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "插入日志记录失败");
		}

	}

	public RESTResponse kennelList(HttpServletRequest request, HttpServletResponse response,Boolean needStar) {
		/** 自 2021-06-09 将官网及APP首页犬舍查询，统一接口逻辑 **/
		String name = request.getParameter("name");
		name = Optional.ofNullable(name).orElseThrow(()->new ZAException(ZAErrorCode.ZA_VALID_FAILED, "请输入犬舍中文或英文名称")).trim();
		if("犬舍".equals(name) || "犬".equals(name) || "舍".equals(name)){
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "暂不支持只输入“犬”，“舍”和“犬舍”，请尝试其它关键字");
		}
		String pageNo = Optional.ofNullable(request.getParameter("pageNo")).orElse("1");
		String pageSize = Optional.ofNullable(request.getParameter("pageSize")).orElse("5");
		Page<Kennel> page = new Page<Kennel>(request,response);
		page.setPageNo(Integer.valueOf(pageNo));
		page.setPageSize(Integer.valueOf(pageSize));
		Page<Kennel> kennelPage = this.pagedSelectByName(page,name);
		Map<String,Member> memberMap = new HashMap<>();
		List<Map<String, Object>> resultList = new ArrayList<>();
		Optional.ofNullable(kennelPage.getList()).ifPresent(kennelList->{
			Set<String> memberCodeSet = kennelList.stream().map(Kennel::getMemberCode).collect(Collectors.toSet());
			List<Member> memberList = new ArrayList<>();
			Optional.ofNullable(memberCodeSet).ifPresent(memberCodes->{
				memberList.addAll(Optional.ofNullable(memberDao.getByMemberCodes(memberCodes)).orElse(Collections.emptyList()));
				memberMap.putAll(memberList.stream().collect(Collectors.toMap(Member::getMemberCode,Function.identity())));
			});
			resultList.addAll(kennelList.stream().map(data -> {
				Map<String, Object> map = new HashMap<String, Object>();
				map.put("id", data.getId());
				map.put("name", data.getName());
				map.put("nameEn", data.getNameEn());
				map.put("memberCode", data.getMemberCode());
				Member member = memberMap.get(data.getMemberCode());
				Optional.ofNullable(member).ifPresent(m->{
					map.put("address",Optional.ofNullable(m.getProvince()).orElse("") + Optional.ofNullable(m.getCity()).orElse(""));
					//姓名长度大于3，隐藏头两个字，其余隐藏头一个字
					map.put("owner", member.getName().replace(member.getName().substring(0,member.getName().length()-1), member.getName().length() > 2 ? "**" : "*"));
				});
				map.put("type", DictUtils.getDictLabel(data.getType(), "kennel_type", ""));
				if(Boolean.valueOf(Optional.ofNullable(needStar).orElse(Boolean.FALSE))) {
					//取消星级犬舍内容2023-03-03
//					map.put("star",getKennelStart(data.getMemberCode()));
					map.put("star","");
				}
				return map;
			}).collect(Collectors.toList()));
		});
		return new RESTResponse("data",new PageBeanResult<Map<String, Object>>(resultList,page.getCount()));
	}
	
	
	public Page<Kennel> pagedSelectByName(Page<Kennel> page,String kennelName) {
		Kennel entity = new Kennel();
		entity.setName(kennelName);
		int pageNo = page.getPageNo();
		int pageSize = page.getPageSize();
		entity.setPage(page);
		page.setList(kennelDao.pagedSelectByName(entity));
		if((pageNo-1)*pageSize>=page.getCount()){
			page.setList(new ArrayList<Kennel>(0));
		}
		return page;
	}

	public String getKennelStart(String memberCode) {
		StringBuilder result = new StringBuilder("");
		try {
			OkHttpClient client = new OkHttpClient.Builder().connectTimeout(10, TimeUnit.SECONDS).readTimeout(20, TimeUnit.SECONDS).build();
			Request request = new Request.Builder().url(KENNEL_START_URL + memberCode).build();
			String data = client.newCall(request).execute().body().string();
			Optional.ofNullable(data).ifPresent(r->{
				JSONObject jsonObject = JSONObject.fromObject(data);
				Optional.ofNullable(jsonObject.get("data")).ifPresent(ob->{
					result.append(JSONObject.fromObject(ob).get("star"));
				});
			});
		} catch (Exception ex) {
			//获取犬舍星级不影响正常业务返回
			//throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬舍星级获取失败");
			System.out.println("获取犬舍星级失败，失败原因："+ex.getMessage());
		}
		return result.toString();
	}
}