package com.cku.oa.sys.service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.sys.code.SysCodeUtil;
import com.cku.oa.sys.dao.AppraiserDao;
import com.cku.oa.sys.entity.Appraiser;
import com.cku.oa.sys.entity.AppraiserPrint;
import com.cku.oa.sys.entity.BaseFile;
import com.cku.oa.sys.entity.Org;
import com.cku.oa.sys.util.FnsUtils;
import com.cku.oa.sys.util.Img2Base64Util;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.StringUtils;

/**
 * 鉴定师表Service
 * @author zhangjunwu
 * @version 2019-10-16
 */
@Service
@Transactional(readOnly = true)
public class AppraiserService extends CrudService<AppraiserDao, Appraiser> {

	@Autowired
	private FileService fileService;
	
	public Appraiser get(String id) {
		return super.get(id);
	}
	
	public List<Appraiser> findList(Appraiser appraiser) {
		return super.findList(appraiser);
	}
	
	public Page<Appraiser> findPage(Page<Appraiser> page, Appraiser appraiser) {
		return super.findPage(page, appraiser);
	}
	
	@Transactional(readOnly = false)
	public void save(Appraiser appraiser) {
		if(StringUtils.isBlank(appraiser.getId())) {
			appraiser.setCertificateNum(SysCodeUtil.getAppraiserCertifiedCode());
			appraiser.setPrintFlag("0");
		}
		super.save(appraiser);
	}
	
	@Transactional(readOnly = false)
	public void delete(Appraiser appraiser) {
		super.delete(appraiser);
	}
	
	@Transactional(readOnly = false)
	public void updatePrintFlag(String appraiserId,String printFlag) {
		Appraiser entity = this.get(appraiserId);
		if(entity == null) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "未能找到该认证师");
		}
		if("1".equals(printFlag) && printFlag.equals(entity.getPrintFlag())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "该证书已打印，请重置后再试");
		}
		dao.updatePrintFlag(appraiserId,printFlag);
	}
	
	private final static SimpleDateFormat SDF_YMD = new SimpleDateFormat("yyyy-MM-dd");
	
	public AppraiserPrint print(String id) {
		Appraiser entity = this.get(id);
		if(entity == null) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "未能找到该认证师");
		}

		AppraiserPrint rest = new AppraiserPrint();
		rest.setId(id);
		rest.setName(entity.getName());
		rest.setNameEn(entity.getNameEn());
		rest.setPhoto(entity.getPhoto());
		rest.setCertificateNum(entity.getCertificateNum());
		rest.setMobile(entity.getMobile());
		rest.setProvince(entity.getProvince());
		rest.setCity(entity.getCity());
		rest.setStreet(entity.getStreet());
		
		rest.setIssuerDate(entity.getCreateDate()==null?"":SDF_YMD.format(entity.getCreateDate()));
		
		Org org = FnsUtils.getOrgById(entity.getOrgId());
		rest.setOrgName(org.getName());
		rest.setOrgAddress(org.getAddress());
		rest.setOrgCode(org.getMemberCode());
		rest.setEndDate(org.getEndDate()==null?"":SDF_YMD.format(org.getEndDate()));
		
		
		//拆分字符，中间加空格
		if(StringUtils.isNotBlank(rest.getCertificateNum())) {
			rest.setCertificateNum(rest.getCertificateNum().replace("", " ").trim());
		}
		if(StringUtils.isNotBlank(rest.getOrgCode())) {
			rest.setOrgCode(rest.getOrgCode().replace("", " ").trim());
		}
		
		ByteArrayOutputStream outPut = null;
		try {
			BaseFile file = fileService.get(entity.getPhoto());
			if(file != null) {
				outPut = new ByteArrayOutputStream();
				fileService.fileDownload(file,outPut);
				rest.setPhotoBase64("data:"+file.getType()+";base64,"+Img2Base64Util.getImgStr(outPut.toByteArray()));
			}
		} catch (Exception e) {
			e.printStackTrace();
		}finally {
			try {
				if(outPut != null)
				outPut.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		return rest;
	}
	
}