package com.cku.oa.sys.web;

import java.math.BigInteger;
import java.time.LocalDate;
import java.time.YearMonth;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.cku.core.ResultDto;
import com.cku.core.ZAErrorCode;
import com.cku.oa.sys.entity.Workday;
import com.cku.oa.sys.service.WorkdayService;
import com.cku.oa.sys.util.LocalDateUtils;
import com.cku.oa.sys.vo.DayData;
import com.cku.oa.sys.vo.MonthData;
import com.cku.oa.sys.vo.YearData;
import com.thinkgem.jeesite.common.config.Global;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.common.web.BaseController;

/**
 * 工作日历表Controller
 * 
 * @author Sunny
 * @version 2024-10-17
 */
@Controller
@RequestMapping(value = "${adminPath}/sys/workday")
public class WorkdayController extends BaseController {

	@Autowired
	private WorkdayService workdayService;

	@ModelAttribute
	public Workday get(@RequestParam(required = false) String id) {
		Workday entity = null;
		if (StringUtils.isNotBlank(id)) {
			entity = workdayService.get(id);
		}
		if (entity == null) {
			entity = new Workday();
		}
		return entity;
	}

	@RequiresPermissions("sys:workday:view")
	@RequestMapping(value = { "list", "" })
	public String list(Workday workday, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<Workday> page = workdayService.findPage(new Page<Workday>(request, response), workday);
		model.addAttribute("page", page);
		return "oa/sys/workday/workdayList";
	}

	@ResponseBody
	@RequiresPermissions("sys:workday:view")
	@RequestMapping(value = "/list/{dateYear}")
	public ResultDto<YearData> list(@PathVariable("dateYear") String dateYear, HttpServletRequest request,
			HttpServletResponse response) {
		List<Workday> workdayList = workdayService.findListByYear(dateYear);
		if (CollectionUtils.isEmpty(workdayList)) {
			return ResultDto.success(YearData.builder().year(dateYear).existDb(false).monthGroup(null).build());
		}
		Map<String, Workday> workdayMap = new HashMap<>();
		workdayMap.putAll(workdayList.stream().collect(Collectors.toMap(Workday::getDateStr, Function.identity())));
		List<List<MonthData>> monthGroup = new ArrayList<>();
		for (int month = 1; month <= 12; month = month + 3) {
			List<MonthData> months = new ArrayList<>();
			for (int tmonth = 0; tmonth < 3; tmonth++) {
				YearMonth yearMonth = YearMonth.of(Integer.parseInt(dateYear), month + tmonth);
				MonthData monthData = MonthData.builder().yearMonth(yearMonth.getMonthValue())
						.yearMonthLabel(yearMonth.getMonthValue()).daysGroup(getDaysInMonth(yearMonth)).build();
				months.add(monthData);
			}
			months.forEach(m -> {
				m.getDaysGroup().forEach(md -> {
					md.forEach(d -> {
						if (workdayMap.containsKey(d.getDayStr())) {
							Workday workday = workdayMap.get(d.getDayStr());
							d.setDayLabel(workday.getName());
							d.setSupplement(Objects.isNull(workday.getOffDay()) || !workday.getOffDay() ? Boolean.TRUE
									: Boolean.FALSE);
							d.setHoliday(Objects.nonNull(workday.getOffDay()) && workday.getOffDay() ? Boolean.TRUE
									: Boolean.FALSE);
						}
					});
				});
			});
			monthGroup.add(months);
		}
		return ResultDto.success(YearData.builder().year(dateYear).existDb(true).monthGroup(monthGroup).build());
	}

	private List<List<DayData>> getDaysInMonth(YearMonth yearMonth) {
		LocalDate firstDayOfMonth = yearMonth.atDay(1);
		int daysInMonth = yearMonth.lengthOfMonth();
		List<DayData> days = new ArrayList<>();
		// 添加前导空日
		for (int i = 1; i < firstDayOfMonth.getDayOfWeek().getValue(); i++) {
			days.add(DayData.builder().day(BigInteger.ZERO.intValue()).dayLabel("").dayStr("")
					.belongsToMonth(Boolean.FALSE).build());
		}
		// 添加当月所有日
		for (int day = 1; day <= daysInMonth; day++) {
			days.add(DayData.builder().day(day).dayLabel(String.valueOf(day))
					.dayStr(LocalDateUtils
							.formatLocalDate(LocalDate.of(yearMonth.getYear(), yearMonth.getMonthValue(), day)))
					.belongsToMonth(Boolean.TRUE).build());
		}
		// 补足 6x7 位 主要满足样式统一
		if (days.size() < 42) {
			while (days.size() < 42) {
				days.add(DayData.builder().day(BigInteger.ZERO.intValue()).dayLabel("").dayStr("")
						.belongsToMonth(Boolean.FALSE).build());
			}
		}
		return IntStream.range(0, (int) Math.ceil(days.size() / (double) 7))
				.mapToObj(i -> days.subList(i * 7, Math.min((i + 1) * 7, days.size()))).collect(Collectors.toList());
	}

	@RequiresPermissions("sys:workday:view")
	@RequestMapping(value = "form")
	public String form(Workday workday, Model model) {
		model.addAttribute("workday", workday);
		return "oa/sys/workday/workdayForm";
	}

	@RequiresPermissions("sys:workday:edit")
	@RequestMapping(value = "save")
	public String save(Workday workday, Model model, RedirectAttributes redirectAttributes) {
		if (!beanValidator(model, workday)) {
			return form(workday, model);
		}
		workdayService.save(workday);
		addMessage(redirectAttributes, "保存工作日历成功");
		return "redirect:" + Global.getAdminPath() + "/sys/workday/?repage";
	}

	@RequiresPermissions("sys:workday:edit")
	@RequestMapping(value = "addWeekend")
	@ResponseBody
	public ResultDto<Boolean> addWeekend(Workday workday) {
		try {
			workdayService.addWeekend(workday);
		} catch (Exception e) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, e.getMessage());
		}

		return ResultDto.success(Boolean.TRUE);
	}

	@RequiresPermissions("sys:workday:edit")
	@RequestMapping(value = "editWeekend")
	@ResponseBody
	public ResultDto<Boolean> editWeekend(Workday workday) {
		try {
			workdayService.editWeekend(workday);
		} catch (Exception e) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, e.getMessage());
		}

		return ResultDto.success(Boolean.TRUE);
	}

	@RequiresPermissions("sys:workday:edit")
	@RequestMapping(value = "deleteWeekend")
	@ResponseBody
	public ResultDto<Boolean> deleteWeekend(Workday workday, Model model, RedirectAttributes redirectAttributes) {
		try {
			workdayService.deleteWeekend(workday);
		} catch (Exception e) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, e.getMessage());
		}
		return ResultDto.success(Boolean.TRUE);
	}

	@RequiresPermissions("sys:workday:edit")
	@RequestMapping(value = "delete")
	public String delete(Workday workday, RedirectAttributes redirectAttributes) {
		workdayService.delete(workday);
		addMessage(redirectAttributes, "删除工作日历成功");
		return "redirect:" + Global.getAdminPath() + "/sys/workday/?repage";
	}

}