package com.cku.restful.v1.open;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import com.auth0.jwt.exceptions.TokenExpiredException;
import com.cku.core.ListResultDto;
import com.cku.core.RESTResponse;
import com.cku.core.ResultDto;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.sys.util.IPLimitUtil;
import com.cku.restful.v1.appreciation.model.RestAppreciationApply;
import com.cku.restful.v1.appreciation.model.RestAppreciationCourse;
import com.cku.restful.v1.appreciation.model.RestAppreciationCourseware;
import com.cku.restful.v1.appreciation.model.RestAppreciationLogin;
import com.cku.restful.v1.appreciation.model.RestAppreciationUser;
import com.cku.restful.v1.appreciation.model.RestAppreciationValidCode;
import com.cku.restful.v1.appreciation.profile.AppreciationUserProfile;
import com.cku.restful.v1.appreciation.service.RestAppreciationService;
import com.cku.restful.v1.sys.web.BaseRestController;
import com.cku.util.TokenUtil;
import com.google.gson.Gson;
import com.thinkgem.jeesite.common.utils.JedisUtils;
import com.thinkgem.jeesite.common.utils.StringUtils;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Api(tags = "鉴赏课")
@RestController
@RequestMapping(value = "/api/v1/open/appreciation")
public class OpenAppreciationController extends BaseRestController {

	@Autowired
	private RestAppreciationService restAppreciationService;

	@ApiOperation(value = "获取价格")
	@GetMapping(value = "/getPrice/{applyType}")
	@ResponseBody
	public ResultDto<String> getPrice(@PathVariable String applyType) throws Exception {
		return restAppreciationService.getPrice(applyType);
	}

	@ApiOperation(value = "获取验证码")
	@PostMapping(value = "/sendCodeMessage")
	@ResponseBody
	public ResultDto<Boolean> sendCodeMessage(@RequestBody RestAppreciationValidCode appreciationValidCode)
			throws Exception {
		return restAppreciationService.sendCodeMessage(appreciationValidCode);
	}

	@ApiOperation(value = "登录")
	@PostMapping(value = "/login")
	@ResponseBody
	public ResultDto<RestAppreciationUser> login(@RequestBody RestAppreciationLogin appreciationLogin)
			throws Exception {
		return restAppreciationService.login(appreciationLogin);
	}

	@ApiOperation(value = "课程列表查询")
	@GetMapping(value = "/courses")
	@ResponseBody
	public ListResultDto<RestAppreciationCourse> courses(HttpServletRequest request) throws Exception {
		// 独立登录模块，在此验证登录信息，符合在继续查询
		AppreciationUserProfile user = this.validateAppreciationUser(request);
		return restAppreciationService.findCourseList(user.getMobile());
	}

	@ApiOperation(value = "课程资料查询")
	@GetMapping(value = "/courses/{coursesId}/coursewares")
	@ResponseBody
	public ListResultDto<RestAppreciationCourseware> coursewares(@PathVariable("coursesId") String coursesId,
			HttpServletRequest request) throws Exception {
		// 独立登录模块，在此验证登录信息，符合在继续查询
		this.validateAppreciationUser(request);
		return restAppreciationService.findCoursewareList(coursesId);
	}

	private AppreciationUserProfile validateAppreciationUser(HttpServletRequest request) {
		String token = request.getHeader("token");
		if (StringUtils.isEmpty(token)) {
			throw new ZAException(ZAErrorCode.ZA_ERC_INVALID_STATUS, ZAErrorCode.ZA_ERC_INVALID_STATUS_MSG);
		}
		try {
			String subject = TokenUtil.parseJwt(token).getSubject();
			AppreciationUserProfile userProfile = new Gson().fromJson(subject, AppreciationUserProfile.class);
			String redisKey = restAppreciationService.generateLoginRdsKey(userProfile.getMobile(),
					userProfile.getMemberCode());
			if (!JedisUtils.exists(redisKey)) {
				throw new TokenExpiredException("登录超时，请稍后再试");
			}
			return userProfile;
		} catch (TokenExpiredException ex) {
			log.info("================解析token失败，Token过期================" + ex.getMessage());
			throw new ZAException(ZAErrorCode.ZA_ERC_INVALID_STATUS, ZAErrorCode.ZA_ERC_INVALID_STATUS_MSG);
		} catch (Exception ex) {
			log.info("================================" + ex.getMessage());
			throw new ZAException(ZAErrorCode.ZA_ERROR, "服务器处理数据发生异常");
		}
	}

	@ApiOperation(value = "报名")
	@PostMapping(value = "/apply")
	@ResponseBody
	public RESTResponse apply(@RequestBody @Valid RestAppreciationApply vo, HttpServletRequest request)
			throws Exception {
		return restAppreciationService.apply(vo, IPLimitUtil.getIpAddr(request));
	}

}
