/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.restful.v1.dog.service;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.PageBeanResult;
import com.cku.core.RESTResponse;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.dao.DogDao;
import com.cku.oa.dog.dao.DogSemenDao;
import com.cku.oa.dog.entity.Dog;
import com.cku.oa.dog.entity.DogSemen;
import com.cku.oa.dog.entity.DogSemenChange;
import com.cku.oa.dog.service.DogSemenChangeService;
import com.cku.oa.dog.service.DogSemenService;
import com.cku.oa.finance.entity.PaymentCart;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.finance.service.PaymentCartService;
import com.cku.oa.finance.service.PaymentChargingItemService;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.LoginServerService;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.restful.v1.sys.utils.BeanUtil;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

@Service
@Transactional(readOnly = true)
public class RestDogSemenChangeService {

	private static final String PAYMENTCHARGINGITEM = "dog_semen_change@free";

	@Autowired
	private MemberService memberService;
	@Autowired
	public DogSemenDao dogSemenDao;
	@Autowired
	public DogSemenChangeService dogSemenChangeService;
	@Autowired
	private DogDao dogDao;
	@Autowired
	public DogSemenService dogSemenService;
	@Autowired
	public PaymentChargingItemService paymentChargingItemService;
	@Autowired
	private PaymentCartService paymentCartService;
	@Autowired
	public LoginServerService loginServerService;

	public RESTResponse get(String id) {
		Member member = UserUtils.getLoginMember();
		DogSemenChange po = dogSemenChangeService.get(id);
		if (po == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
		}
		if (!po.getOriginMemberCode().equals(member.getMemberCode())
				&& !po.getNewMemberCode().equals(member.getMemberCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "id有误");
		}
		JSONObject data = new JSONObject();
		data.put("id", getString(po.getId()));
		dogSemenService.setDogInfo(po.getDogId(), data);
		data.put("originMemberCode", getString(po.getOriginMemberCode()));
		data.put("originMemberName", getString(po.getOriginMemberName()));
		data.put("newMemberCode", getString(po.getNewMemberCode()));
		data.put("newMemberName", getString(po.getNewMemberName()));
		data.put("createDate", getString(com.cku.util.DateUtils.dateToStr(po.getCreateDate())));
		data.put("businessState", getString(po.getBusinessState()));
		data.put("reviewState", getString(po.getReviewState()));
		data.put("reviewRemarks", getString(po.getReviewRemarks()));
		JSONArray ja = new JSONArray();
		for (String semenCode : po.getSemenCode().split(",")) {
			DogSemen dogSemen = dogSemenDao.getBySemenCode(semenCode);
			JSONObject jo = new JSONObject();
			jo.put("id", getString(dogSemen.getId()));
			jo.put("originalSemenCode", getString(dogSemen.getOriginalSemenCode()));
			jo.put("semenCode", getString(dogSemen.getSemenCode()));
			ja.add(jo);
		}
		data.put("dogSemenList", ja);
		data.put("semenNumber", ja.size());
		return new RESTResponse("data", data);
	}

	public RESTResponse getList(HttpServletRequest request) {
		Member member = UserUtils.getLoginMember();
		// 获取参数
		String pedigreeCertified = request.getParameter("pedigreeCertified");
		String originMemberCode = request.getParameter("originMemberCode");
		String newMemberCode = request.getParameter("newMemberCode");
		String semenCode = request.getParameter("semenCode");
		String type = request.getParameter("type");
		String pageNo = request.getParameter("pageNo");
		String pageSize = request.getParameter("pageSize");
		// 参数校验
		int pageNoInt = 1;
		int pageSizeInt = 10;
		try {
			if (StringUtils.isNotBlank(pageNo)) {
				pageNoInt = Integer.parseInt(pageNo);
			}
			if (StringUtils.isNotBlank(pageSize)) {
				pageSizeInt = Integer.parseInt(pageSize);
			}
		} catch (Exception e) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "pageNo和pageSize必须都为数字");
		}
		if (!"origin".equals(type) && !"new".equals(type)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "type参数有误，不能为空且只能为origin或new");
		}
		DogSemenChange example = new DogSemenChange();
		// 设置分页参数
		Page<DogSemenChange> page = new Page<DogSemenChange>();
		page.setPageNo(pageNoInt);
		page.setPageSize(pageSizeInt);
		// 设置查询条件
		if ("origin".equals(type)) {
			example.setOriginMemberCode(member.getMemberCode());
			example.setPaymentState("2");
			if (StringUtils.isNotBlank(newMemberCode)) {
				example.setNewMemberCodeAndNewMemberNameLike(newMemberCode);
			}
		} else if ("new".equals(type)) {
			example.setNewMemberCode(member.getMemberCode());
			if (StringUtils.isNotBlank(newMemberCode)) {
				example.setOriginMemberCodeAndOriginMemberNameLike(originMemberCode);
			}
		}
		example.setPage(page);
		if (StringUtils.isNotBlank(pedigreeCertified)) {
			example.setPedigreeCertifiedLike(pedigreeCertified);
		}
		if (StringUtils.isNotBlank(semenCode)) {
			example.setSemenCodeLike(semenCode);
		}
		// 根据当前登录是否杜高会员来判断个人狗列表是杜高犬还是CKU犬
		if (StringUtils.isBlank(example.getDogBreed())) {
			// 根据当前登录端查询犬种
			example.getSqlMap().put("dogBreedMap", loginServerService.dataScopeFilter());
		} else {
			example.getSqlMap().put("dogBreedMap", " ='" + example.getDogBreed() + "'");
		}
		// 查询并组装返回对象
		Page<DogSemenChange> result = dogSemenChangeService.findPage(page, example);
		List<DogSemenChange> dogSemenChangeList = result.getList();
		List<JSONObject> list = new ArrayList<JSONObject>();
		for (DogSemenChange po : dogSemenChangeList) {
			JSONObject vo = new JSONObject();
			vo.put("id", getString(po.getId()));
			dogSemenService.setDogInfo(po.getDogId(), vo);
			vo.put("originMemberCode", getString(po.getOriginMemberCode()));
			vo.put("originMemberName", getString(po.getOriginMemberName()));
			vo.put("newMemberCode", getString(po.getNewMemberCode()));
			vo.put("newMemberName", getString(po.getNewMemberName()));
			vo.put("createDate", getString(com.cku.util.DateUtils.dateToStr(po.getCreateDate())));
			vo.put("businessState", getString(po.getBusinessState()));
			String reviewRemarks = "";
			if ("2".equals(po.getCkuReviewState())) {
				reviewRemarks = po.getCkuReviewRemarks();
			} else if ("2".equals(po.getReviewState())) {
				reviewRemarks = po.getReviewRemarks();
			}
			vo.put("reviewRemarks", reviewRemarks);
			list.add(vo);
		}
		long count = result.getCount();
		return new RESTResponse("list", new PageBeanResult<JSONObject>(list, count));
	}

	@Transactional(readOnly = false)
	public RESTResponse post(String json) {
		// 获取参数
		JSONObject jo = JSONObject.fromObject(json);
		// 参数校验
		validateParameter(jo);
		// 构造持久对象
		DogSemenChange po = buildPo(jo);
		// 业务校验
		dogSemenChangeService.validateBusiness(po);
		// 写入业务表
		dogSemenChangeService.save(po);
		// 写入购物车
		PaymentCart paymentCart = buildPaymentCart(po);
		paymentCartService.addToCart(paymentCart, "1014");
		return new RESTResponse("data", paymentCart.getId());
	}

	@Transactional(readOnly = false)
	public RESTResponse put(String json) {
		// 获取参数
		DogSemenChange vo = BeanUtil.toBean(json, DogSemenChange.class);
		// 犬精子变更审核参数校验
		putValidateParameter(vo);
		// 原精子主人审核
		dogSemenChangeService.originMemberAudit(vo);
		return new RESTResponse();
	}

	private void putValidateParameter(DogSemenChange vo) {
		if (StringUtils.isBlank(vo.getId())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬只精子变更id为空");
		}
		if (StringUtils.isBlank(vo.getReviewState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "审核状态为空");
		}
		// if ("2".equals(vo.getReviewState()) &&
		// StringUtils.isBlank(vo.getReviewRemarks())) {
		// throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "审核备注为空");
		// }
		if (!"1".equals(vo.getReviewState()) && !"2".equals(vo.getReviewState())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "审核状态只能为1或2");
		}
	}

	/**
	 * @description: 参数校验
	 * @author: laiguanglong
	 * @date: 2016年9月13日 下午4:09:54
	 */
	private void validateParameter(JSONObject jo) {
		if (!jo.containsKey("dogSemenId")) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子id为空");
		}
		if (jo.getJSONArray("dogSemenId") == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "精子id为空");
		}
	}

	/**
	 * @description: 构造持久对象
	 * @author: laiguanglong
	 * @date: 2016年9月13日 下午5:56:52
	 */
	private DogSemenChange buildPo(JSONObject jo) {
		Member member = UserUtils.getLoginMember();
		JSONArray ja = jo.getJSONArray("dogSemenId");
		String[] ids = new String[ja.size()];
		for (int i = 0, len = ja.size(); i < len; i++) {
			ids[i] = ja.getString(i);
		}
		return dogSemenChangeService.buildPo(ids, member.getMemberCode());
	}

	/**
	 * @description: 根据血统证书号查询犬只可用精子
	 * @author: laiguanglong
	 * @date: 2017/8/30 17:50
	 */
	public RESTResponse getDogSemenByPedigreeCertifiedAndMemberCode(HttpServletRequest request) {
		String pedigreeCertified = request.getParameter("pedigreeCertified");
		String memberCode = request.getParameter("memberCode");
		if (StringUtils.isBlank(pedigreeCertified)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号为空");
		}
		if (StringUtils.isBlank(memberCode)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "会员号为空");
		}
		Member applyMember = UserUtils.getLoginMember();
		if (memberCode.toUpperCase().equals(applyMember.getMemberCode().toUpperCase())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "会员号不能为当前登录会员号");
		}
		Member member = memberService.getByMemberCode(memberCode);
		if (member == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "会员号有误");
		}
		// 新旧犬主有效专业会员校验
		dogSemenChangeService.checkMember(applyMember, member);
		// 精子主人为有效专业会员
		Dog dog = dogDao.getByPedigreeCertifiedCode(pedigreeCertified);
		if (dog == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "血统证书号有误");
		}
		loginServerService.validDogType(dog);
		List<DogSemen> dogSemenList = dogSemenChangeService.getAvailableDogSemen(dog.getId(), memberCode);
		if (dogSemenList.size() == 0) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "没有找到匹配的精子记录！");
		}
		JSONObject data = new JSONObject();
		dogSemenService.setDogInfo(dog.getId(), data);
		JSONArray ja = new JSONArray();
		for (DogSemen dogSemen : dogSemenList) {
			if (dogSemenChangeService.isApply(dogSemen.getSemenCode())) {
				continue;
			}
			JSONObject jo = new JSONObject();
			jo.put("dogSemenId", getString(dogSemen.getId()));
			jo.put("originalSemenCode", getString(dogSemen.getOriginalSemenCode()));
			jo.put("semenCode", getString(dogSemen.getSemenCode()));
			ja.add(jo);
		}
		data.put("dogSemenList", ja);
		data.put("dogSemenListCount", ja.size());
		data.put("msg", ja.size() == 0 ? "该犬只无留存有效精子" : "");
		return new RESTResponse("data", data);
	}

	public RESTResponse getPrice() {
		PaymentChargingItem paymentChargingItem = getPaymentChargingItem();
		JSONObject jo = new JSONObject();
		jo.put("price", paymentChargingItem.getPrice());
		return new RESTResponse("data", jo);
	}

	private PaymentChargingItem getPaymentChargingItem() {
		return paymentChargingItemService.getByShortName(PAYMENTCHARGINGITEM);
	}

	private PaymentCart buildPaymentCart(DogSemenChange po) {
		PaymentCart paymentCart = new PaymentCart();
		PaymentChargingItem paymentChargingItem = getPaymentChargingItem();
		paymentCart.setChargingItemId(paymentChargingItem.getId());
		paymentCart.setChargingItemName(paymentChargingItem.getName());
		paymentCart.setNum("1");
		paymentCart.setPrice(paymentChargingItem.getPrice());
		paymentCart.setTotalPrice(paymentChargingItem.getPrice());
		paymentCart.setBusinessTable("dog_semen_change");
		paymentCart.setBusinessIds(po.getId());
		return paymentCart;
	}

	private String getString(String str) {
		return str == null ? "" : str;
	}
}