/**
 * Copyright &copy; 2012-2014 <a href="https://github.com/thinkgem/jeesite">JeeSite</a> All rights reserved.
 */
package com.cku.oa.sys.web.user;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.authz.AuthorizationException;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.cku.core.RESTResponse;
import com.cku.core.ResultDto;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.activity.service.SysMemberActivityRecordService;
import com.cku.oa.finance.dao.PaymentOrderDao;
import com.cku.oa.finance.entity.PaymentOrder;
import com.cku.oa.finance.service.PaymentOrderService;
import com.cku.oa.kennel.dao.KennelAllianceDao;
import com.cku.oa.kennel.dao.KennelDao;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.kennel.entity.KennelAlliance;
import com.cku.oa.sys.entity.Org;
import com.cku.oa.sys.entity.user.CabSysUserThirdRef;
import com.cku.oa.sys.entity.user.IEnum;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.entity.user.MemberCardTypeEnum;
import com.cku.oa.sys.entity.user.MemberLimitCount;
import com.cku.oa.sys.request.MemberRestPwdRequest;
import com.cku.oa.sys.service.OrgService;
import com.cku.oa.sys.service.UserService;
import com.cku.oa.sys.service.user.MemberLimitCountService;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.oa.sys.vo.MemberRenewalRecordVO;
import com.cku.oa.tcb.dao.TcbKennelDao;
import com.cku.oa.tcb.entity.TcbKennel;
import com.thinkgem.jeesite.common.config.Global;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.common.web.BaseController;
import com.thinkgem.jeesite.modules.sys.entity.Office;
import com.thinkgem.jeesite.modules.sys.entity.User;
import com.thinkgem.jeesite.modules.sys.service.SystemService;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

import net.sf.json.JSONObject;

/**
 * 用户会员信息表Controller
 * 
 * @author fanhuibin
 * @version 2016-07-13
 */
@Controller
@Validated
@RequestMapping(value = "${adminPath}/sys/user/member")
public class MemberController extends BaseController {

	@Autowired
	private MemberService memberService;
	@Autowired
	private UserService userService;
	@Autowired
	private SystemService systemService;
	@Autowired
	private KennelAllianceDao kennelAllianceDao;
	@Autowired
	private PaymentOrderService paymentOrderService;
	@Autowired
	private PaymentOrderDao paymentOrderDao;
	@Autowired
	private TcbKennelDao tcbKennelDao;
	@Autowired
	private MemberLimitCountService memberLimitCountService;
	@Autowired
	private KennelDao kennelDao;
	@Autowired
	private OrgService orgService;

	@Autowired
	private SysMemberActivityRecordService sysMemberActivityRecordService;

	@ModelAttribute
	public Member get(@RequestParam(required = false) String id) {
		Member entity = null;
		if (StringUtils.isNotBlank(id)) {
			entity = memberService.get(id);
		}
		if (entity == null) {
			entity = new Member();
		}
		return entity;
	}

	@RequiresPermissions("sys:user:member:view")
	@RequestMapping(value = { "list", "" })
	public String list(Member member, HttpServletRequest request, HttpServletResponse response, Model model,
			@ModelAttribute("importError") String importError) {
		Office office = UserUtils.getUser().getOffice();
		if ("100000020".equals(office.getCode())) {
			member.setContractCode("tc");
		}
		if (StringUtils.isNotBlank(member.getMemberCode()) && member.getMemberCode().contains("&middot;")) {
			member.setMemberCode(member.getMemberCode().replace("&middot;", "·"));
		}
		Page<Member> page = memberService.findPage(new Page<Member>(request, response), member);

		// 遍历所有list，如果是优生繁育联盟的，添加标记
		long now = System.currentTimeMillis();
		Map<String, TcbKennel> tcbKennelMap = new HashMap<>();
		Map<String, KennelAlliance> kennelAllianceMap = new HashMap<>();
		Map<String, Kennel> kennelMap = new HashMap<>();
		/**
		 * modify by zhangjunwu 修改老版本循环查库逻辑 start
		 */
		List<Member> members = page.getList();
		if (members != null && !members.isEmpty()) {
			List<String> memberCodes = members.stream().map(entity -> entity.getMemberCode())
					.collect(Collectors.toList());
			if (memberCodes != null && !memberCodes.isEmpty()) {
				List<TcbKennel> tcbKennels = tcbKennelDao.getByMemberCodes(memberCodes);
				if (tcbKennels != null) {
					tcbKennelMap.putAll(tcbKennels.stream()
							.collect(Collectors.toMap(TcbKennel::getMemberCode, Function.identity())));
				}
				List<KennelAlliance> kennelAlliances = kennelAllianceDao.getByMemberCodes(memberCodes);
				if (kennelAlliances != null) {
					kennelAllianceMap.putAll(kennelAlliances.stream()
							.collect(Collectors.toMap(KennelAlliance::getMemberCode, Function.identity())));
				}
				List<Kennel> kennels = kennelDao.getKennelByMemberCodes(memberCodes);
				if (kennels != null) {
					kennelMap.putAll(
							kennels.stream().collect(Collectors.toMap(Kennel::getMemberCode, Function.identity())));
				}
			}
		}
		members.forEach(m -> {
			KennelAlliance kennelAlliance = kennelAllianceMap.get(m.getMemberCode());
			if (kennelAlliance != null && kennelAlliance.getStartDate() != null && kennelAlliance.getEndDate() != null
					&& kennelAlliance.getStartDate().getTime() < now && kennelAlliance.getEndDate().getTime() > now) {
				m.setKennelAlliance(kennelAlliance);
			}
			TcbKennel tcbKennel = tcbKennelMap.get(m.getMemberCode());
			if (tcbKennel != null && tcbKennel.getStartDate() != null && tcbKennel.getEndDate() != null
					&& tcbKennel.getStartDate().getTime() < now && tcbKennel.getEndDate().getTime() > now) {
				m.setTcbKennel(tcbKennel);
			}
			m.setKennel(kennelMap.get(m.getMemberCode()));
		});
		/**
		 * modify by zhangjunwu 修改老版本循环查库逻辑 end
		 */

		model.addAttribute("page", page);
		model.addAttribute("importError", importError);
		model.addAttribute("member", member);
		return "oa/sys/user/memberList";
	}

	@RequiresPermissions("sys:user:member:view")
	@RequestMapping(value = { "search" })
	public String search(Member member, HttpServletRequest request, HttpServletResponse response, Model model) {
		Page<Member> page = new Page<Member>();
		if (member.getMemberCode() != null && member.getName() != null) {
			page = memberService.findPage(new Page<Member>(request, response), member);

			for (Member m : page.getList()) {
				if (StringUtils.isNotBlank(m.getStreet())) {
					m.setStreet(m.getStreet().replace("'", "‘"));
					m.setStreet(m.getStreet().replace("\n", " "));
				}
				if (StringUtils.isNotBlank(m.getAddress())) {
					m.setAddress(m.getAddress().replace("\n", " "));
				}
			}
		}

		model.addAttribute("page", page);
		return "oa/sys/user/memberSearch";
	}

	@RequiresPermissions("sys:user:member:view")
	@RequestMapping(value = "view")
	public String view(Member member, Model model, HttpServletRequest request, HttpServletResponse response) {
		model.addAttribute("member", member);
		String memberCode = member.getMemberCode();
		PaymentOrder pyo = new PaymentOrder();
		pyo.setMemberCode(memberCode);
		List<PaymentOrder> findList = paymentOrderService.findList(pyo);
		model.addAttribute("list", findList);
		Map<String, String> map = paymentOrderDao.getConsumeMsgByMemberCode(memberCode);
		model.addAttribute("in_price", map.get("in_price"));
		model.addAttribute("out_price", map.get("out_price"));
		model.addAttribute("three_price", map.get("three_price"));
		if (StringUtils.isNoneBlank(member.getSingleClubId())) {
			Org org = orgService.get(member.getSingleClubId());
			if (org != null) {
				member.setSingleClub(org.getName());
			}
		}
		return "oa/sys/user/memberView";
	}

	@RequiresPermissions("sys:user:member:view")
	@RequestMapping(value = "form")
	public String form(Member member, Model model) {

		model.addAttribute("member", member);
		model.addAttribute("province", UserUtils.getProvince());
		model.addAttribute("city", UserUtils.getCityByProvince(member.getProvince()));
		if (StringUtils.isNoneBlank(member.getSingleClubId())) {
			Org org = orgService.get(member.getSingleClubId());
			if (org != null) {
				member.setSingleClub(org.getName());
			}
		}
		if (member.getId() != null && !member.getId().equals("")) {
			String payItem = memberService.getPayItem(member);
			model.addAttribute("payItem", payItem);
			return "oa/sys/user/memberForm";
		} else {
			return "oa/sys/user/memberAdd";
		}
	}

	/**
	 * 淘宠宝会员添加跳转页
	 * 
	 * @param member
	 * @param model
	 * @return
	 */
	@RequiresPermissions("sys:user:member:view")
	@RequestMapping(value = "tcform")
	public String tcform(Member member, Model model) {
		model.addAttribute("member", member);
		model.addAttribute("province", UserUtils.getProvince());
		model.addAttribute("city", UserUtils.getCityByProvince(member.getProvince()));
		return "oa/sys/user/membertcAdd";

	}

	/**
	 * 淘宠宝会员添加保存
	 * 
	 * @param member
	 * @param model
	 * @param redirectAttributes
	 * @return
	 */
	@RequiresPermissions("sys:user:member:tcsave")
	@RequestMapping(value = "tcsave")
	public String tcsave(Member member, Model model, RedirectAttributes redirectAttributes) {
		member.setName(StringEscapeUtils.unescapeHtml4(member.getName()));
		boolean hasPermission = false;
		if (StringUtils.isEmpty(member.getId())) {
			hasPermission = SecurityUtils.getSubject().isPermitted("sys:user:member:tcadd");
		} else {
			hasPermission = SecurityUtils.getSubject().isPermitted("sys:user:member:tcedit");
		}
		if (!hasPermission) {
			throw new AuthorizationException("用户权限不足");
		}

		if (!beanValidator(model, member)) {
			return form(member, model);
		}

		try {
			member.setMemberType("basic");
			memberService.authenticationVild(member);
			memberService.tcsave(member);
		} catch (Exception e) {
			addMessage(redirectAttributes, e.getMessage());
		}
		return "redirect:" + Global.getAdminPath() + "/sys/user/member/?repage";
	}

	@RequiresPermissions("sys:user:member:edit")
	@RequestMapping(value = "save")
	@ResponseBody
	public ResultDto<String> save(Member member, Model model, RedirectAttributes redirectAttributes) {
		ResultDto<String> result = ResultDto.success("ok");
		// 由于注册外籍会员的时候会员姓名会带&符号，如C.PRAKIT & T.CHAIVAT，所以需要转回原始字符串
		try {
			member.setName(StringEscapeUtils.unescapeHtml4(member.getName()));
			boolean hasPermission = false;
			if (StringUtils.isEmpty(member.getId())) {
				hasPermission = SecurityUtils.getSubject().isPermitted("sys:user:member:add");
			} else {
				hasPermission = SecurityUtils.getSubject().isPermitted("sys:user:member:edit");
			}
			if (!hasPermission) {
				throw new AuthorizationException("用户权限不足");
			}

			if (!beanValidator(model, member)) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "参数校验失败");
			}
			if (member.getIsNewRecord()) {
				// 进行实名验证
				memberService.authenticationVild(member);
			}
			member.setMemberType(IEnum.getMessage(MemberCardTypeEnum.class, member.getMemberCardType()));
			memberService.save(member);
			result.setMsg("保存成功");
		} catch (Exception ex) {
			ex.printStackTrace();
			result = ResultDto.error(ZAErrorCode.ZA_ERROR, ex.getMessage());
		}
		return result;
	}

	@RequiresPermissions("sys:user:member:edit")
	@RequestMapping(value = "delete")
	public String delete(Member member, RedirectAttributes redirectAttributes) {
		try {
			memberService.delete(member);
			addMessage(redirectAttributes, "删除用户会员成功");
		} catch (ZAException e) {
			e.printStackTrace();
			addMessage(redirectAttributes, e.getMessage());
		}
		return "redirect:" + Global.getAdminPath() + "/sys/user/member/?repage";
	}

	@RequiresPermissions("sys:user:member:view")
	@RequestMapping(value = { "bind", "bind/list" })
	public String bind(Member member, HttpServletRequest request, HttpServletResponse response, Model model) {
		/*
		 * if(member.getUser() != null && member.getUser().getLoginName() != null &&
		 * !member.getUser().getLoginName().equals("")){ User u =
		 * UserUtils.getByLoginNameAndUserType(member.getUser().getLoginName(),"2");
		 * member.setUser(u); }
		 */
		Page<Member> page = memberService.findPage(new Page<Member>(request, response), member);
		if (!CollectionUtils.isEmpty(page.getList())) {
			List<Member> haveUserMemberList = page.getList().stream()
					.filter(entity -> Objects.nonNull(entity.getUser())).collect(Collectors.toList());
			List<CabSysUserThirdRef> cabSysUserList = null;
			if (!CollectionUtils.isEmpty(haveUserMemberList)) {
				cabSysUserList = memberService.findByThirdIds(haveUserMemberList.stream()
						.map(entity -> entity.getUser().getId()).collect(Collectors.toList()));
			}
			if (!CollectionUtils.isEmpty(cabSysUserList)) {
				Map<String, String> cabSysUserMap = cabSysUserList.stream()
						.collect(Collectors.toMap(CabSysUserThirdRef::getThirdId, CabSysUserThirdRef::getMobile));
				haveUserMemberList.forEach(m -> {
					m.setLoginMobile(cabSysUserMap.get(m.getUser().getId()));
				});
			}
		}
		model.addAttribute("page", page);
		return "oa/sys/user/memberBindList";
	}

	@RequiresPermissions("sys:user:member:view")
	@RequestMapping(value = "bind/form")
	public String bindForm(Member member, Model model) {
		model.addAttribute("member", member);
		model.addAttribute("province", UserUtils.getProvince());
		model.addAttribute("city", UserUtils.getCityByProvince(member.getProvince()));
		return "oa/sys/user/memberBindForm";
	}

	@RequiresPermissions("sys:user:member:edit")
	@RequestMapping(value = "/registerAccount/form")
	public String registerAccountForm(Member member, Model model) {
		model.addAttribute("member", memberService.get(member.getId()));
		return "oa/sys/user/memberRegisterAccountForm";
	}

	@RequiresPermissions("sys:user:member:edit")
	@RequestMapping(value = "/registerAccount")
	@ResponseBody
	public RESTResponse registerAccount(Member member, Model model, HttpServletResponse response) throws Exception {
		RESTResponse result = new RESTResponse();
		try {
			userService.newRegistAndBindMember(member.getMobile(), member.getId());
			result.setMsg("创建成功！");
		} catch (Exception e) {
			e.printStackTrace();
			result.setRc(ZAErrorCode.ZA_ERROR);
			result.setMsg(e.getMessage());
		}
		return result;
	}

	@RequiresPermissions("sys:user:member:cardEditTime")
	@RequestMapping(value = "editCardTime/form")
	public String editCardTimeForm(Member member, Model model) {
		model.addAttribute("member", member);
		model.addAttribute("province", UserUtils.getProvince());
		model.addAttribute("city", UserUtils.getCityByProvince(member.getProvince()));
		return "oa/sys/user/memberCardTimeEditForm";
	}

	@RequiresPermissions("sys:user:member:editMemberType")
	@RequestMapping(value = "editMemberType/form")
	public String editMemberType(Member member, Model model) {
		model.addAttribute("member", member);
		return "oa/sys/user/memberTypeEditForm";
	}

	/**
	 * 修改会员类型
	 * 
	 * @param
	 * @return
	 * @throws IOException
	 */
	@RequiresPermissions("sys:user:member:editMemberType")
	@RequestMapping(value = "editMember/edit")
	@ResponseBody
	public void editMember(Member member, @RequestParam String type, @RequestParam String remarklogs, Model model,
			HttpServletResponse response) throws IOException {
		JSONObject result = new JSONObject();
		// 查询会员
		try {
			memberService.editMemberType(member, type, remarklogs);
			String mes = "会员：" + member.getMemberCode() + "类型转换成功！";
			result.put("rc", 0);
			result.put("msg", mes);
			result.put("successUrl", "/sys/user/member/list?message=" + URLEncoder.encode(mes));
		} catch (Exception e) {
			result.put("rc", 1);
			result.put("msg", e.getMessage());

		}
		response.setContentType("application/json;charset=UTF-8");
		response.getWriter().write(result.toString());
	}

	/**
	 * 修改会员、犬舍截止日期
	 * 
	 * @param model
	 * @return
	 * @throws IOException
	 */
	@RequestMapping(value = "editCardTime/edit")
	@ResponseBody
	public void editCardTime(@RequestParam String memberCode, @RequestParam String memberEndDate,
			@RequestParam String kennelEndTime, Model model, HttpServletResponse response) throws IOException {
		JSONObject result = new JSONObject();
		try {
			memberService.cardTimeEdit(memberCode, memberEndDate, kennelEndTime);
			String mes = "会员：" + memberCode + "截止日期修改成功";
			result.put("rc", 0);
			result.put("msg", mes);
			result.put("successUrl", "/sys/user/member/list?message=" + URLEncoder.encode(mes));
		} catch (ZAException e) {
			result.put("rc", 1);
			result.put("msg", e.getMessage());

		}
		response.setContentType("application/json;charset=UTF-8");
		response.getWriter().write(result.toString());
	}

	@RequiresPermissions("sys:user:member:cardEdit")
	@RequestMapping(value = "editCard/form")
	public String editCardForm(Member member, Model model) {
		model.addAttribute("member", member);
		model.addAttribute("province", UserUtils.getProvince());
		model.addAttribute("city", UserUtils.getCityByProvince(member.getProvince()));
		return "oa/sys/user/memberCardEditForm";
	}

	/**
	 * 修改会员身份信息
	 * 
	 * @param model
	 * @return
	 * @throws IOException
	 */
	@RequiresPermissions("sys:user:member:cardEdit")
	@RequestMapping(value = "editCard/edit")
	@ResponseBody
	public void editCard(@RequestParam String memberCode, @RequestParam String memberName, @RequestParam String cardNo,
			@RequestParam String validType, @RequestParam String safeMobile, Model model, HttpServletResponse response)
			throws IOException {
		JSONObject result = new JSONObject();
		try {
			memberService.cardEdit(memberCode, StringEscapeUtils.unescapeHtml4(memberName), cardNo, validType,
					safeMobile);
			String mes = "会员：" + memberCode + "修改身份信息成功";
			result.put("rc", 0);
			result.put("msg", mes);
			result.put("successUrl", "/sys/user/member/list?message=" + URLEncoder.encode(mes));
		} catch (ZAException e) {
			result.put("rc", 1);
			result.put("msg", e.getMessage());

		}
		response.setContentType("application/json;charset=UTF-8");
		response.getWriter().write(result.toString());
	}

	/**
	 * 绑定用户与会员
	 */
	@RequiresPermissions("sys:user:member:bind")
	@RequestMapping(value = "bind/bind")
	@ResponseBody
	public void bindSave(@RequestParam String memberCode, @RequestParam String loginName, Model model,
			HttpServletResponse response) throws IOException {
		JSONObject result = new JSONObject();
		try {
			memberService.bindUser(memberCode, loginName, null);
			String mes = "会员：" + memberCode + "绑定用户成功";
			result.put("rc", 0);
			result.put("msg", mes);
			result.put("successUrl", "/sys/user/member/bind?message=" + URLEncoder.encode(mes));
		} catch (ZAException e) {
			result.put("rc", 1);
			result.put("msg", e.getMessage());

		}
		response.setContentType("application/json;charset=UTF-8");
		response.getWriter().write(result.toString());
	}

	/**
	 * 用户与会员解绑
	 * 
	 * @param member
	 * @param model
	 * @param redirectAttributes
	 * @return
	 */
	@RequiresPermissions("sys:user:member:unbind")
	@RequestMapping(value = "bind/unbind")
	public String unbindSave(Member member, Model model, RedirectAttributes redirectAttributes) {
		// if (!beanValidator(model, member)){
		// return form(member, model);
		// }
		memberService.unbindUser(member);
		addMessage(redirectAttributes, "会员与网站帐号解绑成功");
		return "redirect:" + Global.getAdminPath() + "/sys/user/member/bind?repage";
	}

	/**
	 * 重置网站用户的密码
	 * 
	 * @return
	 */
	@RequiresPermissions("sys:user:member:resetPassword")
	@RequestMapping(value = "/password/reset")
	@ResponseBody
	public ResultDto<String> resetPassword(@Valid @RequestBody MemberRestPwdRequest request) {
		return ResultDto.success(systemService.resetMemberPassword(request));
	}

	@RequiresPermissions("sys:user:member:resetPassword")
	@RequestMapping(value = "/password/resetForm")
	public String resetPasswordForm(Member member, Model model) {
		Member dbMember = null;
		if (StringUtils.isNotBlank(member.getId())) {
			dbMember = memberService.get(member.getId());
		} else if (StringUtils.isNotBlank(member.getMemberCode())) {
			dbMember = memberService.getByMemberCode(member.getMemberCode());
		}
		if (Objects.nonNull(dbMember) && Objects.nonNull(dbMember.getUser())) {
			CabSysUserThirdRef cabUser = memberService.findByThirdId(dbMember.getUser().getId());
			if (Objects.nonNull(cabUser)) {
				dbMember.setLoginMobile(cabUser.getMobile());
			}
		}
		model.addAttribute("member", dbMember);
		return "oa/sys/user/memberResetPwdForm";
	}

	@RequiresPermissions("sys:user:member:state")
	@RequestMapping(value = "login/state")
	public String changeLoginState(Member member, Model model, RedirectAttributes redirectAttributes) {
		if (member.getUser() != null && member.getUser().getId() != null && !member.getUser().equals("")) {
			User user = UserUtils.get(member.getUser().getId());
			if (user.getLoginFlag().equals(Global.YES)) {
				user.setLoginFlag(Global.NO);
			} else {
				user.setLoginFlag(Global.YES);
			}
			systemService.saveUser(user);
		}
		addMessage(redirectAttributes, "用户的登录状态已经被修改");
		return "redirect:" + Global.getAdminPath() + "/sys/user/member/bind?repage";
	}

	/**
	 *
	 * @Description：根据会员号获取会员姓名和状态
	 * @author: zhuoHeng
	 * @version: 2016年10月24日 下午3:03:00
	 */
	@RequestMapping(value = "getByMemberCode")
	@ResponseBody
	public Member getByMemberCode(HttpServletRequest requset) {
		String memberCode = requset.getParameter("memberCode");
		return memberService.getByMemberCode(memberCode);
	}

	/**
	 * 根据会员号查看会员信息
	 * 
	 * @Author chaixueteng
	 * @2016年11月8日上午10:25:24
	 */
	@RequestMapping(value = "toMemberView")
	public String toMemberView(String memberCode, Model model, HttpServletRequest request,
			HttpServletResponse response) {
		Member byMemberCode = memberService.getByMemberCode(memberCode);
		model.addAttribute("member", byMemberCode);
		PaymentOrder pyo = new PaymentOrder();
		pyo.setMemberCode(memberCode);
		List<PaymentOrder> findList = paymentOrderService.findList(pyo);
		model.addAttribute("list", findList);
		Map<String, String> map = paymentOrderDao.getConsumeMsgByMemberCode(memberCode);
		model.addAttribute("in_price", map.get("in_price"));
		model.addAttribute("out_price", map.get("out_price"));
		return "oa/sys/user/memberView";
	}

	/**
	 * 重置网站用户的身份验证次数
	 * 
	 * @param member
	 * @param model
	 * @param redirectAttributes
	 * @return
	 */
	@RequiresPermissions("sys:user:member:resetCardLimit")
	@RequestMapping(value = "card/reset")
	public String resetCardLimit(Member member, Model model, RedirectAttributes redirectAttributes) {
		Member member2 = memberService.get(member.getId());
		MemberLimitCount memberLimitCount = memberLimitCountService.findByMemberCode(member2.getMemberCode());
		if (memberLimitCount != null) {
			memberLimitCount.setCheckIDCardLimit(0);
			memberLimitCountService.save(memberLimitCount);
		}
		addMessage(redirectAttributes, "用户身份验证次数已成功重置为0");
		return "redirect:" + Global.getAdminPath() + "/sys/user/member/bind?repage";
	}

	/**
	 * @Description： 导入会员信息的模板
	 * 
	 * @author: yuanshuai
	 * @date: 2019/5/29 17:28
	 */
	@RequiresPermissions("sys:user:member:import")
	@RequestMapping(value = "import/template")
	public String importFileTemplate(HttpServletResponse response, RedirectAttributes redirectAttributes) {
		try {
			return memberService.importFileTemplate(response);
		} catch (Exception e) {
			addMessage(redirectAttributes, "导入模板下载失败！失败信息：" + e.getMessage());
		}
		return "redirect:" + Global.getAdminPath() + "/sys/user/member/?repage";
	}

	/**
	 * 导入会员信息
	 *
	 * @author: yuanshuai
	 * @date: 2019/5/29 17:28
	 */
	@RequiresPermissions("sys:user:member:import")
	@RequestMapping(value = "import", method = RequestMethod.POST)
	public String importFile(MultipartFile file, RedirectAttributes redirectAttributes) {
		try {
			// 会员信息导入
			memberService.importMember(file);
			addMessage(redirectAttributes, "导入会员信息成功！");
		} catch (Exception e) {
			redirectAttributes.addFlashAttribute("importError", "导入会员信息失败！失败信息：##" + e.getMessage());
		}
		return "redirect:" + Global.getAdminPath() + "/sys/user/member/?repage";
	}

	@RequiresPermissions("sys:user:member:dogNoseEdit")
	@RequestMapping(value = "dogNose/form")
	public String dogNoseEdit(Member member, Model model) {
		model.addAttribute("member", member);
		return "oa/sys/user/dogNoseEditForm";
	}

	@RequiresPermissions("sys:user:member:dogNoseEdit")
	@RequestMapping(value = "dogNose/edit")
	@ResponseBody
	public void dogNoseEdit(@RequestParam String memberCode, @RequestParam String dogNoseRepeatedDetection,
			@RequestParam String dogNoseLivenessDetection, Model model, HttpServletRequest request,
			HttpServletResponse response) throws IOException {
		JSONObject result = new JSONObject();
		try {
			memberService.dogNoseEdit(memberCode, dogNoseRepeatedDetection, dogNoseLivenessDetection, request);
			String mes = "会员：" + memberCode + "修改身份信息成功";
			result.put("rc", 0);
			result.put("msg", mes);
			result.put("successUrl", "/sys/user/member/list?message=" + URLEncoder.encode(mes));
		} catch (ZAException e) {
			result.put("rc", 1);
			result.put("msg", e.getMessage());

		}
		response.setContentType("application/json;charset=UTF-8");
		response.getWriter().write(result.toString());
	}

	@RequestMapping(value = "renewalRecords")
	public String renewalRecords(Member member, Model model) {
		model.addAttribute("member", member);
		List<MemberRenewalRecordVO> list = sysMemberActivityRecordService.getListByMemberId(member);
		model.addAttribute("list", list);
		return "oa/sys/user/renewalRecords";
	}

	@RequiresPermissions("sys:user:member:edit:bind")
	@RequestMapping(value = "/edit/bind/form")
	public String memberEditBindForm(Member member, Model model) {

		Member dbMember = null;
		if (StringUtils.isNotBlank(member.getId())) {
			dbMember = memberService.get(member.getId());
		} else if (StringUtils.isNotBlank(member.getMemberCode())) {
			dbMember = memberService.getByMemberCode(member.getMemberCode());
		}
		if (Objects.nonNull(dbMember) && Objects.nonNull(dbMember.getUser())) {
			CabSysUserThirdRef cabUser = memberService.findByThirdId(dbMember.getUser().getId());
			if (Objects.nonNull(cabUser)) {
				dbMember.setLoginMobile(cabUser.getMobile());
			}
		}
		model.addAttribute("member", dbMember);
		return "oa/sys/user/memberEditBindForm";
	}

	@RequiresPermissions("sys:user:member:edit:bind")
	@RequestMapping(value = "/edit/bind/relation")
	@ResponseBody
	public ResultDto<String> memberEditBindRelation(Member member, Model model) {
		try {
			return memberService.editBindRelation(member);
		} catch (ZAException zaex) {
			return ResultDto.error(zaex.get_code(), zaex.getMessage());
		} catch (Exception ex) {
			return ResultDto.error(ZAErrorCode.ZA_ERROR, "修改绑定关系失败,失败原因：" + ex.getMessage());
		}
	}
}