package com.cku.oa.sys.service;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import com.alibaba.fastjson.JSONObject;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.sys.dao.WorkdayDao;
import com.cku.oa.sys.entity.Workday;
import com.cku.oa.sys.util.DateUtils;
import com.dtflys.forest.Forest;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;

/**
 * 工作日历表Service
 * 
 * @author Sunny
 * @version 2024-10-17
 */
@Service
@Transactional(readOnly = true)
public class WorkdayService extends CrudService<WorkdayDao, Workday> {

	private final static String HOLIDAY_API_URL = "https://api.jiejiariapi.com/v1/holidays/";

	public Workday get(String id) {
		return super.get(id);
	}

	public List<Workday> findList(Workday workday) {
		return super.findList(workday);
	}

	public List<Workday> findListByYear(String dateYear) {
		return this.dao.selectByDateYear(dateYear);
	}

	public Page<Workday> findPage(Page<Workday> page, Workday workday) {
		return super.findPage(page, workday);
	}

	@Transactional(readOnly = false)
	public void save(Workday workday) {
		super.save(workday);
	}

	@Transactional(readOnly = false)
	public void addWeekend(Workday workday) {
		List<Workday> workdayList = this.dao.selectByDateYear(workday.getDateStr());
		if (!CollectionUtils.isEmpty(workdayList)) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "已存在".concat(workday.getDateStr()).concat("年的数据，请删除后重试！"));
		}
		List<Workday> dateList = new ArrayList<>();
		try {
			String holidaysResult = Forest.get(HOLIDAY_API_URL.concat(workday.getDateStr())).setRetryEnabled(true)
					.setMaxRetryCount(3).setConnectTimeout(10000).setReadTimeout(10000).execute(String.class);
			if (StringUtils.isNotEmpty(holidaysResult)) {
				JSONObject holidays = JSONObject.parseObject(holidaysResult);
				holidays.keySet().forEach(key -> {
					JSONObject holiday = holidays.getJSONObject(key);
					Workday day = new Workday();
					day.setDateStr(holiday.getString("date"));
					day.setName(holiday.getString("name"));
					day.setOffDay(holiday.getBoolean("isOffDay"));
					day.preInsert();
					dateList.add(day);
				});
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		// 获取当前年份
		Calendar calendar = Calendar.getInstance();
		int currentYear = calendar.get(Calendar.YEAR);
		// 设置为当前年的第一天
		calendar.set(currentYear, Calendar.JANUARY, 1);
		// 循环遍历整个年份
		List<String> dateStrList = new ArrayList<>();
		if (!CollectionUtils.isEmpty(dateList)) {
			dateStrList.addAll(dateList.stream().map(Workday::getDateStr).collect(Collectors.toList()));
		}
		while (calendar.get(Calendar.YEAR) == currentYear) {
			int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
			String currentDateStr = DateUtils.dateToStr(calendar.getTime());
			if ((dayOfWeek == Calendar.SATURDAY || dayOfWeek == Calendar.SUNDAY)
					&& !dateStrList.contains(currentDateStr)) {
				Workday entity = new Workday();
				entity.setDateStr(currentDateStr);
				entity.setName(dayOfWeek == Calendar.SATURDAY ? "周六" : "周日");
				entity.setOffDay(Boolean.TRUE);
				entity.preInsert();
				dateList.add(entity);
			}
			// 移动到下一天
			calendar.add(Calendar.DAY_OF_MONTH, 1);
		}

		if (!CollectionUtils.isEmpty(dateList)) {
			dateList.forEach(d -> {
				this.dao.insert(d);
			});
		}
	}

	@Transactional(readOnly = false)
	public void editWeekend(Workday workday) {

	}

	@Transactional(readOnly = false)
	public void deleteWeekend(Workday workday) {
		this.dao.deleteByDateYear(workday.getDateStr());
	}

	@Transactional(readOnly = false)
	public void delete(Workday workday) {
		super.delete(workday);
	}

	public boolean validOffday(Date date) {
		if (Objects.isNull(date)) {
			return Boolean.FALSE;
		}
		Workday workday = this.dao.selectByDateStr(DateUtils.dateToStr(date));
		return Objects.nonNull(workday) && workday.getOffDay() ? workday.getOffDay() : Boolean.FALSE;
	}

}