package com.cku.oa.statistics.service;

import com.alibaba.fastjson.JSONObject;
import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.finance.cab.dao.CABFinanceStatisticsDao;
import com.cku.oa.finance.dao.PaymentOrderDao;
import com.cku.oa.finance.dao.PaymentOrderDetailDao;
import com.cku.oa.finance.dao.PaymentRecordDao;
import com.cku.oa.finance.dao.SaPaymentDetailTotalDao;
import com.cku.oa.finance.entity.PaymentOrder;
import com.cku.oa.finance.entity.PaymentOrderDetail;
import com.cku.oa.finance.entity.PaymentRecord;
import com.cku.oa.finance.entity.SaPaymentDetailTotal;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.kennel.service.KennelService;
import com.cku.oa.statistics.strategy.RelationShipEntity;
import com.cku.oa.statistics.strategy.YearToMonthSplitEntity;
import com.cku.oa.statistics.strategy.YearToMonthSplitUtils;
import com.cku.oa.sys.dao.OrgDao;
import com.cku.oa.sys.entity.Org;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.OrgService;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.oa.trainschool.dao.TrainingInstitutionDao;
import com.cku.oa.trainschool.entity.TrainingInstitution;
import com.thinkgem.jeesite.common.utils.Collections3;
import com.thinkgem.jeesite.common.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;

import static com.cku.oa.statistics.strategy.InitSplitStrategy.PAYMENT_TYPE_CAB;
import static com.cku.oa.statistics.strategy.InitSplitStrategy.PAYMENT_TYPE_CKU;

@Service
public class SplitOrderService {

	private Logger logger = LoggerFactory.getLogger(SplitOrderService.class);
	@Autowired
	private PaymentOrderDetailDao paymentOrderDetailDao;
	@Autowired
	private PaymentRecordDao paymentRecordDao;
	@Autowired
	private CABFinanceStatisticsDao cabFinanceStatisticsDao;
	@Autowired
	private OrgService orgService;
	@Autowired
	private TrainingInstitutionDao trainingInstitutionDao;
	@Autowired
	private MemberService memberService;
	@Autowired
	private KennelService kennelService;
	@Autowired
	private SaPaymentDetailTotalDao saPaymentDetailTotalDao;
	@Autowired
	private PaymentOrderDao paymentOrderDao;
	@Autowired
	private OrgDao orgDao;

	public static final String[] ONE_TO_MANY = { "324", "278", "179", "134", "151", "210", "16", "153", "17", "135",
			"14", "15", "131", "130", "129", "19", "230", "277", "57053cefa38149a8a04d4ae6d8a68741", "279", "26", "239",
			"877", "76", "13", "816", "218", "191", "194", "50", "193", "192", "49", "33", "34", "35", "101", "29",
			"85", "36", "37", "38", "201", "b2ed91bbe772425ba6ecbcf3000d7e8d", "62818c904fed4d06bd215080bdd7e652", "94",
			"98", "e957c1373c744b26a5866b795a15bf48", "317", "319", "2bb9549794cc4757996b64574ea5fa4f", "325",
			"45b14635dddb434ea894cc8fd3ea0129", "a5142d9719824e6a987fa090e0570b2c", "2f73bbd1890844cc86ae692c68e6cdfa",
			"24d78920df4e4c50bb8d03e1a4706bde", "9285afb4c0a147e380024edcab89399b", "e81a20bb6968434689cb4a42700112e3",
			"98a5e1027c054645bd5af654ca2ac736", "d923d3ce76c24fa8bd6f695cb4885270", "0cbf012cf2cb4c59a1db0731a59cb575",
			"0cbf012cf2cb4c59a1db0731a59cb575", "5fe44bed80dc478c95a0fe8f37498c7e", "4fb73289dd464530ad8657c9551e8fbe",
			"73b9ae46bc6b4aaf8026c16a1d3ed339", "fe25a3dedb2541a5a4a4ec34092a7f86", "a3025dcc3cb04f74a8084e372bf3a26f",
			"eb2a7ab2c36b49e986776ad40dac16e0", "58e08a2616e645bd9ba66bcab8f129a2",
			//2023.06.01
			"a0db99c555d446eb8ea281704fcfb91a",
			//2023.07.03
			"0d6afd1eef2d45ad8a2c406e597be910",
			"ac627bcf9f1c49bc8122b1028027b94c",
			//2023.11.01
			"bed7367a15b346918d1091705b2db556",
			"265a2cbd74464dc58bda3b3a86cd8518",
			//2024.02.01
			"97d6e885a6324ee89e370d50536c682b","909c65809b4a41678752ce67231ca50c","469cce7c03bd4856aad83adee6a457c2",
			"996f695d345446e2bf19b43f5126eaf2","16d692b476584d2eb58c2853cb70f0a5","3cccd62892f542b9a928a2485dda86da",
			"e39af5bea0b2497084d0aba8654dda65","481063d4b6674dff815a38fef49cf160","72cde4561c574548b287252bc5420c6c",
			//2024.08.01
			//美容
			"b95300a5267642789b10212a9b74e1ff",
			//2024.12.02
			//美容
			"4dfd50872ca94491a182b98434585845","af1f49c8c77b4a9e8011673670dd402e",
			//2025.01.06
			//美容
			"ef98d59a4ded42ca9f3e6588e8716581","c1fc52daecf946eca38e7d67b813dfda","cbccf91be86843269ef9c4e6c7134286",
			"8a6c195ab66443b6b890c473a1f3d8bd","def26f02ccb641a58751a84f58474e36","ed5f46d4fa7844dd9ed3044e2fbfddca",
			"b598dc92b5bb4b9b96587f5dc36e2381",
			//2025.02.06
			//培训
			"acd9d8af9a2442eb8e25d94b95a9652f",
			//2025.07.01
			//培训
			"0513a5133406480aa17c7c41a470785f","5b4f7bf7eec34f50a77184f329a08f87","aa2866c302f645a29f2c227040210d65",
			//2025.08.01
			//培训
			"4009f250481c44e19c560929be6f0660"
	};

	public static final String[] MATH = {
//犬只参赛费：合办
			"140", "220", "142", "222", };

	public static final String[] REFUND_PROCESS = { "143", "144", "145", "146", "147", "148", "155", "156", "157",
			"158", "163", "199", "227", "233", "257", "281", "297", "305", "305zz", "306", "306zz",
			"2d82a6f76824418aa5f00c0e6652b155", "4b3a80da07fe40e28d3d5e683715ec02", "dc60531b20074952a44ce060ccd7430e",
			"f288b22d0f06402daa7df0024dd71aea", "ff1651a694724ad6bdc66d74de44b256" };

	public static final String[] NO_PROCESS = { "0",
//            犬只参赛费：新合办、自办
			"45", "368", "221", "141", "219", "372", "299", "371", "370", "367", "299zz", "369",
//            原生犬参赛费、原生犬鉴定参赛费
			"267", "259", "270", "262", "301", "303", "268", "260", "269", "261", "301zz", "303zz", "241", "12", "99",
			"181", "999", "132", "17855720b480431c81181bd082018d23", "76f54bdd760c43d797a9b9c835ca20ee", "242", "258",
			"133", "207", "4c115d68975444548a82ca8bba9bcd04", "243", "244", "245", "246", "225", "249", "22", "247",
			"125", "185", "21", "23", "126", "128", "25", "137", "24", "136", "240", "127", "56", "275", "276", "229",
			"328", "de20ce434cfc42f7b32b65b3953b886f", "c7351b6e4f6049cdbc6f1ff1b50382a0",
			"9dae72e17dbd43bdaa361ce8f64c563d", "71", "28", "166", "bb6610483b8245dc821bbb9e8b260b92", "118",
			"4b3a80da07fe40e28d3d5e683715ec02", "5809c0665c6c4bb39ed02c749fa1e134", "223", "234", "281", "224",
			"9925e1079fc44ed4a2ff096353f0a674", "f0eef5f34b864a9294e3d4e03561179a", "e53a5c2687e24bd4ae84e6ee84fa80cc",
			"119", "195", "157", "121", "148", "117", "189", "73989b7f0c874729b76d5289b702064a",
			"a995e065c4294b25b2ebbd6f56a31795", "305", "d64061366fa245bc902c542807e2d998", "155", "190",
			"25c04f060c1f4314bb7b7d18aeb7b915", "305zz", "272", "273", "274", "271", "263", "266", "265", "264", "156",
			"158", "257", "233", "298", "111", "112", "51", "67", "52", "68", "53", "227", "214", "54", "235", "215",
			"206", "312", "256", "167", "169", "84", "601", "602", "282", "f288b22d0f06402daa7df0024dd71aea", "280",
			"208", "80", "104", "10", "2", "5", "8", "60", "42715470bfad4b289b734354ba92a148", "143", "58", "217", "64",
			"238", "202", "226", "251", "250", "18", "285", "17", "20", "248", "144", "313", "314", "315", "326", "163",
			"e53a5c2687e24bd4ae84e6ee84fa804d", "212", "177", "178", "145", "32", "138b44dbd2c14394beb420799cb3d1c4",
			"39", "72", "40", "59523b7d19f54b82940f07ef6ac343a9", "41", "83", "30", "992c2008746f43c2be9c484d150008a4",
			"998", "31", "59", "146", "311", "300zz", "300", "33a073bd32b64c58b77744d887a5384e",
			"225d91b7b40d476ca6f7d7665dfaaf40", "a064682336fd4443853a70d9362460be", "97e8411e1f2f40989bc41a5d092f8f98",
			"9e3d51bf43754ea190f7f68da9aae817", "e53a5c2687e24bd4ae84e6ee84fa80zz", "04828ea19eb14f9291b8e0fa4ec6cdzz",
			"e53a5c2687e24bd4ae84e6ee84fa80cq", "e53a5c2687e24bd4ae84e6ee84fa80hz", "5dbe60c1ac8149b3a38e072d14032e8c",
			"e53a5c2687e24bd4ae84e6ee84fa80cs", "04828ea19eb14f9291b8e0fa4ec6cd02", "254", "171", "170", "92", "186",
			"43", "86", "255", "172", "44", "283", "198", "211", "159", "205", "456", "284", "252", "366", "253", "108",
			"109", "147", "8f6e5ea2c4db4cfaa7e465435df06173", "307", "310", "316", "318", "321", "322", "323",
			"238c11d98bcf484aa3929f480c71bd9f", "fb9d498f2dc24ddfa5b50112f086c672", "20b96343190e4ba9b914b79b413f2f66",
			"8b4a949131224404a81c3421a100bdde", "90a05d12fb784643a4ab919761f1449d", "4ba9605460a84b32b3a00ef71c924e7f",
			"3faf6501d752492791690fd24091e639", "fc1c9bc6e20f4ac3b61d2e7b3fec270a", "b2272f463e48444fb1abe810ec276b6a",
//            2020-06-04新增
			"43a76668f7924f49aef7c7df33dcf5f6", "e1499d3b6b184238b668bbbe843cec94", "391c19bcccbe429590861a4669ace058",
			"c821501786ee4986bc079a0e58d0ed46", "8c77ea7c693c4765b96685db22843d8a", "ca8c0e71ec9b4efc906656bcd6d9d5a9",
			"472d7c0affcc4e5ba1725b4729344346", "0d1d2749270443ecb20899ed200f92a7", "16e814e6409d4c759d909fcb52d30063",
			"bafe7753d9f343528bfad0bdd466ce44", "f5353f0d1e474d3dac4b9bca3d4c84a4", "bc35a8e93d54474c8b3b6252c1e3309b",
			"73aef57be260432d9f4dedca1c367486", "b196aba404ce4ff2b331d32ba1e3152f", "99904",
//            20200630
			"bc2a3c91eac54db497e62e8b94ea8525", "98b615f2007c40dd999421bbb2916808", "d410b11179f1479c843d337d876d6807",
			"f5b0c97b819b4853a72cab5224e543ee", "f7676fbe0bdc4188a2a74ef475cbf694", "381809578f254f33b08bed8d0059a1d2",
			"9b0957dc92af438eb0e9046fdc763962", "02ed5464fdaf414cb157d94e1f3cfa1c", "c4b4ef07cd9d4324b981d74f704ab399",
			"6d84b2d699f64431b1e9c6f887d7944f", "34d80b3bdbe44c5380b2e82dfd8a94ef", "c15785a93e224ef9ab9e3059ef652f45",
			"ba6b3b03a6af4a138dd3c3e03800ab1c", "fb84ccb10ced4a74a33dd5679b72cfb8", "61c851d0d3064676b51c2846a9d1e580",
			"2644aff00fda43f5a269ed392ccc28d2", "24987cf2e7cd44cb959bfc05b4f43fc4", "921cb8e18bb648a09e13a9e81bec8578",
			"49031d34292a40fab24c004b0ea1aa46", "6adc404e0ec8423891986d4c7fe0dd15",
//            20200927
			"1303518712e54a3bb9f3339d479aa75d", "14fc3fb75ea842829fc056d60e32f2c8", "dc60531b20074952a44ce060ccd7430e",
			"861ae984947b4136989c34dbbe7d36c6", "af50807b2c5a4fbdb4ecb3d52a0ab536", "de747e854e0d4c559b4f31eb1878787c",
			"3bf959b482aa459bbe5730fcb74ac24f", "e0bebb6702e4429a97ccb1e25140ed35", "ac91fdf697fa4f9ea9afd081144947c7",
			"b00b4f7d018245ce9a24c0320ea22988", "1e8e703ca4ea429cb57f7db1d0a82774", "ff1651a694724ad6bdc66d74de44b256",
			"16f9033a3c7c400981d30253b637c4d0", "ad0a591e8d934b308acbd42eaa05c154", "4500c7fc381c47e094fd0d364ea65fe3",
			"f6876716d0c542e79610ac4b32f5bb0f", "31dfd9693ca44b69b3cd659b2795933c", "861e3ef534514375b3774b91cacf916d",
			"0e769cce9fdf4d4f88a72d0130cea779",	"fe535c0a0be546379351b9f50444da3a", "ab8245b66e9e436d8a5cfec5b465eb0c",
			"2bdd35b2a0c7455aa14f7c3fe559e593", "166055e31e9649a39fb76c44ca13b058", "b57a51b31cad4fc8a3d1270e2546b997",
			"48724e0394824927bdfc8ddffaf197c4", "5349519101be4fc9b2ee81677019f21b", "790a4f21f33f4ac5903a452c92cfa15d",
			"8af3b7250f974929921b89756c0622c6",
			//2022.04.01培训新增
			"02cd9ffbbe6944fb99b7b7ceb53cc78a", "0bac5dfadd4444f8a69ed86391d21c8d","26a2df03973e439ca04781e769b617ab",
			"299dde5e324548faa27d911d6bfd23e6", "36e6676f38364d30a3a88889936801f0", "3b5112bdc7ac48ba85b64b9a41755d4c",
			"4437f551fb8c4da386e38a48f355b4be", "54ee7bc98fbf424b8a41b438e61602c4", "66e8abc81cec4079bdd9cfcd6a8d4864",
			"88a72b0bdb684355bbcb2e58cbe8b8bb", "8b79c519ad0e4a97b88d62053b43e715", "8ea476cd6d1b42daa1a0e42f3c2fbdb9",
			"9393b0f613dc4bb38ff493c23c3cc371", "a2dfc4f22fe34a539f7636438b69669e", "a5c2e883d4c94ece8b356efdc48a0e1e",
			"b259f6c402144bf89bc0731323320c23", "d65aaea898b7447f8d5d8a6fcda6ed6c", "e2ff70b6c00244eabfa54c6e58d48423",
			"e8c26fce2f93416f8d3ce3f4152b199c", "f15f0cdb4ef5410caec4c0a4d281ffb6", "f5162ebc9cc2440a993a86aa50ee72a9",
			"f806c52b2b9e4ad6805de9f96ffdf89f",
			//2022.06.01事务部新增
			"4ef0700f36a34eb5827abdb4603c52c0",
			//2022.07.01美容部新增
			"1a64f8a05672438b94bfb5c4c5ea87d3", "7ca3a44681494ed59a62c10b78a0536c", "86a430923fb944bc987e201e44c9913c",
			"a0e0550c59f74c0284508931ed53585c", "ce9342b829bf4ba0b916ee74f56f2ea9", "db6672386128489699a845f2174393a2",
			"f73187aed03f4955b6a72296b2e3b892",
			//2022.12.01新增基因会员培训收费项
			//培训
			"ffb7a3efeda2487897a5e8ff1e2e481f","89d66c56b2fa489e9ef9ce459cc42e49","3a3d3551db0b4e1aab72579b99602865",
			"3e2661bc5bb84c5e93c78be42ce02104","b1611c6f4eac4588b0c4419ef83b5144","a2099580df654fd099b115f502dc8a77",
			"71c9017661ef41168bcdc03410128d2b","17850f6f19c04c30904188575af6eb35",
			//会员
			"6ec3d0ec24c1438b8b31e36af9927df7",
			//基因
			"55a411fc9df04891b36d6b6a50203961","a279c4dfd4dd4a28a6de083243ace888","059b606e9bac456ea4abf4518125ee3c",
			"35e7c3ec85a349fd9126e98042e0a1f3","b6f31fa5f4e84226b1c9567fe22b01d1","bf7016542dc64e008613b5b350ced17e",
			//2023.04.03
			//培训
			"54882179cba04b988cbdab61b1d9a7e3",
			//基因
			"c2a14ef1c2fc44d6977c69073b4bc802",
			//2023.06.01
			//基因
			"24fed019e8c64b7789cfa56a856f81c7", "c6a72dbf9f354d939f2292c5e35732f0",
			//地方俱乐部
			"9f742a03f73749e3a3d50c9f5941910c",
			//繁殖
			"e905032b77cf4a8b9bb3e53e693bdbac","971e1864b6a54525ba3f66e56b3c10e5","cf701ff5b169405bb4dab90623728042",
			"17bf072dcae74f7db3d489e06a3de3a4","42486b62a04f40da8ae269c3efa142d2","034db20398ee4bea828876e7e72a6f8c",
			"3f53d07d1006491189b0a0974fb2e443","56444644f3a64d1b96e53236a85e1948","272e0249fba64ff6b4e22e897ca37ec2",
			//2023.12.01
			//繁殖
			"ef6d1c5c10ed4df69f24aa740d40a0d2",
			//2024.02.01
			//繁殖
			"ee1726c2d7594fb9858eb97cbcbde53f",
			//培训
			"48f8b77c3ad94ac1bb17aa91ef199f27","6a91b4bbe7e94d248c6e174c5bdef8d5",
			//美容
			"28d74262c59f49d58198b9361f534b51",
			//2024.03.01
			//美容
			"4329827604114c69bddfc8322d5ef016",
			//招商
			"2b4dde7c558349378d375971feed4272",
			//2024.05.06
			//美容
			"0743608d8f3545a9a833226077c4add0",
			//2024.06.03
			//繁殖
			"4df8d3ef339a4ea8b6f1167894567eea",
			//2024.07.01
			//培训
			"35d3c83b9a17455090b2a1485e154372",
			//2024.08.01
			//美容
			"c70b9835ee7748a79420f331fdfb102e",
			//繁殖
			"4f4f0654e9cc4b3fac8403532b2865b1","4625f60e8ac44d37b23ce2d972d1b532",
			//2024.09.02
			//繁殖
			"e8c03b7eb6174c99b5686e2171892182",
			//美容
			"7bb64fc1ef374a748dfedcb453fd9702",
			//2025.01.06
			//繁育基数
			"6943096af97d481c88d050855dcb30fd","0c653b8b33bc42a78f91f2e5ae715054",
			//美容
			"0a1fba1486234c13ad8fec7c6ca8d70f","1a6dd8c513234b15a068498ca9121b8e",
			"421cb05b25994be0abb197b87c45f178",
			//2025.04.01
			//合作单位（赛事）
			"e70f60d0bcd04ac5b2bc1198542ba0ba","a641edbdcf0e442bab94842c0c08b4a0",
			//2025.06.03
			"4bffbf17744f4ad4ab70d1f8f15e8295",
			//2025.07.01
			//美容
			"e0928b89071d4f1499a716446329befb","3df58af369db4d9d825bb018d1becb89",
			//培训
			"0650a285501943369e561a1f05891349","973b8836a2ed4884a895d5bcea9291a9","ea51106dab0849bca544ff21f8da992b",
			//2025.08.01
			//培训
			"9e82db321c7e45dc95ae36f9526442f7",
			//繁育基数
			"950f5aec851047788611ad5bb57ff8a6", "c60cbd6e329d4848931cb78bb2ecf8d0", "53091d337a374be8ad2aece2f9e02ef7",
	};

	public static final String[] YEAR_TO_MONTH_12 = { "237", "1", "105", "106", "27",
			"292a74d8249d44a88365fee44a2818b2", "644132ab3e06413e8a01e46d59afa17a", "102", "103", "200",
			"2c4e95dbc3ce4d5b898247f172594866", "e2e26a94169c41e588551ce9d9197fd6", "de2d33b8bb014867968c31689dbc71bb",
			"3b51f5e483c348c7b465086060032324", "286", "287", "288", "289", "290", "291", "292", "293", "294", "295",
			"296", "1111", "63", "96b80592faeb41a18a7343c63490ffc0", "64b70616c06746e595dfe29b5c388dae","005bf368da2447d2bcdaf4a66a81ac33",
			//2024.02.01
			//培训
			"1ee5686f97f6480c94e09d6332791cb5",
			//美容
			"3ca9e19215284fb49f92b147f502535d",
			//2024.05.06
			//美容
			"0b9839845d0949ddbcbd0f391a887f77",
			//2024.10.08
			//会员
			"07bdf5aacd0f4dc1a3d7370c2f8e2ede",
	};
	public static final String[] YEAR_TO_MONTH_2 = { "320", };
	public static final String[] YEAR_TO_MONTH_3 = { "308", "309", };
	public static final String[] YEAR_TO_MONTH_36 = { "d1ad21c0d8c5413380808014fb7bb03d", };
	public static final String[] YEAR_TO_MONTH_60 = { "95", "46779ba4c36c4de399911e0c6ed75c68","2654257cdfa1442c8f69df320b04de73", };

	public static final String[] YEAR_TO_MONTH_WITH_TWO_12 = { "3", "6",};
	public static final String[] YEAR_TO_MONTH_WITH_TWO_24 = { "da8f743b648c4f369fc19ebb31662ad1",};
	public static final String[] YEAR_TO_MONTH_WITH_TWO_120 = { "7", };

	@Async("splitOrder")
	@Transactional(propagation = Propagation.REQUIRES_NEW,readOnly = false, rollbackFor = Exception.class)
	public void initSplitAsync(String orderCode) {
		this.initSplit(orderCode);
	}

	@Transactional(readOnly = false, rollbackFor = Exception.class)
	public void initSplit(String orderCode) {
		List<PaymentOrderDetail> list = paymentOrderDetailDao.findByOrderCode(orderCode);
		PaymentRecord paymentRecord = paymentRecordDao.getByRunningNumber(orderCode);
		if (Collections3.isEmpty(list) && paymentRecord == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "拆分错误：找不到该订单");
		} else {
			if (!Collections3.isEmpty(list)) {
				List<String> itemList = new ArrayList<>();
				for (PaymentOrderDetail detail : list) {
					if (!itemList.contains(detail.getChargingItemId()) && !"0".equals(detail.getChargingItemId())) {
						itemList.add(detail.getChargingItemId());
					}
				}
				for (String s : itemList) {
					split(s, orderCode);
				}
			}
			if (paymentRecord != null) {
				split(paymentRecord.getChargingItemId(), paymentRecord.getRunningNumber());
			}
		}
	}

	@Transactional(readOnly = false)
	public void split(String itemId, String orderCode) {
//        拆分关系
		HashMap<String, RelationShipEntity> relationShipMap = new HashMap<String, RelationShipEntity>(2 << 8);
		HashMap<String, String> ship = cabFinanceStatisticsDao.relationShipGetById(itemId);
		if (ship != null) {
			relationShipMap.put((String) ship.get("id"),
					new RelationShipEntity((String) ship.get("id"),
							ship.get("CAB") != null ? (String) ship.get("CAB") : null,
							ship.get("CKU") != null ? (String) ship.get("CKU") : null, (String) ship.get("dept")));
			// 查询拆分规则
			HashMap<String, BigDecimal> chargingPriceMap = new HashMap<String, BigDecimal>(2 << 8);
			if (StringUtils.isNotBlank(ship.get("CKU"))) {
				List<Map> ckulist = cabFinanceStatisticsDao
						.ckuChargeItemPriceGetByIdList(Arrays.asList(ship.get("CKU").split(",")));
				for (Map ckum : ckulist) {
					chargingPriceMap.put((String) ckum.get("id"), (BigDecimal) ckum.get("price"));

				}
			}

			if (StringUtils.isNotBlank(ship.get("CAB"))) {
				List<Map> cablist = cabFinanceStatisticsDao
						.cabChargeItemPriceGetByIdList(Arrays.asList(ship.get("CAB").split(",")));
				for (Map cabm : cablist) {
					chargingPriceMap.put((String) cabm.get("id"), (BigDecimal) cabm.get("price"));
				}
			}

			// 获取订单和流水
			List<PaymentOrderDetail> details = paymentOrderDetailDao.findByOrderCodeAndItemId(orderCode, itemId);
			List<PaymentRecord> recordList = paymentRecordDao.getByRunningNumberAndItemId(orderCode, itemId);

			if (Arrays.asList(MATH).contains(itemId)) {
				if (details != null && details.size() > 0)
					initSplitMathOrderDetail(details, itemId, relationShipMap, chargingPriceMap);
				if (recordList != null && recordList.size() > 0)
					initSplitMathRecord(recordList, itemId, relationShipMap, chargingPriceMap);
			} else if (Arrays.asList(ONE_TO_MANY).contains(itemId)) {
				if (details != null && details.size() > 0) {
					for (PaymentOrderDetail orderDetail : details) {
						orderDetail.setSplitBeginDate(orderDetail.getAddTime());
					}
					initSplitOneToManyOrderDetail(details, itemId, relationShipMap, chargingPriceMap);
				}
				if (recordList != null && recordList.size() > 0) {
					for (PaymentRecord record : recordList) {
						record.setSplitBeginDate(record.getPaymentTime());
					}
					initSplitOneTwoManyRecord(recordList, itemId, relationShipMap, chargingPriceMap);
				}
			} else if (Arrays.asList(NO_PROCESS).contains(itemId)) {
				if (details != null && details.size() > 0 && !"0".equals(itemId))
					initSplitNoProcessOrderDetail(details, itemId, relationShipMap, chargingPriceMap);
				if (recordList != null && recordList.size() > 0)
					initSplitNoProcessRecord(recordList, itemId, relationShipMap, chargingPriceMap);
			} else if (Arrays.asList(YEAR_TO_MONTH_2).contains(itemId)) {
				if (details != null && details.size() > 0) {
					for (PaymentOrderDetail orderDetail : details) {
						orderDetail.setSplitBeginDate(orderDetail.getAddTime());
					}
					initSplitYearToMonthOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 2);

				}
				if (recordList != null && recordList.size() > 0) {
					for (PaymentRecord record : recordList) {
						record.setSplitBeginDate(record.getPaymentTime());
					}
					initSplitYearToMonthRecord(recordList, itemId, relationShipMap, chargingPriceMap, 2);
				}
			} else if (Arrays.asList(YEAR_TO_MONTH_3).contains(itemId)) {
				if (details != null && details.size() > 0) {
					for (PaymentOrderDetail orderDetail : details) {
						orderDetail.setSplitBeginDate(orderDetail.getAddTime());
					}
					initSplitYearToMonthOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 3);
				}

				if (recordList != null && recordList.size() > 0) {
					for (PaymentRecord record : recordList) {
						record.setSplitBeginDate(record.getPaymentTime());
					}
					initSplitYearToMonthRecord(recordList, itemId, relationShipMap, chargingPriceMap, 3);

				}
			} else if (Arrays.asList(YEAR_TO_MONTH_36).contains(itemId)) {
				if (details != null && details.size() > 0) {
					for (PaymentOrderDetail orderDetail : details) {
						orderDetail.setSplitBeginDate(orderDetail.getAddTime());
					}
					initSplitYearToMonthOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 36);
				}

				if (recordList != null && recordList.size() > 0) {
					for (PaymentRecord record : recordList) {
						record.setSplitBeginDate(record.getPaymentTime());
					}
					initSplitYearToMonthRecord(recordList, itemId, relationShipMap, chargingPriceMap, 36);

				}
			} else if (Arrays.asList(YEAR_TO_MONTH_WITH_TWO_120).contains(itemId)) {
				if (details != null && details.size() > 0) {
					for (PaymentOrderDetail orderDetail : details) {
						orderDetail.setSplitBeginDate(orderDetail.getAddTime());
					}
					initSplitYearToMonthWithTowOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 120,
							"cab");

				}
				if (recordList != null && recordList.size() > 0) {
					for (PaymentRecord record : recordList) {
						record.setSplitBeginDate(record.getPaymentTime());
					}
					initSplitYearToMonthWithTowRecord(recordList, itemId, relationShipMap, chargingPriceMap, 120,
							"cab");
				}
			} else if (Arrays.asList(YEAR_TO_MONTH_WITH_TWO_24).contains(itemId)) {
				if (details != null && details.size() > 0) {
					for (PaymentOrderDetail orderDetail : details) {
						orderDetail.setSplitBeginDate(orderDetail.getAddTime());
					}
					initSplitYearToMonthWithTowOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 120,
							"cab");

				}
				if (recordList != null && recordList.size() > 0) {
					for (PaymentRecord record : recordList) {
						record.setSplitBeginDate(record.getPaymentTime());
					}
					initSplitYearToMonthWithTowRecord(recordList, itemId, relationShipMap, chargingPriceMap, 120,
							"cab");
				}
			} else if (Arrays.asList(YEAR_TO_MONTH_WITH_TWO_12).contains(itemId)) {
				if ("3".equals(itemId)) {
					if (details != null && details.size() > 0) {
						for (PaymentOrderDetail orderDetail : details) {
							String hrmember = orderDetail.getMemberCode();
							Member member = memberService.getByMemberCode(hrmember);
							if (member == null || member.getMemberType() == null
									|| (!"professional".equals(member.getMemberType()))
									|| member.getMemberEndDate() == null) {
								orderDetail.setSplitBeginDate(orderDetail.getAddTime());
							} else {

								if (hrmember.indexOf("HZ") > -1
										&& "sys_member".equals(orderDetail.getBusinessTable())) {
									// 俱乐部帮会员办理
									Member forMember = memberService.get(orderDetail.getBusinessIds());
									orderDetail.setSplitBeginDate(YearToMonthSplitUtils
											.getBeforeDate(forMember.getMemberEndDate(), orderDetail.getAddTime()));
								} else {
									orderDetail.setSplitBeginDate(YearToMonthSplitUtils
											.getBeforeDate(member.getMemberEndDate(), orderDetail.getAddTime()));
								}
							}

						}
						initSplitYearToMonthWithTowOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");
					}

					if (recordList != null && recordList.size() > 0) {
						for (PaymentRecord record : recordList) {
							String hrmember = record.getMemberCode();
							Member member = memberService.getByMemberCode(hrmember);
							if (member == null || member.getMemberType() == null
									|| (!"professional".equals(member.getMemberType()))
									|| member.getMemberEndDate() == null) {
								record.setSplitBeginDate(record.getPaymentTime());
							} else {
								record.setSplitBeginDate(YearToMonthSplitUtils.getBeforeDate(member.getMemberEndDate(),
										record.getPaymentTime()));
							}
						}
						initSplitYearToMonthWithTowRecord(recordList, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");
					}
				} else if ("6".equals(itemId)) {
					if (details != null && details.size() > 0) {
						for (PaymentOrderDetail orderDetail : details) {
							String hrmember = orderDetail.getMemberCode();
							Kennel kennel = kennelService.getKennelByMemberCode(hrmember);
							if (kennel == null || (!"year".equals(kennel.getType())) || kennel.getEndTime() == null) {
								orderDetail.setSplitBeginDate(orderDetail.getAddTime());
							} else {
								orderDetail.setSplitBeginDate(YearToMonthSplitUtils.getBeforeDate(kennel.getEndTime(),
										orderDetail.getAddTime()));
							}

						}
						initSplitYearToMonthWithTowOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");
					}
					if (recordList != null && recordList.size() > 0) {
						for (PaymentRecord record : recordList) {
							String hrmember = record.getMemberCode();
							Kennel kennel = kennelService.getKennelByMemberCode(hrmember);
							if (kennel == null || (!"year".equals(kennel.getType())) || kennel.getEndTime() == null) {
								record.setSplitBeginDate(record.getPaymentTime());
							} else {
								record.setSplitBeginDate(YearToMonthSplitUtils.getBeforeDate(kennel.getEndTime(),
										record.getPaymentTime()));
							}
						}
						initSplitYearToMonthWithTowRecord(recordList, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");
					}

				}
			} else if (Arrays.asList(YEAR_TO_MONTH_60).contains(itemId)) {
				if (details != null && details.size() > 0) {
					for (PaymentOrderDetail orderDetail : details) {
						orderDetail.setSplitBeginDate(orderDetail.getAddTime());
					}
					initSplitYearToMonthOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 60);

				}

				if (recordList != null && recordList.size() > 0) {
					for (PaymentRecord record : recordList) {
						record.setSplitBeginDate(record.getPaymentTime());
					}
					initSplitYearToMonthRecord(recordList, itemId, relationShipMap, chargingPriceMap, 60);

				}
			} else if (Arrays.asList(YEAR_TO_MONTH_12).contains(itemId)) {
				if (itemId.equals("1")) {
					if (details != null && details.size() > 0) {
						for (PaymentOrderDetail orderDetail : details) {
							String hrmember = orderDetail.getMemberCode();
							Member member = memberService.getByMemberCode(hrmember);
							if (member == null || member.getMemberType() == null
									|| (!"basic".equals(member.getMemberType())) || member.getMemberEndDate() == null) {
								orderDetail.setSplitBeginDate(orderDetail.getAddTime());
							} else {
								if (hrmember.indexOf("HZ") > -1
										&& "sys_member".equals(orderDetail.getBusinessTable())) {
									// 俱乐部帮会员办理
									Member forMember = memberService.get(orderDetail.getBusinessIds());
									orderDetail.setSplitBeginDate(YearToMonthSplitUtils
											.getBeforeDate(forMember.getMemberEndDate(), orderDetail.getAddTime()));
								} else {
									orderDetail.setSplitBeginDate(YearToMonthSplitUtils
											.getBeforeDate(member.getMemberEndDate(), orderDetail.getAddTime()));
								}
							}

						}
						initSplitYearToMonthOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 12);
					}
					if (recordList != null && recordList.size() > 0) {
						for (PaymentRecord record : recordList) {
							String hrmember = record.getMemberCode();
							Member member = memberService.getByMemberCode(hrmember);
							if (member == null || member.getMemberType() == null
									|| (!"basic".equals(member.getMemberType())) || member.getMemberEndDate() == null) {
								record.setSplitBeginDate(record.getPaymentTime());
							} else {
								record.setSplitBeginDate(YearToMonthSplitUtils.getBeforeDate(member.getMemberEndDate(),
										record.getPaymentTime()));
							}
						}
						initSplitYearToMonthRecord(recordList, itemId, relationShipMap, chargingPriceMap, 12);
					}

				} else if (itemId.equals("105")) {
					if (details != null && details.size() > 0) {
						for (PaymentOrderDetail orderDetail : details) {
							String hrmember = orderDetail.getMemberCode();
							Member member = memberService.getByMemberCode(hrmember);
							if (member == null || member.getGroomerType() == null
									|| (!"reserve".equals(member.getGroomerType()))
									|| member.getGroomerEndDate() == null) {
								orderDetail.setSplitBeginDate(orderDetail.getAddTime());
							} else {
								orderDetail.setSplitBeginDate(YearToMonthSplitUtils
										.getBeforeDate(member.getGroomerEndDate(), orderDetail.getAddTime()));
							}

						}
						initSplitYearToMonthWithTowOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");

					}
					if (recordList != null && recordList.size() > 0) {
						for (PaymentRecord record : recordList) {
							String hrmember = record.getMemberCode();
							Member member = memberService.getByMemberCode(hrmember);
							if (member == null || member.getGroomerType() == null
									|| (!"reserve".equals(member.getGroomerType()))
									|| member.getGroomerEndDate() == null) {
								record.setSplitBeginDate(record.getPaymentTime());
							} else {
								record.setSplitBeginDate(YearToMonthSplitUtils.getBeforeDate(member.getGroomerEndDate(),
										record.getPaymentTime()));
							}
						}
						initSplitYearToMonthWithTowRecord(recordList, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");

					}
				} else if (itemId.equals("106")) {
					if (details != null && details.size() > 0) {
						for (PaymentOrderDetail orderDetail : details) {
							String hrmember = orderDetail.getMemberCode();
							Member member = memberService.getByMemberCode(hrmember);
							if (member == null || member.getGroomerType() == null
									|| (!"formal".equals(member.getGroomerType()))
									|| member.getGroomerEndDate() == null) {
								orderDetail.setSplitBeginDate(orderDetail.getAddTime());
							} else {
								orderDetail.setSplitBeginDate(YearToMonthSplitUtils
										.getBeforeDate(member.getGroomerEndDate(), orderDetail.getAddTime()));
							}

						}
						initSplitYearToMonthWithTowOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");
					}
					if (recordList != null && recordList.size() > 0) {
						for (PaymentRecord record : recordList) {
							String hrmember = record.getMemberCode();
							Member member = memberService.getByMemberCode(hrmember);
							if (member == null || member.getGroomerType() == null
									|| (!"formal".equals(member.getGroomerType()))
									|| member.getGroomerEndDate() == null) {
								record.setSplitBeginDate(record.getPaymentTime());
							} else {
								record.setSplitBeginDate(YearToMonthSplitUtils.getBeforeDate(member.getGroomerEndDate(),
										record.getPaymentTime()));
							}
						}
						initSplitYearToMonthWithTowRecord(recordList, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");

					}
				} else if (itemId.equals("27")) {
					if (details != null && details.size() > 0) {
						for (PaymentOrderDetail orderDetail : details) {
							String yymember = orderDetail.getMemberCode();
							TrainingInstitution trainingInstitution = null;
							if (yymember.indexOf("MRHZ") > -1) {
								trainingInstitution = trainingInstitutionDao.getBySchoolMemberCode(yymember);
							} else {
								trainingInstitution = trainingInstitutionDao.getByMemberCode(yymember);
							}
							if (trainingInstitution == null || trainingInstitution.getEndDate() == null) {
								orderDetail.setSplitBeginDate(orderDetail.getAddTime());
							} else {
								orderDetail.setSplitBeginDate(YearToMonthSplitUtils
										.getBeforeDate(trainingInstitution.getEndDate(), orderDetail.getAddTime()));
							}

						}
						initSplitYearToMonthWithTowOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");
					}

					if (recordList != null && recordList.size() > 0) {
						for (PaymentRecord record : recordList) {
							String yymember = record.getMemberCode();
							TrainingInstitution trainingInstitution = null;
							if (yymember.indexOf("MRHZ") > -1) {
								trainingInstitution = trainingInstitutionDao.getBySchoolMemberCode(yymember);
							} else {
								trainingInstitution = trainingInstitutionDao.getByMemberCode(yymember);
							}
							if (trainingInstitution == null || trainingInstitution.getEndDate() == null) {
								record.setSplitBeginDate(record.getPaymentTime());
							} else {
								record.setSplitBeginDate(YearToMonthSplitUtils
										.getBeforeDate(trainingInstitution.getEndDate(), record.getPaymentTime()));
							}
						}
						initSplitYearToMonthWithTowRecord(recordList, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");

					}
				} else if (itemId.equals("102") || itemId.equals("103")) {
					if (details != null && details.size() > 0) {
						for (PaymentOrderDetail orderDetail : details) {
							String yymember = orderDetail.getMemberCode();
							TrainingInstitution trainingInstitution = trainingInstitutionDao.getByMemberCode(yymember);
							if (trainingInstitution == null || trainingInstitution.getEndDate() == null) {
								orderDetail.setSplitBeginDate(orderDetail.getAddTime());
							} else {
								orderDetail.setSplitBeginDate(YearToMonthSplitUtils
										.getBeforeDate(trainingInstitution.getEndDate(), orderDetail.getAddTime()));
							}
						}
						initSplitYearToMonthWithTowOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");
					}

					if (recordList != null && recordList.size() > 0) {
						for (PaymentRecord record : recordList) {
							String yymember = record.getMemberCode();
							TrainingInstitution trainingInstitution = trainingInstitutionDao.getByMemberCode(yymember);
							if (trainingInstitution == null || trainingInstitution.getEndDate() == null) {
								record.setSplitBeginDate(record.getPaymentTime());
							} else {
								record.setSplitBeginDate(YearToMonthSplitUtils
										.getBeforeDate(trainingInstitution.getEndDate(), record.getPaymentTime()));
							}
						}
						initSplitYearToMonthWithTowRecord(recordList, itemId, relationShipMap, chargingPriceMap, 12,
								"cab");

					}
				} else {
					if (details != null && details.size() > 0) {
						for (PaymentOrderDetail orderDetail : details) {
							orderDetail.setSplitBeginDate(orderDetail.getAddTime());
						}
						initSplitYearToMonthOrderDetail(details, itemId, relationShipMap, chargingPriceMap, 12);
					}
					if (recordList != null && recordList.size() > 0) {
						for (PaymentRecord record : recordList) {
							record.setSplitBeginDate(record.getPaymentTime());
						}
						initSplitYearToMonthRecord(recordList, itemId, relationShipMap, chargingPriceMap, 12);
					}
				}
			}
		}
	}

	public void initSplitYearToMonthWithTowOrderDetail(List<PaymentOrderDetail> orderDetailList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap, int step,
			String stepType) {
		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		orderDetailList.forEach(r -> {

			if (stepType.equals("cab")) {
				BigDecimal cabPrice = chargingPriceMap.get(relationShipEntity.getCab());
				BigDecimal cabTotalPrice = chargingPriceMap.get(relationShipEntity.getCab())
						.multiply(new BigDecimal(r.getNum()));
				List<YearToMonthSplitEntity> list = YearToMonthSplitUtils.splitYearToMonth(r.getSplitBeginDate(), step,
						cabTotalPrice, cabPrice);
				for (YearToMonthSplitEntity ytm : list) {
					SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
					saDetail.setPaymentNum(r.getNum());
					saDetail.setMemberCode(r.getMemberCode());
					saDetail.setChargeDate(ytm.getChargeDate());
					saDetail.setPrice(ytm.getPrice());
					saDetail.setRunningNumber(r.getOrderCode());
					saDetail.setPaymentMoney(ytm.getPaymentMoney());
					saDetail.setType(PAYMENT_TYPE_CAB);
					saDetail.setSourceCharge(id);
					saDetail.setChargeItemId(relationShipEntity.getCab());
					saDetail.setOrderId(r.getId());
					saDetail.setPaymentTime(
							(r.getPaymentOrder() != null && r.getPaymentOrder().getPaymentTime() != null)
									? r.getPaymentOrder().getPaymentTime()
									: r.getAddTime());
					saDetail.setDeptId(relationShipEntity.getDept());
					saDetail.setInstallmentEndDate(ytm.getInstallmentEndDate());
					saDetail.setInstallmentBeginDate(ytm.getInstallmentBeginDate());
					saDetail.setChargeGroupId(r.getBusinessId());
					PaymentOrder order = paymentOrderDao.getByorderCode(r.getOrderCode());
					saDetail.setPaymentWay(order.getPaymentWay());
					saDetail.setPayMemberCode(getPayMemberCodeByOrder(r.getOrderCode()));
					saDetail.preInsert();
					saPaymentDetailTotalDao.insert(saDetail);
				}

				SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
				saDetail.setMemberCode(r.getMemberCode());
				saDetail.setChargeDate(r.getAddTime());
				saDetail.setRunningNumber(r.getOrderCode());
				saDetail.setType(PAYMENT_TYPE_CKU);
				saDetail.setSourceCharge(id);
				saDetail.setChargeItemId(relationShipEntity.getCku());
				BigDecimal ckuPrice = chargingPriceMap.get(relationShipEntity.getCku());
				saDetail.setPaymentNum(r.getNum());
				saDetail.setPrice(ckuPrice == null ? BigDecimal.ZERO.toString() : ckuPrice.toString());
				saDetail.setPaymentMoney(
						new BigDecimal(r.getNum()).multiply(ckuPrice == null ? BigDecimal.ZERO : ckuPrice).toString());
				saDetail.setOrderId(r.getId());
				saDetail.setPaymentTime((r.getPaymentOrder() != null && r.getPaymentOrder().getPaymentTime() != null)
						? r.getPaymentOrder().getPaymentTime()
						: r.getAddTime());
				saDetail.setDeptId(relationShipEntity.getDept());
				saDetail.setChargeGroupId(r.getBusinessId());
				PaymentOrder order = paymentOrderDao.getByorderCode(r.getOrderCode());
				saDetail.setPaymentWay(order.getPaymentWay());
				saDetail.setPayMemberCode(getPayMemberCodeByOrder(r.getOrderCode()));
				saDetail.preInsert();
				saPaymentDetailTotalDao.insert(saDetail);
			}

		});
	}

	public void initSplitYearToMonthWithTowRecord(List<PaymentRecord> recordList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap, int step,
			String stepType) {
		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		recordList.forEach(r -> {
			if (stepType.equals("cab")) {
				BigDecimal cabPrice = chargingPriceMap.get(relationShipEntity.getCab());
				Integer chargingItemNum = r.getChargingItemNum() == null ? 1 : r.getChargingItemNum();
				BigDecimal cabTotalPrice = chargingPriceMap.get(relationShipEntity.getCab())
						.multiply(new BigDecimal(chargingItemNum));
				List<YearToMonthSplitEntity> list = null;

				// 如果不是扣费输入分摊区间，走正常拆分 按支付时间向后拆分
				if (StringUtils.isBlank(r.getIsInstallment()) || "0".equals(r.getIsInstallment())) {
					list = YearToMonthSplitUtils.splitYearToMonth(r.getSplitBeginDate(), step, cabTotalPrice, cabPrice);
				} else {
					// 如果输入分摊区间，走新规拆分 按支付时间 和 拆分时间做对比
					list = YearToMonthSplitUtils.splitYearToMonthWithInstallment(r.getSplitBeginDate(),
							r.getInstallmentStartDate(), r.getInstallmentEndDate(), step, cabTotalPrice, cabPrice);
				}
				for (YearToMonthSplitEntity ytm : list) {
					SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
					saDetail.setPaymentNum(r.getChargingItemNum() == null ? "1" : r.getChargingItemNum().toString());
					saDetail.setMemberCode(r.getMemberCode());
					saDetail.setChargeDate(ytm.getChargeDate());
					saDetail.setPrice(ytm.getPrice());
					saDetail.setRunningNumber(r.getRunningNumber());
					saDetail.setPaymentMoney(ytm.getPaymentMoney());

					saDetail.setType(PAYMENT_TYPE_CAB);
					saDetail.setChargeItemId(relationShipEntity.getCab());
					saDetail.setSourceCharge(id);
					saDetail.setRecordId(r.getId());
					saDetail.setPaymentTime(r.getPaymentTime());
					saDetail.setDeptId(relationShipEntity.getDept());
					saDetail.setInstallmentBeginDate(ytm.getInstallmentBeginDate());
					saDetail.setInstallmentEndDate(ytm.getInstallmentEndDate());
					saDetail.setPaymentWay(r.getPaymentWay());
					saDetail.setPayMemberCode(getPaymemberCodeByRecord(r));
					saDetail.preInsert();
					saPaymentDetailTotalDao.insert(saDetail);
				}
				SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
				saDetail.setPaymentNum(r.getChargingItemNum() == null ? "1" : r.getChargingItemNum().toString());
				saDetail.setMemberCode(r.getMemberCode());
				saDetail.setChargeDate(r.getCreateDate());
				saDetail.setRunningNumber(r.getRunningNumber());
				saDetail.setPaymentMoney(r.getPaymentAmount());
				saDetail.setType(PAYMENT_TYPE_CKU);
				saDetail.setChargeItemId(relationShipEntity.getCku());
				saDetail.setSourceCharge(id);
				BigDecimal ckuPrice = chargingPriceMap.get(relationShipEntity.getCku());
				saDetail.setPrice(ckuPrice == null ? "0.00" : ckuPrice.toString());
				int num = r.getChargingItemNum() == null ? 1 : r.getChargingItemNum();
				if (ckuPrice == null) {
					saDetail.setPaymentMoney("0.00");
				} else {
					saDetail.setPaymentMoney(new BigDecimal(num).multiply(ckuPrice).toString());
				}
				saDetail.setRecordId(r.getId());
				saDetail.setPaymentTime(r.getPaymentTime());
				saDetail.setDeptId(relationShipEntity.getDept());
				saDetail.setPaymentWay(r.getPaymentWay());
				saDetail.setPayMemberCode(getPaymemberCodeByRecord(r));
				saDetail.preInsert();
				saPaymentDetailTotalDao.insert(saDetail);
			}

		});
	}

	public void initSplitYearToMonthRecord(List<PaymentRecord> recordList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap,
			int step) {
		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		recordList.forEach(r -> {
			List<YearToMonthSplitEntity> list = null;
			// 如果不是扣费输入分摊区间，走正常拆分 按支付时间向后拆分
			if (StringUtils.isBlank(r.getIsInstallment()) || "0".equals(r.getIsInstallment())) {
				list = YearToMonthSplitUtils.splitYearToMonth(r.getSplitBeginDate(), step,
						new BigDecimal(r.getPaymentAmount()), new BigDecimal(r.getPrice()));
			} else {
				// 如果输入分摊区间，走新规拆分 按支付时间 和 拆分时间做对比
				list = YearToMonthSplitUtils.splitYearToMonthWithInstallment(r.getSplitBeginDate(),
						r.getInstallmentStartDate(), r.getInstallmentEndDate(), step,
						new BigDecimal(r.getPaymentAmount()), new BigDecimal(r.getPrice()));
			}

			for (YearToMonthSplitEntity ytm : list) {
				SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
				saDetail.setPaymentNum(r.getChargingItemNum() == null ? "1" : r.getChargingItemNum().toString());
				saDetail.setMemberCode(r.getMemberCode());
				saDetail.setChargeDate(ytm.getChargeDate());
				saDetail.setPrice(ytm.getPrice());
				saDetail.setRunningNumber(r.getRunningNumber());
				saDetail.setPaymentMoney(ytm.getPaymentMoney());
				if (relationShipEntity.getCab() != null && relationShipEntity.getCab().length() > 0) {
					saDetail.setType(PAYMENT_TYPE_CAB);
					saDetail.setChargeItemId(relationShipEntity.getCab());
				} else {
					saDetail.setType(PAYMENT_TYPE_CKU);
					saDetail.setChargeItemId(relationShipEntity.getCku());
				}
				saDetail.setSourceCharge(id);
				saDetail.setRecordId(r.getId());
				saDetail.setPaymentTime(r.getPaymentTime());
				saDetail.setDeptId(relationShipEntity.getDept());
				saDetail.setInstallmentBeginDate(ytm.getInstallmentBeginDate());
				saDetail.setInstallmentEndDate(ytm.getInstallmentEndDate());
				saDetail.setPaymentWay(r.getPaymentWay());
				saDetail.setPayMemberCode(getPaymemberCodeByRecord(r));
				saDetail.preInsert();
				saPaymentDetailTotalDao.insert(saDetail);
			}

		});
	}

	public void initSplitYearToMonthOrderDetail(List<PaymentOrderDetail> orderDetailList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap,
			int step) {
		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		orderDetailList.forEach(r -> {
			List<YearToMonthSplitEntity> list = YearToMonthSplitUtils.splitYearToMonth(r.getSplitBeginDate(), step,
					new BigDecimal(r.getTotalPrice()), new BigDecimal(r.getPrice()));
			for (YearToMonthSplitEntity ytm : list) {
				SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
				saDetail.setPaymentNum(r.getNum());
				saDetail.setMemberCode(r.getMemberCode());
				saDetail.setChargeDate(ytm.getChargeDate());
				saDetail.setPrice(ytm.getPrice());
				saDetail.setRunningNumber(r.getOrderCode());
				saDetail.setPaymentMoney(ytm.getPaymentMoney());
				if (relationShipEntity.getCab() != null && relationShipEntity.getCab().length() > 0) {
					saDetail.setType(PAYMENT_TYPE_CAB);
					saDetail.setChargeItemId(relationShipEntity.getCab());
				} else {
					saDetail.setType(PAYMENT_TYPE_CKU);
					saDetail.setChargeItemId(relationShipEntity.getCku());
				}
				saDetail.setSourceCharge(id);
				saDetail.setOrderId(r.getId());
				saDetail.setPaymentTime((r.getPaymentOrder() != null && r.getPaymentOrder().getPaymentTime() != null)
						? r.getPaymentOrder().getPaymentTime()
						: r.getAddTime());
				saDetail.setDeptId(relationShipEntity.getDept());
				saDetail.setInstallmentEndDate(ytm.getInstallmentEndDate());
				saDetail.setInstallmentBeginDate(ytm.getInstallmentBeginDate());
				saDetail.setChargeGroupId(r.getBusinessId());
				PaymentOrder order = paymentOrderDao.getByorderCode(r.getOrderCode());
				saDetail.setPaymentWay(order.getPaymentWay());
				saDetail.setPayMemberCode(getPayMemberCodeByOrder(r.getOrderCode()));
				saDetail.preInsert();
				saPaymentDetailTotalDao.insert(saDetail);
			}
		});
	}

	public void initSplitOneToManyOrderDetail(List<PaymentOrderDetail> orderDetailList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap) {
		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		orderDetailList.forEach(r -> {

			List<String> insertCharging = new ArrayList<String>();
			List<BigDecimal> insertPrice = new ArrayList<BigDecimal>();
			// 新规实际收款价格
			List<BigDecimal> insertPay = new ArrayList<BigDecimal>();
			BigDecimal chargingPrice = new BigDecimal(0);
			if (relationShipEntity.getCku() != null && relationShipEntity.getCku().length() > 0) {
				String[] ckuArr = relationShipEntity.getCku().split(",");
				for (String cku : ckuArr) {
					insertCharging.add(cku);
					insertPrice.add(chargingPriceMap.get(cku));
					insertPay.add(chargingPriceMap.get(cku));
					chargingPrice = chargingPrice
							.add(chargingPriceMap.get(cku) == null ? new BigDecimal(0) : chargingPriceMap.get(cku));

				}

			}
			if (relationShipEntity.getCab() != null && relationShipEntity.getCab().length() > 0) {
				String[] cabArr = relationShipEntity.getCab().split(",");
				for (String cab : cabArr) {
					insertCharging.add(cab);
					insertPrice.add(chargingPriceMap.get(cab));
					insertPay.add(chargingPriceMap.get(cab));
					chargingPrice = chargingPrice
							.add(chargingPriceMap.get(cab) == null ? new BigDecimal(0) : chargingPriceMap.get(cab));

				}
			}

			try {
				BigDecimal realPrice = new BigDecimal(r.getTotalPrice())
						.divide(new BigDecimal(Integer.parseInt(r.getNum())));

				// 如果不相等 eg 1 相同情况有 正常
				if (chargingPrice.compareTo(realPrice) != 0 && chargingPrice.compareTo(new BigDecimal(0)) > 0
						&& realPrice.compareTo(new BigDecimal(0)) > 0) {
					BigDecimal yinzi = realPrice.divide(chargingPrice, 2, BigDecimal.ROUND_HALF_UP);
					for (int i = 0; i < insertPrice.size(); i++) {
						insertPay.set(i, insertPrice.get(i).multiply(yinzi));
					}
				} else if (realPrice.compareTo(new BigDecimal(0)) == 0) {
					for (int i = 0; i < insertPrice.size(); i++) {
						insertPay.set(i, new BigDecimal(0));
					}
				}

				for (int i = 0; i < insertCharging.size(); i++) {
					SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
					saDetail.setPaymentNum(r.getNum());
					saDetail.setMemberCode(r.getMemberCode());
					saDetail.setChargeDate(r.getCreateDate());
					saDetail.setRunningNumber(r.getOrderCode());
					if (insertCharging.get(i).indexOf("cab") > -1) {
						saDetail.setType(PAYMENT_TYPE_CAB);
					} else {
						saDetail.setType(PAYMENT_TYPE_CKU);
					}
					saDetail.setSourceCharge(id);
					saDetail.setChargeItemId(insertCharging.get(i));
					saDetail.setPrice(insertPrice.get(i).toString());
					saDetail.setPaymentMoney(
							insertPay.get(i).multiply(new BigDecimal(Integer.parseInt(r.getNum()))).toString());

					saDetail.setOrderId(r.getId());
					saDetail.setPaymentTime(
							(r.getPaymentOrder() != null && r.getPaymentOrder().getPaymentTime() != null)
									? r.getPaymentOrder().getPaymentTime()
									: r.getAddTime());
					saDetail.setDeptId(relationShipEntity.getDept());
					saDetail.setChargeGroupId(r.getBusinessId());
					PaymentOrder order = paymentOrderDao.getByorderCode(r.getOrderCode());
					saDetail.setPaymentWay(order.getPaymentWay());
					saDetail.setPayMemberCode(getPayMemberCodeByOrder(r.getOrderCode()));
					saDetail.preInsert();
					saPaymentDetailTotalDao.insert(saDetail);
				}

			} catch (Exception e) {
				System.out.println("数据处理错误：总价或者数量少" + r.getId());
				e.printStackTrace();
			}

		});
	}

	public void initSplitOneTwoManyRecord(List<PaymentRecord> recordList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap) {

		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		recordList.forEach(r -> {
			// 拆分的收费项列表
			List<String> insertCharging = new ArrayList<String>();
			// 新规价格
			List<BigDecimal> insertPrice = new ArrayList<BigDecimal>();
			// 新规实际收款价格
			List<BigDecimal> insertPay = new ArrayList<BigDecimal>();
			// 新规总价
			BigDecimal chargingPrice = new BigDecimal(0);
			if (relationShipEntity.getCku() != null && relationShipEntity.getCku().length() > 0) {
				String[] ckuArr = relationShipEntity.getCku().split(",");
				for (String cku : ckuArr) {
					insertCharging.add(cku);
					System.out.println(" ====== cku: " + cku);
					System.out.println(" ====== chargingPriceMap:" + JSONObject.toJSONString(chargingPriceMap));
					insertPrice.add(chargingPriceMap.get(cku));
					insertPay.add(chargingPriceMap.get(cku));
					if (chargingPriceMap.get(cku) != null) {
						chargingPrice = chargingPrice.add(chargingPriceMap.get(cku));
					}
				}

			}
			if (relationShipEntity.getCab() != null && relationShipEntity.getCab().length() > 0) {
				String[] cabArr = relationShipEntity.getCab().split(",");
				for (String cab : cabArr) {
					insertCharging.add(cab);
					insertPrice.add(chargingPriceMap.get(cab));
					insertPay.add(chargingPriceMap.get(cab));
					if (chargingPriceMap.get(cab) != null) {
						chargingPrice = chargingPrice.add(chargingPriceMap.get(cab));
					}
				}
			}

			try {
				BigDecimal realPrice = new BigDecimal(r.getPaymentAmount())
						.divide(new BigDecimal(r.getChargingItemNum() == null ? 1 : r.getChargingItemNum()));

				// 如果不相等 eg 1 相同情况有 正常
				if (chargingPrice.compareTo(realPrice) != 0 && chargingPrice.compareTo(new BigDecimal(0)) > 0
						&& realPrice.compareTo(new BigDecimal(0)) > 0) {
					BigDecimal yinzi = realPrice.divide(chargingPrice, 2, BigDecimal.ROUND_HALF_UP);
					for (int i = 0; i < insertPrice.size(); i++) {
						insertPay.set(i, insertPrice.get(i).multiply(yinzi));
					}
				} else if (realPrice.compareTo(new BigDecimal(0)) == 0) {
					for (int i = 0; i < insertPrice.size(); i++) {
						insertPay.set(i, new BigDecimal(0));
					}
				}

				for (int i = 0; i < insertCharging.size(); i++) {
					SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
					saDetail.setPaymentNum(r.getChargingItemNum() == null ? "1" : r.getChargingItemNum().toString());
					saDetail.setMemberCode(r.getMemberCode());
					saDetail.setChargeDate(r.getCreateDate());
					saDetail.setRunningNumber(r.getRunningNumber());

					if (insertCharging.get(i).indexOf("cab") > -1) {
						saDetail.setType(PAYMENT_TYPE_CAB);
					} else {
						saDetail.setType(PAYMENT_TYPE_CKU);
					}
					saDetail.setSourceCharge(id);
					saDetail.setChargeItemId(insertCharging.get(i));
					saDetail.setPrice(insertPrice.get(i).toString());
					saDetail.setPaymentMoney(insertPay.get(i)
							.multiply(new BigDecimal(r.getChargingItemNum() == null ? 1 : r.getChargingItemNum()))
							.toString());

					saDetail.setRecordId(r.getId());
					saDetail.setPaymentTime(r.getPaymentTime());
					saDetail.setDeptId(relationShipEntity.getDept());
					saDetail.setPaymentWay(r.getPaymentWay());
					saDetail.setPayMemberCode(getPaymemberCodeByRecord(r));
					saDetail.preInsert();
					saPaymentDetailTotalDao.insert(saDetail);
				}

			} catch (Exception e) {
				System.out.println("数据处理错误：总价或者数量少" + r.getId());
				e.printStackTrace();
			}
		});
	}

	public void initSplitNoProcessOrderDetail(List<PaymentOrderDetail> orderDetailList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap) {
		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		orderDetailList.forEach(r -> {
			SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
			saDetail.setPaymentNum(r.getNum());
			saDetail.setMemberCode(r.getMemberCode());
			saDetail.setChargeDate(r.getAddTime());
			saDetail.setRunningNumber(r.getOrderCode());
			saDetail.setPaymentMoney(r.getTotalPrice());
			saDetail.setSourceCharge(id);
			if (relationShipEntity.getCab() != null && relationShipEntity.getCab().length() > 0) {
				saDetail.setType(PAYMENT_TYPE_CAB);
				saDetail.setChargeItemId(relationShipEntity.getCab());
				BigDecimal price = chargingPriceMap.get(relationShipEntity.getCab());

				if (price.compareTo(new BigDecimal(0)) == 0) {
					if ((id.equals("242") || id.equals("12")) && !r.getPrice().equals(price)) {
						saDetail.setPrice(r.getPrice());
					} else {
						saDetail.setPrice(r.getPrice());
					}
				} else {
					if ((id.equals("242") || id.equals("12")) && !r.getPrice().equals(price)) {
						saDetail.setPrice(r.getPrice());
					} else {
						saDetail.setPrice(price.toString());
					}
				}
			} else {
				saDetail.setType(PAYMENT_TYPE_CKU);
				saDetail.setChargeItemId(relationShipEntity.getCku());
				BigDecimal price = chargingPriceMap.get(relationShipEntity.getCku());
				if (price.compareTo(new BigDecimal(0)) == 0) {
					saDetail.setPrice(r.getPrice());
				} else {
					saDetail.setPrice(price.toString());
				}
			}
			saDetail.setOrderId(r.getId());
			saDetail.setPaymentTime((r.getPaymentOrder() != null && r.getPaymentOrder().getPaymentTime() != null)
					? r.getPaymentOrder().getPaymentTime()
					: r.getAddTime());
			saDetail.setDeptId(relationShipEntity.getDept());
			saDetail.setChargeGroupId(r.getBusinessId());
			PaymentOrder order = paymentOrderDao.getByorderCode(r.getOrderCode());
			saDetail.setPaymentWay(order.getPaymentWay());
			saDetail.setPayMemberCode(getPayMemberCodeByOrder(r.getOrderCode()));
			saDetail.preInsert();
			saPaymentDetailTotalDao.insert(saDetail);
		});
	}

	public void initSplitNoProcessRecord(List<PaymentRecord> recordList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap) {
		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		recordList.forEach(r -> {
			SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
			saDetail.setPaymentNum(r.getChargingItemNum() == null ? "1" : r.getChargingItemNum().toString());
			saDetail.setMemberCode(r.getMemberCode());
			saDetail.setChargeDate(r.getCreateDate());
			saDetail.setRunningNumber(r.getRunningNumber());
			saDetail.setPaymentMoney(r.getPaymentAmount());
			saDetail.setSourceCharge(id);
			if (relationShipEntity.getCab() != null && relationShipEntity.getCab().length() > 0) {
				saDetail.setType(PAYMENT_TYPE_CAB);
				saDetail.setChargeItemId(relationShipEntity.getCab());
				BigDecimal price = chargingPriceMap.get(relationShipEntity.getCab());
				if (price.compareTo(new BigDecimal(0)) == 0) {
					saDetail.setPrice(r.getPrice());
				} else {
					saDetail.setPrice(price.toString());
				}

			} else {
				saDetail.setType(PAYMENT_TYPE_CKU);
				saDetail.setChargeItemId(relationShipEntity.getCku());
				BigDecimal price = chargingPriceMap.get(relationShipEntity.getCku());
				if (price.compareTo(new BigDecimal(0)) == 0) {
					saDetail.setPrice(r.getPrice());
				} else {
					saDetail.setPrice(price.toString());
				}
			}

			saDetail.setRecordId(r.getId());
			saDetail.setPaymentTime(r.getPaymentTime());
			saDetail.setDeptId(relationShipEntity.getDept());
			saDetail.setPaymentWay(r.getPaymentWay());
			saDetail.setPayMemberCode(getPaymemberCodeByRecord(r));
			saDetail.preInsert();
			saPaymentDetailTotalDao.insert(saDetail);
		});
	}

	public void initSplitMathOrderDetail(List<PaymentOrderDetail> orderDetailList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap) {
		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		orderDetailList.forEach(r -> {
			int feeLong = new BigDecimal(r.getTotalPrice()).intValue();
			int num = Integer.parseInt(r.getNum());
			HashMap<String, Integer> countMap = new HashMap();
			countMap.put("280", 0);
			countMap.put("380", 0);
			if (splitShow(feeLong, num, countMap)) {
				if (countMap.get("280") > 0) {
					SaPaymentDetailTotal saDetail280 = new SaPaymentDetailTotal();
					saDetail280.setPaymentNum(countMap.get("280").toString());
					saDetail280.setMemberCode(r.getMemberCode());
					saDetail280.setChargeDate(r.getAddTime());
					saDetail280.setPrice("280");
					saDetail280.setRunningNumber(r.getOrderCode());
					saDetail280.setPaymentMoney((280 * countMap.get("280")) + "");
					saDetail280.setType(PAYMENT_TYPE_CAB);
					saDetail280.setSourceCharge(id);
					saDetail280.setChargeItemId("cab2192");
					saDetail280.setOrderId(r.getId());
					saDetail280.setPaymentTime(
							(r.getPaymentOrder() != null && r.getPaymentOrder().getPaymentTime() != null)
									? r.getPaymentOrder().getPaymentTime()
									: r.getAddTime());
					saDetail280.setDeptId(relationShipEntity.getDept());
					saDetail280.setChargeGroupId(r.getBusinessId());
					PaymentOrder order = paymentOrderDao.getByorderCode(r.getOrderCode());
					saDetail280.setPaymentWay(order.getPaymentWay());
					saDetail280.setPayMemberCode(getPayMemberCodeByOrder(r.getOrderCode()));
					saDetail280.preInsert();
					saPaymentDetailTotalDao.insert(saDetail280);
				}

				if (countMap.get("380") > 0) {
					SaPaymentDetailTotal saDetail380 = new SaPaymentDetailTotal();
					saDetail380.setPaymentNum(countMap.get("380").toString());
					saDetail380.setMemberCode(r.getMemberCode());
					saDetail380.setChargeDate(r.getAddTime());
					saDetail380.setPrice("380");
					saDetail380.setRunningNumber(r.getOrderCode());
					saDetail380.setPaymentMoney((380 * countMap.get("380")) + "");
					saDetail380.setType(PAYMENT_TYPE_CAB);
					saDetail380.setSourceCharge(id);
					saDetail380.setChargeItemId("cab2191");
					saDetail380.setOrderId(r.getId());
					saDetail380.setPaymentTime(
							(r.getPaymentOrder() != null && r.getPaymentOrder().getPaymentTime() != null)
									? r.getPaymentOrder().getPaymentTime()
									: r.getAddTime());
					saDetail380.setDeptId(relationShipEntity.getDept());
					saDetail380.setChargeGroupId(r.getBusinessId());
					PaymentOrder order = paymentOrderDao.getByorderCode(r.getOrderCode());
					saDetail380.setPaymentWay(order.getPaymentWay());
					saDetail380.setPayMemberCode(getPayMemberCodeByOrder(r.getOrderCode()));
					saDetail380.preInsert();
					saPaymentDetailTotalDao.insert(saDetail380);
				}

			} else {

				SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
				saDetail.setPaymentNum(r.getNum());
				saDetail.setMemberCode(r.getMemberCode());
				saDetail.setChargeDate(r.getAddTime());
				saDetail.setPrice(r.getPrice());
				saDetail.setRunningNumber(r.getOrderCode());
				saDetail.setPaymentMoney(r.getTotalPrice());
				saDetail.setType(PAYMENT_TYPE_CAB);
				saDetail.setSourceCharge(id);
				saDetail.setChargeItemId("cab2193");
				saDetail.setOrderId(r.getId());

				saDetail.setPaymentTime((r.getPaymentOrder() != null && r.getPaymentOrder().getPaymentTime() != null)
						? r.getPaymentOrder().getPaymentTime()
						: r.getAddTime());
				saDetail.setDeptId(relationShipEntity.getDept());
				saDetail.setChargeGroupId(r.getBusinessId());
				PaymentOrder order = paymentOrderDao.getByorderCode(r.getOrderCode());
				saDetail.setPaymentWay(order.getPaymentWay());
				saDetail.setPayMemberCode(getPayMemberCodeByOrder(r.getOrderCode()));
				saDetail.preInsert();
				saPaymentDetailTotalDao.insert(saDetail);
			}
			;

		});
	}

	public void initSplitMathRecord(List<PaymentRecord> recordList, String id,
			HashMap<String, RelationShipEntity> relationShipMap, HashMap<String, BigDecimal> chargingPriceMap) {
		RelationShipEntity relationShipEntity = relationShipMap.get(id);
		recordList.forEach(r -> {
			int feeLong = new BigDecimal(r.getPaymentAmount()).intValue();
			int num = r.getChargingItemNum();
			HashMap<String, Integer> countMap = new HashMap();
			countMap.put("280", 0);
			countMap.put("380", 0);
			if (splitShow(feeLong, num, countMap)) {
				if (countMap.get("280") > 0) {
					SaPaymentDetailTotal saDetail280 = new SaPaymentDetailTotal();
					saDetail280.setPaymentNum(countMap.get("280").toString());
					saDetail280.setMemberCode(r.getMemberCode());
					saDetail280.setChargeDate(r.getCreateDate());
					saDetail280.setPrice("280");
					saDetail280.setRunningNumber(r.getRunningNumber());
					saDetail280.setPaymentMoney((280 * countMap.get("280")) + "");
					saDetail280.setType(PAYMENT_TYPE_CAB);
					saDetail280.setSourceCharge(id);
					saDetail280.setChargeItemId("cab2192");
					saDetail280.setRecordId(r.getId());
					saDetail280.setPaymentTime(r.getPaymentTime());
					saDetail280.setDeptId(relationShipEntity.getDept());
					saDetail280.setPaymentWay(r.getPaymentWay());
					saDetail280.setPayMemberCode(getPaymemberCodeByRecord(r));
					saDetail280.preInsert();
					saPaymentDetailTotalDao.insert(saDetail280);
				}
				if (countMap.get("380") > 0) {
					SaPaymentDetailTotal saDetail380 = new SaPaymentDetailTotal();
					saDetail380.setPaymentNum(countMap.get("380").toString());
					saDetail380.setMemberCode(r.getMemberCode());
					saDetail380.setChargeDate(r.getCreateDate());
					saDetail380.setPrice("380");
					saDetail380.setRunningNumber(r.getRunningNumber());
					saDetail380.setPaymentMoney((380 * countMap.get("380")) + "");
					saDetail380.setType(PAYMENT_TYPE_CAB);
					saDetail380.setSourceCharge(id);
					saDetail380.setChargeItemId("cab2191");
					saDetail380.setRecordId(r.getId());
					saDetail380.setPaymentTime(r.getPaymentTime());
					saDetail380.setDeptId(relationShipEntity.getDept());
					saDetail380.setPaymentWay(r.getPaymentWay());
					saDetail380.setPayMemberCode(getPaymemberCodeByRecord(r));
					saDetail380.preInsert();
					saPaymentDetailTotalDao.insert(saDetail380);
				}

			} else {
				SaPaymentDetailTotal saDetail = new SaPaymentDetailTotal();
				saDetail.setPaymentNum(r.getChargingItemNum().toString());
				saDetail.setMemberCode(r.getMemberCode());
				saDetail.setChargeDate(r.getCreateDate());
				saDetail.setPrice(r.getPrice());
				saDetail.setRunningNumber(r.getRunningNumber());
				saDetail.setPaymentMoney(r.getPaymentAmount());
				saDetail.setType(PAYMENT_TYPE_CAB);
				saDetail.setSourceCharge(id);
				saDetail.setChargeItemId("cab2193");
				saDetail.setRecordId(r.getId());
				saDetail.setPaymentTime(r.getPaymentTime());
				saDetail.setDeptId(relationShipEntity.getDept());
				saDetail.setPaymentWay(r.getPaymentWay());
				saDetail.setPayMemberCode(getPaymemberCodeByRecord(r));
				saDetail.preInsert();
				saPaymentDetailTotalDao.insert(saDetail);
			}
			;
		});
	}

	private boolean splitShow(int feeLong, int num, HashMap<String, Integer> countMap) {
		boolean boo = true;
		if (feeLong != 0 && feeLong % 280 == 0) {
			countMap.put("280", feeLong / 280);
		} else if (feeLong != 0 && feeLong % 380 == 0) {
			countMap.put("380", feeLong / 380);
		} else if (feeLong != 0 && feeLong % 660 == 0) {
			countMap.put("280", feeLong / 660);
			countMap.put("380", feeLong / 660);
		} else if (feeLong != 0 && feeLong % 940 == 0) {
			countMap.put("280", 2 * feeLong / 940);
			countMap.put("380", feeLong / 940);
		} else if (feeLong != 0 && feeLong % 1040 == 0) {
			countMap.put("280", feeLong / 1040);
			countMap.put("380", 2 * feeLong / 1040);
		} else if (feeLong != 0 && feeLong % 1220 == 0) {
			countMap.put("280", 3 * feeLong / 1220);
			countMap.put("380", feeLong / 1220);
		} else if (feeLong != 0 && feeLong % 1420 == 0) {
			countMap.put("280", feeLong / 1420);
			countMap.put("380", 3 * feeLong / 1420);
		} else if (feeLong != 0 && feeLong % 1500 == 0) {
			countMap.put("280", 4 * feeLong / 1500);
			countMap.put("380", feeLong / 1500);
		} else if (feeLong != 0 && feeLong % 1800 == 0) {
			countMap.put("280", feeLong / 1800);
			countMap.put("380", 4 * feeLong / 1800);
		} else {
			boo = false;
			System.out.println(
					"-------------------------------------------------------------------------------------------------------------");
		}
		return boo;
	}

	/**
	 * 获取付款人会员号
	 * 
	 * @param record
	 * @return
	 */
	public String getPaymemberCodeByRecord(PaymentRecord record) {
		String userId = record.getUpdateBy().getId();
		Member member = memberService.getByUserId(userId);
		if (member != null) {
			return member.getMemberCode();
		}
		Org org = orgDao.getOrgByUserId(userId);
		if (org != null) {
			return org.getMemberCode();
		}
		return record.getMemberCode();
	}

	/**
	 * 获取付款人会员号
	 * 
	 * @param orderCode
	 * @return
	 */
	public String getPayMemberCodeByOrder(String orderCode) {
		PaymentRecord record = paymentRecordDao.getByRunningNumber(orderCode);
		if(Objects.isNull(record)) {
			return "";
		}
		return getPaymemberCodeByRecord(record);
	}

}
