package com.cku.oa.dogsporting.service;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.sys.code.SysCodeUtil;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.common.utils.IdcardUtils;
import com.thinkgem.jeesite.common.utils.excel.ExportExcel;
import com.thinkgem.jeesite.common.utils.excel.ImportExcel;
import com.thinkgem.jeesite.modules.sys.utils.DictUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.cku.oa.dogsporting.entity.DogSportingMembers;
import com.cku.oa.dogsporting.dao.DogSportingMembersDao;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;

/**
 * 犬运动会员Service
 *
 * @author yuanshuai
 * @version 2021-06-15
 */
@Service
@Transactional(readOnly = true)
public class DogSportingMembersService extends CrudService<DogSportingMembersDao, DogSportingMembers> {

	public DogSportingMembers get(String id) {
		return super.get(id);
	}

	public List<DogSportingMembers> findList(DogSportingMembers dogSportingMembers) {
		return super.findList(dogSportingMembers);
	}

	public Page<DogSportingMembers> findPage(Page<DogSportingMembers> page, DogSportingMembers dogSportingMembers) {
		return super.findPage(page, dogSportingMembers);
	}

	@Transactional(readOnly = false)
	public void save(DogSportingMembers dogSportingMembers) {
		parameterValidate(dogSportingMembers);
		if (dogSportingMembers.getIsNewRecord()) {
			dogSportingMembers.setMemberCode(SysCodeUtil.getCDSCMemberCode());
		}
		super.save(dogSportingMembers);
	}

	@Transactional(readOnly = false)
	public void delete(DogSportingMembers dogSportingMembers) {
		super.delete(dogSportingMembers);
	}

	/**
	 * 参数校验
	 *
	 * @param dogSportingMembers
	 * @author yuanshuai
	 * @date 2021/6/21 14:06
	 */
	private void parameterValidate(DogSportingMembers dogSportingMembers) {
		//中文正则
		String regex = "[\\u4e00-\\u9fa5]+";
		if (dogSportingMembers.getName().length() > 10 || !dogSportingMembers.getName().matches(regex)) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：会员姓名需为中文，长度10个字符内");
		}
		if (dogSportingMembers.getMobile().length() != 11 || !dogSportingMembers.getMobile().matches("^1\\d{10}$")) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：电话号码需为11位数字");
		}
		if ("id_card".equals(dogSportingMembers.getCardType()) && !IdcardUtils.validateCard(dogSportingMembers.getCardNo())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：会员证件编号不符合身份证规范");
		}
		if(StringUtils.isBlank(dogSportingMembers.getCardNo())){
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：会员证件编号为空");
		}
		if (!StringUtils.isBlank(dogSportingMembers.getCardNo()) && checkCardNoByMemberCode(dogSportingMembers)) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：会员证件编号已存在");
		}
		if(StringUtils.isBlank(dogSportingMembers.getMemberType())){
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：会员类型不能为空");
		}
		if(StringUtils.isBlank(dogSportingMembers.getGender())){
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：会员性别不能为空");
		}
		if(StringUtils.isBlank(dogSportingMembers.getProvince())){
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：会员省份不能为空");
		}
		if(StringUtils.isBlank(dogSportingMembers.getCity())){
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：会员城市不能为空");
		}
		if(StringUtils.isBlank(dogSportingMembers.getArea())){
			throw new ZAException(ZAErrorCode.ZA_ERROR, "会员" + dogSportingMembers.getName() + "：会员区域不能为空");
		}
	}

	/**
	 * 证件编号查重
	 *
	 * @author yuanshuai
	 * @date 2021/6/21 15:30
	 */
	private boolean checkCardNoByMemberCode(DogSportingMembers dogSportingMembers) {
		return dao.getCardNoCountByMemberCode(dogSportingMembers.getCardNo(), dogSportingMembers.getMemberCode()) > 0;
	}


	/**
	 * 导入模板
	 *
	 * @author yuanshuai
	 * @date 2021/6/22 17:18
	 */
	public String importFileTemplate(HttpServletResponse response) throws IOException {
		List<DogSportingMembers> list = Lists.newArrayList();
		DogSportingMembers member = new DogSportingMembers();
		member.setName("姓名");
		member.setGender("男/女");
		member.setMobile("电话");
		member.setCardType("护照/身份证");
		member.setCardNo("110101xxxxxxxxxxxx");
		member.setMemberType("基础会员/专业会员");
		member.setCardNo("导入会员无实名校验");
		member.setProvince("xx省");
		member.setCity("xx市");
		member.setArea("xx区");
		member.setStreet("xxxxxx街道");
		member.setRemarks("备注");
		list.add(member);
		new ExportExcel("犬运动会员信息", DogSportingMembers.class, 2).setDataList(list).write(response, "犬运动会员数据导入模板.xlsx").dispose();
		return null;
	}

	/**
	 * 导入
	 *
	 * @author yuanshuai
	 * @date 2021/6/22 17:30
	 */
	@Transactional(readOnly = false)
	public void importMember(MultipartFile file) throws IOException, InvalidFormatException, IllegalAccessException, InstantiationException {
		//计数器
		AtomicInteger count = new AtomicInteger();
		//读取excel
		ImportExcel ei = new ImportExcel(file, 1, 0);
		List<DogSportingMembers> list = ei.getDataList(DogSportingMembers.class);
		list.forEach(member -> {
			if (StringUtils.isBlank(member.getName())) {
				return;
			}
			parameterValidate(member);
			if (!"基础会员".equals(member.getMemberType()) && !"专业会员".equals(member.getMemberType())) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "会员类型需为基础会员或专业会员");
			}
			if (!"男".equals(member.getGender()) && !"女".equals(member.getGender())) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "会员性别需为男或女");
			}
			if (!"护照".equals(member.getCardType()) && !"身份证".equals(member.getCardType())) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "会员证件类型需为护照或身份证");
			}
			member.setMemberType(DictUtils.getDictValue(member.getMemberType(), "dog_sporting_member_type", null));
			member.setGender(DictUtils.getDictValue(member.getGender(), "user_gender", null));
			member.setCardType(DictUtils.getDictValue(member.getCardType(), "user_card_type", null));
			save(member);
			count.getAndIncrement();
		});
		if (count.get() <= 0) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "导入失败，Excel中无数据");
		}
	}

	/**
	 * 根据会员号获取会员信息
	 *
	 * @author yuanshuai
	 * @date 2021/6/22 17:42
	 */
	public DogSportingMembers getByMemberCode(String memberCode) {
		if (!StringUtils.isBlank(memberCode)) {
			return dao.getByMemberCode(memberCode);
		} else {
			return null;
		}
	}

	/**
	 * 根据手机号获取会员信息
	 *
	 * @author yuanshuai
	 * @date 2021/6/23 14:44
	 */
	public DogSportingMembers getByMemberMobile(String mobile) {
		if (!StringUtils.isBlank(mobile)) {
			return dao.getByMemberMobile(mobile);
		} else {
			return null;
		}
	}

	/**
	 * 导出
	 *
	 * @author yuanshuai
	 * @date 2021/7/8 11:53
	 */
	public List<DogSportingMembers> export(DogSportingMembers dogSportingMembers) {
		List<DogSportingMembers> list = findList(dogSportingMembers);
		list.forEach(member -> {
			if (StringUtils.isBlank(member.getName())) {
				return;
			}
			member.setMemberType(DictUtils.getDictLabel(member.getMemberType(), "dog_sporting_member_type", null));
			member.setGender(DictUtils.getDictLabel(member.getGender(), "user_gender", null));
			member.setCardType(DictUtils.getDictLabel(member.getCardType(), "user_card_type", null));
		});
		return list;
	}
}