package com.cku.oa.report.web;

import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.collections.CollectionUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.kennel.entity.Kennel;
import com.cku.oa.report.entity.ReportMemberArea;
import com.cku.oa.report.entity.ReportRule;
import com.cku.oa.report.service.ReportMemberAreaService;
import com.cku.oa.report.vo.ReportMemberAreaExportVo;
import com.cku.oa.report.vo.ReportMemberAreaVo;
import com.cku.oa.report.vo.ReportMemberVO;
import com.cku.oa.statistics.service.ShowOfficeRegionService;
import com.cku.oa.statistics.vo.RegionVo;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.oa.sys.util.LocalDateUtils;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.utils.DateUtils;
import com.thinkgem.jeesite.common.utils.StringUtils;
import com.thinkgem.jeesite.common.utils.excel.ExportAttachedTitle;
import com.thinkgem.jeesite.common.utils.excel.ExportExcel;
import com.thinkgem.jeesite.common.web.BaseController;

import edu.emory.mathcs.backport.java.util.Arrays;

/**
 * 会员统计报表Controller
 * 
 * @author Sunny
 * @version 2022-04-15
 */
@Controller
@RequestMapping(value = "${adminPath}/report/reportMemberArea")
public class ReportMemberAreaController extends BaseController {

	@Autowired
	private ReportMemberAreaService reportMemberAreaService;

	@Autowired
	private MemberService memberService;

	@Autowired
	private ShowOfficeRegionService showOfficeRegionService;

	@ModelAttribute
	public ReportMemberArea get(@RequestParam(required = false) String id) {
		ReportMemberArea entity = null;
		if (StringUtils.isNotBlank(id)) {
			entity = reportMemberAreaService.get(id);
		}
		if (entity == null) {
			entity = new ReportMemberArea();
		}
		return entity;
	}

	@RequiresPermissions("report:reportMemberArea:view")
	@RequestMapping(value = { "list", "" })
	public String list(ReportMemberArea reportMemberArea, HttpServletRequest request, HttpServletResponse response,
			Model model) {
		if (StringUtils.isNotEmpty(reportMemberArea.getPeriodYear())
				&& StringUtils.isNotEmpty(reportMemberArea.getPeriodType())
				&& StringUtils.isNotEmpty(reportMemberArea.getPeriodNo())) {

			Page<ReportMemberArea> page = reportMemberAreaService
					.findPage(new Page<ReportMemberArea>(request, response), reportMemberArea);
			if (CollectionUtils.isNotEmpty(page.getList())) {
				List<ReportMemberArea> reportList = page.getList();
				reportList.forEach(report -> {
					ReportMemberAreaVo vo = new ReportMemberAreaVo();
					initReportVo(report.getValidGroup(), vo.getValidTotal());
					initReportVo(report.getAddedGroup(), vo.getAddedTotal());
					initReportVo(report.getLossGroup(), vo.getLossTotal());
					initReportVo(report.getRenewalGroup(), vo.getRenewalTotal());
					initReportVo(report.getKeepGroup(), vo.getKeepTotal());
					report.setReportVo(vo);
				});

				List<ReportMemberArea> totalList = reportMemberAreaService.findTotal(reportMemberArea.getOfficeName(),
						reportMemberArea.getPeriodYear(), reportMemberArea.getPeriodType(),
						reportMemberArea.getPeriodNo());
				ReportMemberAreaVo totalVo = new ReportMemberAreaVo();
				totalList.forEach(entity -> {
					this.initTotalReportVo(entity, totalVo);
				});

				totalVo.getValidTotal().put("合计",
						totalVo.getValidTotal().values().stream().reduce(BigDecimal.ZERO, BigDecimal::add));
				totalVo.getAddedTotal().put("合计",
						totalVo.getAddedTotal().values().stream().reduce(BigDecimal.ZERO, BigDecimal::add));
				totalVo.getRenewalTotal().put("合计",
						totalVo.getRenewalTotal().values().stream().reduce(BigDecimal.ZERO, BigDecimal::add));
				totalVo.getLossTotal().put("合计",
						totalVo.getLossTotal().values().stream().reduce(BigDecimal.ZERO, BigDecimal::add));
				totalVo.getKeepTotal().put("合计",
						totalVo.getKeepTotal().values().stream().reduce(BigDecimal.ZERO, BigDecimal::add));
				model.addAttribute("total", totalVo);
			}

			model.addAttribute("page", page);
		}
		return "oa/report/reportMemberAreaList";
	}

	private void initReportVo(String strGroup, Map<String, BigDecimal> voMap) {
		List<String> cardList = Arrays.asList(strGroup.split(","));
		cardList.forEach(c -> {
			List<String> cardVal = Arrays.asList(c.split(":"));
			voMap.put(cardVal.get(0), voMap.get(cardVal.get(0)).add(new BigDecimal(cardVal.get(1))));
		});
		voMap.put("合计", voMap.values().stream().reduce(BigDecimal.ZERO, BigDecimal::add));

	}

	private void initTotalReportVo(ReportMemberArea report, ReportMemberAreaVo totalVo) {
		totalVo.getValidTotal().put(report.getCardType(),
				totalVo.getValidTotal().get(report.getCardType()).add(report.getValidTotal()));
		totalVo.getAddedTotal().put(report.getCardType(),
				totalVo.getAddedTotal().get(report.getCardType()).add(report.getAddedTotal()));
		totalVo.getRenewalTotal().put(report.getCardType(),
				totalVo.getRenewalTotal().get(report.getCardType()).add(report.getRenewalTotal()));
		totalVo.getLossTotal().put(report.getCardType(),
				totalVo.getLossTotal().get(report.getCardType()).add(report.getLossTotal()));
		totalVo.getKeepTotal().put(report.getCardType(),
				totalVo.getKeepTotal().get(report.getCardType()).add(report.getKeepTotal()));
	}

	@RequiresPermissions("report:reportMemberArea:export")
	@RequestMapping(value = "export")
	public void export(ReportMemberArea reportMemberArea, HttpServletRequest request, HttpServletResponse response,
			Model model) throws Exception {
		if (StringUtils.isNotEmpty(reportMemberArea.getPeriodYear())
				&& StringUtils.isNotEmpty(reportMemberArea.getPeriodType())
				&& StringUtils.isNotEmpty(reportMemberArea.getPeriodNo())) {
			List<ReportMemberArea> reportList = reportMemberAreaService.findList(reportMemberArea);
			Map<String, List<ReportMemberAreaExportVo>> exportMap = new HashMap<>();
			String currentDate = DateUtils.getDate();
			if (CollectionUtils.isNotEmpty(reportList)) {
				LocalDate statisticsEndDate = LocalDate.of(Integer.valueOf(reportMemberArea.getPeriodYear()),
						Integer.valueOf(reportMemberArea.getPeriodType())
								* Integer.valueOf(reportMemberArea.getPeriodNo()),
						1);
				LocalDate statisticsStartDate = statisticsEndDate
						.plusMonths(1 - Integer.valueOf(reportMemberArea.getPeriodType()));
				Map<String, List<ReportMemberArea>> reportMap = reportList.stream()
						.collect(Collectors.groupingBy(ReportMemberArea::getOfficeName));
				reportMap.forEach((key, val) -> {
					List<ReportMemberAreaExportVo> exportList = new ArrayList<>();
					exportList.addAll(val.stream().map(r -> {
						ReportMemberAreaExportVo re = initReportExportVo(r);
						re.setExportDate(currentDate);
						re.setStatisticsStartDate(LocalDateUtils.formatLocalDateYM(statisticsStartDate));
						re.setStatisticsEndDate(LocalDateUtils.formatLocalDateYM(statisticsEndDate));
						return re;
					}).collect(Collectors.toList()));
					exportMap.put(key, exportList);
				});
			}
			AtomicInteger index = new AtomicInteger(0);
			ExportExcel excel = null;
			List<ExportAttachedTitle> attachedTitleList = new ArrayList<ExportAttachedTitle>() {
				private static final long serialVersionUID = 6353543934771984288L;
				{
					add(new ExportAttachedTitle("有效", 4, 4));
					add(new ExportAttachedTitle("新增", 8, 4));
					add(new ExportAttachedTitle("流失", 12, 4));
					add(new ExportAttachedTitle("续费", 16, 4));
					add(new ExportAttachedTitle("留存", 20, 4));
				}
			};
			for (Map.Entry<String, List<ReportMemberAreaExportVo>> entry : exportMap.entrySet()) {
				if (excel == null) {
					excel = new ExportExcel(entry.getKey(), ReportMemberAreaExportVo.class, 1, attachedTitleList);
					excel.setDataList(entry.getValue(), index.getAndAdd(1));
				} else {
					excel.addSheet(entry.getKey(), ReportMemberAreaExportVo.class, attachedTitleList);
					excel.setDataList(entry.getValue(), index.getAndAdd(1));
				}
			}

			ReportRule rule = reportMemberAreaService.findReportRule(reportMemberArea.getOfficeName(),
					reportMemberArea.getPeriodYear(), reportMemberArea.getPeriodType(), reportMemberArea.getPeriodNo());
			if (Objects.nonNull(rule)) {
				excel.addSheet("统计规则说明", ReportRule.class);
				excel.setDataList(Arrays.asList(new ReportRule[] { rule }), index.getAndAdd(1));
			}
			String fileName = "会员报表-" + currentDate + ".xlsx";
			excel.write(response, fileName).dispose();
		} else {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "未选中导出条件年度及周期");
		}
	}

	private ReportMemberAreaExportVo initReportExportVo(ReportMemberArea report) {
		ReportMemberAreaExportVo exportVo = new ReportMemberAreaExportVo();
		exportVo.setProvince(report.getProvince());
		// 有效
		List<String> cardList = Arrays.asList(report.getValidGroup().split(","));
		cardList.forEach(c -> {
			List<String> cardVal = Arrays.asList(c.split(":"));
			switch (cardVal.get(0)) {
			case "银卡":
				exportVo.setValidSilverCard(cardVal.get(1));
				break;
			case "金卡":
				exportVo.setValidGoldenCard(cardVal.get(1));
				break;
			case "铂金卡":
				exportVo.setValidPlatinumCard(cardVal.get(1));
				break;
			case "紫金卡":
				exportVo.setValidPurpleCard(cardVal.get(1));
				break;
			default:
				break;
			}
		});
		// 新增
		cardList = Arrays.asList(report.getAddedGroup().split(","));
		cardList.forEach(c -> {
			List<String> cardVal = Arrays.asList(c.split(":"));
			switch (cardVal.get(0)) {
			case "银卡":
				exportVo.setAddSilverCard(cardVal.get(1));
				break;
			case "金卡":
				exportVo.setAddGoldenCard(cardVal.get(1));
				break;
			case "铂金卡":
				exportVo.setAddPlatinumCard(cardVal.get(1));
				break;
			case "紫金卡":
				exportVo.setAddPurpleCard(cardVal.get(1));
				break;
			default:
				break;
			}
		});
		// 流失
		cardList = Arrays.asList(report.getLossGroup().split(","));
		cardList.forEach(c -> {
			List<String> cardVal = Arrays.asList(c.split(":"));
			switch (cardVal.get(0)) {
			case "银卡":
				exportVo.setLossSilverCard(cardVal.get(1));
				break;
			case "金卡":
				exportVo.setLossGoldenCard(cardVal.get(1));
				break;
			case "铂金卡":
				exportVo.setLossPlatinumCard(cardVal.get(1));
				break;
			case "紫金卡":
				exportVo.setLossPurpleCard(cardVal.get(1));
				break;
			default:
				break;
			}
		});
		// 续费
		cardList = Arrays.asList(report.getRenewalGroup().split(","));
		cardList.forEach(c -> {
			List<String> cardVal = Arrays.asList(c.split(":"));
			switch (cardVal.get(0)) {
			case "银卡":
				exportVo.setRenewalSilverCard(cardVal.get(1));
				break;
			case "金卡":
				exportVo.setRenewalGoldenCard(cardVal.get(1));
				break;
			case "铂金卡":
				exportVo.setRenewalPlatinumCard(cardVal.get(1));
				break;
			case "紫金卡":
				exportVo.setRenewalPurpleCard(cardVal.get(1));
				break;
			default:
				break;
			}
		});
		// 留存
		cardList = Arrays.asList(report.getKeepGroup().split(","));
		cardList.forEach(c -> {
			List<String> cardVal = Arrays.asList(c.split(":"));
			switch (cardVal.get(0)) {
			case "银卡":
				exportVo.setKeepSilverCard(cardVal.get(1));
				break;
			case "金卡":
				exportVo.setKeepGoldenCard(cardVal.get(1));
				break;
			case "铂金卡":
				exportVo.setKeepPlatinumCard(cardVal.get(1));
				break;
			case "紫金卡":
				exportVo.setKeepPurpleCard(cardVal.get(1));
				break;
			default:
				break;
			}
		});
		return exportVo;
	}

	@RequiresPermissions("report:memberList:view")
	@RequestMapping(value = { "memberList" })
	public String memberList(ReportMemberVO member, HttpServletRequest request, HttpServletResponse response,
			Model model) {
		member.setKey(
				StringUtils.isNotBlank(member.getKey()) ? member.getKey().replace("&middot;", "·") : member.getKey());
		Page<ReportMemberVO> page = new Page<>();
		// 确认该会员可以查询的省份
		List<RegionVo> list = showOfficeRegionService.findRegionList();
		List<String> provincelist = new ArrayList<>();
		if (list.size() > 0) {
			provincelist = list.stream().map(a -> a.getValue()).collect(Collectors.toList());
			member.setProvinces(provincelist);
			page = memberService.findReportMemberPage(new Page<ReportMemberVO>(request, response), member);

		}
		model.addAttribute("provincelist", provincelist);
		model.addAttribute("member", member);
		model.addAttribute("page", page);
		return "oa/report/reportMemberList";

	}

	/**
	 * 导出界面
	 * 
	 * @param model
	 * @return
	 */
	@RequiresPermissions("report:memberList:export")
	@RequestMapping(value = "exportForm")
	public String exportForm(ReportMemberVO member, Model model) {
		model.addAttribute("member", member);
		return "oa/report/exportForm";
	}

	private SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");

	@RequiresPermissions("report:memberList:export")
	@RequestMapping(value = "memberExport")
	public void memberExport(ReportMemberVO member, HttpServletRequest request, HttpServletResponse response,
			Model model) throws Exception {
		if (member.getMemberEndDateStart() == null || member.getMemberEndDateEnd() == null
				|| StringUtils.isBlank(member.getIsAdded())) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "请选择导出时间和导出类型！");
		}
		// 确认该会员可以查询的省份
		List<RegionVo> provincelist = showOfficeRegionService.findRegionList();
		if (provincelist.size() == 0) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "登录账号没有大区数据权限！");
		}
		List<String> provinces = provincelist.stream().map(a -> a.getValue()).collect(Collectors.toList());
		member.setProvinces(provinces);
		// 查询
		List<ReportMemberVO> list = memberService.findReportMemberListFExport(member);
		list.stream().forEach(a -> {
			// 完整地址
			if (StringUtils.isNotBlank(a.getRegisterStreet())
					&& a.getRegisterStreet().startsWith(a.getRegisterProvince())) {
				a.setCompleteAddress(a.getRegisterStreet());
			} else if (StringUtils.isNotBlank(a.getRegisterProvince())
					&& a.getRegisterProvince().equals(a.getRegisterCity())) {
				a.setCompleteAddress(a.getRegisterProvince() + a.getRegisterArea() + a.getRegisterStreet());
			} else {
				a.setCompleteAddress(
						a.getRegisterProvince() + a.getRegisterCity() + a.getRegisterArea() + a.getRegisterStreet());
			}
			a.setCompleteAddress(a.getCompleteAddress().replace("null", ""));
		});
		// 对结果进行大区分组
		Map<String, List<ReportMemberVO>> reportMap = list.stream()
				.collect(Collectors.groupingBy(ReportMemberVO::getOfficeName));
		ExportExcel excel = null;
		int i = 0;
		for (String key : reportMap.keySet()) {
			if (excel == null) {
				excel = new ExportExcel(key, ReportMemberVO.class, reportMap.keySet().size());
			} else {
				excel.addSheet(key, ReportMemberVO.class);
			}
			excel.setDataList(reportMap.get(key), i);
			i++;
		}

		String fileName = ("0".equals(member.getIsAdded()) ? "新增" : ("1".equals(member.getIsAdded()) ? "到期" : ""))
				+ "会员|" + sdf.format(member.getMemberEndDateStart()) + "-" + sdf.format(member.getMemberEndDateEnd())
				+ ".xlsx";
		excel.write(response, fileName).dispose();
	}

	@RequiresPermissions("report:memberList:export")
	@RequestMapping(value = "exportReportMember")
	public void exportReportMember(ReportMemberVO member, HttpServletRequest request, HttpServletResponse response,
			Model model) throws Exception {
		
		if (StringUtils.isNotBlank(member.getRegisterProvince())) {
			List<String> provinces = new ArrayList<>();
			provinces.add(member.getRegisterProvince());
			member.setProvinces(provinces);
		} else {
			// 确认该会员可以查询的省份
			List<RegionVo> provincelist = showOfficeRegionService.findRegionList();
			if (provincelist.size() == 0) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "登录账号没有大区数据权限！");
			}
			List<String> provinces = provincelist.stream().map(a -> a.getValue()).collect(Collectors.toList());
			member.setProvinces(provinces);
		}
		
		// 查询
		List<ReportMemberVO> list = memberService.findExportReportMemberList(member);
		list.stream().forEach(a -> {
			// 完整地址
			if (StringUtils.isNotBlank(a.getRegisterStreet())
					&& a.getRegisterStreet().startsWith(a.getRegisterProvince())) {
				a.setCompleteAddress(a.getRegisterStreet());
			} else if (StringUtils.isNotBlank(a.getRegisterProvince())
					&& a.getRegisterProvince().equals(a.getRegisterCity())) {
				a.setCompleteAddress(a.getRegisterProvince() + a.getRegisterArea() + a.getRegisterStreet());
			} else {
				a.setCompleteAddress(
						a.getRegisterProvince() + a.getRegisterCity() + a.getRegisterArea() + a.getRegisterStreet());
			}
			a.setCompleteAddress(a.getCompleteAddress().replace("null", ""));
			a.setExpires("0".equals(a.getIsExpires()) ? "无效" : "有效");
		});
		new ExportExcel("会员信息", ReportMemberVO.class, 1).setDataList(list).write(response, "会员信息表"+ LocalDate.now() +".xlsx").dispose();
	}

}