package com.cku.oa.order.service.impl;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.common.FinanceBusinessIdConstants;
import com.cku.oa.common.PaymentItemIdConstants;
import com.cku.oa.common.SKUIDConstants;
import com.cku.oa.dog.entity.DogBirthCertificate;
import com.cku.oa.dog.entity.DogChip;
import com.cku.oa.dog.enums.DnaParentsCompareApplySourceEnum;
import com.cku.oa.dog.service.DogBirthCertificateService;
import com.cku.oa.dog.service.DogChipService;
import com.cku.oa.order.service.BaseSkuBusinessService;
import com.cku.oa.order.service.IOrderBusinessService;
import com.cku.oa.order.vo.*;
import com.cku.oa.sys.entity.Org;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.OrgService;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.oa.sys.util.DateUtils;
import com.cku.oa.sys.util.MemberUtil;
import com.cku.restful.v1.dog.service.RestGoldCertifiedService;
import com.cku.restful.v1.dog.vo.request.GoldCertifiedApplyDetailRequest;
import com.cku.restful.v1.dog.vo.request.GoldCertifiedApplyRequest;
import com.cku.restful.v2.order.vo.request.OrderApplyRequest;
import com.google.common.collect.Lists;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 金质证书订单处理类
 *
 * @author yuanshuai
 * @date 2023/6/18 15:19
 */
@Service
public class GoldCertifiedOrderServiceImpl extends BaseSkuBusinessService<GoldCertifiedBusinessVo> implements IOrderBusinessService<GoldCertifiedApplyRequest> {

	@Autowired
	private RestGoldCertifiedService restGoldCertifiedService;
	@Autowired
	private MemberService memberService;
	@Autowired
	private DogChipService dogChipService;
	@Autowired
	private DogBirthCertificateService dogBirthCertificateService;
	@Autowired
	private OrgService orgService;


	@Override
	public void businessValidate(OrderApplyRequest orderRequest, GoldCertifiedApplyRequest businessRequest, Member member) {
		//参数校验
		validateParam(businessRequest);
		//获取金质证书详细信息
		GoldCertifiedApplyDetailRequest detailRequest = GoldCertifiedApplyDetailRequest.builder().build();
		if (DnaParentsCompareApplySourceEnum.PEDIGREE_CERTIFIED_UPGRADE.code().equals(businessRequest.getApplySource())) {
			//血统证书升级
			detailRequest.setPedigreeCertified(businessRequest.getPedigreeCertified());
		} else if (DnaParentsCompareApplySourceEnum.BIRTH_APPLY.code().equals(businessRequest.getApplySource())) {
			//出生纸申请
			detailRequest.setBirthCode(businessRequest.getBirthCode());
		}
		GoldCertifiedBusinessVo businessVo = restGoldCertifiedService.findAndValidateApplyDetailByDogCode(detailRequest, member, Boolean.TRUE);
		//业务校验
		validateBusiness(businessRequest, businessVo, member);
	}

	private void validateBusiness(GoldCertifiedApplyRequest businessRequest, GoldCertifiedBusinessVo businessVo, Member member) {
		// 当前账号会员有效
		if (!MemberUtil.isActiveMember(member)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "当前会员为无效会员，请进行续费后再申请");
		}
		if (0 == businessRequest.getDnaApplyChannel()) {
			if (StringUtils.isBlank(businessRequest.getHospitalCode())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "合作机构编号为空");
			}
			// 合作医院
			Org org = orgService.get(businessRequest.getHospitalCode());
			if (Objects.isNull(org)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "合作机构编号有误");
			}
		}
		// 新会员校验
		if (StringUtils.isNotBlank(businessRequest.getNewMemberCode())) {
			Member newMember = memberService.getByMemberCode(businessRequest.getNewMemberCode());
			if (Objects.isNull(newMember)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "新犬主会员号有误");
			}
			int cardType = UserUtils.getMemberCardType(newMember);
			if (0 == cardType) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "新犬主需为CKU爱宠银卡会员或以上会员，现新犬主为CKU爱宠普卡会员，请升级后继续办理该业务");
			}
			if (30 == cardType
					&& Objects.nonNull(newMember.getKennel())
					&& !newMember.getMemberEndDate().equals(newMember.getKennel().getEndTime())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "新犬主会员截至日期和犬舍截至日期不同步，需登录宠爱王国APP完成会员日期同步后方可继续办理该业务");
			}
			// 判断新犬主与申请人为同一站点会员
			if (!((StringUtils.isBlank(member.getMemberSite()) && StringUtils.isBlank(newMember.getMemberSite()))
					|| (member.getMemberSite() == null ? "" : member.getMemberSite()).equals(newMember.getMemberSite()))) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "新犬主与申请人会员站点不同，不能进行新生犬登记");
			}
		}
	}

	/**
	 * 请求参数校验
	 *
	 * @author yuanshuai
	 * @date 2023/6/18 15:48
	 */
	private void validateParam(GoldCertifiedApplyRequest request) {
		//参数校验
		if (1 != request.getApplySource() && 2 != request.getApplySource()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "申请来源只能为血统证书升级或者登记卡申请");
		}
		if (0 != request.getDnaApplyChannel() && 1 != request.getDnaApplyChannel()) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "DNA办理方式只能为机构办理或自检");
		}
		if (0 == request.getDnaApplyChannel() && StringUtils.isBlank(request.getHospitalCode())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "选择机构办理时，必须选择一个合作机构");
		}
		if (2 == request.getApplySource()) {
			if (Objects.isNull(request.getEcertFlag())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "电子证书标记位不能为空");
			} else if (0 != request.getEcertFlag() && 1 != request.getEcertFlag()) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "电子证书标记位只能为是或否");
			}
		}
	}

	@Override
	public List<OrderSkuVo> preOrder(OrderApplyRequest orderRequest, GoldCertifiedApplyRequest businessRequest, Member member) {
		//参数校验
		validateParam(businessRequest);
		//获取金质证书详细信息
		GoldCertifiedApplyDetailRequest detailRequest = GoldCertifiedApplyDetailRequest.builder().build();
		if (DnaParentsCompareApplySourceEnum.PEDIGREE_CERTIFIED_UPGRADE.code().equals(businessRequest.getApplySource())) {
			//血统证书升级
			detailRequest.setPedigreeCertified(businessRequest.getPedigreeCertified());
		} else if (DnaParentsCompareApplySourceEnum.BIRTH_APPLY.code().equals(businessRequest.getApplySource())) {
			//出生纸申请
			detailRequest.setBirthCode(businessRequest.getBirthCode());
		}
		//构建金质证书SKU业务参数
		GoldCertifiedBusinessVo businessVo = restGoldCertifiedService.findAndValidateApplyDetailByDogCode(detailRequest, member, Boolean.TRUE);
		//请求参数
		businessVo.setApplySource(businessRequest.getApplySource());
		businessVo.setNickname(businessRequest.getNickname());
		businessVo.setNewMemberCode(businessRequest.getNewMemberCode());
		businessVo.setEcertFlag(businessRequest.getEcertFlag());
		businessVo.setDnaApplyChannel(businessRequest.getDnaApplyChannel());
		businessVo.setHospitalCode(businessRequest.getHospitalCode());
		//业务校验
		validateBusiness(businessRequest, businessVo, member);
		//sku列表
		List<OrderSkuVo> skuVoList = Lists.newArrayList();
		//dna参数列表
		List<OrderBusinessVo> dnaArchiveBusinessVoList = Lists.newArrayList();
		//构建DNA存档SKU业务参数
		DnaArchiveBusinessVo dnaArchiveBusinessVo = DnaArchiveBusinessVo.builder()
				.dogId(businessVo.getDogId())
				.pedigreeCertified(businessVo.getPedigreeCertified())
				.birthCode(businessVo.getBirthCode())
				.overseasPedigreeCertified(businessVo.getOverseasPedigreeCertified())
				.applySource(businessRequest.getApplySource())
				.dnaApplyChannel(businessRequest.getDnaApplyChannel())
				.hospitalId(businessRequest.getHospitalCode())
				.build();
		//证书费
		if (DnaParentsCompareApplySourceEnum.BIRTH_APPLY.code().equals(businessVo.getApplySource())) {
			//出生纸申请时添加证书费
			//根据犬只年龄取不同的SKU
			if (DateUtils.isInXMounth(businessVo.getBirthdate(), 12)) {
				//添加证书费SKU
				skuVoList.add(OrderSkuVo.builder()
						.skuId(SKUIDConstants.GOLD_CERTIFIED_LT1)
						.num(1)
						.businessList(Stream.of(businessVo)
								.collect(Collectors.toList()))
						.build());
			} else if (DateUtils.isInXMounth(businessVo.getBirthdate(), 24)) {
				//添加证书费SKU
				skuVoList.add(OrderSkuVo.builder()
						.skuId(SKUIDConstants.GOLD_CERTIFIED_GT1)
						.num(1)
						.businessList(Stream.of(businessVo)
								.collect(Collectors.toList()))
						.build());
			} else {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "犬只超过两岁无法办理金质证书业务");
			}
		} else if (DnaParentsCompareApplySourceEnum.PEDIGREE_CERTIFIED_UPGRADE.code().equals(businessVo.getApplySource())) {
			//血统证书升级时添加证书补做费
			skuVoList.add(OrderSkuVo.builder()
					.skuId(SKUIDConstants.GOLD_CERTIFIED_REDO)
					.num(1)
					.businessList(Stream.of(businessVo)
							.collect(Collectors.toList()))
					.build());
		}
		//自身DNA存档费用
		if (0 == businessVo.getDna()) {
			if (DnaParentsCompareApplySourceEnum.BIRTH_APPLY.code().equals(businessVo.getApplySource())
					&& !DateUtils.isInXMounth(businessVo.getBirthdate(), 12)
					&& DateUtils.isInXMounth(businessVo.getBirthdate(), 24)) {
				//渠道为出生纸申请且犬只大于1岁小于两岁，DNA使用500SKU
				skuVoList.add(OrderSkuVo.builder()
						.skuId(SKUIDConstants.DNA_ARCHIVE_500)
						.num(1)
						.businessList(Stream.of(dnaArchiveBusinessVo)
								.collect(Collectors.toList()))
						.build());
			} else {
				//渠道为血统证书升级或出生纸申请且一岁以内犬只，DNA使用280SKU
				dnaArchiveBusinessVoList.add(dnaArchiveBusinessVo);
			}
		}
		//构建父犬DNA存档参数
		if (0 == businessVo.getMaleDna()) {
			dnaArchiveBusinessVoList.add(DnaArchiveBusinessVo.builder()
					.dogId(businessVo.getMaleDogId())
					.pedigreeCertified(businessVo.getMalePedigreeCertified())
					.birthCode(businessVo.getMaleBirthCode())
					.overseasPedigreeCertified(businessVo.getMaleOverseasPedigreeCertified())
					.applySource(businessRequest.getApplySource())
					.dnaApplyChannel(businessRequest.getDnaApplyChannel())
					.hospitalId(businessRequest.getHospitalCode())
					.build());
		}
		//构建母犬DNA存档参数
		if (0 == businessVo.getFemaleDna()) {
			dnaArchiveBusinessVoList.add(DnaArchiveBusinessVo.builder()
					.dogId(businessVo.getFemaleDogId())
					.pedigreeCertified(businessVo.getFemalePedigreeCertified())
					.birthCode(businessVo.getFemaleBirthCode())
					.overseasPedigreeCertified(businessVo.getFemaleOverseasPedigreeCertified())
					.applySource(businessRequest.getApplySource())
					.dnaApplyChannel(businessRequest.getDnaApplyChannel())
					.hospitalId(businessRequest.getHospitalCode())
					.build());
		}
		//添加DNA存档SKU
		if (dnaArchiveBusinessVoList.size() > 0) {
			skuVoList.add(OrderSkuVo.builder()
					.skuId(SKUIDConstants.DNA_ARCHIVE_280)
					.num(dnaArchiveBusinessVoList.size())
					.businessList(dnaArchiveBusinessVoList)
					.build());
		}
		//添加亲子比对SKU
		skuVoList.add(OrderSkuVo.builder()
				.skuId(SKUIDConstants.DNA_PARENTS_COMPARE)
				.num(1)
				.businessList(Stream.of(DnaParentsCompareBusinessVo.builder()
								.dogId(businessVo.getDogId())
								.pedigreeCertified(businessVo.getPedigreeCertified())
								.birthCode(businessVo.getBirthCode())
								.applySource(businessRequest.getApplySource())
								.dnaApplyChannel(businessRequest.getDnaApplyChannel())
								.hospitalId(businessRequest.getHospitalCode())
								.build())
						.collect(Collectors.toList()))
				.build());
		return skuVoList;
	}


	@Override
	@Transactional(readOnly = false)
	public void processBusinessPrePay(OrderApplyRequest orderRequest, GoldCertifiedBusinessVo businessData, OrderSkuVo skuVo) {
		if (DnaParentsCompareApplySourceEnum.BIRTH_APPLY.code().equals(businessData.getApplySource())) {
			//如果办理方式为出生纸申请则添加血统证书记录
			if (StringUtils.isBlank(businessData.getBirthCode())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "登记卡编号为空");
			}
			DogBirthCertificate dogBirth = dogBirthCertificateService.getByBirthCerRegCode(businessData.getBirthCode());
			if (Objects.isNull(dogBirth)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "没有查到相关犬只信息");
			}
			//新犬主
			Member newMember = null;
			if (StringUtils.isNotBlank(businessData.getNewMemberCode())) {
				newMember = memberService.getByMemberCode(businessData.getNewMemberCode());
			}
			//构建血统证书和更新出生纸
			DogChip dogChip = buildDogChip(businessData, dogBirth, skuVo.getOrderCode(), skuVo.getUnitPrice(), skuVo.getMember(), newMember);
			// 业务校验
			dogChipService.validateBusiness(dogChip, dogBirth);
			//更新出生纸信息
			dogBirthCertificateService.save(dogBirth);
			//保存血统证书信息
			dogChipService.save(dogChip);
		}
	}

	/**
	 * 构建血统证书记录并更新出生纸数据
	 *
	 * @author yuanshuai
	 * @date 2023/6/20 15:11
	 */
	private DogChip buildDogChip(GoldCertifiedBusinessVo businessData, DogBirthCertificate dogBirth, String orderCode, BigDecimal price, Member member, Member newMember) {
		DogChip dogChip = new DogChip();
		dogChip.setBirthCode(businessData.getBirthCode());
		dogChip.setPaymentMemberCode(member.getMemberCode());
		//dna办理方式为机构办理
		if (0 == businessData.getDnaApplyChannel()) {
			if (StringUtils.isBlank(businessData.getHospitalCode())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "合作机构编号为空");
			}
			// 合作医院
			Org org = orgService.get(businessData.getHospitalCode());
			if (Objects.isNull(org)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "合作机构编号有误");
			}
			dogChip.setHospitalName(org.getName());
			dogChip.setHospitalCode(org.getId());
			// 更新出生纸的临时会员信息及addUser字段
			dogBirth.setAddUser(org.getUser().getLoginName());
		}
		//如果有新犬主更新出生纸
		if (Objects.nonNull(newMember)) {
			dogBirth.setTempMemberCode(newMember.getMemberCode());
			dogBirth.setTempUserName(newMember.getName());
		}
		//订单号
		dogChip.setRunningNumber(orderCode);
		//证书费用
		dogChip.setPrice(price.toString());
		//呼名
		dogChip.setCallName(businessData.getNickname());
		//电子化标记
		dogChip.setEcertFlag(businessData.getEcertFlag().toString());
		//办理方式
		dogChip.setHaddleWay(businessData.getDnaApplyChannel() == 0 ? "1" : "0");
		//默认亲子比对
		dogChip.setIsDnaParentsCompare("1");
		//默认非整窝
		dogChip.setIsAll("0");
		//默认位点分析
		dogChip.setDnaSaveWay("1");
		//初始化默认值
		dogChip.setReviewState("0");
		dogChip.setPaymentState("1");
		dogChip.setProcessState("0");
		dogChip.setPrintState("0");
		dogChip.setStickerFlag("0");
		return dogChip;
	}

	@Override
	public void processChargingItemPrice(OrderApplyRequest orderRequest, List<GoldCertifiedBusinessVo> businessRequest, List<ChargingItemVo> itemList) {
		//获取业务参数
		GoldCertifiedBusinessVo businessVo = businessRequest.get(0);
		//查找出生纸是否有抵扣50的情况
		if (DnaParentsCompareApplySourceEnum.BIRTH_APPLY.code().equals(businessVo.getApplySource())) {
			if (StringUtils.isBlank(businessVo.getBirthCode())) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "登记卡编号为空");
			}
			DogBirthCertificate dogBirth = dogBirthCertificateService.getByBirthCerRegCode(businessVo.getBirthCode());
			if (Objects.isNull(dogBirth)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "没有查到相关犬只信息");
			}
			//犬只年龄小于6月龄且有抵扣标记则抵扣50元
			if (DateUtils.isInXMounth(dogBirth.getDogBirthday(), 6) && "1".equals(dogBirth.getDeduction())) {
				itemList.stream()
						.filter(i -> PaymentItemIdConstants.GOLD_CERTIFIED_LT1.equals(i.getId()))
						.findFirst()
						.ifPresent(item -> item.setPrice(item.getPrice().subtract(BigDecimal.valueOf(50))));
			}
		}
	}

}
