package com.cku.oa.groomer.service;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.constant.Constants;
import com.cku.oa.groomer.dao.GroomerQualificationCertificateChangeDao;
import com.cku.oa.groomer.entity.GroomerCertificateChangeRule;
import com.cku.oa.groomer.entity.GroomerQualificationCertificateChange;
import com.cku.oa.sys.service.user.MemberService;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 美容资格证书换发Service
 *
 * @author yuanshuai
 * @version 2024-02-18
 */
@Service
@Transactional(readOnly = true)
public class GroomerQualificationCertificateChangeService extends CrudService<GroomerQualificationCertificateChangeDao, GroomerQualificationCertificateChange> {

	@Autowired
	private GroomerCertificateChangeRuleService groomerCertificateChangeRuleService;
	@Autowired
	private GroomerQualificationCertificateService groomerQualificationCertificateService;
	@Autowired
	private MemberService memberService;

	public GroomerQualificationCertificateChange get(String id) {
		return super.get(id);
	}

	public List<GroomerQualificationCertificateChange> findList(GroomerQualificationCertificateChange groomerQualificationCertificateChange) {
		return super.findList(groomerQualificationCertificateChange);
	}

	public Page<GroomerQualificationCertificateChange> findPage(Page<GroomerQualificationCertificateChange> page, GroomerQualificationCertificateChange groomerQualificationCertificateChange) {
		return super.findPage(page, groomerQualificationCertificateChange);
	}

	/**
	 * 根据会员号查询
	 *
	 * @author yuanshuai
	 * @since 2024/2/19 16:26
	 */
	public List<GroomerQualificationCertificateChange> findListByMemberCode(String memberCode) {
		GroomerQualificationCertificateChange search = new GroomerQualificationCertificateChange();
		search.setMemberCode(memberCode);
		return super.findList(search);
	}

	@Transactional(readOnly = false)
	public void save(GroomerQualificationCertificateChange groomerQualificationCertificateChange) {
		super.save(groomerQualificationCertificateChange);
	}

	@Transactional(readOnly = false)
	public void delete(GroomerQualificationCertificateChange groomerQualificationCertificateChange) {
		super.delete(groomerQualificationCertificateChange);
	}

	@Transactional(readOnly = false)
	public void saveData(GroomerQualificationCertificateChange groomerQCC) {
		//查询换发后CKU证书等级
		GroomerCertificateChangeRule rule = groomerCertificateChangeRuleService.findRuleByCondition(1, groomerQCC.getCertificateSource(), groomerQCC.getCertificateLevel());
		if (Objects.isNull(rule)) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该证书来源和级别暂时无法换发CKU证书");
		}
		groomerQCC.setCkuCertificateLevel(rule.getCkuCertificateLevel());
		//业务校验
		validateBusiness(groomerQCC);
		//判断是否新增
		if (StringUtils.isBlank(groomerQCC.getId())) {
			//新增
			//初始化数据
			initData(groomerQCC);
		} else {
			//修改
			GroomerQualificationCertificateChange po = get(groomerQCC.getId());
			//处理审核
			if (!po.getReviewState().equals(groomerQCC.getReviewState())) {
				//修改审核状态
				Date now = new Date();
				groomerQCC.setReviewTime(now);
				if (!"1".equals(po.getReviewState()) && "1".equals(groomerQCC.getReviewState())) {
					//审核通过
					groomerQCC.setProccessState("1");
					groomerQCC.setProccessTime(now);
					//生成资格证书
					groomerQualificationCertificateService.generateQualificationCertificate(groomerQCC);
				}
			}
		}
		save(groomerQCC);
	}

	/**
	 * 业务校验
	 *
	 * @author yuanshuai
	 * @since 2024/2/19 16:19
	 */
	private void validateBusiness(GroomerQualificationCertificateChange groomerQCC) {
		//重复校验
		GroomerQualificationCertificateChange changeSearch = new GroomerQualificationCertificateChange();
		changeSearch.setMemberCode(groomerQCC.getMemberCode());
		changeSearch.setCertificateSource(groomerQCC.getCertificateSource());
		changeSearch.setCertificateLevel(groomerQCC.getCertificateLevel());
		List<GroomerQualificationCertificateChange> duplicateList = findList(changeSearch);
		int count;
		if (StringUtils.isNotBlank(groomerQCC.getId())) {
			count = (int) duplicateList.stream().filter(t -> !t.getId().equals(groomerQCC.getId())).count();
		} else {
			count = duplicateList.size();
		}
		if (count > Constants.NUM_ZERO) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该会员已经申请过相同国家及级别换发业务，请勿重复申请!");
		}
		//CKU证书重复校验
		if (groomerQualificationCertificateService.repeatValidateByMemberAndLevel(groomerQCC.getMemberCode(), groomerQCC.getCkuCertificateLevel())) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "该会员已有" + groomerQCC.getCkuCertificateLevel() + "级CKU美容资格证书，无法换发同样级别!");
		}
	}

	/**
	 * 初始化数据
	 *
	 * @author yuanshuai
	 * @since 2024/2/19 15:20
	 */
	private void initData(GroomerQualificationCertificateChange groomerQCC) {
		groomerQCC.setPaymentState("1");
		groomerQCC.setReviewState("0");
		groomerQCC.setProccessState("0");
	}
}